package net.wicp.tams.common.kubernetes.duckula;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ArrayUtils;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.FreemarkUtil;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.jdbc.JdbcAssit;
import net.wicp.tams.common.apiext.jdbc.JdbcConnection;
import net.wicp.tams.common.constant.DbType;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.kubernetes.apiserver.KubeClientTams;
import net.wicp.tams.common.kubernetes.beans.DucklaHelmInfo;
import net.wicp.tams.common.kubernetes.beans.HelmChartInfo;
import net.wicp.tams.common.kubernetes.helm.HelmClient;
import net.wicp.tams.common.kubernetes.helm.po.HelmInst;

/****
 * 安装duckula工具
 * 
 * @author Andy.zhou
 *
 */
@Slf4j
public class DuckulaInstall {

	private final Map<String, String> params;

	public Map<String, String> getParams() {
		return params;
	}

	private final String initInstallPre = "duckula-init";

	private final KubeClientTams tams;

	private final HelmClient helmClient;

	public KubeClientTams getTams() {
		return tams;
	}

	public HelmClient getHelmClient() {
		return helmClient;
	}

	private boolean isInitK8s = false;// 是否已初始化k8s?有些任务的前置条件

	public DuckulaInstall(File k8sConfig, Properties properties) {
		Conf.overProp(properties);
		log.info("============收到的配置项======");
		for (Object key : properties.keySet()) {
			log.info("===={}={}", key, properties.getProperty(String.valueOf(key)));
		}
		log.info("============00 配置信息======");
		this.params = ConfigItem.proParams();
		for (String key : this.params.keySet()) {// 最终参与的
			System.out.println("key=" + key + "   value=" + this.params.get(key));
		}
		if (k8sConfig == null) {
			this.tams = new KubeClientTams();// 拿容器里的权限
			// 由于helm必须要一个config文件，但容器中运行的环境有可能不存在config文件 ，所以需要下面逻辑
			File config = getConfig();
			this.helmClient = new HelmClient(config.getPath());// 默认就是这个名称空间
		} else {
			this.tams = new KubeClientTams(k8sConfig.getPath());// 传入的配置文件
			this.helmClient = new HelmClient(k8sConfig.getPath());
		}
	}

	public DuckulaInstall(File k8sConfig, String configConext) {
		this(k8sConfig, IOUtil.StringToProperties(configConext));
	}

	public DuckulaInstall(String configConext) {
		this(null, configConext);
	}

	public DuckulaInstall(File k8sConfig, File config) {
		this(k8sConfig, IOUtil.fileToProperties(config));
	}

	public DuckulaInstall(File config) {
		this(null, config);
	}

	/***
	 * 安装其它软件
	 * 
	 * @param ducklaHelmInfo
	 * @return
	 */
	public Result install(DucklaHelmInfo ducklaHelmInfo) {
		String[] checkParams = ducklaHelmInfo.getChecker().checkParams(this.params);
		if (ArrayUtils.isNotEmpty(checkParams)) {
			return Result.getError("下列参数不符合要求：" + CollectionUtil.arrayJoin(checkParams, ","));
		}
		if (ducklaHelmInfo.isNeedInitK8s() && !this.isInitK8s) {// 需要初始化k8s
			initK8s();
		}
		// 检查是否已经安装了此同名实例，为了避免冲突，需要加些前缀或后缀之类
		List<HelmInst> listTenant = this.helmClient.listInstance(ducklaHelmInfo.getHelmInstanceName(),
				ConfigItem.k8s_namespace.getValue());
		if (CollectionUtils.isEmpty(listTenant)) {
			File osPathForValue = new File(
					UploadSubDir.helmvalue.getOsPathDir(ducklaHelmInfo.getHelmInstanceName() + ".yaml"));// 最终的helm
																											// value值
			if (!osPathForValue.exists()) {
				String clustervalue = null;
				try {
					clustervalue = IOUtil.slurp(IOUtil.fileToInputStream(ducklaHelmInfo.getHelmValueTempPath(),
							ducklaHelmInfo.getHelmClassz()));
				} catch (IOException e) {
					e.printStackTrace();
				}
				clustervalue = FreemarkUtil.getInst().doProcessByTemp(clustervalue, this.params);
				log.info(clustervalue);
				DuckulaInstall.saveFile(osPathForValue.getPath(), clustervalue);
			}

			File targetFile = new File(UploadSubDir.helmchart.getOsPathDir(ducklaHelmInfo.getHelmChartFileName()));

			File file = DuckulaInstall.copyCharts(ducklaHelmInfo.getHelmClassz(), ducklaHelmInfo.getHelmChartPath(),
					targetFile);

			HelmChartInfo helmChartInfoForTenant = HelmChartInfo.parseChartTgz(file.getPath());
			Result result = this.helmClient.install(ducklaHelmInfo.getHelmInstanceName(), helmChartInfoForTenant,
					osPathForValue.getPath());
			log.info("============完成安装：" + ducklaHelmInfo.getHelmInstanceName() + "======");
			return result;
		} else {
			log.info("============跳过安装：" + ducklaHelmInfo.getHelmInstanceName() + "======");
			return Result.getSuc();
		}
	}

	/***
	 * 全新安装所有模块（存在的模块不安装）
	 * 
	 * @return
	 */
	public Result installNewDuckula() {
		UploadSubDir.clearDir();// 删除本地缓存，全新安装
		// 数据库连接测试
		Result initK8s = initK8s();
		if (!initK8s.isSuc()) {
			return initK8s;
		}
		Result installOps = installOps();
		if (!installOps.isSuc()) {
			return installOps;
		}
		Result initTenant = initTenant();
		return initTenant;
	}

	public Result initTenant() {
		checkConfig(ConfigItem.k8s_namespace, ConfigItem.k8s_secret_docker_enable, ConfigItem.duckula_deployId,
				ConfigItem.k8s_secret_ssl_enable, ConfigItem.docker_image_repository, ConfigItem.duckula_defaultdb,
				ConfigItem.db_port, ConfigItem.db_host, ConfigItem.db_username, ConfigItem.db_password,
				ConfigItem.duckula_agent_imageGroup, ConfigItem.duckula_agent_imageVersion,
				ConfigItem.duckula_agent_ingress_enable, ConfigItem.duckula_agent_ingressHost,
				ConfigItem.duckula_operator_imageGroup, ConfigItem.duckula_operator_imageVersion);
		List<HelmInst> listTenant = this.helmClient.listInstance("duckula-init-tenant",
				ConfigItem.k8s_namespace.getValue());
		if (CollectionUtils.isEmpty(listTenant)) {
			File initTenantHelmValue = getTenantInitHelmValue();
			File file = copyCharts("duckula-init-tenant-0.1.0.tgz");
			HelmChartInfo helmChartInfoForTenant = HelmChartInfo.parseChartTgz(file.getPath());
			Result result = helmClient.install("duckula-init-tenant", helmChartInfoForTenant,
					initTenantHelmValue.getPath());
			log.info("============15 执行租户初始化======");
			return result;
		} else {
			log.info("============15 跳过执行租户初始化======");
			return Result.getSuc();
		}
	}

	public Result installDolphinscheduler() {
		checkConfig(ConfigItem.docker_image_repository, ConfigItem.ds_images_imageGroup,
				ConfigItem.ds_images_imageVersion, ConfigItem.db_host, ConfigItem.db_port, ConfigItem.db_username,
				ConfigItem.db_password, ConfigItem.ds_defaultdb, ConfigItem.ds_ingressHost);
		List<HelmInst> listTenant = this.helmClient.listInstance("dolphinscheduler",
				ConfigItem.k8s_namespace.getValue());
		if (CollectionUtils.isEmpty(listTenant)) {
			File osPathForValue = new File(UploadSubDir.helmvalue.getOsPathDir("ds.yaml"));
			if (!osPathForValue.exists()) {
				String clustervalue = null;
				try {
					clustervalue = IOUtil.slurp(
							IOUtil.fileToInputStream("/duckula/values-dolphinscheduler.yaml", DuckulaInstall.class));
				} catch (IOException e) {
					e.printStackTrace();
				}
				clustervalue = FreemarkUtil.getInst().doProcessByTemp(clustervalue, params);
				log.info(clustervalue);
				saveFile(UploadSubDir.helmvalue.getOsPathDir("ds.yaml"), clustervalue);
			}

			File file = copyCharts("dolphinscheduler-2.0.3.tgz");
			HelmChartInfo helmChartInfoForTenant = HelmChartInfo.parseChartTgz(file.getPath());
			Result result = helmClient.install("dolphinscheduler", helmChartInfoForTenant, osPathForValue.getPath());
			log.info("============16 执行ds配置======");
			return result;
		} else {
			log.info("============16 跳过执行ds配置======");
			return Result.getSuc();
		}
	}

	public Result installOps() {
		checkConfig(ConfigItem.docker_image_repository, ConfigItem.duckula_ops_imageGroup,
				ConfigItem.duckula_ops_imageVersion, ConfigItem.k8s_secret_docker_enable, ConfigItem.duckula_ops_cpu,
				ConfigItem.duckula_ops_memory, ConfigItem.duckula_ops_ingressHost, ConfigItem.duckula_ops_ingressHost,
				ConfigItem.db_host, ConfigItem.db_port, ConfigItem.duckula_defaultdb, ConfigItem.db_username,
				ConfigItem.db_password, ConfigItem.duckula_ops_ingressHost, ConfigItem.duckula_deployId,
				ConfigItem.k8s_namespace);
		Connection mysqlconn = initConn();
		log.info("============01 数据库连接检查通过======");
		List<HelmInst> listInstance = this.helmClient.listInstance("duckula-ops", ConfigItem.k8s_namespace.getValue());
		if (CollectionUtils.isEmpty(listInstance)) {
			try {
				initOpsDb(mysqlconn);// 先初始化数据库
			} finally {
				JdbcAssit.close(mysqlconn);
			}
			File opsHelmValue = getOpsHelmValue();
			File copyCharts = copyCharts("duckula-ops-0.1.0.tgz");
			HelmChartInfo helmChartInfoForOps = HelmChartInfo.parseChartTgz(copyCharts.getPath());
			Result result = helmClient.install("duckula-ops", helmChartInfoForOps, opsHelmValue.getPath());
			log.info("============14 执行ops配置======");
			return result;
		} else {
			log.info("============14 跳过执行ops配置======");
			return Result.getSuc();
		}
	}

	public Connection initConn() {
		// 测试连通性，特别是没有添加白明单情况。
		boolean testConn = DbType.mysql.testConn(ConfigItem.db_host.getValue(),
				Integer.parseInt(ConfigItem.db_port.getValue()), ConfigItem.db_username.getValue(),
				ConfigItem.db_password.getValue());
		if (!testConn) {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_conn_fail, "请确认数据库的连通，包括检查白明单，以mysql client能进行连接为准。");
		}
		String url = DbType.mysql.geturl(ConfigItem.db_host.getValue(), Integer.parseInt(ConfigItem.db_port.getValue()),
				ConfigItem.duckula_defaultdb.getValue());
		Connection mysqlconn = JdbcConnection.getConnectionMyql(url, ConfigItem.db_username.getValue(),
				ConfigItem.db_password.getValue());
		if (mysqlconn == null) {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_conn_fail, "连接失败，请确认地址：" + url);
		}
		return mysqlconn;
	}

	public Result initK8s() {
		// 不能用公共方法，防死循环
		List<String> errorItems = new ArrayList<>();
		for (ConfigItem configItem : ConfigItem.findConfigItem("k8s_")) {
			if (configItem.isNotNull() && StringUtil.isNull(this.params.get(configItem.name()))) {
				errorItems.add(configItem.getItemName());
			}
		}
		if (errorItems.size() > 0) {
			throw new ProjectExceptionRuntime(ExceptAll.Project_default,
					"下列参数检查不通过，不能为空：" + CollectionUtil.listJoin(errorItems, ","));
		}
		boolean hasCluster = false;
		List<String> initNamespaces = new ArrayList<String>();// 已经初始化过的名称空间
		List<HelmInst> listInstanceGloble = this.helmClient.listInstanceGloble(initInstallPre + "-");
		for (HelmInst helmInst : listInstanceGloble) {
			hasCluster = hasCluster || (helmInst.getName().equals(initInstallPre + "-cluster")
					|| helmInst.getName().equals(initInstallPre + "-all"));
			if (helmInst.getName().equals(initInstallPre + "-namespace")
					|| helmInst.getName().equals(initInstallPre + "-all")) {
				initNamespaces.add(helmInst.getNamespace());
			}
		}
		if (!hasCluster) {// 集群初始化，在配置的第一个名称空间里执行
			File copyCharts = copyCharts("duckula-init-k8s-0.1.0.tgz");
			HelmChartInfo helmChartInfoForK8s = HelmChartInfo.parseChartTgz(copyCharts.getPath());
			Map<String, String> paramsmap = CollectionUtil
					.newMapStr(new String[] { "scope", "cluster", "namespace", ConfigItem.k8s_namespace.getValue() });
			File clusterInitHelmValue = getClusterInitHelmValue();
			Result result = this.helmClient.install(initInstallPre + "-cluster", ConfigItem.k8s_namespace.getValue(),
					helmChartInfoForK8s, clusterInitHelmValue.getPath(), paramsmap);
			log.info("============12 执行cluster配置======");
			if (!result.isSuc()) {
				this.isInitK8s = false;
				return result;
				// throw new ProjectExceptionRuntime(ExceptAll.Project_default, "安装集群资源失败,原因：" +
				// result.getMessage());
			}
		} else {
			log.info("============12 跳过执行cluster配置======");
		}
		// 租户名称空间初始化，
		if (!initNamespaces.contains(ConfigItem.k8s_namespace.getValue())) {// 已经初始化过
			File copyCharts = copyCharts("duckula-init-k8s-0.1.0.tgz");
			HelmChartInfo helmChartInfoForTeant = HelmChartInfo.parseChartTgz(copyCharts.getPath());
			// ServiceAccount不需要创建，因为在做config时已创建了
			Map<String, String> paramsmap = CollectionUtil.newMapStr(new String[] { "scope", "namespace", "namespace",
					ConfigItem.k8s_namespace.getValue(), "needsa", "no" });
			File clusterInitHelmValue = getClusterInitHelmValue();
			Result result = this.helmClient.install(initInstallPre + "-namespace", ConfigItem.k8s_namespace.getValue(),
					helmChartInfoForTeant, clusterInitHelmValue.getPath(), paramsmap);
			log.info("============13 执行namespace配置======");
			if (!result.isSuc()) {
				// throw new ProjectExceptionRuntime(ExceptAll.Project_default, "安装名称空间资源失败,原因："
				// + result.getMessage());
				this.isInitK8s = false;
				return result;
			}
		} else {
			log.info("============13 跳过执行namespace配置======");
		}
		this.isInitK8s = true;
		return Result.getSuc();
	}

	public File getClusterInitHelmValue() {
		File osPathForValue = new File(UploadSubDir.helmvalue.getOsPathDir("cluster.yaml"));
		if (osPathForValue.exists()) {
			return osPathForValue;
		}
		String clustervalue = null;
		try {
			clustervalue = IOUtil
					.slurp(IOUtil.fileToInputStream("/duckula/values-duckula-init-k8s.yaml", DuckulaInstall.class));
		} catch (Exception e) {
			log.error("读values-duckula-init-k8s.yaml失败", e);
			return null;
		}
		clustervalue = FreemarkUtil.getInst().doProcessByTemp(clustervalue, this.params);
		log.info(clustervalue);
		saveFile(UploadSubDir.helmvalue.getOsPathDir("cluster.yaml"), clustervalue);
		log.info("============02 生成cluster的配置======");
		return osPathForValue;
	}

	public File getTenantInitHelmValue() {
		File osPathForValue = new File(UploadSubDir.helmvalue.getOsPathDir("tenant.yaml"));
		if (osPathForValue.exists()) {
			return osPathForValue;
		}
		String tenantvalue = null;
		try {
			tenantvalue = IOUtil
					.slurp(IOUtil.fileToInputStream("/duckula/values-duckula-init-tenant.yaml", DuckulaInstall.class));
		} catch (Exception e) {
			log.error("读values-duckula-init-tenant.yaml失败", e);
			return null;
		}
		tenantvalue = FreemarkUtil.getInst().doProcessByTemp(tenantvalue, this.params);
		log.info(tenantvalue);
		saveFile(UploadSubDir.helmvalue.getOsPathDir("tenant.yaml"), tenantvalue);
		log.info("============03 生成namespace的配置======");
		return osPathForValue;
	}

	public File getOpsHelmValue() {
		File osPathForValue = new File(UploadSubDir.helmvalue.getOsPathDir("ops.yaml"));
		if (osPathForValue.exists()) {
			return osPathForValue;
		}
		String opsvalue = null;
		try {
			opsvalue = IOUtil
					.slurp(IOUtil.fileToInputStream("/duckula/values-duckula3-ops.yaml", DuckulaInstall.class));
		} catch (Exception e) {
			log.error("读values-duckula3-ops.yaml失败", e);
			return null;
		}
		opsvalue = FreemarkUtil.getInst().doProcessByTemp(opsvalue, this.params);
		log.info(opsvalue);
		saveFile(UploadSubDir.helmvalue.getOsPathDir("ops.yaml"), opsvalue);
		log.info("============04 生成ops的配置======");
		return osPathForValue;
	}

	private File getConfig() {
		File file = new File(UploadSubDir.k8sconfig.getOsPathDir("config"));
		if (file.exists()) {
			return file;
		}

		boolean exitNamespace = this.tams.exitNamespace(ConfigItem.k8s_namespace.getValue());
		if (!exitNamespace) {
			Result namespace = tams.createNamespace(ConfigItem.k8s_namespace.getValue());
			if (!namespace.isSuc()) {
				throw new ProjectExceptionRuntime(ExceptAll.Project_default,
						"创建名称空间" + ConfigItem.k8s_namespace.getValue() + "失败");
			}
		}
		// 集群名称写死为"duckula"
		Result proKubeConfig = this.tams.proKubeConfig("duckula", ConfigItem.k8s_namespace.getValue(), initInstallPre,
				true);// 租户code就是k8s的帐号
		if (proKubeConfig.isSuc()) {
			try {
				FileUtils.write(file, String.valueOf(proKubeConfig.retObjs(0)));
			} catch (IOException e) {
				throw new ProjectExceptionRuntime(ExceptAll.Project_default, "不能创建集群的config", e);
			}
		}
		return file;
	}

	private void checkConfig(ConfigItem... checkItems) {
		List<String> errorItems = new ArrayList<>();
		for (ConfigItem configItem : checkItems) {
			if (configItem.isNotNull() && StringUtil.isNull(this.params.get(configItem.name()))) {
				errorItems.add(configItem.getItemName());
			}
		}
		if (errorItems.size() > 0) {
			throw new ProjectExceptionRuntime(ExceptAll.Project_default,
					"下列参数检查不通过，不能为空：" + CollectionUtil.listJoin(errorItems, ","));
		}
		if (!this.isInitK8s) {
			initK8s();
		}
	}

	private static <T> File copyCharts(Class<T> classz, String oriPath, File targetFile) {

		if (!targetFile.exists()) {
			// String path = IOUtil.mergeFolderAndFilePath("/duckula/charts", fileName);
			try {
				FileUtils.copyInputStreamToFile(IOUtil.fileToInputStream(oriPath, classz), targetFile);
			} catch (IOException e) {
				throw new ProjectExceptionRuntime(ExceptAll.Project_default, "不存在此文件:" + targetFile.getPath(), e);
			}
		}
		return targetFile;
	}

	private static <T> File copyCharts(String fileName) {
		File file = new File(UploadSubDir.helmchart.getOsPathDir(fileName));
		String path = IOUtil.mergeFolderAndFilePath("/duckula/charts", fileName);
		return copyCharts(DuckulaInstall.class, path, file);
	}

	private static void saveFile(String path, String context) {
		try {
			File file = new File(path);
			if (file.exists()) {
				FileUtils.forceDelete(file);
			}
			FileUtils.write(file, context);
		} catch (IOException e) {
			throw new ProjectExceptionRuntime(ExceptAll.Project_default, "写文件【" + path + "】失败", e);
		}
	}

	private static void initOpsDb(Connection mysqlconn) {
		try {
			// 初始化
			JdbcAssit.exeSQLScript(mysqlconn, "/duckula/duckula-init.sql", DuckulaInstall.class, false, true);
			int indexOf = ConfigItem.duckula_flink_ingressHost.getValue().indexOf(".");
			String ingressFlinkSubdomain = ConfigItem.duckula_flink_ingressHost.getValue().substring(0, indexOf);
			String ingressSuffix = ConfigItem.duckula_flink_ingressHost.getValue().substring(indexOf);
			String configContext = "";// config的内容。
			mysqlconn.setAutoCommit(false);
			// 部署信息
			String deployInsertSql = "REPLACE INTO  `common_deploy` VALUES (1344624860489924610, 'bigdata-idc', 'k8s', 'idc', '"
					+ ConfigItem.k8s_namespace.getValue() + "', '', '', '"
					+ ConfigItem.k8s_secret_docker_configjson.getValue() + "', '"
					+ ConfigItem.docker_image_repository.getValue() + "', '', '', '" + ingressSuffix + "', '"
					+ ingressFlinkSubdomain + "', '" + configContext
					+ "', '/data/duckula-data/upload/k8sconfig/cluster/1344624860489924610', NULL, '', 22, '', '', '"
					+ ConfigItem.k8s_nas_storageClass.getValue()
					+ "', 'yes', 64, '', '', NULL, '私有化部署', now(), now(), 1, 1)";
			Result execSql = JdbcAssit.execSql(mysqlconn, deployInsertSql);

			if (!execSql.isSuc() && !execSql.getMessage().contains("Duplicate entry")) {// 主键重复，不管
				throw new ProjectExceptionRuntime(ExceptAll.Project_default, "执行SQL失败：" + execSql.getMessage());
			}
			// 租户信息
			String agentUrl = StringUtil.isNull(ConfigItem.duckula_agent_ingressHost.getValue()) ? ""
					: String.format("%s://%s",
							"true".equals(ConfigItem.k8s_ingress_ssl_enable.getValue()) ? "https" : "http",
							ConfigItem.duckula_agent_ingressHost.getValue());// agent地址
			Result execTenantSql = JdbcAssit.execSql(mysqlconn,
					"REPLACE INTO `sys_tenant` VALUES (0, 'common', '公共资源', 'admin', 'yes', '2021-06-27 09:39:16', 'test', 1344624860489924610, '"
							+ ConfigItem.k8s_namespace.getValue() + "', NULL, '" + agentUrl + "', '', '', '')");
			if (!execTenantSql.isSuc() && !execTenantSql.getMessage().contains("Duplicate entry")) {// 主键重复，不管
				throw new ProjectExceptionRuntime(ExceptAll.Project_default, "执行SQL失败：" + execTenantSql.getMessage());
			}
			// 版本信息
			Result agentSql = JdbcAssit.execSql(mysqlconn,
					"REPLACE INTO `common_version` VALUES (6, 'duckula租户agent', NULL, 'rjzjh/duckulaagent', 'duckula-agent', '"
							+ ConfigItem.duckula_agent_imageVersion.getValue()
							+ "', '', 'duckula', 'duckulaagent', 'yes', '2022-05-28 21:16:39', 'yes', '"
							+ ConfigItem.duckula_agent_imageVersion.getValue() + "', NULL, 'yes', '用户上传文件等管理')");
			if (!agentSql.isSuc() && !agentSql.getMessage().contains("Duplicate entry")) {// 主键重复，不管
				throw new ProjectExceptionRuntime(ExceptAll.Project_default, "执行SQL失败：" + agentSql.getMessage());
			}

			Result operatorSql = JdbcAssit.execSql(mysqlconn,
					"REPLACE INTO `common_version` VALUES (7, 'duckula租户的operator', NULL, 'rjzjh/duckulaoperator', 'operator-flink', '"
							+ ConfigItem.duckula_operator_imageVersion.getValue()
							+ "', '', 'duckula', 'duckulaoperator', 'yes', '2023-06-05 18:38:24', 'yes', '"
							+ ConfigItem.duckula_operator_imageVersion.getValue() + "', '', 'yes', '租户的k8s管理')");
			if (!operatorSql.isSuc() && !operatorSql.getMessage().contains("Duplicate entry")) {// 主键重复，不管
				throw new ProjectExceptionRuntime(ExceptAll.Project_default, "执行SQL失败：" + operatorSql.getMessage());
			}

			Result opsSql = JdbcAssit.execSql(mysqlconn,
					"REPLACE INTO `common_version` VALUES (12, 'ops控制台', NULL, 'rjzjh/duckulaops', 'duckula-ops', '"
							+ ConfigItem.duckula_ops_imageVersion.getValue()
							+ "', '', 'duckula', 'duckulaops', 'yes', '2023-06-27 18:14:04', 'yes', '"
							+ ConfigItem.duckula_ops_imageVersion.getValue() + "', '', 'yes', 'ops控制台')");
			if (!opsSql.isSuc() && !opsSql.getMessage().contains("Duplicate entry")) {// 主键重复，不管
				throw new ProjectExceptionRuntime(ExceptAll.Project_default, "执行SQL失败：" + opsSql.getMessage());
			}

			mysqlconn.commit();
			log.info("============14.0 数据库初始化======");
		} catch (Throwable e) {
			try {
				mysqlconn.rollback();
			} catch (SQLException e1) {
			}
			log.info("============14.0 数据库初始化失败，请手动执行======");
		}
	}

}
