package net.wicp.tams.common.kubernetes.apiserver;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import io.fabric8.kubernetes.api.model.Container;
import io.fabric8.kubernetes.api.model.ContainerPort;
import io.fabric8.kubernetes.api.model.GenericKubernetesResource;
import io.fabric8.kubernetes.api.model.GenericKubernetesResourceList;
import io.fabric8.kubernetes.api.model.HasMetadata;
import io.fabric8.kubernetes.api.model.IntOrString;
import io.fabric8.kubernetes.api.model.Namespace;
import io.fabric8.kubernetes.api.model.NamespaceBuilder;
import io.fabric8.kubernetes.api.model.ObjectMeta;
import io.fabric8.kubernetes.api.model.OwnerReference;
import io.fabric8.kubernetes.api.model.Pod;
import io.fabric8.kubernetes.api.model.PodList;
import io.fabric8.kubernetes.api.model.PodTemplateSpec;
import io.fabric8.kubernetes.api.model.Quantity;
import io.fabric8.kubernetes.api.model.ReplicationController;
import io.fabric8.kubernetes.api.model.ReplicationControllerSpec;
import io.fabric8.kubernetes.api.model.ResourceQuota;
import io.fabric8.kubernetes.api.model.ResourceQuotaBuilder;
import io.fabric8.kubernetes.api.model.Secret;
import io.fabric8.kubernetes.api.model.SecretList;
import io.fabric8.kubernetes.api.model.Service;
import io.fabric8.kubernetes.api.model.ServiceAccount;
import io.fabric8.kubernetes.api.model.ServiceBuilder;
import io.fabric8.kubernetes.api.model.ServiceList;
import io.fabric8.kubernetes.api.model.ServicePort;
import io.fabric8.kubernetes.api.model.ServiceSpec;
import io.fabric8.kubernetes.api.model.apps.Deployment;
import io.fabric8.kubernetes.api.model.apps.DeploymentList;
import io.fabric8.kubernetes.api.model.batch.v1.Job;
import io.fabric8.kubernetes.api.model.batch.v1.JobList;
import io.fabric8.kubernetes.client.ConfigBuilder;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientBuilder;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.dsl.FilterNested;
import io.fabric8.kubernetes.client.dsl.FilterWatchListDeletable;
import io.fabric8.kubernetes.client.dsl.NonNamespaceOperation;
import io.fabric8.kubernetes.client.dsl.PodResource;
import io.fabric8.kubernetes.client.dsl.Resource;
import io.fabric8.kubernetes.client.dsl.RollableScalableResource;
import io.fabric8.kubernetes.client.dsl.ScalableResource;
import io.fabric8.kubernetes.client.dsl.ServiceResource;
import io.fabric8.kubernetes.client.dsl.base.CustomResourceDefinitionContext;
import io.kubernetes.client.Exec;
import io.kubernetes.client.PodLogs;
import io.kubernetes.client.openapi.ApiClient;
import io.kubernetes.client.openapi.ApiException;
import io.kubernetes.client.openapi.Configuration;
import io.kubernetes.client.openapi.apis.AppsV1Api;
import io.kubernetes.client.openapi.apis.RbacAuthorizationV1Api;
import io.kubernetes.client.openapi.models.V1ClusterRoleBinding;
import io.kubernetes.client.openapi.models.V1Deployment;
import io.kubernetes.client.openapi.models.V1DeploymentList;
import io.kubernetes.client.openapi.models.V1DeploymentSpec;
import io.kubernetes.client.openapi.models.V1ObjectMeta;
import io.kubernetes.client.openapi.models.V1RoleBinding;
import io.kubernetes.client.openapi.models.V1RoleRef;
import io.kubernetes.client.openapi.models.V1Subject;
import io.kubernetes.client.util.Yaml;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.FreemarkUtil;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.NumberUtil;
import net.wicp.tams.common.apiext.PwdUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.yaml.YamlAssist;
import net.wicp.tams.common.beans.StreamCopyBean;
import net.wicp.tams.common.callback.IProcess;
import net.wicp.tams.common.constant.dic.YesOrNo;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.http.HttpClient;
import net.wicp.tams.common.http.HttpResult;
import net.wicp.tams.common.kubernetes.K8sAssit;
import net.wicp.tams.common.kubernetes.apiserver.crd.ICrdDef;
import net.wicp.tams.common.kubernetes.beans.PodPhase;

/***
 * 
 * @author Andy.zhou
 *
 */
@Slf4j
public class KubeClientTams {
	private final KubernetesClient client;
	private final ApiClient apiClient;

	private void apiClientInit(ApiClient apiClient) {
		Configuration.setDefaultApiClient(apiClient);
		this.apiClient.setReadTimeout(300000);// 5分钟
	}

	public KubernetesClient getClient() {
		return client;
	}

	/***
	 * 通过Conf的配置得到client
	 */
	public KubeClientTams() {
		String incluster = Conf.get("common.kubernetes.incluster");
		log.info("是否在容器中运行:{}", incluster);
		if (!YesOrNo.yes.name().equals(incluster)) {// 不在集群中，需要检查有没有k8s的config文件。
			String k8sConfig = IOUtil.mergeFolderAndFilePath(System.getProperty("user.home"), "/.kube/config");
			if (!new File(k8sConfig).exists()) {
				throw new ProjectExceptionRuntime(ExceptAll.Project_default, "没有在k8s容器中运行，需要有k8s的认证文件：~/.kube/config");
			}
		}
		this.client = initClient();
		this.apiClient = ApiClientManager.getApiClient();
		apiClientInit(apiClient);
	}

	/**
	 * 通过配置文件的内容和context得到client
	 * 
	 * @param kubeconfigstr kubeconfig内容
	 * @param context       启用哪个context,传空则为当前context
	 */
	public KubeClientTams(String kubeconfigstr, String context) {
		Validate.notEmpty(kubeconfigstr, "k8s的认证文件不能为空");
		this.client = getClientForKube(kubeconfigstr, context);
		this.apiClient = new ApiClientCreator(kubeconfigstr, context).createObject();
		apiClientInit(apiClient);
	}

	/***
	 * 配置文件路径得到client
	 * 
	 * @param pathTypePath 配置文件
	 */
	public KubeClientTams(String pathTypePath) {
		this(IOUtil.slurpPathTypePath(pathTypePath), null);
	}

	public KubeClientTams(KubernetesClient client, ApiClient apiClient) {
		this.client = client;
		this.apiClient = apiClient;
		apiClientInit(apiClient);
	}

	private KubernetesClient initClient() {
		KubernetesClient client = null;
		if (StringUtil.isNotNull(Conf.get("common.kubernetes.kubeconfig.path"))) {
			String kubeconfigstr = IOUtil.slurp(Conf.get("common.kubernetes.kubeconfig.path"));
			String context = IOUtil.slurp(Conf.get("common.kubernetes.kubeconfig.context"));
			client = getClientForKube(kubeconfigstr, context);
		} else if (StringUtil.isNotNull(Conf.get("common.kubernetes.apiserver.master.url"))) {
			ConfigBuilder config = new ConfigBuilder()
					.withMasterUrl(Conf.get("common.kubernetes.apiserver.master.url"));
			if (StringUtil.isNotNull(Conf.get("common.kubernetes.apiserver.master.username"))) {
				config.withUsername(Conf.get("common.kubernetes.apiserver.master.username"));
				config.withPassword(Conf.get("common.kubernetes.apiserver.master.password"));
			}
			config.withTrustCerts(true);
			client = new KubernetesClientBuilder().withConfig(config.build()).build();

		} else {
			client = new KubernetesClientBuilder().build();// 默认读./kube信息，支持在容器内执行。
		}
		return client;
	}

	private KubernetesClient getClientForKube(String kubeconfigstr, String context) {
		try {
			io.fabric8.kubernetes.client.Config config = StringUtil.isNull(context)
					? io.fabric8.kubernetes.client.Config.fromKubeconfig(kubeconfigstr)
					: io.fabric8.kubernetes.client.Config.fromKubeconfig(context, kubeconfigstr, null);
			KubernetesClient client = new KubernetesClientBuilder().withConfig(config).build();
			return client;
		} catch (Exception e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_conn_error, "", e);
		}
	}

	private String getUseNamespaceName(String namespace) {
		String namespaceTrue = StringUtil.isNull(namespace) ? Conf.get("common.kubernetes.apiserver.namespace.default")
				: namespace;
		return namespaceTrue;
	}

	public Namespace getNamespace(String namespace) {
		Namespace myns = this.client.namespaces().withName(namespace).get();
		return myns;
	}

	public ServiceList getServices(String namespace) {
		ServiceList myNsServices = this.client.services().inNamespace(getUseNamespaceName(namespace)).list();
		return myNsServices;
	}

	public ServiceList getServices() {
		return getServices(null);
	}

	public Service getService(String namespace, String serviceName) {
		Service myservice = this.client.services().inNamespace(getUseNamespaceName(namespace)).withName(serviceName)
				.get();
		return myservice;
	}

	// https://github.com/fabric8io/kubernetes-client/blob/master/kubernetes-examples/src/main/java/io/fabric8/kubernetes/examples/kubectl/equivalents/CustomResourceCreateDemoTypeless.java
	public GenericKubernetesResource createCusObject(ICrdDef crdVesion, String version, String namespace,
			InputStream fileInputStream) {
		try {
			CustomResourceDefinitionContext context = crdVesion.getCrd();
			NonNamespaceOperation<GenericKubernetesResource, GenericKubernetesResourceList, Resource<GenericKubernetesResource>> inNamespace = this.client
					.genericKubernetesResources(context).inNamespace(namespace);
			Resource<GenericKubernetesResource> dummyObject = inNamespace.load(fileInputStream);
			GenericKubernetesResource create = dummyObject.create();
			return create;
		} catch (Throwable e) {
			if (e.getMessage().contains("404 page not found")) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_pagenotfound, "没有布署api:" + e.getMessage());
			}
			if (e.getMessage().contains("already exists")) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_exists, "已存在此类型的同名资源:" + e.getMessage());
			}
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败", e);
		}
	}

	public GenericKubernetesResource createCusObject(ICrdDef crdVesion, String namespace, InputStream fileInputStream) {
		return createCusObject(crdVesion, null, namespace, fileInputStream);
	}

	/****
	 * 查看日志
	 * 
	 * @param pod
	 * @return
	 */
	public InputStream viewLog(Pod pod) {
		PodLogs logs = new PodLogs(this.apiClient);
		// 5秒500行,不指定容器
		this.apiClient.setReadTimeout(300000);// 5分钟
		try {
			// InputStream is = logs.streamNamespacedPodLog(v1Pod);
			// 只查前500条日志
			InputStream is = logs.streamNamespacedPodLog(pod.getMetadata().getNamespace(), pod.getMetadata().getName(),
					pod.getSpec().getContainers().get(0).getName(), null, 500, false);
			// BufferedReader reader = new BufferedReader(new InputStreamReader(is));
			return is;
			// ByteStreams.copy(is, System.out);
			// String line;
			// while((line = reader.readLine()) != null) {
			// System.out.println("====="+line);
			// }
			// System.out.println(2222);
		} catch (Exception e) {
			log.error("获取日志失败", e);
			return null;
		}
	}

	/**
	 * 通过deploymentName随机查看其中一个pod的日志
	 * 
	 * @param namespace
	 * @param deploymentName
	 * @return
	 */
	public InputStream viewLog(String namespace, String deploymentName) {
		Pod pod = getPodByDeployment(namespace, deploymentName);
		return viewLog(pod);
	}

	/***
	 * 复制文件到Pod
	 * 
	 * @param namespace
	 * @param deploymentName
	 * @param osFilepath
	 * @param podFilePath
	 * @return
	 */
	public Boolean copyFileToPod(String namespace, String deploymentName, String osFilepath, String podFilePath) {
		Pod pod = getPodByDeployment(namespace, deploymentName);
		Boolean upload = this.client.pods().inNamespace(pod.getMetadata().getNamespace())
				.withName(pod.getMetadata().getName()).file(osFilepath).upload(new File(podFilePath).toPath());
		return upload;
	}

	/***
	 * 复制文件从pod
	 * 
	 * @param namespace
	 * @param deploymentName
	 * @param podFilePath
	 * @return
	 */
	public InputStream copyFileFromPod(String namespace, String deploymentName, String podFilePath) {
		Pod pod = getPodByDeployment(namespace, deploymentName);
		InputStream inputStream = this.client.pods().inNamespace(pod.getMetadata().getNamespace())
				.withName(pod.getMetadata().getName()).file(podFilePath).read();
		return inputStream;
	}

	// kubectl exec po/flink-jobmanager-5d84d6b644-nbflx -n default --
	// /opt/flink/bin/sql-client.sh
	// 执行shell命令
	public Process execshell(String namespace, String podname, String... cmdargs) {
		Validate.notBlank(podname, "节点名必填");
		Validate.notEmpty(cmdargs, "命令参数不能为空");
//		final Options options = new Options();
//	    options.addOption(new Option("p", "pod", true, "The name of the pod"));
//	    options.addOption(new Option("n", "namespace", true, "The namespace of the pod"));
//	    CommandLineParser parser = new DefaultParser();
//	    CommandLine cmd = parser.parse(options, cmdOther);
		String[] args = null;
		if (cmdargs[0] != "sh") {// 兼容没有传 sh -c 的情况
			args = CollectionUtil.arrayMerge(String[].class, new String[] { "sh", "-c" }, cmdargs, false);
		} else {
			args = cmdargs;
		}
		Process proc = execPod(namespace, podname, args);
		return proc;
	}

	private Process execPod(String namespace, String podname, String[] args) {
		Exec exec = new Exec(this.apiClient);
		boolean tty = System.console() != null;
		Process proc;
		try {
			proc = exec.exec(StringUtil.hasNull(namespace, "default"), podname, args, true, tty);
		} catch (ApiException e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other,
					"执行shell失败：" + CollectionUtil.arrayJoin(args, " "));
		} catch (IOException e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "执行shell失败,网络或io方面问题");
		}
		return proc;
	}

	/**
	 * 执行flink sql程序
	 * 
	 * @param namespace
	 * @param podname
	 * @return
	 */
	public Process execFlinkSqlClient(String namespace, String podname, String shellcmd) {
		String[] paramsTrue;
		if (StringUtil.isNull(shellcmd)) {
			paramsTrue = new String[] { "sh", "-c", "/opt/flink/bin/sql-client.sh embedded" };
		} else {
			paramsTrue = new String[] { "sh", "-c", shellcmd };
		}
		return execshell(namespace, podname, paramsTrue);
	}

	/***
	 * 执行tams定制的sqlclient
	 * 
	 * @param namespace
	 * @param podname
	 * @param params       初始化文件的参数，如：tenantId:1,operate:2 表示租户为1，用户为2
	 * @param initFilePath 初始化文件地址
	 * @return
	 */
	public Process execFlinkSqlClientTams(String namespace, String podname, String params, String initFilePath) {
		String cmd = String.format("/opt/flink/bin/sql-client.sh embedded  -i   " + initFilePath + " -p %s", params);
		return execFlinkSqlClient(namespace, podname, cmd);
	}

	/***
	 * 登陆pod(没有成功)
	 * 
	 * @param namespace
	 * @param podname
	 * @return
	 */
	public Process execPod(String namespace, String podname) {
//		String[] paramsTrue= new String[] { "sh","-c","/bin/sh"};
		// "sh","-c","/bin/sh"
		String[] paramsTrue = new String[] { "sh", "-s", "/bin/bash" };
		return execPod(namespace, podname, paramsTrue);
	}

	public void execShellToSystemIO(String namespace, String podname, String... cmdargs) {
		Process proc = execshell(namespace, podname, cmdargs);
		execShellToCallback(proc, StreamCopyBean.procToSystem(proc), cmdargs);
	}

	/***
	 * 执行shell命令，并反回结果
	 * 
	 * @param namespace k8s
	 * @param podname   容器名
	 * @param needError 是否需要告警和错误信息
	 * @param cmd       执行的命令
	 * @return
	 */
	public String execShellToResult(String namespace, String podname, boolean needError, String cmd) {
		Process proc = execshell(namespace, podname, cmd);
		try {
			String retstr = IOUtil.slurp(proc.getInputStream());
			return retstr;
		} catch (IOException e) {
			log.error("执行命令失败:" + cmd, e);
			return null;
		}
		// 异步才用下面这种方式
//		StringBuffer buff=new StringBuffer();
//		execShellToCallback(proc, new IProcess() {
//			
//			@Override
//			public void doOutputStream(OutputStream outputStream) {
//			}
//			
//			@Override
//			public void doInputStream(InputStream inputStream) {
//				try {
//					buff.append(IOUtil.slurp(inputStream));
//				} catch (Exception e) {
//					log.error("执行命令失败："+cmd,e);
//				}
//			}
//			
//			@Override
//			public void doErrorStream(InputStream errorStream) {
//				if(needError) {
//					try {
//						buff.append(IOUtil.slurp(errorStream));
//					} catch (Exception e) {
//						log.error("执行命令失败："+cmd,e);
//					}
//				}
//			}
//		}, cmd);
//		return buff.toString();
	}

	// 反回空值即为失败
	public String execShellToResult(String namespace, String podname, String cmd) {
		return execShellToResult(namespace, podname, false, cmd);
	}

	/***
	 * 调用服务的restfull接口，如果是非本地集群，那么要求被调用方安装curl
	 * 
	 * @param isLocalCluser 是否本地k8s集群（用户方与被调用方在同一个集群），
	 * @param port          端口，默认端口，本地集群是server的默认端口80，非本地就是port里面的服务默认端口8080
	 * @param namespace     名称空间
	 * @param serviceName   服务名
	 * @param relaUrl       相对地址
	 * @return null表示失败
	 */
	public String callRestServiceToResult(boolean isLocalCluser, Integer port, String namespace, String serviceName,
			String relaUrl) {
		String retstr = null;
		if (!isLocalCluser) {
			retstr = callRestServiceToResultForExecGet(null, port, namespace, serviceName, relaUrl);
		} else {
			HttpResult doGet = callRestServiceToResult(port, namespace, serviceName, relaUrl);
			if (doGet.getResult(null).isSuc()) {
				retstr = doGet.getBodyStr();
			}
		}
		return retstr;
	}

	public HttpResult callRestServiceToResult(Integer port, String namespace, String serviceName, String relaUrl,
			String... params) {
		String httpurl = K8sAssit.getInnerUrl(port, namespace, serviceName, relaUrl, params);
		HttpResult doGet = HttpClient.doGet(httpurl);
		return doGet;
	}

	// 直接调某pod的curl本机地址来得到信息
//	public String callRestServiceToResultForExec(Integer port, String namespace, String serviceName, String relaUrl,
//			String... params) {
////		Pod pod = this.getPodByService(namespace, serviceName);
////		String relaUrlTrue = StringUtil.packageUrlParams(relaUrl, params);
////		String retstr = this.execShellToResult(namespace, pod.getMetadata().getName(), "curl "
////				+ IOUtil.mergeFolderAndFilePath("http://localhost:" + (port == null ? 8080 : port), relaUrlTrue));
////		return retstr;
//		return callRestServiceToResultForExec(null,port, namespace, serviceName, relaUrl, params);
//	}

	/// 在指定pod上通过curl调用别的机器上的服务
	private String callRestServiceToResultForExec(Pod pod, Integer port, String namespace, String serviceName,
			String relaUrl, String modth, String... params) {
		Pod podTrue = pod == null ? this.getPodByService(namespace, serviceName) : pod;
		String relaUrlTrue = StringUtil.packageUrlParams(relaUrl, params);
		String serviceUrl = pod == null ? "localhost" : String.format("%s.%s", serviceName, namespace);
		int portTrue = port == null ? 80 : port;

		String modthPathStr = "";
		if ("PATCH".equalsIgnoreCase(modth)) {
			modthPathStr = "--request PATCH";
		}
		// 支持&等转义要加双引号
		String retstr = this.execShellToResult(podTrue.getMetadata().getNamespace(), podTrue.getMetadata().getName(),
				"curl  \"" + modthPathStr
						+ IOUtil.mergeFolderAndFilePath("http://" + serviceUrl + ":" + portTrue, relaUrlTrue) + "\"");
		return retstr;
	}

	public String callRestServiceToResultForExecGet(Pod pod, Integer port, String namespace, String serviceName,
			String relaUrl, String... params) {
		return callRestServiceToResultForExec(pod, port, namespace, serviceName, relaUrl, "GET", params);
	}

	public String callRestServiceToResultForExecPatch(Pod pod, Integer port, String namespace, String serviceName,
			String relaUrl, String... params) {
		return callRestServiceToResultForExec(pod, port, namespace, serviceName, relaUrl, "PATCH", params);
	}

	public void execShellToCallback(Process proc, IProcess callback, String... cmdargs) {

		// BufferedWriter write = new BufferedWriter(new
		// OutputStreamWriter(proc.getOutputStream()));
		// BufferedReader reader = new BufferedReader(new
		// InputStreamReader(proc.getInputStream()));
		Thread in = new Thread(new Runnable() {
			public void run() {
				try {
					callback.doOutputStream(proc.getOutputStream());
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		});
		in.start();

		Thread out = new Thread(new Runnable() {
			public void run() {
				try {
					callback.doInputStream(proc.getInputStream());
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		});
		out.start();

		Thread error = new Thread(new Runnable() {
			public void run() {
				try {
					callback.doErrorStream(proc.getErrorStream());
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		});
		error.start();

		try {
			proc.waitFor();
			// wait for any last output; no need to wait for input thread
			out.join();
			error.join();
		} catch (InterruptedException e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "启动线程有问题");
		}
		proc.destroy();
	}

	/***
	 * 删除自定义的资源
	 * 
	 * @param crdVesion
	 * @param version   版本
	 * @param namespace
	 * @param name
	 * @return
	 */
	public boolean deleteCusObject(ICrdDef crdVesion, String version, String namespace, String name) {
		try {
			CustomResourceDefinitionContext context = crdVesion.getCrd();
			boolean empty = this.client.genericKubernetesResources(context).inNamespace(namespace).withName(name)
					.delete().isEmpty();
			return !empty;
		} catch (KubernetesClientException e) {
			if (e.getCode() == 404) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_pagenotfound,
						"没有布署应用:" + name + ": " + e.getMessage());
			}
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败");
		} catch (Throwable e) {
			if (e.getMessage().contains("already exists")) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_exists, "已存在此类型的同名资源:" + e.getMessage());
			}
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败");
		}
	}

	public boolean deleteCusObject(ICrdDef crdVesion, String namespace, String name) {
		return deleteCusObject(crdVesion, null, namespace, name);
	}

	/***
	 * 查询用户自定义的资源
	 * 
	 * @param crdVesion
	 * @param version   版本
	 * @param namespace
	 * @param name
	 * @return
	 */
	public List<GenericKubernetesResource> selCusObject(ICrdDef crdVesion, String version, String namespace,
			String name) {
		CustomResourceDefinitionContext context = crdVesion.getCrd();
		List<GenericKubernetesResource> retlist = null;
		if (StringUtil.isNotNull(namespace) && StringUtil.isNotNull(name)) {
			GenericKubernetesResource resource = this.client.genericKubernetesResources(context).inNamespace(namespace)
					.withName(name).get();
			retlist = resource == null ? new ArrayList<>()
					: Arrays.asList(new GenericKubernetesResource[] { resource });
		} else if (StringUtil.isNotNull(namespace) && StringUtil.isNull(name)) {
			retlist = this.client.genericKubernetesResources(context).inNamespace(namespace).list().getItems();
		} else if (StringUtil.isNull(namespace) && StringUtil.isNotNull(name)) {
			GenericKubernetesResource resource = this.client.genericKubernetesResources(context).withName(name).get();
			retlist = resource == null ? new ArrayList<>()
					: Arrays.asList(new GenericKubernetesResource[] { resource });
		} else if (StringUtil.isNull(namespace) && StringUtil.isNull(name)) {
			retlist = this.client.genericKubernetesResources(context).list().getItems();
		}
		return retlist;
	}

	public List<GenericKubernetesResource> selCusObject(ICrdDef crdVesion, String namespace, String name) {
		return selCusObject(crdVesion, crdVesion.getVersions()[0], namespace, name);
	}

	// 定制，得到flink的crd
//	public static CustomResourceDefinitionContext getFlinkCRD(CrdVesion crdVesion) {
//		CustomResourceDefinitionContext context = new CustomResourceDefinitionContext.Builder()
//				.withGroup(crdVesion.getGroup())
//				// .withKind("Dummy")
//				.withName(crdVesion.getCrdName()).withPlural(crdVesion.getPlural()).withScope(crdVesion.getScope())
//				.withVersion(crdVesion.getVersions()[0]).build();
//		return context;
//	}

	public Service getService(String serviceName) {
		return getService(null, serviceName);
	}

	/***
	 * 通过名字得到RC
	 * 
	 * @param namespace
	 * @param rcName
	 * @return
	 */
	public ReplicationController getRC(String namespace, String rcName) {
		ReplicationController gotRc = this.client.replicationControllers().inNamespace(getUseNamespaceName(namespace))
				.withName(rcName).get();
		return gotRc;
	}

	public ReplicationController getRC(String rcName) {
		return getRC(null, rcName);
	}

	/***
	 * 得到service
	 * 
	 * @param namespace
	 * @param rcName
	 * @return
	 */
	public Service getSVC(String namespace, String rcName) {
		Service service = this.client.services().inNamespace(getUseNamespaceName(namespace)).withName(rcName).get();
		return service;
	}

	public boolean delService(String namespace, String serviceName) {
		boolean optResult = !this.client.services().inNamespace(getUseNamespaceName(namespace)).withName(serviceName)
				.delete().isEmpty();
		return optResult;
	}

	public boolean delService(String serviceName) {
		return delService(null, serviceName);
	}

	public Result createNamespace(String namespace) {
		try {
			Namespace namespaceRet = this.client.namespaces()
					.resource(new NamespaceBuilder().withNewMetadata().withName(namespace).endMetadata().build())
					.create();
			return Result.getSuc().setRetObjs(namespaceRet);
		} catch (Exception e) {
			return Result.getError(e.getMessage());
		}
	}

	public boolean exitNamespace(String namespace) {
		Namespace namespace2 = this.client.namespaces().withName(namespace).get();
		return namespace2 != null;
	}

	/**
	 * 删除名称空间
	 * 
	 * @param namespace
	 * @return true:成功 false:没有此名称空间
	 */
	public boolean delNamespace(String namespace) {
		boolean retobj = !this.client.namespaces().withName(namespace).delete().isEmpty();
		return retobj;
	}

	public boolean delPodByName(String namespace, String podName) {
		return !this.client.pods().inNamespace(getUseNamespaceName(namespace)).withName(podName).delete().isEmpty();
	}

	public boolean delPodByName(String podName) {
		return delPodByName(null, podName);
	}

	public boolean delPodByLabel(String namespace, String key, String value) {
		return !this.client.pods().inNamespace(getUseNamespaceName(namespace)).withLabel(key, value).delete().isEmpty();
	}

	public boolean delPodByLabel(String key, String value) {
		return delPodByLabel(null, key, value);
	}

	/***
	 * 拿到deployment对应的某个pod
	 * 
	 * @param namespace      名称空间
	 * @param deploymentName
	 * @return
	 */
	public Pod getPodByDeployment(String namespace, String deploymentName, String... status) {
		List<Pod> items = getPodListByDeployment(namespace, deploymentName, status);
		return CollectionUtils.isEmpty(items) ? null : items.get(NumberUtil.random(items.size()));
	}

	public List<Pod> getPodListByDeployment(String namespace, String deploymentName, String... status) {
		// 通过 deploymentName、namespace 获取matchLabels
		Deployment deployment = this.client.apps().deployments().inNamespace(namespace).withName(deploymentName).get();
		if (deployment == null) {
			return null;
		}
		Map<String, String> matchLabels = deployment.getSpec().getTemplate().getMetadata().getLabels();

		String[] podAry = ArrayUtils.isEmpty(status) ? new String[] { "Running" } : status;

		// 获取PodList
		List<Pod> items = this.client.pods().inNamespace(namespace).withLabels(matchLabels).list().getItems();
		CollectionUtils.filter(items, new Predicate() {
			@Override
			public boolean evaluate(Object object) {
				Pod pod = (Pod) object;
				return ArrayUtils.contains(podAry, pod.getStatus().getPhase());
			}
		});
		return items;
	}

	public Pod getPodByJob(String namespace, String jobName, String... status) {
		List<Pod> items = getPodListByJob(namespace, jobName, status);
		return CollectionUtils.isEmpty(items) ? null : items.get(NumberUtil.random(items.size()));
	}

	public List<Pod> getPodListByJob(String namespace, String jobName, String... status) {
		// 通过 deploymentName、namespace 获取matchLabels
		Job job = this.client.batch().v1().jobs().inNamespace(namespace).withName(jobName).get();
		if (job == null) {
			return null;
		}
		Map<String, String> matchLabels = job.getSpec().getTemplate().getMetadata().getLabels();
		String[] podAry = ArrayUtils.isEmpty(status) ? new String[] { "Running" } : status;
		// 获取PodList
		List<Pod> items = this.client.pods().inNamespace(namespace).withLabels(matchLabels).list().getItems();
		CollectionUtils.filter(items, new Predicate() {
			@Override
			public boolean evaluate(Object object) {
				Pod pod = (Pod) object;
				return ArrayUtils.contains(podAry, pod.getStatus().getPhase());
			}
		});
		return items;
	}

	/**
	 * 通过service拿到pod
	 * 
	 * @param namespace
	 * @param serviceName
	 * @return
	 */
	public Pod getPodByService(String namespace, String serviceName) {
		List<Pod> items = getPodListByService(namespace, serviceName);
		return CollectionUtils.isEmpty(items) ? null : items.get(NumberUtil.random(items.size()));
	}

	public List<Pod> getPodListByService(String namespace, String serviceName) {
		// 通过 deploymentName、namespace 获取matchLabels
		Map<String, String> matchLabels = this.client.services().inNamespace(namespace).withName(serviceName).get()
				.getSpec().getSelector();
		// 获取PodList
		List<Pod> items = this.client.pods().inNamespace(namespace).withLabels(matchLabels).list().getItems();
		CollectionUtils.filter(items, new Predicate() {
			@Override
			public boolean evaluate(Object object) {
				Pod pod = (Pod) object;
				return "Running".equals(pod.getStatus().getPhase());
			}
		});
		return items;
	}

	/****
	 * 查询pod，跟据过滤条件
	 * 
	 * @param namespace
	 * @param filtermap
	 * @param lable
	 * @param ids
	 * @return
	 */
	public List<Pod> getPodListByIds(String namespace, Map<String, String> filtermap, String lable, String... ids) {
		FilterNested<FilterWatchListDeletable<Pod, PodList, PodResource>> withLabels = this.client.pods()
				.inNamespace(namespace).withNewFilter();
		if (MapUtils.isNotEmpty(filtermap)) {
			withLabels = withLabels.withLabels(filtermap);
			if (ArrayUtils.isNotEmpty(ids)) {
				withLabels = withLabels.withLabelIn(lable, ids);
			}
		} else {
			if (ArrayUtils.isNotEmpty(ids)) {
				withLabels = withLabels.withLabelIn(lable, ids);
			}
		}
		FilterWatchListDeletable<Pod, PodList, PodResource> endFilter = withLabels.endFilter();
		PodList list = endFilter.list();
		return list.getItems();
	}

	/***
	 * 
	 * @param namespace
	 * @param filtermap
	 * @param lable
	 * @param ids
	 * @return
	 */
	public List<Job> getJobListByIds(String namespace, Map<String, String> filtermap, String lable, String... ids) {
		NonNamespaceOperation<Job, JobList, ScalableResource<Job>> withLabels = this.client.batch().v1().jobs()
				.inNamespace(namespace);
		FilterWatchListDeletable<Job, JobList, ScalableResource<Job>> queryListWatch = null;
		if (MapUtils.isNotEmpty(filtermap)) {
			queryListWatch = withLabels.withLabels(filtermap);
			if (ArrayUtils.isNotEmpty(ids)) {
				queryListWatch = queryListWatch.withLabelIn(lable, ids);
			}
		} else {
			if (ArrayUtils.isNotEmpty(ids)) {
				queryListWatch = withLabels.withLabelIn(lable, ids);
			}
		}
		return queryListWatch == null ? new ArrayList<Job>() : queryListWatch.list().getItems();
	}

	public List<Pair<String, PodPhase>> queryPodStatus(String namespace, Map<String, String> filtermap, String lable,
			String[] lableNames) {
//		PodList podList = this.client.pods().withLabels(labels).withLabel("app.kubernetes.io/instance", "rjzjh").withLabelIn("appname", ids).list();
		List<Pod> pods = getPodListByIds(namespace, filtermap, lable, lableNames);
		List<Pair<String, PodPhase>> retlist = new ArrayList<Pair<String, PodPhase>>();
		for (String lableName : lableNames) {
			PodPhase elePhase = PodPhase.noExit;
			for (Pod pod : pods) {
				String podname = pod.getMetadata().getName();
				if (podname.startsWith(lableName + "-")) {
					String[] groups = podname.split("-");
					String[] subgroups = ArrayUtils.subarray(groups, 0, groups.length - 2);
					// "duckula-init-agent-6c47b7bfc4-p2l9z"变为"duckula-init-agent-"
					// t-dtdt-saasproduct-06-d64bbfb68-t25dx
					// String podnamesub = podname.substring(0, podname.length()-16);
					if (CollectionUtil.arrayJoin(subgroups, "-").equals(lableName)) {// 防止
																						// "mall-rebate-tag"和"mall-rebate"类似的问题导致的判断不准确
						elePhase = PodPhase.valueOf(pod.getStatus().getPhase());
						break;
					}
				}
			}
			retlist.add(Pair.of(lableName, elePhase));
		}
		return retlist;
	}

	public List<Pair<String, PodPhase>> queryJobStatus(String namespace, Map<String, String> filtermap, String lable,
			String[] lableNames) {
//		PodList podList = this.client.pods().withLabels(labels).withLabel("app.kubernetes.io/instance", "rjzjh").withLabelIn("appname", ids).list();
		List<Job> jobList = getJobListByIds(namespace, filtermap, lable, lableNames);
		List<Pair<String, PodPhase>> retlist = new ArrayList<Pair<String, PodPhase>>();
		for (String lableName : lableNames) {
			PodPhase elePhase = PodPhase.noExit;
			for (Job job : jobList) {
				if (job.getMetadata().getName().equals(lableName)) {
					if (job.getStatus().getActive() != null && job.getStatus().getActive() == 1) {// 因为只有一个pod，所以写死为1
						elePhase = PodPhase.Running;
					} else if (job.getStatus().getSucceeded() != null && job.getStatus().getSucceeded() == 1) {// 因为只有一个pod，所以写死为1
						elePhase = PodPhase.Succeeded;
					} else {
						elePhase = PodPhase.Unknown;
					}
					break;
				}
			}
			retlist.add(Pair.of(lableName, elePhase));
		}
		return retlist;
	}

	/***
	 * 通过label得到pod
	 * 
	 * @param namespace
	 * @param labelmap
	 * @return
	 */
	public List<String> selPodNameByLabel(String namespace, Map<String, String> labelmap) {
		List<Pod> podList = getPodListByIds(namespace, labelmap, null);
		List<String> retlist = new ArrayList<String>();
		for (Pod pod : podList) {
			retlist.add(pod.getMetadata().getName());
		}
		return retlist;
	}

	/***
	 * 通过label得到service
	 * 
	 * @param namespace
	 * @param labelmap
	 * @return
	 */
	public List<String> selServiceByLabel(String namespace, Map<String, String> labelmap) {
		NonNamespaceOperation<Service, ServiceList, ServiceResource<Service>> serviceSel = this.client.services()
				.inNamespace(getUseNamespaceName(namespace));
		FilterWatchListDeletable<Service, ServiceList, ServiceResource<Service>> withLabel = null;
		for (String key : labelmap.keySet()) {
			if (withLabel == null) {
				withLabel = serviceSel.withLabel(key, labelmap.get(key));
			} else {
				withLabel = withLabel.withLabel(key, labelmap.get(key));
			}
		}
		ServiceList list = withLabel.list();
		List<String> retlist = new ArrayList<String>();
		for (Service service : list.getItems()) {
			retlist.add(service.getMetadata().getName());
		}
		return retlist;
	}

	public void createRc(String namespace, String filePath) throws ProjectException {
		FileInputStream inputstream;
		try {
			inputstream = new FileInputStream(filePath);
		} catch (FileNotFoundException e) {
			throw new ProjectException(ExceptAll.Project_default, "文件不存在");
		}
		createRc(namespace, inputstream);
	}

	/***
	 * 跟据文件创建RC TODO 测试
	 * 
	 * @param fileInputStream 输入流
	 * @throws ProjectException
	 */
	@SuppressWarnings("unchecked")
	public void createRc(String namespace, FileInputStream fileInputStream) throws ProjectException {
		JSONObject jsonObject = YamlAssist.readYaml(fileInputStream);
		ReplicationController rc = new ReplicationController();
		if (jsonObject.containsKey("metadata")) {
			JSONObject metadataJson = jsonObject.getJSONObject("metadata");
			if (!metadataJson.containsKey("name")) {
				throw new ProjectException(ExceptAll.param_error, "metadata需要有name元素");
			}
			ObjectMeta metadata = new ObjectMeta();
			metadata.setName(metadataJson.getString("name"));
			if (metadataJson.containsKey("labels")) {
				JSONObject labels = metadataJson.getJSONObject("labels");
				Map<String, String> lables = new HashMap<String, String>();
				for (String key : labels.keySet()) {
					lables.put(key, labels.getString(key));
				}
				metadata.setLabels(lables);
			}
			rc.setMetadata(metadata);
		}
		JSONObject specJson = jsonObject.getJSONObject("spec");
		ReplicationControllerSpec spec = rc.getSpec();
		if (specJson.containsKey("replicas")) {
			spec.setReplicas(specJson.getInteger("replicas"));
		}
		if (specJson.containsKey("selector")) {
			Map<String, String> selector = JSONObject.toJavaObject(specJson.getJSONObject("selector"), Map.class);
			spec.setSelector(selector);
		}

		PodTemplateSpec newTemplate = spec.getTemplate();
		JSONObject templateJson = specJson.getJSONObject("template");
		if (templateJson.containsKey("metadata")) {
			ObjectMeta withNewMetadata = newTemplate.getMetadata();
			JSONObject jsonObject2 = templateJson.getJSONObject("metadata").getJSONObject("labels");
			Map<String, String> labels = JSON.toJavaObject(jsonObject2, Map.class);
			withNewMetadata.setLabels(labels);
		}
		// PodSpec innerSpec = newTemplate.getSpec();
		JSONObject innerSpecJson = templateJson.getJSONObject("spec");
		JSONArray containersJson = innerSpecJson.getJSONArray("containers");
		for (int i = 0; i < containersJson.size(); i++) {
			JSONObject containerJson = containersJson.getJSONObject(i);
			Container addNewContainer = new Container();
			addNewContainer.setName(containerJson.getString("name"));
			addNewContainer.setImage(containerJson.getString("image"));
			JSONArray portAry = containerJson.getJSONArray("ports");
			List<ContainerPort> ports = new ArrayList<ContainerPort>();
			for (int j = 0; j < portAry.size(); j++) {
				ContainerPort temport = new ContainerPort();
				temport.setContainerPort(portAry.getJSONObject(i).getInteger("containerPort"));
				ports.add(temport);
			}
			addNewContainer.setPorts(ports);
		}
		rc = this.client.replicationControllers().inNamespace(getUseNamespaceName(namespace)).resource(rc).create();
	}

	/***
	 * 创建Service TODO 测试
	 * 
	 * @param namespace
	 * @param fileInputStream
	 * @throws ProjectException
	 */
	@SuppressWarnings("unchecked")
	public void createSvc(String namespace, FileInputStream fileInputStream) throws ProjectException {
		JSONObject jsonObject = YamlAssist.readYaml(fileInputStream);
		Service createNew = new ServiceBuilder().build();
		JSONObject metadata = jsonObject.getJSONObject("metadata");

		ObjectMeta meta = new ObjectMeta();
		meta.setName(metadata.getString("name"));
		createNew.setMetadata(meta);
		JSONObject specjson = jsonObject.getJSONObject("spec");
		ServiceSpec withNewSpec = createNew.getSpec();
		if (specjson.containsKey("type")) {
			withNewSpec.setType(specjson.getString("type"));
		}

		if (specjson.containsKey("selector")) {
			Map<String, String> seleMap = JSON.toJavaObject(specjson.getJSONObject("selector"), Map.class);
			withNewSpec.setSelector(seleMap);
		}
		if (specjson.containsKey("ports")) {
			JSONArray array = specjson.getJSONArray("ports");
			List<ServicePort> ports = new ArrayList<>();
			for (int i = 0; i < array.size(); i++) {
				JSONObject tempobj = array.getJSONObject(i);
				ServicePort temp = new ServicePort();
				if (tempobj.containsKey("name")) {
					temp.setName(tempobj.getString("name"));
				}
				if (tempobj.containsKey("nodePort")) {
					temp.setNodePort(tempobj.getInteger("nodePort"));
				}
				if (tempobj.containsKey("port")) {
					temp.setPort(tempobj.getInteger("port"));
				}
				if (tempobj.containsKey("protocol")) {
					temp.setProtocol(tempobj.getString("protocol"));
				}
				if (tempobj.containsKey("targetPort")) {
					IntOrString input = new IntOrString(tempobj.getInteger("targetPort"));
					temp.setTargetPort(input);
				}
				ports.add(temp);
			}
			withNewSpec.setPorts(ports);
		}
		Service service = new ServiceBuilder().withSpec(withNewSpec).build();
		service = this.client.services().inNamespace(getUseNamespaceName(namespace)).resource(service).create();
	}

	public void createSvc(String namespace, String filePath) throws ProjectException {
		FileInputStream inputstream;
		try {
			inputstream = new FileInputStream(filePath);
		} catch (FileNotFoundException e) {
			throw new ProjectException(ExceptAll.Project_default, "文件不存在");
		}
		createSvc(namespace, inputstream);
	}

	/***
	 * 创建命名空间配额
	 * 
	 * @param namespace
	 * @param minCpu
	 * @param minMemory
	 * @param maxCpu
	 * @param maxMemory
	 * @param pods
	 */
	public void createResourcequota(String namespace, int minCpu, int minMemory, int maxCpu, int maxMemory, int pods) {
		Map<String, Quantity> parma = new HashMap<>();
		parma.put("pods", new Quantity(String.valueOf(pods)));
		parma.put("requests.cpu", new Quantity(String.valueOf(minCpu)));
		parma.put("requests.memory", new Quantity(String.valueOf(minMemory)));
		parma.put("limits.cpu", new Quantity(String.valueOf(maxCpu)));
		parma.put("limits.memory", new Quantity(String.valueOf(maxMemory)));
		ResourceQuota quota = new ResourceQuotaBuilder().withNewMetadata().withName(getUseNamespaceName(namespace))
				.endMetadata().withNewSpec().addToHard(parma).endSpec().build();
		this.client.resourceQuotas().inNamespace(getUseNamespaceName(namespace)).resource(quota).create();
	}

	/**
	 * 删除命名空间配额
	 * 
	 * @param namespace
	 */
	public void delResourcequota(String namespace) {
		this.client.resourceQuotas().inNamespace(getUseNamespaceName(namespace)).delete();
	}

	public boolean delRc(String namespace, String rcName) {
		return !this.client.replicationControllers().inNamespace(getUseNamespaceName(namespace)).withName(rcName)
				.delete().isEmpty();
	}

	/***
	 * 删除svc
	 * 
	 * 
	 * @param namespace
	 * @param svcName
	 * @return
	 */
	public boolean delSvc(String namespace, String svcName) {
		return !this.client.services().inNamespace(getUseNamespaceName(namespace)).withField("metadata.name", svcName)
				.delete().isEmpty();
	}

	/***
	 * 更新容器的image TODO 测试
	 * 
	 * @param namespace
	 * @param rcName
	 * @param containerIndex 这个pod的第几个容器
	 * @param imageValue
	 */
	public void updateRcImage(String namespace, String rcName, int containerIndex, String imageValue) {
		ReplicationControllerSpec spec = this.client.replicationControllers()
				.inNamespace(getUseNamespaceName(namespace)).withName(rcName).edit().getSpec();
		Container container = spec.getTemplate().getSpec().getContainers().get(containerIndex);
		container.setImage(imageValue);
		RollableScalableResource<ReplicationController> withName = getClient().replicationControllers()
				.inNamespace((getUseNamespaceName(namespace))).withName(rcName);
		withName.updateImage(imageValue);
		withName.rolling();
//		getClient().replicationControllers().inNamespace((getUseNamespaceName(namespace))).withName(rcName) .rolling()
//				.updateImage(imageValue);
	}

	/**
	 * 更新副本
	 * 
	 * @param namespace
	 * @param deploymentName
	 * @param numberOfReplicas
	 * @throws ApiException
	 */
	public void scaleDeployment(String namespace, String deploymentName, int numberOfReplicas) throws ApiException {
		AppsV1Api appsV1Api = new AppsV1Api(this.apiClient);
		// 获取V1DeploymentList
		V1DeploymentList listNamespacedDeployment = appsV1Api.listNamespacedDeployment(getUseNamespaceName(namespace),
				deploymentName, null, null, null, null, null, null, null, null, Boolean.FALSE);

		List<V1Deployment> appsV1DeploymentItems = listNamespacedDeployment.getItems();

		Optional<V1Deployment> findedDeployment = appsV1DeploymentItems.stream()
				.filter((V1Deployment deployment) -> deployment.getMetadata().getName().equals(deploymentName))
				.findFirst();

		findedDeployment.ifPresent((V1Deployment deploy) -> {
			try {
				V1DeploymentSpec newSpec = deploy.getSpec().replicas(numberOfReplicas);
				V1Deployment newDeploy = deploy.spec(newSpec);
				appsV1Api.replaceNamespacedDeployment(deploymentName, getUseNamespaceName(namespace), newDeploy, null,
						null, null, null);
			} catch (ApiException ex) {
				log.warn("Scale the pod failed for Deployment:" + deploymentName, ex);
			}
		});
	}

	// protected abstract Object setImage(String imageValue);

	/***
	 * 滚动更新image TODO 测试
	 * 
	 * @param namespace
	 * @param rcName
	 * @param imageValue
	 * @return
	 */
	public Result updateImageRolling(String namespace, String rcName, String imageValue) {
		try {
			RollableScalableResource<ReplicationController> withName = this.client.replicationControllers()
					.inNamespace(namespace).withName(rcName);
			withName.updateImage(imageValue);
			withName.rolling();
			return Result.getSuc();
		} catch (Exception e) {
			return Result.getError(e.getMessage());
		}
	}

	/***
	 * 创建或更新RC
	 * 
	 * @param namespace  命名空间
	 * @param imageValue image值
	 * @param rcPath     rc的文件路径
	 * @return 创建或更新成功
	 */
	public Result createOrUpdateImageRolling(String namespace, String imageValue, String rcPath) {
		JSONObject json = YamlAssist.readYaml(rcPath);
		String rcName = json.getJSONObject("metadata").getString("name");
		ReplicationController queryRc = getRC(namespace, rcName);
		if (queryRc == null) {
			try {
				createRc(namespace, rcPath);
				return Result.getSuc();
			} catch (ProjectException e) {
				return Result.getError(e.getMessage());
			}
		} else {
			return updateImageRolling(namespace, rcName, imageValue);
		}
	}

	/***
	 * 得到deploymentName
	 */
	public Deployment getDeploymentByName(String namespace, String name) {
		Deployment deployment = this.client.apps().deployments().inNamespace(StringUtil.hasNull(namespace, "default"))
				.withName(name).get();
		return deployment;
	}

	/***
	 * 通过label查找惟一的deployment
	 * 
	 * @param namespace
	 * @param label
	 * @param value
	 * @return
	 */
	public Deployment getDeploymentByLabelOne(String namespace, String label, String value) {
		DeploymentList list = getDeploymentByLabel(namespace, label, value);
		if (list == null || list.getItems().size() != 1) {
			throw new ProjectExceptionRuntime(ExceptAll.param_notfit,
					"Deployment预期只有一个，现已查出:" + list.getItems().size());
		}
		return list.getItems().get(0);
	}

	public DeploymentList getDeploymentByLabel(String namespace, String label, String value) {
		DeploymentList deployments = this.client.apps().deployments()
				.inNamespace(StringUtil.hasNull(namespace, "default")).withLabel(label, value).list();
		return deployments;
	}

	public DeploymentList getDeploymentByLabel(String namespace, String label) {
		DeploymentList deployments = this.client.apps().deployments()
				.inNamespace(StringUtil.hasNull(namespace, "default")).withLabel(label).list();
		return deployments;
	}

	/***
	 * 通过freemark模板和参数部署deployment
	 * 
	 * @param namespace
	 * @param context
	 * @param params
	 * @return
	 */
	public V1Deployment installDeploymentByTemp(String namespace, String context, Map<String, Object> params) {
		try {
			String result = FreemarkUtil.getInst().doProcessByTemp(context, params);
			V1Deployment yamlSvc = (V1Deployment) Yaml.load(result);
			AppsV1Api appsV1Api = new AppsV1Api(this.apiClient);
			V1Deployment v1Deployment = appsV1Api.createNamespacedDeployment(namespace, yamlSvc, "true", null, null,
					null);
			return v1Deployment;
		} catch (ApiException e) {
			if ("Conflict".equals(e.getMessage())) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_deploy_conflict);
			} else {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, e.getMessage());
			}
		} catch (Exception e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_deploy_excetion, "部署task失败", e);
		}
	}

	/***
	 * 得到资源的owner
	 * 
	 * @param md 资源,如pod等
	 * @return null表示没有owner
	 */
	public OwnerReference getControllerOf(HasMetadata md) {
		List<OwnerReference> ownerReferences = md.getMetadata().getOwnerReferences();
		for (OwnerReference ownerReference : ownerReferences) {
			if (ownerReference.getController().equals(Boolean.TRUE)) {
				return ownerReference;
			}
		}
		return null;
	}

	///////////////////////////////////////////////////////////////////////// 为了创建config需要添加的方法////////////////////////////////////////////////////////////////
	/**
	 * 查找并创建SA，创建sa名称saName，绑定到集群角色cluster-admin
	 * 
	 * @param namespace
	 * @param saName
	 * @return
	 */
	public ServiceAccount findCreateSa(String namespace, String saName) {
		ServiceAccount serviceAccount = this.client.serviceAccounts().inNamespace(namespace).withName(saName).get();
		if (serviceAccount == null) {
			ServiceAccount sa = new ServiceAccount();
			ObjectMeta objectMeta = new ObjectMeta();
			objectMeta.setName(saName);
			sa.setMetadata(objectMeta);
			serviceAccount = this.client.serviceAccounts().inNamespace(namespace).resource(sa).create();// .create(sa);
		}
		return serviceAccount;
	}

	// 把同名称空间下的帐号赋于管理员角色
	/***
	 * 使用Secret类型是kubernetes.io/service-account-token，用于被serviceaccount引用。serviceaccout
	 * 创建时 Kubernetes 会默认创建对应的 secret，即使删除也会重建。 查看：kubectl get RoleBinding
	 * 
	 * @param namespace
	 * @param bindname
	 * @param sas
	 * @return
	 */
	public Result bindAdminRoleForSa(String namespace, String bindname, String... sas) {
		RbacAuthorizationV1Api rolebind = new RbacAuthorizationV1Api(this.apiClient);
		V1RoleBinding body = new V1RoleBinding();
		body.setKind("RoleBinding");
		body.setApiVersion("rbac.authorization.k8s.io/v1");
		V1ObjectMeta metadata = new V1ObjectMeta();
		metadata.setName(bindname);
		metadata.setNamespace(namespace);
		body.setMetadata(metadata);
		V1RoleRef roleRef = new V1RoleRef();
		roleRef.setApiGroup("rbac.authorization.k8s.io");
		roleRef.setKind("ClusterRole");
		roleRef.setName("admin");
		body.setRoleRef(roleRef);
		List<V1Subject> subjects = new ArrayList<V1Subject>();
		for (String sa : sas) {
			V1Subject temp = new V1Subject();
			temp.setKind("ServiceAccount");
			temp.setName(sa);
			temp.setNamespace(namespace);
			subjects.add(temp);
		}
		body.setSubjects(subjects);
		try {
			rolebind.createNamespacedRoleBinding(namespace, body, "true", null, null, null);
			return Result.getSuc();
		} catch (ApiException e) {
//			{
//				  "kind": "Status",
//				  "apiVersion": "v1",
//				  "metadata": {
//				    
//				  },
//				  "status": "Failure",
//				  "message": "rolebindings.rbac.authorization.k8s.io \"common-admin-binder\" already exists",
//				  "reason": "AlreadyExists",
//				  "details": {
//				    "name": "common-admin-binder",
//				    "group": "rbac.authorization.k8s.io",
//				    "kind": "rolebindings"
//				  },
//				  "code": 409
//				}
			if (e.getCode() == 409) {
				return Result.getSuc();
			} else {
				log.error("创建角色绑定关系失败:" + e.getResponseBody(), e);
				return Result.getError("创建角色绑定关系失败" + e.getMessage());
			}
		}
	}

	// 配置集群权限
	public Result bindClusterAdminRoleForSa(String namespace, String bindname, String... sas) {
		RbacAuthorizationV1Api rolebind = new RbacAuthorizationV1Api(this.apiClient);
		V1ClusterRoleBinding body = new V1ClusterRoleBinding();
		body.setKind("ClusterRoleBinding");
		body.setApiVersion("rbac.authorization.k8s.io/v1");
		V1ObjectMeta metadata = new V1ObjectMeta();
		metadata.setName(bindname);
		metadata.setNamespace(namespace);
		body.setMetadata(metadata);
		V1RoleRef roleRef = new V1RoleRef();
		roleRef.setApiGroup("rbac.authorization.k8s.io");
		roleRef.setKind("ClusterRole");
		roleRef.setName("cluster-admin");
		body.setRoleRef(roleRef);
		List<V1Subject> subjects = new ArrayList<V1Subject>();
		for (String sa : sas) {
			V1Subject temp = new V1Subject();
			temp.setKind("ServiceAccount");
			temp.setName(sa);
			temp.setNamespace(namespace);
			subjects.add(temp);
		}
		body.setSubjects(subjects);
		try {
			rolebind.createClusterRoleBinding(body, "true", null, null, null);
			return Result.getSuc();
		} catch (ApiException e) {
//			{
//				  "kind": "Status",
//				  "apiVersion": "v1",
//				  "metadata": {
//				    
//				  },
//				  "status": "Failure",
//				  "message": "rolebindings.rbac.authorization.k8s.io \"common-admin-binder\" already exists",
//				  "reason": "AlreadyExists",
//				  "details": {
//				    "name": "common-admin-binder",
//				    "group": "rbac.authorization.k8s.io",
//				    "kind": "rolebindings"
//				  },
//				  "code": 409
//				}
			if (e.getCode() == 409) {
				return Result.getSuc();
			} else {
				log.error("创建角色绑定关系失败:" + e.getResponseBody(), e);
				return Result.getError("创建角色绑定关系失败" + e.getMessage());
			}
		}
	}

	/***
	 * 通过它可以拿到 data.ca.crt 和 data.token(base加密后的密文),通过 kubectl describe得到的token才是原文.
	 * 
	 * @param namespace
	 * @param accountName
	 * @return
	 */
	public Secret getSecretAccount(String namespace, String accountName) {
		SecretList list = this.client.secrets().inNamespace(namespace).list();
		List<Secret> items = list.getItems();
		CollectionUtils.filter(items, new Predicate() {
			@Override
			public boolean evaluate(Object object) {
				Secret temp = (Secret) object;
				return MapUtils.isNotEmpty(temp.getMetadata().getAnnotations()) && accountName
						.equals(temp.getMetadata().getAnnotations().get("kubernetes.io/service-account.name"));
			}
		});
		return CollectionUtils.isEmpty(items) ? null : items.get(0);
	}

	/***
	 * 通过帐号创建配置文件
	 * 
	 * @param namespace
	 * @param accountName
	 * @param isCluster   true:是集群管理员，false:名称空间管理员
	 * @return RetObjs[0]为config的内容
	 */
	public Result proKubeConfig(String clusterName, String namespace, String accountName, boolean isCluster) {
		ServiceAccount account = this.findCreateSa(namespace, accountName);// 找到或创建帐号
		// 集群资源，注意不要与其它名称空间注册的资源冲突。
		String bindName = isCluster
				? String.format("%s-%s-clusteradmin-binder", account.getMetadata().getName(), namespace)
				: String.format("%s-%s-admin-binder", account.getMetadata().getName(), namespace);// 按规则取一个绑定名
		Result result = isCluster ? this.bindClusterAdminRoleForSa(namespace, bindName, account.getMetadata().getName())
				: this.bindAdminRoleForSa(namespace, bindName, account.getMetadata().getName());
		if (!result.isSuc()) {
			return result;
		}
		Secret accountSecret = this.getSecretAccount(namespace, account.getMetadata().getName());// 找到这个帐号的secret,里面有CA和token
		String proKubeConfig = this.proKubeConfig(clusterName, accountSecret);
		Result suc = Result.getSuc();
		suc.setRetObjs(proKubeConfig);
		return suc;
	}

	/***
	 * 生成kubeconfig文件,测试通过
	 * 
	 * @param secret
	 * @param clusterName 集群的名称
	 * @return
	 */
	public String proKubeConfig(String clusterName, Secret secret) {
//		Config conifg = new Config();
//		List<NamedCluster> clusters = new ArrayList<NamedCluster>();
//		Cluster cluster = new Cluster();
//		cluster.setServer("https://192.168.21.9:1025");
//		cluster.setCertificateAuthorityData("");
//		clusters.add(new NamedCluster(cluster, "k8s-idc-dev"));
//		conifg.setClusters(clusters);
		try {
			Map<String, Object> params = new HashMap<String, Object>();
			params.put("cluster_certificate_authority_data", secret.getData().get("ca.crt"));
			params.put("cluster_server", this.client.getMasterUrl());
			// params.put("cluster_name",
			// this.client.getConfiguration().getCurrentContext().getContext().getCluster());//
			// 指定这个集群的名字,用连接的集群名字来定
			params.put("cluster_name", clusterName);// 指定这个集群的名字,用连接的集群名字来定
			params.put("user_name", secret.getMetadata().getAnnotations().get("kubernetes.io/service-account.name"));
			params.put("namespace", secret.getMetadata().getNamespace());
			params.put("context_name", clusterName);// 把当前配置的context当目标的context
			params.put("user_token", new String(PwdUtil.fromBase64(secret.getData().get("token"))));
			String slurp = IOUtil.slurp(IOUtil.fileToInputStream("/template/configSingle.yaml", KubeClientTams.class));
			String doProcessByTemp = FreemarkUtil.getInst().doProcessByTemp(slurp, params);
			return doProcessByTemp;
		} catch (IOException e) {
			log.error("生成config失败", e);
			return null;
		}
	}

}
