package net.wicp.tams.common.kubernetes.apiserver;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Map;

import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.SafeConstructor;

import io.kubernetes.client.openapi.ApiClient;
import io.kubernetes.client.util.Config;
import io.kubernetes.client.util.KubeConfig;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.constant.dic.YesOrNo;
import net.wicp.tams.common.thread.threadlocal.ObjectCreator;

/***
 * 使用k8s的原生sdk
 * 
 * @author Andy.zhou
 *
 */
public class ApiClientCreator implements ObjectCreator<ApiClient> {
	private final String config;
	private final String context;

	public ApiClientCreator(String config, String context) {
		this.config = config;
		this.context = context;
	}

	public ApiClientCreator(String config) {
		this(config, null);
	}

	public ApiClientCreator() {
		this(null, null);
	}

	@Override
	public ApiClient createObject() {
		ApiClient apiClient;
		try {
			// apiClient = Config.defaultClient();
			if (StringUtil.isNotNull(config)) {
				KubeConfig config = userKubeConfig(this.config, this.context);
				apiClient = Config.fromConfig(config);
			} else if (YesOrNo.find(Conf.get("common.kubernetes.incluster")) == YesOrNo.yes) {// 是运行在集群中
				apiClient = Config.fromCluster();
			} else {// 运行在集群外面
				apiClient = Config.defaultClient();// 默认的配置
			}
		} catch (IOException e) {
			throw new RuntimeException("create apiClient error", e);
		}
		return apiClient;
	}

	@SuppressWarnings("unchecked")
	public static KubeConfig userKubeConfig(String yamlStr, String context) {
		Yaml yaml = new Yaml(new SafeConstructor());
		Object config = yaml.load(yamlStr);
		Map<String, Object> configMap = (Map<String, Object>) config;
		String currentContext = (String) configMap.get(StringUtil.hasNull(context, "current-context"));
		ArrayList<Object> contexts = (ArrayList<Object>) configMap.get("contexts");
		ArrayList<Object> clusters = (ArrayList<Object>) configMap.get("clusters");
		ArrayList<Object> users = (ArrayList<Object>) configMap.get("users");
		Object preferences = configMap.get("preferences");
		KubeConfig kubeConfig = new KubeConfig(contexts, clusters, users);
		kubeConfig.setContext(currentContext);
		kubeConfig.setPreferences(preferences);
		return kubeConfig;
	}

}