package common.kubernetes.apiserver.crd;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;

import org.apache.commons.lang3.tuple.Pair;

import common.kubernetes.apiserver.KubeClientTams;
import io.fabric8.kubernetes.client.CustomResource;
import io.fabric8.kubernetes.client.CustomResourceList;
import io.fabric8.kubernetes.client.dsl.MixedOperation;
import io.fabric8.kubernetes.client.dsl.Resource;
import io.fabric8.kubernetes.client.informers.ResourceEventHandler;
import io.fabric8.kubernetes.client.informers.SharedIndexInformer;
import io.fabric8.kubernetes.client.informers.SharedInformerFactory;
import io.fabric8.kubernetes.client.informers.cache.Cache;
import io.fabric8.kubernetes.client.informers.cache.Lister;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.ReflectAssist;
import net.wicp.tams.common.constant.OptType;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

/***
 * 辅助开发Operate
 * 
 * @author Andy
 *
 */
@Slf4j
public abstract class CRDController<CR extends CustomResource<S, T>, CRS extends CustomResourceList<CR>, S, T> {

	protected final KubeClientTams client;
	// 定义内部操作
	protected final MixedOperation<CR, CRS, Resource<CR>> crClient;
	// 共享工厂信息
	protected final SharedInformerFactory informerFactory;
	// 共享自定义类型的信息
	protected final SharedIndexInformer<CR> crSharedIndexInformer;
	// 自定义类型的list
	protected final Lister<CR> crLister;
	// 名称空间的
	protected final String namespace;
	// CRD的定义信息
	protected final ICrdRealize crdRealize;

	protected final BlockingQueue<Pair<String, OptType>> workqueue;

	/***
	 * 初始化
	 */
	public abstract void init();

	/***
	 * 需要实现的主要接口,处理
	 * 
	 * @param cr  如果为删除,则为空
	 * @param key keey
	 * @return
	 */
	public abstract Result reconcile(CR cr, Pair<String, OptType> key);

	public abstract T getCrStatus(CR cr);

	/****
	 * 在启动前做检查
	 * 
	 * @return
	 */
	public abstract Result checkWithPreRun();

	/***
	 * 重要，判断是否需要处理修改事件。
	 * 
	 * @param oldSpec
	 * @param oldStatus
	 * @param newSpec
	 * @param newStatus
	 * @return
	 */
	public abstract Result needReconcile(S oldSpec, T oldStatus, S newSpec, T newStatus);

	public CRDController(ICrdRealize crdRealize, KubeClientTams client, String namespace) {
		this.client = client;
		this.informerFactory = client.getClient().informers();
		this.namespace = namespace;
		this.crdRealize = crdRealize;
		this.crClient = client.getClient().customResources(this.crdRealize.getCrClass(), this.crdRealize.getCrsClass());
		// 10分钟检查一次
		this.crSharedIndexInformer = this.informerFactory.sharedIndexInformerFor(this.crdRealize.getCrClass(),
				10 * 60 * 1000);// 10分种
		this.workqueue = new ArrayBlockingQueue<>(1024);
		this.crLister = new Lister<>(this.crSharedIndexInformer.getIndexer(), namespace);
		this.crSharedIndexInformer.addEventHandler(new ResourceEventHandler<CR>() {
			Map<String, String> addUidVersionMap = new HashMap<String, String>();

			@Override
			public void onAdd(CR cr) {
				log.info("add: the resource[{}],name[{}] ", crdRealize.getCrClass().getSimpleName(),
						cr.getMetadata().getName());
				if (crdRealize.getCrdName().equals(cr.getCRDName())) {
					addUidVersionMap.put(cr.getMetadata().getUid(), cr.getMetadata().getResourceVersion());
				}
				enqueuePodSet(cr, OptType.insert);
			}

			/***
			 * 更新了状态也会导致产生update事件
			 */
			@Override
			public void onUpdate(CR oldCR, CR newCR) {
				// 如果执行了一次新增，且修改事件的old是新增就可以跳过了
				if (crdRealize.getCrdName().equals(newCR.getCRDName())
						&& addUidVersionMap.containsKey(newCR.getMetadata().getUid()) && addUidVersionMap
								.get(newCR.getMetadata().getUid()).equals(oldCR.getMetadata().getResourceVersion())) {
					addUidVersionMap.remove(newCR.getMetadata().getUid());
					return;
				}
				if (!needReconcile(oldCR.getSpec(), oldCR.getStatus(), newCR.getSpec(), newCR.getStatus()).isSuc()) {
					return;
				}
				// List<ManagedFieldsEntry> fields = newCR.getMetadata().getManagedFields();
				// 最后一次修改时间
				// String time = fields.get(fields.size() - 1).getTime();// 2021-11-17T13:14:10Z
				// 创建时间
				// String creationTimestamp = newCR.getMetadata().getCreationTimestamp();//
				// 2021-11-17T13:14:10Z
				log.info("update: the old resource[{}],new resource[{}] ", oldCR.getSpec(), newCR.getSpec());
				enqueuePodSet(newCR, OptType.update);
			}

			@Override
			public void onDelete(CR cr, boolean b) {
				log.info("delete: the resource[{}],name[{}] ", crdRealize.getCrClass().getSimpleName(),
						cr.getMetadata().getName());
				enqueuePodSet(cr, OptType.delete);
			}
		});
		// 子controller初始化，需要在start完成初始化，因为可能init里也有Informers
		init();
		// 启动监听
		informerFactory.startAllRegisteredInformers();
		//
		informerFactory
				.addSharedInformerEventListener(exception -> log.error("Exception occurred, but caught", exception));

	}

	// 启动控制器
	public final void run() {
		log.info("Starting CR controller");
		// 没有准备好
		while (!crSharedIndexInformer.hasSynced()) {
			// 资源是否准备好?
		}
		Result checkRs = (Result) ReflectAssist.invokeMothedRetry(this, "checkWithPreRun", 5);
		if (!checkRs.isSuc()) {
			throw new ProjectExceptionRuntime(ExceptAll.Project_default, "检查不成功，不能启动");
		}
		while (true) {
			try {
				log.info("trying to fetch item from workqueue...");
				if (workqueue.isEmpty()) {
					log.info("Work Queue is empty");
				}
				Pair<String, OptType> key = workqueue.take();
				Objects.requireNonNull(key.getLeft(), "key can't be null");
				log.info(String.format("Got %s", key.getLeft()));
				if (key != null && key.getLeft().isEmpty() || (!key.getLeft().contains("/"))) {
					log.warn(String.format("invalid resource key: %s", key));
				} else {
					Result rcRs = null;
					if (key.getRight() == OptType.delete) {
						rcRs = reconcile(null, key);
					} else {
						// Get the PodSet resource's name from key which is in format namespace/name
						String name = key.getLeft().split("/")[1];
						CR cr = this.crLister.get(key.getLeft().split("/")[1]);
						// 在workqueue但没有找到对应的资源,有可能在处理前就删除掉了
						if (cr == null) {
							log.info(String.format("CR %s in workqueue no longer exists", name));
							return;
						}
						rcRs = reconcile(cr, key);
						if (rcRs.isSuc()) {// 子业务处理成功,并且不是删除操作
							// Update PodSet status
							T crStatus = getCrStatus(cr);
							if (crStatus != null) {
								cr.setStatus(crStatus);
								crClient.inNamespace(cr.getMetadata().getNamespace())
										.withName(cr.getMetadata().getName()).patchStatus(cr);
							}
						}
					}
				}
			} catch (InterruptedException interruptedException) {
				Thread.currentThread().interrupt();
				log.info("controller interrupted..");
			}
		}
	}

	private void enqueuePodSet(CR cr, OptType opt) {
		log.info("enqueueCR(" + cr.getMetadata().getName() + ")");
		String key = Cache.metaNamespaceKeyFunc(cr);
		log.info(String.format("Going to enqueue key %s", key));
		if (key != null && !key.isEmpty()) {
			log.info("Adding item to workqueue");
			workqueue.add(Pair.of(key, opt));
		}
	}

}
