package common.kubernetes.helm;

import java.util.List;

import org.apache.commons.lang3.ArrayUtils;

import common.kubernetes.helm.po.ChartInst;
import common.kubernetes.helm.po.HelmInst;
import common.kubernetes.helm.po.RepoInst;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.callback.IConvertObj;
import net.wicp.tams.common.constant.dic.intf.IEnumCombobox;

/***
 * 是与否枚举
 * 
 * @author andy.zhou
 * 
 */
public enum HelmCmd implements IEnumCombobox {
	install("安装", "helm install", new IConvertObj<Result>() {
		@Override
		public Result getObj(String value, String formate) {
			if (value.contains("Error:")) {
				return Result.getError(value);
			} else {
				// 如果已存在报："duckula" already exists with the same configuration, skipping
				return Result.getSuc(value);
			}
		}
	}),

	list("列出chart包", "helm list", new IConvertObj<Result>() {
		@Override
		public Result getObj(String value, String formate) {
			if (value.contains("Error:")) {
				return Result.getError(value);
			} else {
				List<HelmInst> retlist = HelmInst.splitList(value);
				Result ret = Result.getSuc();
				if (retlist.size() > 0) {
					ret.setRetObjs(retlist.toArray(new HelmInst[retlist.size()]));
				}
				return ret;
			}
		}
	}),

	delete("卸载安装包", "helm delete", new IConvertObj<Result>() {
		@Override
		public Result getObj(String value, String formate) {
			if (value.contains("Error:")) {// Error: uninstall: Release not loaded: aaaaaa: release: not found
				return Result.getError(value);
			} else {// release "aaa" uninstalled
				return Result.getSuc(value);
			}
		}
	}),

	// 由于 Helm 官方版本对插件的支持有限，环境变量 HELM_REPO_USERNAME 和 HELM_REPO_PASSWORD
	// 是必须设置的，配置本地仓库映射结束后，不需要再次设置环境变量。
	repoadd("安装本地chart库", "helm repo add --username ${HELM_REPO_USERNAME} --password ${HELM_REPO_PASSWORD}",
			new IConvertObj<Result>() {
				@Override
				public Result getObj(String value, String formate) {
					if (value.contains("Error:")) {// Error: uninstall: Release not loaded: aaaaaa: release: not found
						return Result.getError(value);
					} else {// release "aaa" uninstalled
						return Result.getSuc(value);
					}
				}
			}),

	repolist("列出已安装的库", "helm repo list", new IConvertObj<Result>() {
		@Override
		public Result getObj(String value, String formate) {
			if (value.contains("Error:")) {// Error: uninstall: Release not loaded: aaaaaa: release: not found
				if (value.contains("no repositories to show")) {// 没找到库，但它是正常的
					return Result.getSuc();
				} else {
					return Result.getError(value);
				}
			} else {// release "aaa" uninstalled
				RepoInst[] insts = RepoInst.splitList(value);
				Result ret = Result.getSuc();
				if (ArrayUtils.isNotEmpty(insts)) {
					ret.setRetObjs(insts);
				}
				return ret;
			}
		}
	}),

	repoupdate("更新本地chart库", "helm repo update", new IConvertObj<Result>() {
		@Override
		public Result getObj(String value, String formate) {
			if (value.contains("Error:")) {
				return Result.getError(value);
			} else {// Update Complete. ⎈Happy Helming!⎈
				return Result.getSuc(value);
			}
		}
	}),

	reposearch("查询找个chart包", "helm search repo", new IConvertObj<Result>() {
		@Override
		public Result getObj(String value, String formate) {
			if (value.contains("Error:")) {
				return Result.getError(value);
			} else {
				ChartInst[] retlist = ChartInst.splitList(value);
				Result ret = Result.getSuc();
				if (ArrayUtils.isNotEmpty(retlist)) {
					ret.setRetObjs(retlist);
				}
				return ret;
			}
		}
	}),

	// helm push <Chart 名称>-<Chart 版本>.tgz <本地仓库名称>
	// helm push <Chart 名称> <本地仓库名称> (使用它来完成)
	chartpush("推送chart包", "helm push", new IConvertObj<Result>() {
		@Override
		public Result getObj(String value, String formate) {
			if (value.contains("Error:")) {
				return Result.getError(value);
			} else {
				return Result.getSuc(value);
			}
		}
	}),
	// helm fetch <本地仓库名称>/<Chart 名称> --version <Chart 版本>
	chartfetch("拉取chart包", "helm fetch", new IConvertObj<Result>() {
		@Override
		public Result getObj(String value, String formate) {
			if (value.contains("Error:")) {
				return Result.getError(value);
			} else {
				return Result.getSuc(value);
			}
		}
	}),
	
	//package打包
	chartPackage("chart打包", "helm package", new IConvertObj<Result>() {
		@Override
		public Result getObj(String value, String formate) {
			if (value.contains("Error:")) {
				return Result.getError(value);
			} else {
				return Result.getSuc(value);
			}
		}
	}),;

	private final String desc;
	private final String cmd;
	private final IConvertObj<Result> convertResult;

	public String getCmd() {
		return cmd;
	}

	private HelmCmd(String desc, String cmd, IConvertObj<Result> convertResult) {
		this.desc = desc;
		this.cmd = cmd;
		this.convertResult = convertResult;
	}

	public String getDesc() {
		return desc;
	}

	public String getName() {
		return this.name();
	}

	@Override
	public String getDesc_zh() {
		return this.desc;
	}

	@Override
	public String getDesc_en() {
		return this.name();
	}

	public IConvertObj<Result> getConvertResult() {
		return convertResult;
	}
}
