package common.kubernetes.helm;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ArrayUtils;

import common.kubernetes.beans.HelmChartInfo;
import common.kubernetes.helm.po.HelmInst;
import common.kubernetes.helm.po.RepoInst;
import common.kubernetes.helm.po.RepoInst.RepoInstBuilder;
import io.fabric8.kubernetes.client.Config;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.OSinfo;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.callback.IConvertValue;
import net.wicp.tams.common.constant.PathType;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

/***
 * 只支持helm3的调用，helm2可以参考TillerClient
 * 
 * @author Andy.zhou
 *
 */
@Slf4j
public class HelmClient {

	private final String configPath;
	private final String namespaceDefault;
	private final String context;

	public HelmClient(String pathTypePath, String context, String namespace) {
		this.configPath = PathType.getPath(pathTypePath);
		String kubeconfigstr = IOUtil.slurp(this.configPath);
		Config config = StringUtil.isNull(context) ? Config.fromKubeconfig(kubeconfigstr)
				: Config.fromKubeconfig(context, kubeconfigstr, null);
		this.context = config.getCurrentContext().getName();
		this.namespaceDefault = StringUtil.hasNull(namespace, config.getCurrentContext().getContext().getNamespace());
	}

	public HelmClient(String pathTypePath, String namespace) {
		this(pathTypePath, null, namespace);
	}

	public HelmClient(String pathTypePath) {
		this(pathTypePath, null, null);
	}

	private static final java.util.Map<String, RepoInst> repomap = new java.util.HashMap<String, RepoInst>();

	public static java.util.Map<String, RepoInst> getAllRepomap() {
		return repomap;
	}

	static {
		// 找出已定义的repo进行注册
		Map<String, Map<String, String>> repoConfMap = Conf.getPreGroup("common.helm.repo.");
		for (String reponame : repoConfMap.keySet()) {
			Map<String, String> repoconf = repoConfMap.get(reponame);
			if (StringUtil.isNull(repoconf.get("url")) || StringUtil.isNull(repoconf.get("username"))
					|| StringUtil.isNull(repoconf.get("password"))) {
				log.error("[%s]repo config url/username/password is not null", reponame);
				continue;
			}
			RepoInst repoInst = RepoInst.builder().name(reponame)
					.username(StringUtil.trimSpace(repoconf.get("username")))
					.password(StringUtil.trimSpace(repoconf.get("password")))
					.url(StringUtil.trimSpace(repoconf.get("url"))).build();
			Result addRepo = HelmClient.addRepo(repoInst);
			if (!addRepo.isSuc()) {
				String errormsg = "[" + reponame + "]repo regist error:" + addRepo.getMessage();
				if ("yes".equals(repoconf.get("forcecheck"))) {
					throw new ProjectExceptionRuntime(ExceptAll.param_error, errormsg);// 如果不能注册就完了？系统运行不下去。
				} else {
					log.error(errormsg);
				}
			} else {
				log.info(addRepo.getMessage());
			}
		}

		// 初始化已存在的repo
		Result call = HelmClient.call(HelmCmd.repolist, null);
		if (call.isSuc() && ArrayUtils.isNotEmpty(call.retObjs())) {
			Object[] retObjs = call.retObjs();
			if (ArrayUtils.isNotEmpty(retObjs)) {
				for (Object object : retObjs) {
					RepoInst ele = (RepoInst) object;
					repomap.put(ele.getName(), ele);
				}
			}
		}
	}

	public static final RepoInst getRepo(String name) {
		if (StringUtil.isNull(name)) {
			return null;
		}
		if (repomap.get(name) == null) {
			synchronized (HelmClient.class) {
				if (repomap.get(name) == null) {
					Map<String, String> map = Conf.getPre(String.format("common.helm.repo.%s", name), true);
					log.info("安装repo{},找到配置项：{}", name, map.size());
					RepoInstBuilder builder = RepoInst.builder().name(name);
					builder.url(map.get("url"));
					if (map.containsKey("username")) {
						builder.username(StringUtil.hasNull(map.get("username"), ""));
					}
					if (map.containsKey("password")) {
						builder.password(StringUtil.hasNull(map.get("password"), ""));
					}
					RepoInst temp = builder.build();
					Result addRepo = addRepo(temp);
					if (addRepo.isSuc()) {
						repomap.put(name, temp);
						log.info("安装repo:{}成功", name);
					} else {
						log.error("安装repo失败,原因：{}", addRepo.getMessage());
					}
				}
			}
		}
		return repomap.get(name);
	}

	public static RepoInst getRepoNoConf(String name, Properties props) {
		if (repomap.containsKey(name) && repomap.get(name) != null) {
			log.warn("已存在此配置,可以调用getDataSource拿到数据源");
			return repomap.get(name);
		}
		Properties propsDefault = Conf.getPreToProp("common.helm.repo.default", true);// 拿到默认配置源
		for (Object key : props.keySet()) {
			propsDefault.put(String.valueOf(key), props.get(key));
		}
		Properties overprops = new Properties();
		for (Object keyobj : propsDefault.keySet()) {
			overprops.put(String.format("common.helm.repo.%s.%s", name, keyobj), propsDefault.get(keyobj));
		}
		// 先放到内存中
		Conf.overProp(overprops);
		return getRepo(name);
	}

	public static Result addRepo(RepoInst repoInst) {
		// 设置环境变量:失败
//		Result setEnvRs = OSinfo.setEnv("helmchart", "HELM_REPO_USERNAME",repoInst.getUsername(),"HELM_REPO_PASSWORD",repoInst.getPassword());
//		if(!setEnvRs.isSuc()) {
//			return setEnvRs;
//		}
		if (OSinfo.isWindows()) {// window是开发环境自己定义环境变量
			Map<HelmFlags, String> values = new HashMap<HelmFlags, String>();
			values.put(HelmFlags.name, repoInst.getName());
			values.put(HelmFlags.url, repoInst.getUrl());
			Result call = HelmClient.call(HelmCmd.repoadd, values);
			return call;
		} else {
			StringBuffer buff = new StringBuffer();
			buff.append("export HELM_REPO_USERNAME='" + repoInst.getUsername() + "';\n");
			buff.append("export HELM_REPO_PASSWORD='" + repoInst.getPassword() + "';\n");
			buff.append(String.format(
					"helm repo add %s %s --username ${HELM_REPO_USERNAME} --password ${HELM_REPO_PASSWORD}",
					repoInst.getName(), repoInst.getUrl()));
			try {
				String shellPath = IOUtil.mergeFolderAndFilePath(System.getProperty("user.home"), "helmchart.sh");
				FileUtils.writeStringToFile(new File(shellPath), buff.toString());
				Result startCmd = OSinfo.startCmd("/bin/sh " + shellPath, HelmCmd.install.getConvertResult());
				return startCmd;
			} catch (IOException e) {
				throw new ProjectExceptionRuntime(ExceptAll.project_datenofitformate, "添加repo失败", e);
			}
		}
	}

	public static Result call(HelmCmd helmCmd, Map<HelmFlags, String> values) {
		try {
			String cmd = HelmFlags.proCommonParmStr(helmCmd, values);
			log.info("the command:{}", cmd);
			Result startCmd = OSinfo.startCmd(cmd, helmCmd.getConvertResult());
			if (!startCmd.isSuc()) {
				log.error("helm call失败,原因：{}", startCmd.getMessage());
				return Result.getError(startCmd.getMessage());
			}
			return startCmd;
		} catch (Throwable e) {
			log.error("helm call失败", e);
			return Result.getError(e.getMessage());
		}
	}

	/***
	 * 安装helmchart包，它会先拉取
	 * 
	 * @param values
	 * @param updateRepo 是否更新索引
	 * @return
	 */
	public static Result install(Map<HelmFlags, String> values, boolean updateRepo) {
		Result updateRs = Result.getSuc();
		if (updateRepo) {
			// 更新索引
			updateRs = HelmClient.updateRepo();
			if (!updateRs.isSuc()) {
				return updateRs;
			}
		}
		Result result = HelmClient.call(HelmCmd.chartfetch, values);
		if (result.isSuc()) {
			// 安装
			Result rs = HelmClient.call(HelmCmd.install, values);
			return rs;
		} else {
			return result;
		}
	}

	/**
	 * 综合2种模式的install方法
	 * 
	 * @param chartInfo
	 * @return
	 */
	public Result install(String name, String namespace, HelmChartInfo chartInfo,String valueFilePath,Map<String, String> paramsmap) {
		if (StringUtil.isNotNull(chartInfo.getUploadFilePath())) {// 本地安装
			return installLocalFile(name, chartInfo.getUploadFilePath(), namespace, valueFilePath, paramsmap);
		} else {
			Map<HelmFlags, String> configmap = configKubeCommon(namespace);
			configmap.put(HelmFlags.chartFilePath,
					String.format("%s/%s", chartInfo.getRepoName(), chartInfo.getName()));
			configmap.put(HelmFlags.version, chartInfo.getAppVersion());
			configmap.put(HelmFlags.values, valueFilePath);
			return install(configmap, false);
		}
	}

	public Result install(String name, HelmChartInfo chartInfo, Map<String, String> paramsmap) {
		return install(name, null, chartInfo,null, paramsmap);
	}
	
	public Result install(String name, HelmChartInfo chartInfo,String valueFilePath) {
		return install(name, null, chartInfo,valueFilePath, null);
	}

	public Result installLocalFile(String name, String chartPath, String namespace, String valueFilePath,
			Map<String, String> setParams) {
		Map<HelmFlags, String> values = configKubeCommon(namespace);
		values.put(HelmFlags.name, name);
		values.put(HelmFlags.chartLocalpath, chartPath);
		if (StringUtil.isNotNull(valueFilePath)) {
			values.put(HelmFlags.values, valueFilePath);
		}
		String mapstrs = CollectionUtil.toPropString(setParams, ",");
		values.put(HelmFlags.set, mapstrs);
		Result rs = HelmClient.call(HelmCmd.install, values);
		return rs;
	}

	public Result installLocalFile(String name, String chartPath, String namespace, String valueFilePath,
			String[] setParams) {
		Map<String, String> parms = CollectionUtil.newMapStr(setParams);
		return installLocalFile(name, chartPath, namespace, valueFilePath, parms);
	}

	public Result installLocalFile(String name, File chartFile, String namespace, File valueFile, String... setParams) {
		return installLocalFile(name, chartFile.getPath(), namespace, valueFile.getPath(), setParams);
	}

	// namespace和context ，来自k8s的配置文件 pathTypeConfig 的默认值
	public Result installLocalFileDefaultNs(String name, String pathTypeChartFilePath, String pathTypeValueFilePath,
			String... setParams) {
		return installLocalFile(name, PathType.getPath(pathTypeChartFilePath), null,
				PathType.getPath(pathTypeValueFilePath), setParams);
	}

	public List<HelmInst> listInstance(String filtername, String namespace) {
		Map<HelmFlags, String> values = configKubeCommon(namespace);
		values.put(HelmFlags.filter, filtername);
		Result rs = HelmClient.call(HelmCmd.list, values);
		HelmInst[] retObjs = (HelmInst[]) rs.retObjs();
		return Arrays.asList(retObjs);
	}

	public List<HelmInst> listInstanceGloble(String filtername) {
		Map<HelmFlags, String> values = configKubeCommon(null);
		values.remove(HelmFlags.namespace);
		values.put(HelmFlags.namespace_all, null);
		values.put(HelmFlags.filter, filtername);
		Result rs = HelmClient.call(HelmCmd.list, values);
		HelmInst[] retObjs = (HelmInst[]) rs.retObjs();
		return Arrays.asList(retObjs == null ? (new HelmInst[0]) : retObjs);
	}

	public Result deleteInstall(String name, String namespace) {
		Map<HelmFlags, String> values = configKubeCommon(namespace);
		values.put(HelmFlags.name, name);
		Result rs = HelmClient.call(HelmCmd.delete, values);
		return rs;
	}

	private Map<HelmFlags, String> configKubeCommon(String namespace) {
		Map<HelmFlags, String> values = new HashMap<HelmFlags, String>();
		values.put(HelmFlags.kubeconfig, this.configPath);
		values.put(HelmFlags.namespace, StringUtil.hasNull(namespace, this.namespaceDefault));
		values.put(HelmFlags.kubecontext, this.context);
		return values;
	}

	public Result deleteInstall(String name) {
		return deleteInstall(name, null);
	}

	/***
	 * 通过id得到某个部署的状态
	 * 
	 * @param values
	 * @return
	 */
	public static IConvertValue<String> getConvert(Map<HelmFlags, String> values) {
		Result call = call(HelmCmd.list, values);
		HelmInst[] helmInsts = (HelmInst[]) call.retObjs();
		return new IConvertValue<String>() {
			@Override
			public String getStr(String keyObj) {
				if (StringUtil.isNull(keyObj)) {
					return "";
				}
				String keyObjTrue = StringUtil.formatRfc1123(keyObj);
				for (HelmInst helmInst : helmInsts) {
					if (keyObjTrue.equals(helmInst.getName())) {
						return helmInst.getStatus();
					}
				}
				return "未部署";
			}
		};
	}

	public static Result updateRepo() {
		Result result = HelmClient.call(HelmCmd.repoupdate, null);
		if (!result.isSuc()) {
			log.warn("更新chart包索引失败，可能会导致搜索chart不准确");
			return result;
		}
		return result;
	}

	/***
	 * 本地搜索chart包，返回的结果 ChartInst[] listCharts =(ChartInst[])
	 * listChartsRs.retObjs();
	 * 
	 * @param chartName   要搜索的包
	 * @param forceupdate 地否要强制更新本地索引
	 * @return
	 */
	public static Result listCharts(String chartName, boolean forceupdate) {
		if (forceupdate) {
			Result result = updateRepo();
			if (!result.isSuc()) {
				return result;
			}
		}
		Map<HelmFlags, String> values = new HashMap<HelmFlags, String>();
		values.put(HelmFlags.name, chartName);
		Result call = HelmClient.call(HelmCmd.reposearch, values);
		return call;
	}

	/***
	 * 推送chart包
	 * 
	 * @param chartFilePath 本地文件名或目录名
	 * @param reponame      本地仓库名
	 * @return
	 */
	public static Result pushChart(String chartFilePath, String reponame) {
		Map<HelmFlags, String> values = new HashMap<HelmFlags, String>();
		values.put(HelmFlags.chartFilePath, chartFilePath);
		values.put(HelmFlags.reponame, reponame);
		Result call = HelmClient.call(HelmCmd.chartpush, values);
		return call;
	}

	/***
	 * 把k8s的config文件保存
	 * 
	 * @param pathTrue 操作系统文件路径，非pathtype路径
	 * @param context  文件内容
	 */
	public static void saveK8sConfFile(String pathTrue, String context) {
		try {
			FileUtils.write(new File(pathTrue), context);
			// 改变权限,导致不能查询helm的状态。 否则会报WARNING: Kubernetes configuration file is
			// group-readable. This is
			// insecure. Location: /data/duckula-data/upload/k8sconfig/1344624860489924609
			if (OSinfo.isLinux()) {
				OSinfo.startCmd("chmod g-r,o-r " + pathTrue, null);
				log.info("修改{}权限成功", pathTrue);
			} else {
				log.info("操作系统为{}，不需要修改{}", OSinfo.getOSname(), pathTrue);
			}
		} catch (IOException e) {
			throw new ProjectExceptionRuntime(ExceptAll.project_other, "保存k8s文件时错误", e);
		}
	}

	public static void saveK8sConfFilePathType(String pathTypeStr, String context) {
		saveK8sConfFile(PathType.getPath(pathTypeStr), context);
	}

}
