package common.kubernetes.helm;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Map;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;

import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.callback.IConvertValue;
import net.wicp.tams.common.constant.PathType;
import net.wicp.tams.common.constant.dic.intf.IEnumCombobox;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

/***
 * 是与否枚举
 * 
 * @author andy.zhou
 * 
 */
public enum HelmFlags implements IEnumCombobox {
	name("名称", "name", "name", true, new IConvertValue<String>() {

		@Override
		public String getStr(String keyObj) {
			// rfc1123规则
			return StringUtil.formatRfc1123(keyObj);
		}

	}, new HelmCmd[] { HelmCmd.install, HelmCmd.delete, HelmCmd.reposearch, HelmCmd.repoadd }, 0), // 参数 -name

	namespace("名称空间", "namespace", "n", true, null, new HelmCmd[] { HelmCmd.install, HelmCmd.list, HelmCmd.delete },
			Integer.MAX_VALUE), // 必须放到最后面

	kubeconfig("k8s配置文件", "kubeconfig", null, true, new IConvertValue<String>() {

		@Override
		public String getStr(String keyObj) {
			return PathType.getPath(keyObj);
		}

	}, new HelmCmd[] { HelmCmd.install, HelmCmd.list, HelmCmd.delete }, 0),

	kubecontext("k8s context", "kube-context", false, new HelmCmd[] { HelmCmd.install, HelmCmd.list, HelmCmd.delete }),

	url("chartrepo地址", null, null, true, null, new HelmCmd[] { HelmCmd.repoadd }, 1),

	filter("过滤", "filter", "f", false, null, new HelmCmd[] { HelmCmd.list }, 0),

	// 不是真正的flags
	reponame("本地仓库名", null, null, true, null, new HelmCmd[] { HelmCmd.chartpush }, Integer.MAX_VALUE - 1),

	// char包本地路径,有了它就不需要chartFilePath+version
	chartLocalpath("chart包路径", null, null, false, new IConvertValue<String>() {

		@Override
		public String getStr(String keyObj) {
			return PathType.getPath(keyObj);
		}

	}, new HelmCmd[] { HelmCmd.install }, Integer.MAX_VALUE - 2),

	// 安装的版本，如果chartFilePath是本地上传的chart目录则不需要传入，如果是本地仓库则必须传入
	version("chart版本", "version", null, false, null, new HelmCmd[] { HelmCmd.install, HelmCmd.chartfetch },
			Integer.MAX_VALUE - 1),

	// <Chart 名称>-<Chart 版本>.tgz 或 <chart目录 名称>
	chartFilePath("本地文件名或目录名", null, null, true, null,
			new HelmCmd[] { HelmCmd.chartpush, HelmCmd.install, HelmCmd.chartfetch }, Integer.MAX_VALUE - 2),

	values("用户自定义的参数", "values", "f", false, new IConvertValue<String>() {

		@Override
		public String getStr(String keyObj) {
			return PathType.getPath(keyObj);
		}

	}, new HelmCmd[] { HelmCmd.install }, Integer.MAX_VALUE - 1);

	private final String desc;
	private final String paramName;
	private final String paramShortName;
	private final boolean notnull;
	private final IConvertValue<String> formart;
	private final HelmCmd[] cmds;
	private final int orderby;

	/**
	 * 产生参数
	 * 
	 * @param values
	 * @return
	 */
	public static String proCommonParmStr(HelmCmd cmd, Map<HelmFlags, String> values) {
		HelmFlags[] flags = HelmFlags.values();
		// 检查不为空
		for (HelmFlags helmFlags : flags) {
			if (ArrayUtils.contains(helmFlags.getCmds(), cmd) && helmFlags.isNotnull()
					&& (values != null && StringUtil.isNull(values.get(helmFlags)))) {
				throw new ProjectExceptionRuntime(ExceptAll.param_error, helmFlags.getDesc() + "不允许为空");
			}
		}
		// 一些特殊检查
		switch (cmd) {
		case install:
			String chartLocalpath = values.get(HelmFlags.chartLocalpath);
			String chartFilePath = values.get(HelmFlags.chartFilePath);
			String version = values.get(HelmFlags.version);
			if (StringUtil.isNull(chartLocalpath) && (StringUtil.isNull(chartFilePath) || StringUtil.isNull(version))) {
				throw new ProjectExceptionRuntime(ExceptAll.param_error,
						"安装chart需要传入chartLocalpath或 chartFilePath和version的组合");
			}
			break;
		case chartfetch:
			if (StringUtil.isNull(values.get(HelmFlags.chartFilePath))
					|| StringUtil.isNull(values.get(HelmFlags.version))) {
				throw new ProjectExceptionRuntime(ExceptAll.param_error, "chartfetch需要chartFilePath和version的组合");
			}
			break;
		default:
			break;
		}

		Arrays.parallelSort(flags, new Comparator<HelmFlags>() {
			@Override
			public int compare(HelmFlags o1, HelmFlags o2) {
				return o1.getOrderby() - o2.getOrderby();
			}
		});

		StringBuffer buff = new StringBuffer(cmd.getCmd());
		if (MapUtils.isNotEmpty(values)) {
			for (int i = 0; i < flags.length; i++) {
				if (!ArrayUtils.contains(flags[i].getCmds(), cmd)) {
					continue;// 不是此命令的flag
				}
				if (!values.containsKey(flags[i]) || StringUtil.isNull(values.get(flags[i]))) {
					continue;// 传入空值，不处理
				}
				String value = flags[i].getFormart() == null ? values.get(flags[i])
						: flags[i].getFormart().getStr(values.get(flags[i]));
				if (StringUtil.isNull(flags[i].getParamName()) && StringUtil.isNull(flags[i].paramShortName)) {// 需要特殊处理
					buff.append(" " + value);
				} else {
					if (flags[i].getParamShortName() != null) {
						buff.append(String.format(" -%s %s", flags[i].getParamShortName(), value));
					} else {
						buff.append(String.format(" --%s %s", flags[i].getParamName(), value));
					}
				}
			}
		}
		return buff.toString();
	}

	private HelmFlags(String desc, String paramName, String paramShortName, boolean notnull,
			IConvertValue<String> formart, HelmCmd[] cmds, int orderby) {
		this.desc = desc;
		this.paramName = paramName;
		this.paramShortName = paramShortName;
		this.notnull = notnull;
		this.formart = formart;
		this.cmds = cmds;
		this.orderby = orderby;
	}

	private HelmFlags(String desc, String paramName, boolean notnull, HelmCmd[] cmds) {
		this(desc, paramName, null, notnull, null, cmds, 0);
	}

	public String getDesc() {
		return desc;
	}

	public String getName() {
		return this.name();
	}

	@Override
	public String getDesc_zh() {
		return this.desc;
	}

	@Override
	public String getDesc_en() {
		return this.name();
	}

	public String getParamName() {
		return paramName;
	}

	public String getParamShortName() {
		return paramShortName;
	}

	public boolean isNotnull() {
		return notnull;
	}

	public IConvertValue<String> getFormart() {
		return formart;
	}

	public HelmCmd[] getCmds() {
		return cmds;
	}

	public int getOrderby() {
		return orderby;
	}
}
