package common.kubernetes.apiserver;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.core.JsonProcessingException;

import common.kubernetes.constant.CrdVesion;
import io.fabric8.kubernetes.api.model.Container;
import io.fabric8.kubernetes.api.model.ContainerPort;
import io.fabric8.kubernetes.api.model.IntOrString;
import io.fabric8.kubernetes.api.model.Namespace;
import io.fabric8.kubernetes.api.model.NamespaceBuilder;
import io.fabric8.kubernetes.api.model.ObjectMeta;
import io.fabric8.kubernetes.api.model.PodTemplateSpec;
import io.fabric8.kubernetes.api.model.Quantity;
import io.fabric8.kubernetes.api.model.ReplicationController;
import io.fabric8.kubernetes.api.model.ReplicationControllerSpec;
import io.fabric8.kubernetes.api.model.ResourceQuota;
import io.fabric8.kubernetes.api.model.ResourceQuotaBuilder;
import io.fabric8.kubernetes.api.model.Service;
import io.fabric8.kubernetes.api.model.ServiceList;
import io.fabric8.kubernetes.api.model.ServicePort;
import io.fabric8.kubernetes.api.model.ServiceSpec;
import io.fabric8.kubernetes.api.model.apps.Deployment;
import io.fabric8.kubernetes.api.model.apps.DeploymentList;
import io.fabric8.kubernetes.client.ConfigBuilder;
import io.fabric8.kubernetes.client.DefaultKubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.dsl.base.CustomResourceDefinitionContext;
import io.fabric8.kubernetes.client.internal.SerializationUtils;
import io.kubernetes.client.openapi.ApiClient;
import io.kubernetes.client.openapi.ApiException;
import io.kubernetes.client.openapi.apis.AppsV1Api;
import io.kubernetes.client.openapi.models.V1Deployment;
import io.kubernetes.client.util.Yaml;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.FreemarkUtil;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.yaml.YamlAssist;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

/***
 * 
 * @author rjzjh
 *
 */
public class KubeClientTams {
	private final DefaultKubernetesClient client;
	private final ApiClient apiClient;

	public DefaultKubernetesClient getClient() {
		return client;
	}

	/***
	 * 通过Conf的配置得到client
	 */
	public KubeClientTams() {
		this.client = initClient();
		this.apiClient = ApiClientManager.getApiClient();
	}

	/**
	 * 通过配置文件的内容和context得到client
	 * 
	 * @param kubeconfigstr kubeconfig内容
	 * @param context       启用哪个context,传空则为当前context
	 */
	public KubeClientTams(String kubeconfigstr, String context) {
		this.client = getClientForKube(kubeconfigstr, context);
		this.apiClient = new ApiClientCreator(kubeconfigstr, context).createObject();
	}

	/***
	 * 配置文件路径得到client
	 * 
	 * @param pathTypePath 配置文件
	 */
	public KubeClientTams(String pathTypePath) {
		String kubeconfigstr = IOUtil.slurpPathTypePath(pathTypePath);
		this.client = getClientForKube(kubeconfigstr, null);
		this.apiClient = new ApiClientCreator(kubeconfigstr, null).createObject();
	}

	public KubeClientTams(DefaultKubernetesClient client, ApiClient apiClient) {
		this.client = client;
		this.apiClient = apiClient;
	}

	private DefaultKubernetesClient initClient() {
		DefaultKubernetesClient client = null;
		if (StringUtil.isNotNull(Conf.get("common.kubernetes.kubeconfig.path"))) {
			String kubeconfigstr = IOUtil.slurp(Conf.get("common.kubernetes.kubeconfig.path"));
			String context = IOUtil.slurp(Conf.get("common.kubernetes.kubeconfig.context"));
			client = getClientForKube(kubeconfigstr, context);
		} else if (StringUtil.isNotNull(Conf.get("common.kubernetes.apiserver.master.url"))) {
			ConfigBuilder config = new ConfigBuilder()
					.withMasterUrl(Conf.get("common.kubernetes.apiserver.master.url"));
			if (StringUtil.isNotNull(Conf.get("common.kubernetes.apiserver.master.username"))) {
				config.withUsername(Conf.get("common.kubernetes.apiserver.master.username"));
				config.withPassword(Conf.get("common.kubernetes.apiserver.master.password"));
			}
			config.withTrustCerts(true);
			client = new DefaultKubernetesClient(config.build());

		} else {
			client = new DefaultKubernetesClient();// 默认读./kube信息
		}
		return client;
	}

	private DefaultKubernetesClient getClientForKube(String kubeconfigstr, String context) {
		try {
			io.fabric8.kubernetes.client.Config config = StringUtil.isNull(context)
					? io.fabric8.kubernetes.client.Config.fromKubeconfig(kubeconfigstr)
					: io.fabric8.kubernetes.client.Config.fromKubeconfig(context, kubeconfigstr, null);
			DefaultKubernetesClient client = new DefaultKubernetesClient(config);
			return client;
		} catch (Exception e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_conn_error, "", e);
		}
	}

	private String getUseNamespaceName(String namespace) {
		String namespaceTrue = StringUtil.isNull(namespace) ? Conf.get("common.kubernetes.apiserver.namespace.default")
				: namespace;
		return namespaceTrue;
	}

	public Namespace getNamespace(String namespace) {
		Namespace myns = this.client.namespaces().withName(namespace).get();
		return myns;
	}

	public ServiceList getServices(String namespace) {
		ServiceList myNsServices = this.client.services().inNamespace(getUseNamespaceName(namespace)).list();
		return myNsServices;
	}

	public ServiceList getServices() {
		return getServices(null);
	}

	public Service getService(String namespace, String serviceName) {
		Service myservice = this.client.services().inNamespace(getUseNamespaceName(namespace)).withName(serviceName)
				.get();
		return myservice;
	}

	// https://github.com/fabric8io/kubernetes-client/blob/master/kubernetes-examples/src/main/java/io/fabric8/kubernetes/examples/kubectl/equivalents/CustomResourceCreateDemoTypeless.java
	public Map<String, Object> createCusObject(CrdVesion crdVesion, String version, String namespace,
			InputStream fileInputStream) {
		try {
			// Create Custom Resource Context
			CustomResourceDefinitionContext context = crdVesion.getCrdByVersion(version);
			// Load from Yaml
			Map<String, Object> dummyObject = this.client.customResource(context).load(fileInputStream);// CustomResourceCreateDemoTypeless.class.getResourceAsStream("/test-customresource.yaml")
			// Create Custom Resource
			Map<String, Object> map = this.client.customResource(context).create(namespace, dummyObject);
			return map;
		} catch (IOException e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败");
		} catch (Throwable e) {
			if (e.getMessage().contains("404 page not found")) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_pagenotfound, "没有布署api:" + e.getMessage());
			}
			if (e.getMessage().contains("already exists")) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_exists, "已存在此类型的同名资源:" + e.getMessage());
			}
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败");
		}
	}

	public Map<String, Object> createCusObject(CrdVesion crdVesion, String namespace, InputStream fileInputStream) {
		return createCusObject(crdVesion, null, namespace, fileInputStream);
	}

	/***
	 * 删除自定义的资源
	 * 
	 * @param crdVesion
	 * @param version   版本
	 * @param namespace
	 * @param name
	 * @return
	 */
	public Map<String, Object> deleteCusObject(CrdVesion crdVesion, String version, String namespace, String name) {
		try {
			CustomResourceDefinitionContext context = crdVesion.getCrdByVersion(version);
			Map<String, Object> map = this.client.customResource(context).delete(namespace, name);
			return map;
		} catch (KubernetesClientException e) {
			if (e.getCode() == 404) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_pagenotfound,
						"没有布署应用:" + name + ": " + e.getMessage());
			}
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败");
		} catch (Throwable e) {
			if (e.getMessage().contains("already exists")) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_exists, "已存在此类型的同名资源:" + e.getMessage());
			}
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败");
		}
	}

	public Map<String, Object> deleteCusObject(CrdVesion crdVesion, String namespace, String name) {
		return deleteCusObject(crdVesion, null, namespace, name);
	}

	/***
	 * 查询用户自定义的资源
	 * 
	 * @param crdVesion
	 * @param version   版本
	 * @param namespace
	 * @param name
	 * @return
	 */
	public Map<String, Object> selCusObject(CrdVesion crdVesion, String version, String namespace, String name) {
		CustomResourceDefinitionContext context = crdVesion.getCrdByVersion(version);
		Map<String, Object> map = null;
		if (StringUtil.isNotNull(namespace) && StringUtil.isNotNull(name)) {
			map = this.client.customResource(context).get(namespace, name);
		} else if (StringUtil.isNotNull(namespace) && StringUtil.isNull(name)) {
			map = this.client.customResource(context).list(namespace);
		} else if (StringUtil.isNull(namespace) && StringUtil.isNotNull(name)) {
			map = this.client.customResource(context).get(name);
		} else if (StringUtil.isNull(namespace) && StringUtil.isNull(name)) {
			map = this.client.customResource(context).list();
		}
		return map;
	}

	// 定制，得到flink的crd
//	public static CustomResourceDefinitionContext getFlinkCRD(CrdVesion crdVesion) {
//		CustomResourceDefinitionContext context = new CustomResourceDefinitionContext.Builder()
//				.withGroup(crdVesion.getGroup())
//				// .withKind("Dummy")
//				.withName(crdVesion.getCrdName()).withPlural(crdVesion.getPlural()).withScope(crdVesion.getScope())
//				.withVersion(crdVesion.getVersions()[0]).build();
//		return context;
//	}

	public Service getService(String serviceName) {
		return getService(null, serviceName);
	}

	/***
	 * 通过名字得到RC
	 * 
	 * @param namespace
	 * @param rcName
	 * @return
	 */
	public ReplicationController getRC(String namespace, String rcName) {
		ReplicationController gotRc = this.client.replicationControllers().inNamespace(getUseNamespaceName(namespace))
				.withName(rcName).get();
		return gotRc;
	}

	public ReplicationController getRC(String rcName) {
		return getRC(null, rcName);
	}

	/***
	 * 得到service
	 * 
	 * @param namespace
	 * @param rcName
	 * @return
	 */
	public Service getSVC(String namespace, String rcName) {
		Service service = this.client.services().inNamespace(getUseNamespaceName(namespace)).withName(rcName).get();
		return service;
	}

	/***
	 * 得到RC对应的yaml文件
	 * 
	 * @param namespace
	 * @param rcName
	 * @return
	 * @throws JsonProcessingException
	 */
	public String getRCYaml(String namespace, String rcName) throws JsonProcessingException {
		ReplicationController gotRc = getRC(namespace, rcName);
		String str = SerializationUtils.dumpAsYaml(gotRc);
		return str;
	}

	public boolean delService(String namespace, String serviceName) {
		Boolean optResult = this.client.services().inNamespace(getUseNamespaceName(namespace)).withName(serviceName)
				.delete();
		return optResult;
	}

	public boolean delService(String serviceName) {
		return delService(null, serviceName);
	}

	public Result createNamespace(String namespace) {
		try {
			Namespace namespaceRet = this.client.namespaces()
					.create(new NamespaceBuilder().withNewMetadata().withName(namespace).endMetadata().build());
			return Result.getSuc().setRetObjs(namespaceRet);
		} catch (Exception e) {
			return Result.getError(e.getMessage());
		}
	}

	public Result delNamespace(String namespace) {
		Boolean retobj = this.client.namespaces().withName("test").delete();
		if (retobj != null && retobj) {
			return Result.getSuc();
		} else {
			return Result.getError("删除名字空间失败");
		}
	}

	public boolean delPodByName(String namespace, String podName) {
		return this.client.pods().inNamespace(getUseNamespaceName(namespace)).withName(podName).delete();
	}

	public boolean delPodByName(String podName) {
		return delPodByName(null, podName);
	}

	public boolean delPodByLabel(String namespace, String key, String value) {
		return this.client.pods().inNamespace(getUseNamespaceName(namespace)).withLabel(key, value).delete();
	}

	public boolean delPodByLabel(String key, String value) {
		return delPodByLabel(null, key, value);
	}

	public void createRc(String namespace, String filePath) throws ProjectException {
		FileInputStream inputstream;
		try {
			inputstream = new FileInputStream(filePath);
		} catch (FileNotFoundException e) {
			throw new ProjectException(ExceptAll.Project_default, "文件不存在");
		}
		createRc(namespace, inputstream);
	}

	/***
	 * 跟据文件创建RC
	 * 
	 * @param fileInputStream 输入流
	 * @throws ProjectException
	 */
	@SuppressWarnings("unchecked")
	public void createRc(String namespace, FileInputStream fileInputStream) throws ProjectException {
		JSONObject jsonObject = YamlAssist.readYaml(fileInputStream);
		// DoneableReplicationController rc =
		ReplicationController rc = this.client.replicationControllers().inNamespace(getUseNamespaceName(namespace))
				.create();
		if (jsonObject.containsKey("metadata")) {
			JSONObject metadataJson = jsonObject.getJSONObject("metadata");
			if (!metadataJson.containsKey("name")) {
				throw new ProjectException(ExceptAll.param_error, "metadata需要有name元素");
			}
			ObjectMeta metadata = new ObjectMeta();
			metadata.setName(metadataJson.getString("name"));
			if (metadataJson.containsKey("labels")) {
				JSONObject labels = metadataJson.getJSONObject("labels");
				Map<String, String> lables = new HashMap<String, String>();
				for (String key : labels.keySet()) {
					lables.put(key, labels.getString(key));
				}
				metadata.setLabels(lables);
			}
			rc.setMetadata(metadata);
		}
		JSONObject specJson = jsonObject.getJSONObject("spec");
		ReplicationControllerSpec spec = rc.getSpec();
		if (specJson.containsKey("replicas")) {
			spec.setReplicas(specJson.getInteger("replicas"));
		}
		if (specJson.containsKey("selector")) {
			Map<String, String> selector = JSONObject.toJavaObject(specJson.getJSONObject("selector"), Map.class);
			spec.setSelector(selector);
		}

		PodTemplateSpec newTemplate = spec.getTemplate();
		JSONObject templateJson = specJson.getJSONObject("template");
		if (templateJson.containsKey("metadata")) {
			ObjectMeta withNewMetadata = newTemplate.getMetadata();
			JSONObject jsonObject2 = templateJson.getJSONObject("metadata").getJSONObject("labels");
			Map<String, String> labels = JSON.toJavaObject(jsonObject2, Map.class);
			withNewMetadata.setLabels(labels);
		}

		// PodSpec innerSpec = newTemplate.getSpec();
		JSONObject innerSpecJson = templateJson.getJSONObject("spec");
		JSONArray containersJson = innerSpecJson.getJSONArray("containers");
		for (int i = 0; i < containersJson.size(); i++) {
			JSONObject containerJson = containersJson.getJSONObject(i);
			Container addNewContainer = new Container();
			addNewContainer.setName(containerJson.getString("name"));
			addNewContainer.setImage(containerJson.getString("image"));
			JSONArray portAry = containerJson.getJSONArray("ports");
			List<ContainerPort> ports = new ArrayList<ContainerPort>();
			for (int j = 0; j < portAry.size(); j++) {
				ContainerPort temport = new ContainerPort();
				temport.setContainerPort(portAry.getJSONObject(i).getInteger("containerPort"));
				ports.add(temport);
			}
			addNewContainer.setPorts(ports);
		}
	}

	/***
	 * 创建Service
	 * 
	 * @param namespace
	 * @param fileInputStream
	 * @throws ProjectException
	 */
	@SuppressWarnings("unchecked")
	public void createSvc(String namespace, FileInputStream fileInputStream) throws ProjectException {
		JSONObject jsonObject = YamlAssist.readYaml(fileInputStream);
		Service createNew = this.client.services().inNamespace(getUseNamespaceName(namespace)).create();
		JSONObject metadata = jsonObject.getJSONObject("metadata");

		ObjectMeta meta = new ObjectMeta();
		meta.setName(metadata.getString("name"));
		createNew.setMetadata(meta);
		JSONObject specjson = jsonObject.getJSONObject("spec");
		ServiceSpec withNewSpec = createNew.getSpec();
		if (specjson.containsKey("type")) {
			withNewSpec.setType(specjson.getString("type"));
		}

		if (specjson.containsKey("selector")) {
			Map<String, String> seleMap = JSON.toJavaObject(specjson.getJSONObject("selector"), Map.class);
			withNewSpec.setSelector(seleMap);
		}
		if (specjson.containsKey("ports")) {
			JSONArray array = specjson.getJSONArray("ports");
			List<ServicePort> ports = new ArrayList<>();
			for (int i = 0; i < array.size(); i++) {
				JSONObject tempobj = array.getJSONObject(i);
				ServicePort temp = new ServicePort();
				if (tempobj.containsKey("name")) {
					temp.setName(tempobj.getString("name"));
				}
				if (tempobj.containsKey("nodePort")) {
					temp.setNodePort(tempobj.getInteger("nodePort"));
				}
				if (tempobj.containsKey("port")) {
					temp.setPort(tempobj.getInteger("port"));
				}
				if (tempobj.containsKey("protocol")) {
					temp.setProtocol(tempobj.getString("protocol"));
				}
				if (tempobj.containsKey("targetPort")) {
					IntOrString input = new IntOrString(tempobj.getInteger("targetPort"));
					temp.setTargetPort(input);
				}
				ports.add(temp);
			}
			withNewSpec.setPorts(ports);
		}
	}

	public void createSvc(String namespace, String filePath) throws ProjectException {
		FileInputStream inputstream;
		try {
			inputstream = new FileInputStream(filePath);
		} catch (FileNotFoundException e) {
			throw new ProjectException(ExceptAll.Project_default, "文件不存在");
		}
		createSvc(namespace, inputstream);
	}

	/***
	 * 创建命名空间配额
	 * 
	 * @param namespace
	 * @param minCpu
	 * @param minMemory
	 * @param maxCpu
	 * @param maxMemory
	 * @param pods
	 */
	public void createResourcequota(String namespace, int minCpu, int minMemory, int maxCpu, int maxMemory, int pods) {
		Map<String, Quantity> parma = new HashMap<>();
		parma.put("pods", new Quantity(String.valueOf(pods)));
		parma.put("requests.cpu", new Quantity(String.valueOf(minCpu)));
		parma.put("requests.memory", new Quantity(String.valueOf(minMemory)));
		parma.put("limits.cpu", new Quantity(String.valueOf(maxCpu)));
		parma.put("limits.memory", new Quantity(String.valueOf(maxMemory)));
		ResourceQuota quota = new ResourceQuotaBuilder().withNewMetadata().withName(getUseNamespaceName(namespace))
				.endMetadata().withNewSpec().addToHard(parma).endSpec().build();
		this.client.resourceQuotas().inNamespace(getUseNamespaceName(namespace)).create(quota);
	}

	/**
	 * 删除命名空间配额
	 * 
	 * @param namespace
	 */
	public void delResourcequota(String namespace) {
		this.client.resourceQuotas().inNamespace(getUseNamespaceName(namespace)).delete();
	}

	public boolean delRc(String namespace, String rcName) {
		return this.client.replicationControllers().inNamespace(getUseNamespaceName(namespace)).withName(rcName)
				.delete();
	}

	/***
	 * 删除svc
	 * 
	 * @param namespace
	 * @param svcName
	 * @return
	 */
	public boolean delSvc(String namespace, String svcName) {
		return this.client.services().inNamespace(getUseNamespaceName(namespace)).withField("metadata.name", svcName)
				.delete();
	}

	/***
	 * 更新容器的image
	 * 
	 * @param namespace
	 * @param rcName
	 * @param containerIndex 这个pod的第几个容器
	 * @param imageValue
	 */
	@SuppressWarnings("deprecation")
	public void updateRcImage(String namespace, String rcName, int containerIndex, String imageValue) {
		ReplicationControllerSpec spec = this.client.replicationControllers()
				.inNamespace(getUseNamespaceName(namespace)).withName(rcName).edit().getSpec();
		Container container = spec.getTemplate().getSpec().getContainers().get(containerIndex);
		container.setImage(imageValue);
		// getClient().replicationControllers().inNamespace((getUseNamespaceName(namespace))).withName(rcName).rolling().updateImage(imageValue);
	}

	// protected abstract Object setImage(String imageValue);

	/***
	 * 滚动更新image
	 * 
	 * @param namespace
	 * @param rcName
	 * @param imageValue
	 * @return
	 */
	public Result updateImageRolling(String namespace, String rcName, String imageValue) {
		try {
			this.client.replicationControllers().inNamespace(namespace).withName(rcName).rolling()
					.updateImage(imageValue);
			return Result.getSuc();
		} catch (Exception e) {
			return Result.getError(e.getMessage());
		}
	}

	/***
	 * 创建或更新RC
	 * 
	 * @param namespace  命名空间
	 * @param imageValue image值
	 * @param rcPath     rc的文件路径
	 * @return 创建或更新成功
	 */
	public Result createOrUpdateImageRolling(String namespace, String imageValue, String rcPath) {
		JSONObject json = YamlAssist.readYaml(rcPath);
		String rcName = json.getJSONObject("metadata").getString("name");
		ReplicationController queryRc = getRC(namespace, rcName);
		if (queryRc == null) {
			try {
				createRc(namespace, rcPath);
				return Result.getSuc();
			} catch (ProjectException e) {
				return Result.getError(e.getMessage());
			}
		} else {
			return updateImageRolling(namespace, rcName, imageValue);
		}
	}

	/***
	 * 得到deploymentName
	 */
	public Deployment getDeploymentByName(String namespace, String name) {
		Deployment deployment = this.client.apps().deployments().inNamespace(StringUtil.hasNull(namespace, "default"))
				.withName(name).get();
		return deployment;
	}

	/***
	 * 通过label查找惟一的deployment
	 * 
	 * @param namespace
	 * @param label
	 * @param value
	 * @return
	 */
	public Deployment getDeploymentByLabelOne(String namespace, String label, String value) {
		DeploymentList list = getDeploymentByLabel(namespace, label, value);
		if (list == null || list.getItems().size() != 1) {
			throw new ProjectExceptionRuntime(ExceptAll.param_notfit,
					"Deployment预期只有一个，现已查出:" + list.getItems().size());
		}
		return list.getItems().get(0);
	}

	public DeploymentList getDeploymentByLabel(String namespace, String label, String value) {
		DeploymentList deployments = this.client.apps().deployments()
				.inNamespace(StringUtil.hasNull(namespace, "default")).withLabel(label, value).list();
		return deployments;
	}

	public DeploymentList getDeploymentByLabel(String namespace, String label) {
		DeploymentList deployments = this.client.apps().deployments()
				.inNamespace(StringUtil.hasNull(namespace, "default")).withLabel(label).list();
		return deployments;
	}

	/***
	 * 通过freemark模板和参数部署deployment
	 * 
	 * @param namespace
	 * @param context
	 * @param params
	 * @return
	 */
	public V1Deployment installDeploymentByTemp(String namespace, String context, Map<String, Object> params) {
		try {
			String result = FreemarkUtil.getInst().doProcessByTemp(context, params);
			V1Deployment yamlSvc = (V1Deployment) Yaml.load(result);
			AppsV1Api appsV1Api = new AppsV1Api(this.apiClient);
			V1Deployment v1Deployment = appsV1Api.createNamespacedDeployment(namespace, yamlSvc, "true", null, null);
			return v1Deployment;
		} catch (ApiException e) {
			if ("Conflict".equals(e.getMessage())) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_deploy_conflict);
			} else {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, e.getMessage());
			}
		} catch (Exception e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_deploy_excetion, "部署task失败", e);
		}
	}

}
