package common.kubernetes.apiserver;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.core.JsonProcessingException;

import common.kubernetes.constant.CrdVesion;
import io.fabric8.kubernetes.api.model.Container;
import io.fabric8.kubernetes.api.model.ContainerPort;
import io.fabric8.kubernetes.api.model.IntOrString;
import io.fabric8.kubernetes.api.model.Namespace;
import io.fabric8.kubernetes.api.model.NamespaceBuilder;
import io.fabric8.kubernetes.api.model.ObjectMeta;
import io.fabric8.kubernetes.api.model.PodSpec;
import io.fabric8.kubernetes.api.model.PodTemplateSpec;
import io.fabric8.kubernetes.api.model.Quantity;
import io.fabric8.kubernetes.api.model.ReplicationController;
import io.fabric8.kubernetes.api.model.ReplicationControllerSpec;
import io.fabric8.kubernetes.api.model.ResourceQuota;
import io.fabric8.kubernetes.api.model.ResourceQuotaBuilder;
import io.fabric8.kubernetes.api.model.Service;
import io.fabric8.kubernetes.api.model.ServiceList;
import io.fabric8.kubernetes.api.model.ServicePort;
import io.fabric8.kubernetes.api.model.ServiceSpec;
import io.fabric8.kubernetes.client.ConfigBuilder;
import io.fabric8.kubernetes.client.DefaultKubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.dsl.base.CustomResourceDefinitionContext;
import io.fabric8.kubernetes.client.internal.SerializationUtils;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.yaml.YamlAssist;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

/***
 * 
 * @author rjzjh
 *
 */
@Slf4j
public abstract class KubeClient {
	public static DefaultKubernetesClient getClient() {
		DefaultKubernetesClient client = null;
		if (StringUtil.isNull(Conf.get("common.kubernetes.apiserver.master.url"))) {
			client = new DefaultKubernetesClient();// 默认读./kube信息
		} else {
			ConfigBuilder config = new ConfigBuilder()
					.withMasterUrl(Conf.get("common.kubernetes.apiserver.master.url"));
			if (StringUtil.isNotNull(Conf.get("common.kubernetes.apiserver.master.username"))) {
				config.withUsername(Conf.get("common.kubernetes.apiserver.master.username"));
				config.withPassword(Conf.get("common.kubernetes.apiserver.master.password"));
			}
			config.withTrustCerts(true);
			client = new DefaultKubernetesClient(config.build());
		}
		return client;
	}

	private static String getUseNamespaceName(String namespace) {
		String namespaceTrue = StringUtil.isNull(namespace) ? Conf.get("common.kubernetes.apiserver.namespace.default")
				: namespace;
		return namespaceTrue;
	}

	public static Namespace getNamespace(String namespace) {
		Namespace myns = getClient().namespaces().withName(namespace).get();
		return myns;
	}

	public static ServiceList getServices(String namespace) {
		ServiceList myNsServices = getClient().services().inNamespace(getUseNamespaceName(namespace)).list();
		return myNsServices;
	}

	public static ServiceList getServices() {
		return getServices(null);
	}

	public static Service getService(String namespace, String serviceName) {
		Service myservice = getClient().services().inNamespace(getUseNamespaceName(namespace)).withName(serviceName)
				.get();
		return myservice;
	}

	// https://github.com/fabric8io/kubernetes-client/blob/master/kubernetes-examples/src/main/java/io/fabric8/kubernetes/examples/kubectl/equivalents/CustomResourceCreateDemoTypeless.java
	public static Map<String, Object> createCusObject(CrdVesion crdVesion, String namespace,
			InputStream fileInputStream) {
		try (final KubernetesClient k8s = getClient()) {
			// Create Custom Resource Context
			CustomResourceDefinitionContext context = getFlinkCRD(crdVesion);
			// Load from Yaml
			Map<String, Object> dummyObject = k8s.customResource(context).load(fileInputStream);// CustomResourceCreateDemoTypeless.class.getResourceAsStream("/test-customresource.yaml")
			// Create Custom Resource
			Map<String, Object> map = k8s.customResource(context).create(namespace, dummyObject);
			return map;
		} catch (IOException e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败");
		} catch (Throwable e) {
			if (e.getMessage().contains("404 page not found")) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_pagenotfound, "没有布署api:" + e.getMessage());
			}
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败");
		}
	}

	// 定制，得到flink的crd
	public static CustomResourceDefinitionContext getFlinkCRD(CrdVesion crdVesion) {
		CustomResourceDefinitionContext context = new CustomResourceDefinitionContext.Builder()
				.withGroup(crdVesion.getGroup())
				// .withKind("Dummy")
				.withName(crdVesion.getCrdName()).withPlural(crdVesion.getPlural()).withScope(crdVesion.getScope())
				.withVersion(crdVesion.getVersions()[0]).build();
		return context;
	}

	public static Service getService(String serviceName) {
		return getService(null, serviceName);
	}

	/***
	 * 通过名字得到RC
	 * 
	 * @param namespace
	 * @param rcName
	 * @return
	 */
	public static ReplicationController getRC(String namespace, String rcName) {
		ReplicationController gotRc = getClient().replicationControllers().inNamespace(getUseNamespaceName(namespace))
				.withName(rcName).get();
		return gotRc;
	}

	public static ReplicationController getRC(String rcName) {
		return getRC(null, rcName);
	}

	/***
	 * 得到service
	 * 
	 * @param namespace
	 * @param rcName
	 * @return
	 */
	public static Service getSVC(String namespace, String rcName) {
		Service service = getClient().services().inNamespace(getUseNamespaceName(namespace)).withName(rcName).get();
		return service;
	}

	/***
	 * 得到RC对应的yaml文件
	 * 
	 * @param namespace
	 * @param rcName
	 * @return
	 * @throws JsonProcessingException
	 */
	public static String getRCYaml(String namespace, String rcName) throws JsonProcessingException {
		ReplicationController gotRc = getRC(namespace, rcName);
		String str = SerializationUtils.dumpAsYaml(gotRc);
		return str;
	}

	public static boolean delService(String namespace, String serviceName) {
		Boolean optResult = getClient().services().inNamespace(getUseNamespaceName(namespace)).withName(serviceName)
				.delete();
		return optResult;
	}

	public static boolean delService(String serviceName) {
		return delService(null, serviceName);
	}

	public static Result createNamespace(String namespace) {
		try {
			Namespace namespaceRet = getClient().namespaces()
					.create(new NamespaceBuilder().withNewMetadata().withName(namespace).endMetadata().build());
			return Result.getSuc().setRetObjs(namespaceRet);
		} catch (Exception e) {
			return Result.getError(e.getMessage());
		}
	}

	public static Result delNamespace(String namespace) {
		Boolean retobj = getClient().namespaces().withName("test").delete();
		if (retobj != null && retobj) {
			return Result.getSuc();
		} else {
			return Result.getError("删除名字空间失败");
		}
	}

	public static boolean delPodByName(String namespace, String podName) {
		return getClient().pods().inNamespace(getUseNamespaceName(namespace)).withName(podName).delete();
	}

	public static boolean delPodByName(String podName) {
		return delPodByName(null, podName);
	}

	public static boolean delPodByLabel(String namespace, String key, String value) {
		return getClient().pods().inNamespace(getUseNamespaceName(namespace)).withLabel(key, value).delete();
	}

	public static boolean delPodByLabel(String key, String value) {
		return delPodByLabel(null, key, value);
	}

	public static void createRc(String namespace, String filePath) throws ProjectException {
		FileInputStream inputstream;
		try {
			inputstream = new FileInputStream(filePath);
		} catch (FileNotFoundException e) {
			throw new ProjectException(ExceptAll.Project_default, "文件不存在");
		}
		createRc(namespace, inputstream);
	}

	/***
	 * 跟据文件创建RC
	 * 
	 * @param fileInputStream 输入流
	 * @throws ProjectException
	 */
	public static void createRc(String namespace, FileInputStream fileInputStream) throws ProjectException {
		JSONObject jsonObject = YamlAssist.readYaml(fileInputStream);
		// DoneableReplicationController rc =
		ReplicationController rc = getClient().replicationControllers().inNamespace(getUseNamespaceName(namespace))
				.create();
		if (jsonObject.containsKey("metadata")) {
			JSONObject metadataJson = jsonObject.getJSONObject("metadata");
			if (!metadataJson.containsKey("name")) {
				throw new ProjectException(ExceptAll.param_error, "metadata需要有name元素");
			}
			ObjectMeta metadata = new ObjectMeta();
			metadata.setName(metadataJson.getString("name"));
			if (metadataJson.containsKey("labels")) {
				JSONObject labels = metadataJson.getJSONObject("labels");
				Map<String, String> lables = new HashMap<String, String>();
				for (String key : labels.keySet()) {
					lables.put(key, labels.getString(key));
				}
				metadata.setLabels(lables);
			}
			rc.setMetadata(metadata);
		}
		JSONObject specJson = jsonObject.getJSONObject("spec");
		ReplicationControllerSpec spec = rc.getSpec();
		if (specJson.containsKey("replicas")) {
			spec.setReplicas(specJson.getInteger("replicas"));
		}
		if (specJson.containsKey("selector")) {
			Map<String, String> selector = JSONObject.toJavaObject(specJson.getJSONObject("selector"), Map.class);
			spec.setSelector(selector);
		}

		PodTemplateSpec newTemplate = spec.getTemplate();
		JSONObject templateJson = specJson.getJSONObject("template");
		if (templateJson.containsKey("metadata")) {
			ObjectMeta withNewMetadata = newTemplate.getMetadata();
			JSONObject jsonObject2 = templateJson.getJSONObject("metadata").getJSONObject("labels");
			Map<String, String> labels = JSON.toJavaObject(jsonObject2, Map.class);
			withNewMetadata.setLabels(labels);
		}

		PodSpec innerSpec = newTemplate.getSpec();
		JSONObject innerSpecJson = templateJson.getJSONObject("spec");
		JSONArray containersJson = innerSpecJson.getJSONArray("containers");
		for (int i = 0; i < containersJson.size(); i++) {
			JSONObject containerJson = containersJson.getJSONObject(i);
			Container addNewContainer = new Container();
			addNewContainer.setName(containerJson.getString("name"));
			addNewContainer.setImage(containerJson.getString("image"));
			JSONArray portAry = containerJson.getJSONArray("ports");
			List<ContainerPort> ports = new ArrayList<ContainerPort>();
			for (int j = 0; j < portAry.size(); j++) {
				ContainerPort temport = new ContainerPort();
				temport.setContainerPort(portAry.getJSONObject(i).getInteger("containerPort"));
				ports.add(temport);
			}
			addNewContainer.setPorts(ports);
		}
	}

	/***
	 * 创建Service
	 * 
	 * @param namespace
	 * @param fileInputStream
	 * @throws ProjectException
	 */
	@SuppressWarnings("unchecked")
	public static void createSvc(String namespace, FileInputStream fileInputStream) throws ProjectException {
		JSONObject jsonObject = YamlAssist.readYaml(fileInputStream);
		Service createNew = getClient().services().inNamespace(getUseNamespaceName(namespace)).create();
		JSONObject metadata = jsonObject.getJSONObject("metadata");

		ObjectMeta meta = new ObjectMeta();
		meta.setName(metadata.getString("name"));
		createNew.setMetadata(meta);
		JSONObject specjson = jsonObject.getJSONObject("spec");
		ServiceSpec withNewSpec = createNew.getSpec();
		if (specjson.containsKey("type")) {
			withNewSpec.setType(specjson.getString("type"));
		}

		if (specjson.containsKey("selector")) {
			Map<String, String> seleMap = JSON.toJavaObject(specjson.getJSONObject("selector"), Map.class);
			withNewSpec.setSelector(seleMap);
		}
		if (specjson.containsKey("ports")) {
			JSONArray array = specjson.getJSONArray("ports");
			List<ServicePort> ports = new ArrayList<>();
			for (int i = 0; i < array.size(); i++) {
				JSONObject tempobj = array.getJSONObject(i);
				ServicePort temp = new ServicePort();
				if (tempobj.containsKey("name")) {
					temp.setName(tempobj.getString("name"));
				}
				if (tempobj.containsKey("nodePort")) {
					temp.setNodePort(tempobj.getInteger("nodePort"));
				}
				if (tempobj.containsKey("port")) {
					temp.setPort(tempobj.getInteger("port"));
				}
				if (tempobj.containsKey("protocol")) {
					temp.setProtocol(tempobj.getString("protocol"));
				}
				if (tempobj.containsKey("targetPort")) {
					IntOrString input = new IntOrString(tempobj.getInteger("targetPort"));
					temp.setTargetPort(input);
				}
				ports.add(temp);
			}
			withNewSpec.setPorts(ports);
		}
	}

	public static void createSvc(String namespace, String filePath) throws ProjectException {
		FileInputStream inputstream;
		try {
			inputstream = new FileInputStream(filePath);
		} catch (FileNotFoundException e) {
			throw new ProjectException(ExceptAll.Project_default, "文件不存在");
		}
		createSvc(namespace, inputstream);
	}

	/***
	 * 创建命名空间配额
	 * 
	 * @param namespace
	 * @param minCpu
	 * @param minMemory
	 * @param maxCpu
	 * @param maxMemory
	 * @param pods
	 */
	public static void createResourcequota(String namespace, int minCpu, int minMemory, int maxCpu, int maxMemory,
			int pods) {
		Map<String, Quantity> parma = new HashMap<>();
		parma.put("pods", new Quantity(String.valueOf(pods)));
		parma.put("requests.cpu", new Quantity(String.valueOf(minCpu)));
		parma.put("requests.memory", new Quantity(String.valueOf(minMemory)));
		parma.put("limits.cpu", new Quantity(String.valueOf(maxCpu)));
		parma.put("limits.memory", new Quantity(String.valueOf(maxMemory)));
		ResourceQuota quota = new ResourceQuotaBuilder().withNewMetadata().withName(getUseNamespaceName(namespace))
				.endMetadata().withNewSpec().addToHard(parma).endSpec().build();
		getClient().resourceQuotas().inNamespace(getUseNamespaceName(namespace)).create(quota);
	}

	/**
	 * 删除命名空间配额
	 * 
	 * @param namespace
	 */
	public static void delResourcequota(String namespace) {
		getClient().resourceQuotas().inNamespace(getUseNamespaceName(namespace)).delete();
	}

	public static boolean delRc(String namespace, String rcName) {
		return getClient().replicationControllers().inNamespace(getUseNamespaceName(namespace)).withName(rcName)
				.delete();
	}

	/***
	 * 删除svc
	 * 
	 * @param namespace
	 * @param svcName
	 * @return
	 */
	public static boolean delSvc(String namespace, String svcName) {
		return getClient().services().inNamespace(getUseNamespaceName(namespace)).withField("metadata.name", svcName)
				.delete();
	}

	/***
	 * 更新容器的image
	 * 
	 * @param namespace
	 * @param rcName
	 * @param containerIndex 这个pod的第几个容器
	 * @param imageValue
	 */
	public static void updateRcImage(String namespace, String rcName, int containerIndex, String imageValue) {
		ReplicationControllerSpec spec = getClient().replicationControllers()
				.inNamespace(getUseNamespaceName(namespace)).withName(rcName).edit().getSpec();
		Container container = spec.getTemplate().getSpec().getContainers().get(containerIndex);
		container.setImage(imageValue);
		// getClient().replicationControllers().inNamespace((getUseNamespaceName(namespace))).withName(rcName).rolling().updateImage(imageValue);
	}

	protected abstract Object setImage(String imageValue);

	/***
	 * 滚动更新image
	 * 
	 * @param namespace
	 * @param rcName
	 * @param imageValue
	 * @return
	 */
	public static Result updateImageRolling(String namespace, String rcName, String imageValue) {
		try {
			getClient().replicationControllers().inNamespace(namespace).withName(rcName).rolling()
					.updateImage(imageValue);
			return Result.getSuc();
		} catch (Exception e) {
			return Result.getError(e.getMessage());
		}
	}

	/***
	 * 创建或更新RC
	 * 
	 * @param namespace  命名空间
	 * @param imageValue image值
	 * @param rcPath     rc的文件路径
	 * @return 创建或更新成功
	 */
	public static Result createOrUpdateImageRolling(String namespace, String imageValue, String rcPath) {
		JSONObject json = YamlAssist.readYaml(rcPath);
		String rcName = json.getJSONObject("metadata").getString("name");
		ReplicationController queryRc = getRC(namespace, rcName);
		if (queryRc == null) {
			try {
				createRc(namespace, rcPath);
				return Result.getSuc();
			} catch (ProjectException e) {
				return Result.getError(e.getMessage());
			}
		} else {
			return updateImageRolling(namespace, rcName, imageValue);
		}
	}
}
