/*
 * Copyright 2004-2007 Gary Bentley 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may 
 * not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 *    http://www.apache.org/licenses/LICENSE-2.0 
 *
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 */
package org.josql.utils;

import java.util.List;
import java.util.Comparator;

import org.josql.Query;
import org.josql.QueryParseException;
import org.josql.QueryExecutionException;

import org.josql.internal.ListExpressionComparator;


public class JoSQLComparator implements Comparator
{

    private Query q = null;
    private Exception exp = null;
    private ListExpressionComparator c = null;

    /**
     * Execute the EXECUTE ON ALL expressions.  
     *
     * @param l The list to execute the expressions on.
     */
    public void doExecuteOn (List   l)
	                     throws QueryExecutionException
    {

	this.q.doExecuteOn (l,
			    Query.ALL);

    }

    /**
     * Clear the cache, it is VITAL that you call this method before you use
     * the comparator (if it has been used before) otherwise data objects will
     * be "left around" and preventing the GC from cleaning them up.
     */
    public void clearCache ()
    {

	if (this.q != null)
	{

	    this.c.clearCache ();

	}

    }

    /**
     * Return whether this comparator uses caching to improve performance.
     *
     * @return <code>true</code> if caching is on.
     * @throws IllegalStateException If the query has not yet been parsed or set.
     */
    public boolean isCaching ()
	                     throws  IllegalStateException
    {

	if ((this.q == null)
	    ||
	    (!this.q.parsed ())
	   )
	{

	    throw new IllegalStateException ("Query has not yet been parsed.");

	}

	return this.c.isCaching ();

    }

    /**
     * Set whether the comparator should use caching to improve performance.
     *
     * @param b Set to <code>true</code> to turn caching on.
     * @throws IllegalStateException If the query has not yet been parsed or set.
     */
    public void setCaching (boolean b)
	                    throws  IllegalStateException
    {

	if ((this.q == null)
	    ||
	    (!this.q.parsed ())
	   )
	{

	    throw new IllegalStateException ("Query has not yet been parsed.");

	}

	this.c.setCaching (b);

    }

    /**
     * Init this filter with the query.
     * 
     * @param q The query.
     * @throws QueryParseException If there is an issue with the parsing of the query.
     */
    public JoSQLComparator (String  q)
	                    throws  QueryParseException
    {

	this.setQuery (q);

    }

    /**
     * Compares the objects as according to the ORDER BY clause.
     *
     * @param o1 The first object.
     * @param o2 The second object.
     */
    public int compare (Object o1,
			Object o2)
    {

	try
	{

	    return c.ci (o1,
			 o2);

	} catch (Exception e) {

	    this.exp = e;

	    return 0;

	}

    }

    /**
     * Init this file filter with the query already built and parsed.
     * 
     * @param q The query.
     * @throws IllegalStateException If the Query object has not been parsed.
     * @throws QueryParseException If the FROM class is not as expected.
     */
    public JoSQLComparator (Query   q)
	                    throws  IllegalStateException,
	                            QueryParseException
    {

	this.setQuery (q);

    }

    /**
     * The {@link Comparator#compare(Object,Object)} method does not allow for 
     * any exceptions to be thrown however since the execution of the ORDER BY clause 
     * on the objects can cause the throwing of a {@link QueryParseException} it should 
     * be captured.  If the exception is thrown then this method will return it.  
     *
     * @return The exception thrown by the execution of the ORDER BY clause in {@link #compare(Object,Object)}
     *         or by sub-class/interface specific methods, this may be null if no exception was thrown.
     */
    public Exception getException ()
    {

	return this.exp;

    }

    /**
     * Set a new Query (string form) for use in this filter.
     *
     * @param q The Query to use.
     * @throws QueryParseException If there is an issue with the parsing of the query, 
     *                             or if the FROM class is not as expected.
     */
    public void setQuery (String  q)
	                  throws  QueryParseException
    {

	this.q = new Query ();
	this.q.parse (q);

	this.c = (ListExpressionComparator) this.q.getOrderByComparator ();

	this.exp = null;

    }

    /**
     * Set a new Query object for use in this filter.
     *
     * @param q The Query to use.
     * @throws IllegalStateException If the Query object has not been parsed.
     * @throws QueryParseException If the FROM class is not as expected.
     */
    public void setQuery (Query   q)
	                  throws  IllegalStateException,
	                          QueryParseException
    {

	if (!q.parsed ())
	{

	    throw new IllegalStateException ("Query has not yet been parsed.");

	}

	this.q = q;

	this.c = (ListExpressionComparator) this.q.getOrderByComparator ();

	this.exp = null;

    }

    /**
     * Get the Query we are using to process objects.
     *
     * @return The Query.
     */
    public Query getQuery ()
    {

	return this.q;

    }

}
