/*
 * Copyright 2004-2007 Gary Bentley 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may 
 * not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 *    http://www.apache.org/licenses/LICENSE-2.0 
 *
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 */
package org.josql.functions;

import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;

import java.util.List;
import java.util.ArrayList;

import org.josql.functions.regexp.RegExpFactory;
import org.josql.functions.regexp.RegExp;

import org.josql.QueryExecutionException;

/**
 * This class holds functions that operate on strings in some way.
 */
public class StringFunctions extends AbstractFunctionHandler
{

    public static final String HANDLER_ID = "_internal_string";

    private RegExpFactory regExpF = null;

    private void initRegExpFactory ()
    {
        
        this.regExpF = new RegExpFactory (this.q);
        
    }


    public boolean regexp (Object o,
			   String re)
	                   throws QueryExecutionException
    {

        if (this.regExpF == null)
        {
            
            this.initRegExpFactory ();
            
        }

	RegExp regexp = this.regExpF.getDefaultInstance (this.q);

	if (regexp == null)
	{

	    throw new QueryExecutionException ("No default regular expression library available for: " +
					       this.regExpF.getDefaultInstanceName ());

	}

	if (o == null)
	{

	    return false;

	}

	String v = o.toString ();

	return regexp.match (re,
			     v);

    }


    public boolean regexp (Object o,
			   String re,
			   String instName)
	                   throws QueryExecutionException
    {

        if (this.regExpF == null)
        {
            
            this.initRegExpFactory ();
            
        }

	RegExp regexp = this.regExpF.getInstance (instName,
                                                  this.q);

	if (regexp == null)
	{

	    throw new QueryExecutionException ("No regular expression library available for: " +
					       instName);

	}

	if (o == null)
	{

	    return false;

	}

	String v = o.toString ();

	return regexp.match (re,
			     v);

    }


    public List grep (File    f,
		      String  s,
		      boolean ignoreCase)
	              throws  QueryExecutionException
    {

	if ((f == null)
	    ||
	    (!f.exists ())
	    ||
	    (f.isDirectory ())
	    ||
	    (!f.canRead ())
	   )
	{

	    return null;

	}

	List retData = new ArrayList ();

	try
	{

	    BufferedReader br = new BufferedReader (new FileReader (f));
	
	    String l = br.readLine ();

	    int lc = 1;
	    
	    String ss = s;

	    if (ignoreCase)
	    {

		ss = s.toLowerCase ();

	    }

	    while (l != null)
	    {

		int ind = -1;

		if (ignoreCase)
		{

		    ind = l.toLowerCase ().indexOf (ss);

		} else {

		    ind = l.indexOf (ss);
		
		}

		if (ind != -1)
		{

		    retData.add (new FileMatch (f,
						lc,
						ind,
						s,
						l));
		    
		}
		
		l = br.readLine ();
		
		lc++;

	    }

	    br.close ();

	} catch (Exception e) {

	    throw new QueryExecutionException ("Unable to read from file: " +
					       f,
					       e);

	}

	return retData;

    }


    public List rgrep (File    f,
		       String  regexp,
		       String  instName)
	               throws  QueryExecutionException
    {

        if (this.regExpF == null)
        {
            
            this.initRegExpFactory ();
            
        }

	RegExp reImpl = this.regExpF.getInstance (instName,
                                                  this.q);

	if (reImpl == null)
	{

	    throw new QueryExecutionException ("No default regular expression library available for: " +
					       instName);

	}

	return this.rgrep (f,
			   regexp,
			   reImpl);

    }


    public List rgrep (File    f,
		       String  regexp)
	               throws  QueryExecutionException
    {

        if (this.regExpF == null)
        {
            
            this.initRegExpFactory ();
            
        }

	RegExp reImpl = this.regExpF.getDefaultInstance (this.q);

	if (reImpl == null)
	{

	    throw new QueryExecutionException ("No default regular expression library available for: " +
					       this.regExpF.getDefaultInstanceName ());

	}

	return this.rgrep (f,
			   regexp,
			   reImpl);

    }

    private List rgrep (File    f,
			String  regexp,
			RegExp  reImpl)
	                throws  QueryExecutionException
    {

	if ((f == null)
	    ||
	    (!f.exists ())
	    ||
	    (f.isDirectory ())
	    ||
	    (!f.canRead ())
	   )
	{

	    return null;

	}

	List retData = new ArrayList ();

	try
	{

	    BufferedReader br = new BufferedReader (new FileReader (f));
	
	    String l = br.readLine ();

	    int lc = 1;
	    
	    while (l != null)
	    {

		if (reImpl.match (regexp,
				  l))
		{

		    retData.add (new FileMatch (f,
						lc,
						-1,
						regexp,
						l));
		    
		}
		
		l = br.readLine ();
		
		lc++;

	    }

	    br.close ();

	} catch (Exception e) {

	    throw new QueryExecutionException ("Unable to read from file: " +
					       f,
					       e);

	}

	return retData;

    }

    public List grep (File   f,
		      String s)
	              throws QueryExecutionException
    {

	return this.grep (f,
			  s,
			  false);

    }

  
    public String trim (Object o,
			Object v)
    {

	if (o == null)
	{

	    return null;

	}

	String os = o.toString ();

	if (v == null)
	{

	    return os.trim ();

	}

	String vs = v.toString ();

	if (os.endsWith (vs))
	{

	    os = os.substring (0,
			       vs.length ());

	}

	if (os.startsWith (vs))
	{

	    os = os.substring (vs.length ());

	}

	return os;

    }

    public double lastIndexOf (Object o,
			       Object i)
    {

	if (o == null)
	{

	    return -1;

	}

	if (i == null)
	{

	    return -1;

	}

	String os = o.toString ();
	String is = i.toString ();

	return os.lastIndexOf (is);

    }  

    /**
     * Return a substring of the passed in object (in a string form).  See {@link #subStr(Object,double,double)}
     * for the full details since this is just a thin-wrapper around that method with the <b>t</b>
     * parameter set to -1.
     *
     * @param o The object to convert to a string and return the substring.
     * @param f The start index.  If this is set to 0 then the entire string is returned.  
     * @return The substring.
     */
    public String subStr (Object o,
			  double f)
    {

	return this.subStr (o,
			    f,
			    -1);

    }


    public String subStr (Object o,
			  double f,
			  double t)
    {

	if (o == null)
	{

	    return null;

	}

	int fi = (int) f;
	int ti = (int) t;

	String s = o.toString ();

	if ((fi < 0)
	    ||
	    (fi > s.length ())
	   )
	{

	    return "";

	}

	if ((ti < fi)
	    ||
	    (ti > s.length ())
	   )
	{

	    ti = -1;

	}

	if (ti == -1)
	{

	    return s.substring (fi);

	}

	return s.substring (fi,
			    ti);

    }

    public double length (Object o)
    {

	if (o == null)
	{

	    return 0;

	}

	if (o instanceof String)
	{

	    return ((String) o).length ();

	}

	return o.toString ().length ();

    }

}
