/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.wicp.tams.common.flink.sqlgateway.jdbc;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.RowIdLifetime;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.tuple.Pair;

import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.constant.StrPattern;
import net.wicp.tams.common.http.flink.FlinkAssit;
import net.wicp.tams.common.http.flink.SqlCmd;
import net.wicp.tams.common.http.flink.SqlGateResultPo;
import net.wicp.tams.common.http.flink.SqlGateWay;

/**
 * Flink JDBC database meta data.
 */
public class FlinkDatabaseMetaData implements DatabaseMetaData {

	private static final String[] SUPPORTED_TABLE_TYPES = new String[] { "TABLE", "VIEW" };
	public static final String DRIVER_VERSION = "0.1";
	public static final String JDBC_VERSION = "4.2";

	private final SqlGateWay sqlGateWay;
	private final FlinkConnection flinkConnection;

	private Pair<String, String> info;// L:productName,R:1.16.0。示例 {"productName":"Apache Flink","version":"1.16.0"}

	public FlinkDatabaseMetaData(SqlGateWay sqlGateWay, FlinkConnection flinkConnection) {
		this.sqlGateWay = sqlGateWay;
		this.flinkConnection = flinkConnection;
	}

	@Override
	public boolean allTablesAreSelectable() throws SQLException {
		return true;
	}

	@Override
	public String getURL() throws SQLException {
		return SymbolConstants.URL_PREFIX + sqlGateWay.getHost() + ":" + sqlGateWay.getPort();
	}

	@Override
	public String getUserName() throws SQLException {
		return null;
	}

	@Override
	public String getDatabaseProductName() throws SQLException {
		return sqlGateWay.getInfo().getLeft();
	}

	@Override
	public String getDatabaseProductVersion() throws SQLException {
		return sqlGateWay.getInfo().getRight();
	}

	@Override
	public String getDriverName() throws SQLException {
		return "Flink Driver";
	}

	@Override
	public String getDriverVersion() throws SQLException {
		return SymbolConstants.DRIVER_VERSION;
	}

	@Override
	public int getDriverMajorVersion() {
		return Integer.valueOf(SymbolConstants.DRIVER_VERSION.split("\\.")[0]);
	}

	@Override
	public int getDriverMinorVersion() {
		return Integer.valueOf(SymbolConstants.DRIVER_VERSION.split("\\.")[1]);
	}

	@Override
	public boolean usesLocalFiles() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#usesLocalFiles is not supported");
	}

	@Override
	public boolean usesLocalFilePerTable() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#usesLocalFilePerTable is not supported");
	}

	@Override
	public boolean supportsMixedCaseIdentifiers() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsMixedCaseIdentifiers is not supported");
	}

	@Override
	public boolean storesUpperCaseIdentifiers() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#storesUpperCaseIdentifiers is not supported");
	}

	@Override
	public boolean storesLowerCaseIdentifiers() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#storesLowerCaseIdentifiers is not supported");
	}

	@Override
	public boolean storesMixedCaseIdentifiers() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#storesMixedCaseIdentifiers is not supported");
	}

	@Override
	public boolean supportsMixedCaseQuotedIdentifiers() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsMixedCaseQuotedIdentifiers is not supported");
	}

	@Override
	public boolean storesUpperCaseQuotedIdentifiers() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#storesUpperCaseQuotedIdentifiers is not supported");
	}

	@Override
	public boolean storesLowerCaseQuotedIdentifiers() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#storesLowerCaseQuotedIdentifiers is not supported");
	}

	@Override
	public boolean storesMixedCaseQuotedIdentifiers() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#storesMixedCaseQuotedIdentifiers is not supported");
	}

	@Override
	public String getIdentifierQuoteString() throws SQLException {
		return "`";
	}

	// schemaPattern 参数没有作用。
	@Override
	public ResultSet getTables(String catalog, String schemaPattern, String tableNamePattern, String[] types)
			throws SQLException {

		if (tableNamePattern == null) {
			tableNamePattern = "%";
		}
		if (types == null) {
			types = SUPPORTED_TABLE_TYPES;
		} else {
			List<String> last = new ArrayList<String>();
			for (String type : types) {
				if (ArrayUtils.contains(SUPPORTED_TABLE_TYPES, type)) {
					last.add(type);
				}
			}
			types = last.toArray(new String[last.size()]);
		}
		List<Map<String, String>> datas = new ArrayList<Map<String, String>>();
		if ("".equals(catalog) || "".equals(schemaPattern)) {
			return new FlinkResultSet(datas, new String[] { "TABLE_CAT", "TABLE_SCHEM", "TABLE_NAME", "TABLE_TYPE",
					"REMARKS", "TYPE_CAT", "TYPE_SCHEM", "TYPE_NAME", "SELF_REFERENCING_COL_NAME", "REF_GENERATION" });
		}
		String oldCatalog = flinkConnection.getCatalog();
		String oldDatabase = flinkConnection.getSchema();

		Pattern javaPattern = Pattern.compile(tableNamePattern.replace("%", ".*").replace("_", ".?"));
		ResultSet databaseResult = getSchemas(catalog, schemaPattern);
		while (databaseResult.next()) {
			String catalogTemp = databaseResult.getString("TABLE_CATALOG");
			String dbTemp = databaseResult.getString("TABLE_SCHEM");
			flinkConnection.setCatalog(catalogTemp);
			flinkConnection.setSchema(dbTemp);

			for (int i = 0; i < types.length; i++) {
				int indexOf = ArrayUtils.indexOf(SUPPORTED_TABLE_TYPES, types[i]);
				SqlGateResultPo rsTemp = indexOf == 0 ? this.sqlGateWay.exec(SqlCmd.tb_show)
						: this.sqlGateWay.exec(SqlCmd.view_show);
				for (String tbele : rsTemp.getSimpleData()) {
					if (javaPattern.matcher(tbele).matches()) {
						Map<String, String> data = new HashMap<String, String>();
						data.put("TABLE_CAT", catalog);
						data.put("TABLE_SCHEM", dbTemp);
						data.put("TABLE_NAME", tbele);
						data.put("TABLE_TYPE", SUPPORTED_TABLE_TYPES[indexOf]);
						datas.add(data);
					}
				}
			}

		}
		flinkConnection.setCatalog(oldCatalog);
		flinkConnection.setSchema(oldDatabase);
		return new FlinkResultSet(datas, new String[] { "TABLE_CAT", "TABLE_SCHEM", "TABLE_NAME", "TABLE_TYPE",
				"REMARKS", "TYPE_CAT", "TYPE_SCHEM", "TYPE_NAME", "SELF_REFERENCING_COL_NAME", "REF_GENERATION" });
	}

	@Override
	public ResultSet getSchemas() throws SQLException {
		return getSchemas(null, null);
	}

	@Override
	public ResultSet getCatalogs() throws SQLException {
		SqlGateResultPo po = this.sqlGateWay.execPutCols(SqlCmd.catalog_show, new String[] { "TABLE_CAT" });//
		if (!po.isSuc()) {
			throw new SQLException(po.getErrors());
		}
		FlinkResultSet resultSet = new FlinkResultSet(po);
		return resultSet;
	}

	@Override
	public ResultSet getTableTypes() throws SQLException {
		List<Map<String, String>> datas = new ArrayList<Map<String, String>>();
		for (String type : SUPPORTED_TABLE_TYPES) {
			datas.add(CollectionUtil.newMapStr("TABLE_TYPE", type));
		}
		return new FlinkResultSet(datas, new String[] { "TABLE_TYPE" });
	}

	@Override
	public ResultSet getColumns(String catalog, String schemaPattern, String tableNamePattern, String columnNamePattern)
			throws SQLException {
		if (columnNamePattern == null) {
			columnNamePattern = "%";
		}
		List<Map<String, String>> datas = new ArrayList<Map<String, String>>();
		ResultSet tableResult = getTables(catalog, schemaPattern, tableNamePattern, SUPPORTED_TABLE_TYPES);
		Pattern javaPattern = Pattern.compile(columnNamePattern.replace("%", ".*").replace("_", ".?"));
		while (tableResult.next()) {
			SqlGateResultPo exec = this.sqlGateWay.exec(SqlCmd.col_show,
					String.format("%s.%s.%s", tableResult.getString("TABLE_CAT"), tableResult.getString("TABLE_SCHEM"),
							tableResult.getString("TABLE_NAME")));
			int index = 1;
			while (exec.next(null)) {
				String colName = exec.getDataColByCurRow("name");
				if (javaPattern.matcher(colName).matches()) {
					Map<String, String> data = new HashMap<String, String>();
					data.put("TABLE_CAT", tableResult.getString("TABLE_CAT"));
					data.put("TABLE_SCHEM", tableResult.getString("TABLE_SCHEM"));
					data.put("TABLE_NAME", tableResult.getString("TABLE_NAME"));
					data.put("COLUMN_NAME", colName);
					// eg:DECIMAL(10, 2) 或 STRING
					String typestr = exec.getDataColByCurRow("type");
					String[] group = StrPattern.coltype.group(typestr,
							new String[] { "datatype", "precision", "scale" });
					int sqlType = FlinkAssit.getSqlType(group[0]);
					data.put("DATA_TYPE", String.valueOf(sqlType));
					data.put("TYPE_NAME", group[0]);
					if (StringUtil.isNotNull(group[1])) {
						data.put("COLUMN_SIZE", group[1]);// TODO int类型
					}
					if (StringUtil.isNotNull(group[2])) {
						data.put("DECIMAL_DIGITS", group[2]);// TODO short类型
					}
					boolean isNotNull = "FALSE".equalsIgnoreCase(exec.getDataColByCurRow("null"));
					data.put("IS_NULLABLE", isNotNull ? "NO" : "YES");
					data.put("NULLABLE", String
							.valueOf(isNotNull ? DatabaseMetaData.columnNoNulls : DatabaseMetaData.columnNullable));
					data.put("ORDINAL_POSITION", String.valueOf(index++));

					data.put("IS_AUTOINCREMENT", "NO");// 自动增加
					data.put("IS_GENERATEDCOLUMN", "NO");// 自动生成
					data.put("NUM_PREC_RADIX", "10");// 基数

					datas.add(data);
				}
			}

		}
		return new FlinkResultSet(datas,
				new String[] { "TABLE_CAT", "TABLE_SCHEM", "TABLE_NAME", "COLUMN_NAME", "DATA_TYPE", "TYPE_NAME",
						"COLUMN_SIZE", "BUFFER_LENGTH", "DECIMAL_DIGITS", "NUM_PREC_RADIX", "NULLABLE", "REMARKS",
						"COLUMN_DEF", "SQL_DATA_TYPE", "SQL_DATETIME_SUB", "CHAR_OCTET_LENGTH", "ORDINAL_POSITION",
						"IS_NULLABLE", "SCOPE_CATALOG", "SCOPE_SCHEMA", "SCOPE_TABLE", "SOURCE_DATA_TYPE",
						"IS_AUTOINCREMENT", "IS_GENERATEDCOLUMN" });
	}

	@Override
	public ResultSet getPrimaryKeys(String catalog, String schema, String table) throws SQLException {
		// "TABLE_CAT","TABLE_SCHEM","TABLE_NAME","COLUMN_NAME","KEY_SEQ","PK_NAME"
		List<Map<String, String>> datas = new ArrayList<Map<String, String>>();
		SqlGateResultPo sqlGateResultPo = this.sqlGateWay.exec(SqlCmd.tb_desc,
				String.format("%s.%s.%s", catalog, schema, table));
		int index = 0;
		while (sqlGateResultPo.next(null)) {
			String keyStr = sqlGateResultPo.getDataColByCurRow("key");
			if (StringUtil.isNotNull(keyStr) && keyStr.startsWith("PRI")) {
				datas.add(CollectionUtil.newMapStr("TABLE_CAT", catalog, "TABLE_SCHEM", schema, "TABLE_NAME", table,
						"COLUMN_NAME", sqlGateResultPo.getDataColByCurRow("name"), "KEY_SEQ", String.valueOf(index++),
						"PK_NAME", sqlGateResultPo.getDataColByCurRow("name")));
			}
		}
		return new FlinkResultSet(datas,
				new String[] { "TABLE_CAT", "TABLE_SCHEM", "TABLE_NAME", "COLUMN_NAME", "KEY_SEQ", "PK_NAME" });
	}

	@Override
	public int getDatabaseMajorVersion() throws SQLException {
		if (info == null) {
			this.info = this.sqlGateWay.getInfo();
		}
		return Integer.valueOf(info.getRight().split("\\.")[0]);
	}

	@Override
	public int getDatabaseMinorVersion() throws SQLException {
		if (info == null) {
			this.info = this.sqlGateWay.getInfo();
		}
		return Integer.valueOf(info.getRight().split("\\.")[1]);
	}

	@Override
	public int getJDBCMajorVersion() throws SQLException {
		return Integer.valueOf(JDBC_VERSION.split("\\.")[0]);
	}

	@Override
	public int getJDBCMinorVersion() throws SQLException {
		return Integer.valueOf(JDBC_VERSION.split("\\.")[1]);
	}

	@Override
	public ResultSet getSchemas(String catalog, String schemaPattern) throws SQLException {
		if (schemaPattern == null) {
			schemaPattern = "%";
		}
		List<Map<String, String>> datas = new ArrayList<Map<String, String>>();
		if ("".equals(catalog)) {
			return new FlinkResultSet(datas, new String[] { "TABLE_SCHEM", "TABLE_CATALOG" });
		}
		List<String> catalogs = new ArrayList<String>();
		if (catalog == null) {
			ResultSet catalogResult = getCatalogs();
			while (catalogResult.next()) {
				catalogs.add(catalogResult.getString("TABLE_CAT"));//
			}
		} else {
			catalogs.add(catalog);
		}

		Pattern javaPattern = Pattern.compile(schemaPattern.replace("%", ".*").replace("_", ".?"));
		String oldCatalog = this.flinkConnection.getCatalog();
		for (String catalogQuery : catalogs) {
			flinkConnection.setCatalog(catalogQuery);
			SqlGateResultPo dblist = this.sqlGateWay.exec(SqlCmd.db_show);
			String[] simpleData = dblist.getSimpleData();
			for (String db : simpleData) {
				if (javaPattern.matcher(db).matches()) {
					datas.add(CollectionUtil.newMapStr("TABLE_SCHEM", db, "TABLE_CATALOG", catalogQuery));
				}
			}
		}
		this.flinkConnection.setCatalog(oldCatalog);
		return new FlinkResultSet(datas, new String[] { "TABLE_SCHEM", "TABLE_CATALOG" });
	}

	@Override
	public boolean isReadOnly() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#isReadOnly is not supported");
	}

	@Override
	public boolean nullsAreSortedHigh() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#nullsAreSortedHigh is not supported");
	}

	@Override
	public boolean nullsAreSortedLow() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#nullsAreSortedLow is not supported");
	}

	@Override
	public boolean nullsAreSortedAtStart() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#nullsAreSortedAtStart is not supported");
	}

	@Override
	public boolean nullsAreSortedAtEnd() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#nullsAreSortedAtEnd is not supported");
	}

	@Override
	public boolean allProceduresAreCallable() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#allProceduresAreCallable is not supported");
	}

	@Override
	public String getSQLKeywords() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getSQLKeywords is not supported");
	}

	@Override
	public String getNumericFunctions() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getNumericFunctions is not supported");
	}

	@Override
	public String getStringFunctions() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getStringFunctions is not supported");
	}

	@Override
	public String getSystemFunctions() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getSystemFunctions is not supported");
	}

	@Override
	public String getTimeDateFunctions() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getTimeDateFunctions is not supported");
	}

	@Override
	public String getSearchStringEscape() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getSearchStringEscape is not supported");
	}

	@Override
	public String getExtraNameCharacters() throws SQLException {
		// 后面可能 beeline
		return "";
	}

	@Override
	public boolean supportsAlterTableWithAddColumn() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsAlterTableWithDropColumn() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsColumnAliasing() throws SQLException {
		return true;
	}

	@Override
	public boolean nullPlusNonNullIsNull() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#nullPlusNonNullIsNull is not supported");
	}

	@Override
	public boolean supportsConvert() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsConvert is not supported");
	}

	@Override
	public boolean supportsConvert(int fromType, int toType) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsConvert is not supported");
	}

	@Override
	public boolean supportsTableCorrelationNames() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsDifferentTableCorrelationNames() throws SQLException {
		return false;
	}

	@Override
	public boolean supportsExpressionsInOrderBy() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsOrderByUnrelated() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsGroupBy() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsGroupByUnrelated() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsGroupByBeyondSelect() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsLikeEscapeClause() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsMultipleResultSets() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsMultipleResultSets is not supported");
	}

	@Override
	public boolean supportsMultipleTransactions() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsMultipleTransactions is not supported");
	}

	@Override
	public boolean supportsNonNullableColumns() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsMinimumSQLGrammar() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsMinimumSQLGrammar is not supported");
	}

	@Override
	public boolean supportsCoreSQLGrammar() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsCoreSQLGrammar is not supported");
	}

	@Override
	public boolean supportsExtendedSQLGrammar() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsExtendedSQLGrammar is not supported");
	}

	@Override
	public boolean supportsANSI92EntryLevelSQL() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsANSI92EntryLevelSQL is not supported");
	}

	@Override
	public boolean supportsANSI92IntermediateSQL() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsANSI92IntermediateSQL is not supported");
	}

	@Override
	public boolean supportsANSI92FullSQL() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsANSI92FullSQL is not supported");
	}

	@Override
	public boolean supportsIntegrityEnhancementFacility() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsIntegrityEnhancementFacility is not supported");
	}

	@Override
	public boolean supportsOuterJoins() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsFullOuterJoins() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsLimitedOuterJoins() throws SQLException {
		return true;
	}

	@Override
	public String getSchemaTerm() throws SQLException {
		return "database";
	}

	@Override
	public String getProcedureTerm() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getProcedureTerm is not supported");
	}

	@Override
	public String getCatalogTerm() throws SQLException {
		return "catalog";
	}

	@Override
	public boolean isCatalogAtStart() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#isCatalogAtStart is not supported");
	}

	@Override
	public String getCatalogSeparator() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getCatalogSeparator is not supported");
	}

	@Override
	public boolean supportsSchemasInDataManipulation() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsSchemasInProcedureCalls() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsSchemasInProcedureCalls is not supported");
	}

	@Override
	public boolean supportsSchemasInTableDefinitions() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsSchemasInIndexDefinitions() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsSchemasInIndexDefinitions is not supported");
	}

	@Override
	public boolean supportsSchemasInPrivilegeDefinitions() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsSchemasInPrivilegeDefinitions is not supported");
	}

	@Override
	public boolean supportsCatalogsInDataManipulation() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsCatalogsInProcedureCalls() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsCatalogsInProcedureCalls is not supported");
	}

	@Override
	public boolean supportsCatalogsInTableDefinitions() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsCatalogsInIndexDefinitions() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsCatalogsInIndexDefinitions is not supported");
	}

	@Override
	public boolean supportsCatalogsInPrivilegeDefinitions() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsCatalogsInPrivilegeDefinitions is not supported");
	}

	@Override
	public boolean supportsPositionedDelete() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsPositionedDelete is not supported");
	}

	@Override
	public boolean supportsPositionedUpdate() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsPositionedUpdate is not supported");
	}

	@Override
	public boolean supportsSelectForUpdate() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsSelectForUpdate is not supported");
	}

	@Override
	public boolean supportsStoredProcedures() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsStoredProcedures is not supported");
	}

	@Override
	public boolean supportsSubqueriesInComparisons() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsSubqueriesInExists() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsSubqueriesInIns() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsSubqueriesInQuantifieds() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsCorrelatedSubqueries() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsUnion() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsUnionAll() throws SQLException {
		return true;
	}

	@Override
	public boolean supportsOpenCursorsAcrossCommit() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsOpenCursorsAcrossCommit is not supported");
	}

	@Override
	public boolean supportsOpenCursorsAcrossRollback() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsOpenCursorsAcrossRollback is not supported");
	}

	@Override
	public boolean supportsOpenStatementsAcrossCommit() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsOpenStatementsAcrossCommit is not supported");
	}

	@Override
	public boolean supportsOpenStatementsAcrossRollback() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsOpenStatementsAcrossRollback is not supported");
	}

	@Override
	public int getMaxBinaryLiteralLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxBinaryLiteralLength is not supported");
	}

	@Override
	public int getMaxCharLiteralLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxCharLiteralLength is not supported");
	}

	@Override
	public int getMaxColumnNameLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxColumnNameLength is not supported");
	}

	@Override
	public int getMaxColumnsInGroupBy() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxColumnsInGroupBy is not supported");
	}

	@Override
	public int getMaxColumnsInIndex() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxColumnsInIndex is not supported");
	}

	@Override
	public int getMaxColumnsInOrderBy() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxColumnsInOrderBy is not supported");
	}

	@Override
	public int getMaxColumnsInSelect() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxColumnsInSelect is not supported");
	}

	@Override
	public int getMaxColumnsInTable() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxColumnsInTable is not supported");
	}

	@Override
	public int getMaxConnections() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxConnections is not supported");
	}

	@Override
	public int getMaxCursorNameLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxCursorNameLength is not supported");
	}

	@Override
	public int getMaxIndexLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxIndexLength is not supported");
	}

	@Override
	public int getMaxSchemaNameLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxSchemaNameLength is not supported");
	}

	@Override
	public int getMaxProcedureNameLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxProcedureNameLength is not supported");
	}

	@Override
	public int getMaxCatalogNameLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxCatalogNameLength is not supported");
	}

	@Override
	public int getMaxRowSize() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxRowSize is not supported");
	}

	@Override
	public boolean doesMaxRowSizeIncludeBlobs() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#doesMaxRowSizeIncludeBlobs is not supported");
	}

	@Override
	public int getMaxStatementLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxStatementLength is not supported");
	}

	@Override
	public int getMaxStatements() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxStatements is not supported");
	}

	@Override
	public int getMaxTableNameLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxTableNameLength is not supported");
	}

	@Override
	public int getMaxTablesInSelect() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxTablesInSelect is not supported");
	}

	@Override
	public int getMaxUserNameLength() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getMaxUserNameLength is not supported");
	}

	@Override
	public int getDefaultTransactionIsolation() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#getDefaultTransactionIsolation is not supported");
	}

	@Override
	public boolean supportsTransactions() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsTransactions is not supported");
	}

	@Override
	public boolean supportsTransactionIsolationLevel(int level) throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsTransactionIsolationLevel is not supported");
	}

	@Override
	public boolean supportsDataDefinitionAndDataManipulationTransactions() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsDataDefinitionAndDataManipulationTransactions is not supported");
	}

	@Override
	public boolean supportsDataManipulationTransactionsOnly() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsDataManipulationTransactionsOnly is not supported");
	}

	@Override
	public boolean dataDefinitionCausesTransactionCommit() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#dataDefinitionCausesTransactionCommit is not supported");
	}

	@Override
	public boolean dataDefinitionIgnoredInTransactions() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#dataDefinitionIgnoredInTransactions is not supported");
	}

	@Override
	public ResultSet getProcedures(String catalog, String schemaPattern, String procedureNamePattern)
			throws SQLException {
		if (procedureNamePattern == null) {
			procedureNamePattern = "%";
		}
		List<Map<String, String>> datas = new ArrayList<Map<String, String>>();
		if ("".equals(catalog) || "".equals(schemaPattern)) {
			return new FlinkResultSet(datas, new String[] { "PROCEDURE_CAT", "PROCEDURE_SCHEM", "PROCEDURE_NAME",
					"REMARKS", "PROCEDURE_TYPE", "SPECIFIC_NAME" });
		}
		String oldCatalog = flinkConnection.getCatalog();
		String oldDatabase = flinkConnection.getSchema();

		Pattern javaPattern = Pattern.compile(procedureNamePattern.replace("%", ".*").replace("_", ".?"));
		ResultSet databaseResult = getSchemas(catalog, schemaPattern);

		while (databaseResult.next()) {
			String catalogTemp = databaseResult.getString("TABLE_CATALOG");
			String dbTemp = databaseResult.getString("TABLE_SCHEM");
			flinkConnection.setCatalog(catalogTemp);
			flinkConnection.setSchema(dbTemp);

			SqlGateResultPo rsTemp = this.sqlGateWay.exec(SqlCmd.fun_show_user);
			for (String funele : rsTemp.getSimpleData()) {
				if (javaPattern.matcher(funele).matches()) {
					Map<String, String> data = new HashMap<String, String>();
					data.put("PROCEDURE_CAT", catalog);
					data.put("PROCEDURE_SCHEM", dbTemp);
					data.put("PROCEDURE_NAME", funele);
					data.put("SPECIFIC_NAME", String.format("%s.%s.%s", catalog, dbTemp, funele));
					datas.add(data);
				}
			}

		}
		flinkConnection.setCatalog(oldCatalog);
		flinkConnection.setSchema(oldDatabase);
		return new FlinkResultSet(datas, new String[] { "PROCEDURE_CAT", "PROCEDURE_SCHEM", "PROCEDURE_NAME", "REMARKS",
				"PROCEDURE_TYPE", "SPECIFIC_NAME" });
	}

	@Override
	public ResultSet getProcedureColumns(String catalog, String schemaPattern, String procedureNamePattern,
			String columnNamePattern) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getProcedureColumns is not supported");
	}

	@Override
	public ResultSet getColumnPrivileges(String catalog, String schema, String table, String columnNamePattern)
			throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getColumnPrivileges is not supported");
	}

	@Override
	public ResultSet getTablePrivileges(String catalog, String schemaPattern, String tableNamePattern)
			throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getTablePrivileges is not supported");
	}

	@Override
	public ResultSet getBestRowIdentifier(String catalog, String schema, String table, int scope, boolean nullable)
			throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getBestRowIdentifier is not supported");
	}

	@Override
	public ResultSet getVersionColumns(String catalog, String schema, String table) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getVersionColumns is not supported");
	}

	@Override
	public ResultSet getImportedKeys(String catalog, String schema, String table) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getImportedKeys is not supported");
	}

	@Override
	public ResultSet getExportedKeys(String catalog, String schema, String table) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getExportedKeys is not supported");
	}

	@Override
	public ResultSet getCrossReference(String parentCatalog, String parentSchema, String parentTable,
			String foreignCatalog, String foreignSchema, String foreignTable) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getCrossReference is not supported");
	}

	@Override
	public ResultSet getTypeInfo() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getTypeInfo is not supported");
	}

	@Override
	public ResultSet getIndexInfo(String catalog, String schema, String table, boolean unique, boolean approximate)
			throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getIndexInfo is not supported");
	}

	@Override
	public boolean supportsResultSetType(int type) throws SQLException {
		return type == ResultSet.TYPE_FORWARD_ONLY;
	}

	@Override
	public boolean supportsResultSetConcurrency(int type, int concurrency) throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsResultSetConcurrency is not supported");
	}

	@Override
	public boolean ownUpdatesAreVisible(int type) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#ownUpdatesAreVisible is not supported");
	}

	@Override
	public boolean ownDeletesAreVisible(int type) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#ownDeletesAreVisible is not supported");
	}

	@Override
	public boolean ownInsertsAreVisible(int type) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#ownInsertsAreVisible is not supported");
	}

	@Override
	public boolean othersUpdatesAreVisible(int type) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#othersUpdatesAreVisible is not supported");
	}

	@Override
	public boolean othersDeletesAreVisible(int type) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#othersDeletesAreVisible is not supported");
	}

	@Override
	public boolean othersInsertsAreVisible(int type) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#othersInsertsAreVisible is not supported");
	}

	@Override
	public boolean updatesAreDetected(int type) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#updatesAreDetected is not supported");
	}

	@Override
	public boolean deletesAreDetected(int type) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#deletesAreDetected is not supported");
	}

	@Override
	public boolean insertsAreDetected(int type) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#insertsAreDetected is not supported");
	}

	@Override
	public boolean supportsBatchUpdates() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsBatchUpdates is not supported");
	}

	@Override
	public ResultSet getUDTs(String catalog, String schemaPattern, String typeNamePattern, int[] types)
			throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getUDTs is not supported");
	}

	@Override
	public Connection getConnection() throws SQLException {
		return this.flinkConnection;
	}

	@Override
	public boolean supportsSavepoints() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsSavepoints is not supported");
	}

	@Override
	public boolean supportsNamedParameters() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsNamedParameters is not supported");
	}

	@Override
	public boolean supportsMultipleOpenResults() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsMultipleOpenResults is not supported");
	}

	@Override
	public boolean supportsGetGeneratedKeys() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsGetGeneratedKeys is not supported");
	}

	@Override
	public ResultSet getSuperTypes(String catalog, String schemaPattern, String typeNamePattern) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getSuperTypes is not supported");
	}

	@Override
	public ResultSet getSuperTables(String catalog, String schemaPattern, String tableNamePattern) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getSuperTypes is not supported");
	}

	@Override
	public ResultSet getAttributes(String catalog, String schemaPattern, String typeNamePattern,
			String attributeNamePattern) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getAttributes is not supported");
	}

	@Override
	public boolean supportsResultSetHoldability(int holdability) throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsResultSetHoldability is not supported");
	}

	@Override
	public int getResultSetHoldability() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getResultSetHoldability is not supported");
	}

	@Override
	public int getSQLStateType() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getSQLStateType is not supported");
	}

	@Override
	public boolean locatorsUpdateCopy() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#locatorsUpdateCopy is not supported");
	}

	@Override
	public boolean supportsStatementPooling() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#supportsStatementPooling is not supported");
	}

	@Override
	public RowIdLifetime getRowIdLifetime() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getRowIdLifetime is not supported");
	}

	@Override
	public boolean supportsStoredFunctionsUsingCallSyntax() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#supportsStoredFunctionsUsingCallSyntax is not supported");
	}

	@Override
	public boolean autoCommitFailureClosesAllResultSets() throws SQLException {
		throw new SQLFeatureNotSupportedException(
				"FlinkDatabaseMetaData#autoCommitFailureClosesAllResultSets is not supported");
	}

	@Override
	public ResultSet getClientInfoProperties() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getClientInfoProperties is not supported");
	}

	@Override
	public ResultSet getFunctions(String catalog, String schemaPattern, String functionNamePattern)
			throws SQLException {
		return getProcedures(catalog, schemaPattern, functionNamePattern);
	}

	@Override
	public ResultSet getFunctionColumns(String catalog, String schemaPattern, String functionNamePattern,
			String columnNamePattern) throws SQLException {
		return getProcedureColumns(catalog, schemaPattern, functionNamePattern, columnNamePattern);
	}

	@Override
	public ResultSet getPseudoColumns(String catalog, String schemaPattern, String tableNamePattern,
			String columnNamePattern) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#getPseudoColumns is not supported");
	}

	@Override
	public boolean generatedKeyAlwaysReturned() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#generatedKeyAlwaysReturned is not supported");
	}

	@Override
	public <T> T unwrap(Class<T> iface) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#unwrap is not supported");
	}

	@Override
	public boolean isWrapperFor(Class<?> iface) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkDatabaseMetaData#isWrapperFor is not supported");
	}

}
