/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.wicp.tams.common.flink.sqlgateway.jdbc;

import java.sql.Array;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.NClob;
import java.sql.PreparedStatement;
import java.sql.SQLClientInfoException;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Savepoint;
import java.sql.Statement;
import java.sql.Struct;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Executor;

import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.constant.StrPattern;
import net.wicp.tams.common.http.flink.SqlGateWay;

/**
 * Flink JDBC connection.
 */
public class FlinkConnection implements Connection {

	private final SqlGateWay sqlGateWay;

	public FlinkConnection(String jdbcUrl) throws Exception {
		if (StringUtil.isNull(jdbcUrl)) {
			throw new IllegalArgumentException("url不能为空");
		}
		String[] urlMetas = StrPattern.jdbcurl.group(jdbcUrl,
				new String[] { "type", "host", "port", "database", "params" });
		if (urlMetas.length != 5 || !"flink".equals(urlMetas[0]) || StringUtil.isNull(urlMetas[1])) {
			throw new IllegalArgumentException("非法的url，格式参考：jdbc:flink://localhost:8083?planner=blink");
		}
		String host = urlMetas[1];
		if (StringUtil.isNotNull(urlMetas[2])) {
			int port;
			@SuppressWarnings("unused")
			String[] params = StringUtil.isNull(urlMetas[4]) ? new String[] {} : urlMetas[4].split("&");
			try {
				port = Integer.valueOf(urlMetas[2]);
			} catch (NumberFormatException e) {
				throw new IllegalArgumentException("端口格式不对：" + urlMetas[2]);
			}
			this.sqlGateWay = new SqlGateWay(host, port);
		} else {
			this.sqlGateWay = new SqlGateWay(String.format("http://%s", host));
		}
	}

	@Override
	public Statement createStatement() throws SQLException {
		return new FlinkStatement(this.sqlGateWay, this);
	}

	@Override
	public void close() throws SQLException {
		if (isClosed()) {
			this.sqlGateWay.close();
		}
	}

	@Override
	public boolean isClosed() throws SQLException {
		return !this.sqlGateWay.isExit();
	}

	@Override
	public DatabaseMetaData getMetaData() throws SQLException {
		return new FlinkDatabaseMetaData(sqlGateWay, this);
	}

	/***
	 * 设置catalog
	 */
	@Override
	public void setCatalog(String catalog) throws SQLException {
		try {
			this.sqlGateWay.setCatalog(catalog);
		} catch (Throwable e) {
			throw new SQLException(e);
		}
	}

	/***
	 * 得到当前的catalog
	 */
	@Override
	public String getCatalog() throws SQLException {
		try {
			return this.sqlGateWay.getCatalog();
		} catch (Throwable e) {
			throw new SQLException(e);
		}
	}

	@Override
	public void setSchema(String schema) throws SQLException {
		try {
			this.sqlGateWay.setDb(schema);
		} catch (Throwable e) {
			throw new SQLException(e);
		}
	}

	@Override
	public String getSchema() throws SQLException {
		try {
			return this.sqlGateWay.getDb();
		} catch (Throwable e) {
			throw new SQLException(e);
		}
	}

	@Override
	public PreparedStatement prepareStatement(String sql) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#prepareStatement is not supported");
	}

	@Override
	public CallableStatement prepareCall(String sql) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#prepareCall is not supported");
	}

	@Override
	public String nativeSQL(String sql) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#nativeSQL is not supported");
	}

	@Override
	public void setAutoCommit(boolean autoCommit) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#setAutoCommit is not supported");
	}

	@Override
	public boolean getAutoCommit() throws SQLException {
		return true;
	}

	@Override
	public void commit() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#commit is not supported");
	}

	@Override
	public void rollback() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#rollback is not supported");
	}

	@Override
	public void setReadOnly(boolean readOnly) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#setReadOnly is not supported");
	}

	@Override
	public boolean isReadOnly() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#isReadOnly is not supported");
	}

	@Override
	public void setTransactionIsolation(int level) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#setTransactionIsolation is not supported");
	}

	@Override
	public int getTransactionIsolation() throws SQLException {
		return Connection.TRANSACTION_NONE;
	}

	@Override
	public SQLWarning getWarnings() throws SQLException {
		return null;
	}

	@Override
	public void clearWarnings() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#clearWarnings is not supported");
	}

	@Override
	public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#createStatement is not supported");
	}

	@Override
	public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency)
			throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#prepareStatement is not supported");
	}

	@Override
	public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#prepareCall is not supported");
	}

	@Override
	public Map<String, Class<?>> getTypeMap() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#getTypeMap is not supported");
	}

	@Override
	public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#setTypeMap is not supported");
	}

	@Override
	public void setHoldability(int holdability) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#setHoldability is not supported");
	}

	@Override
	public int getHoldability() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#getHoldability is not supported");
	}

	@Override
	public Savepoint setSavepoint() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#setSavepoint is not supported");
	}

	@Override
	public Savepoint setSavepoint(String name) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#setSavepoint is not supported");
	}

	@Override
	public void rollback(Savepoint savepoint) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#rollback is not supported");
	}

	@Override
	public void releaseSavepoint(Savepoint savepoint) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#releaseSavepoint is not supported");
	}

	@Override
	public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability)
			throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#createStatement is not supported");
	}

	@Override
	public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency,
			int resultSetHoldability) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#prepareStatement is not supported");
	}

	@Override
	public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency,
			int resultSetHoldability) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#prepareCall is not supported");
	}

	@Override
	public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#prepareStatement is not supported");
	}

	@Override
	public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#prepareStatement is not supported");
	}

	@Override
	public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#prepareStatement is not supported");
	}

	@Override
	public Clob createClob() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#createClob is not supported");
	}

	@Override
	public Blob createBlob() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#createBlob is not supported");
	}

	@Override
	public NClob createNClob() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#createNClob is not supported");
	}

	@Override
	public SQLXML createSQLXML() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#createSQLXML is not supported");
	}

	@Override
	public boolean isValid(int timeout) throws SQLException {
		// TODO 暂不支持超时
		return sqlGateWay.heartbeat();
	}

	@Override
	public void setClientInfo(String name, String value) throws SQLClientInfoException {
		throw new SQLClientInfoException();
	}

	@Override
	public void setClientInfo(Properties properties) throws SQLClientInfoException {
		throw new SQLClientInfoException();
	}

	@Override
	public String getClientInfo(String name) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#getClientInfo is not supported");
	}

	@Override
	public Properties getClientInfo() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#getClientInfo is not supported");
	}

	@Override
	public Array createArrayOf(String typeName, Object[] elements) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#createArrayOf is not supported");
	}

	@Override
	public Struct createStruct(String typeName, Object[] attributes) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#createStruct is not supported");
	}

	@Override
	public void abort(Executor executor) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#abort is not supported");
	}

	@Override
	public void setNetworkTimeout(Executor executor, int milliseconds) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#setNetworkTimeout is not supported");
	}

	@Override
	public int getNetworkTimeout() throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#getNetworkTimeout is not supported");
	}

	@Override
	public <T> T unwrap(Class<T> iface) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#unwrap is not supported");
	}

	@Override
	public boolean isWrapperFor(Class<?> iface) throws SQLException {
		throw new SQLFeatureNotSupportedException("FlinkConnection#isWrapperFor is not supported");
	}

}
