/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.flink.k8s;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.flink.table.api.Schema;
import org.apache.flink.table.api.Schema.Builder;
import org.apache.flink.table.catalog.AbstractCatalog;
import org.apache.flink.table.catalog.CatalogBaseTable;
import org.apache.flink.table.catalog.CatalogBaseTable.TableKind;
import org.apache.flink.table.catalog.CatalogDatabase;
import org.apache.flink.table.catalog.CatalogDatabaseImpl;
import org.apache.flink.table.catalog.CatalogFunction;
import org.apache.flink.table.catalog.CatalogPartition;
import org.apache.flink.table.catalog.CatalogPartitionSpec;
import org.apache.flink.table.catalog.CatalogTable;
import org.apache.flink.table.catalog.ObjectPath;
import org.apache.flink.table.catalog.exceptions.CatalogException;
import org.apache.flink.table.catalog.exceptions.DatabaseAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotEmptyException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotExistException;
import org.apache.flink.table.catalog.exceptions.FunctionAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.FunctionNotExistException;
import org.apache.flink.table.catalog.exceptions.PartitionAlreadyExistsException;
import org.apache.flink.table.catalog.exceptions.PartitionNotExistException;
import org.apache.flink.table.catalog.exceptions.PartitionSpecInvalidException;
import org.apache.flink.table.catalog.exceptions.TableAlreadyExistException;
//20220623 andy 确保不是org.apache.flink.table.api.TableNotExistException异常
import org.apache.flink.table.catalog.exceptions.TableNotExistException;
import org.apache.flink.table.catalog.exceptions.TableNotPartitionedException;
import org.apache.flink.table.catalog.exceptions.TablePartitionedException;
import org.apache.flink.table.catalog.stats.CatalogColumnStatistics;
import org.apache.flink.table.catalog.stats.CatalogTableStatistics;
import org.apache.flink.table.expressions.Expression;
import org.apache.flink.table.types.DataType;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;

import io.fabric8.kubernetes.api.model.GenericKubernetesResource;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.jdbc.MySqlColBean;
import net.wicp.tams.common.apiext.jdbc.MySqlTbBean;
import net.wicp.tams.common.constant.DateFormatCase;
import net.wicp.tams.common.constant.FieldFormart;
import net.wicp.tams.common.constant.Middleware;
import net.wicp.tams.common.constant.MiddlewareOption;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.flink.k8s.constant.CatalogAssit;
import net.wicp.tams.common.kubernetes.driver.KubernetesConnection;

/***
 * 需要下列CRD： fcdb fctb fcview
 * 
 * @author andy
 *
 */
@Slf4j
public class KubernetesCatalog extends AbstractCatalog {

	private static final String PRE_TB = "t_";
	private static final String PRE_VIEW = "v_";
	private static final String LABLE_CATALOG = "catalog";
	private static final String LABLE_DB = "db";
	private static final String LABLE_TB = "tb";
	private final String namespace;

	private final Set<String> dbs = new TreeSet<>();

	private KubernetesConnection connection;

	public KubernetesCatalog(String catalogName, String defaultDatabase, String namespace) {
		super(catalogName, defaultDatabase);
		this.namespace = namespace;
	}

	@Override
	public void open() throws CatalogException {
		Conf.overProp("common.kubernetes.crd.enum.crd1", "net.wicp.tams.common.flink.k8s.CrdVesion");
		try {
			// this.connection = new KubernetesConnection("jdbc:k8s://localhost");
			this.connection = new KubernetesConnection("jdbc:k8s://localhost?context=yaboshi&conf=~/.kube/config");
			this.connection.setSchema(this.namespace);
		} catch (Exception e) {
			log.error("获得连接失败", e);
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_conn_fail, e);
		}
		// 检查安装CRD资源，需要有安装CRD资源的权限
		if (!this.connection.getClientTams().existCrd(CrdVesion.fcdb)) {
			this.connection.getClientTams().installCrd("/crd/fcdb.yaml", KubernetesCatalog.class);
		}
		if (!this.connection.getClientTams().existCrd(CrdVesion.fctb)) {
			this.connection.getClientTams().installCrd("/crd/fctb.yaml", KubernetesCatalog.class);
		}
		if (!this.connection.getClientTams().existCrd(CrdVesion.fcview)) {
			this.connection.getClientTams().installCrd("/crd/fcview.yaml", KubernetesCatalog.class);
		}
		cacheDb();
		try {// 创建默认的db和duckula，有可能这2个库是同一个，也有可能不是。
			this.createDatabase(this.getDefaultDatabase(), null, true);
		} catch (CatalogException | DatabaseAlreadyExistException e1) {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_exec_fail, "初始化default库失败", e1);
		}
	}

	// 重新拉缓存
	public void cacheDb() {
		// 缓存catalog名和db名
		List<GenericKubernetesResource> rslist = connection.getClientTams().selCusObject(CrdVesion.fcdb, this.namespace,
				null);
		if (CollectionUtils.isEmpty(rslist)) {// 没有库就自动创建一个默认库
			this.connection.getClientTams().createCusObject(CrdVesion.fcdb, this.namespace,
					IOUtil.fileToInputStream("/crd/fcdb-default.yaml", KubernetesCatalog.class));
			rslist = connection.getClientTams().selCusObject(CrdVesion.fcdb, this.namespace, null);
		}
		Map<String, List<String>> catalogtodb = new HashMap<String, List<String>>();
		for (GenericKubernetesResource rs : rslist) {
			Map<String, String> labels = rs.getMetadata().getLabels();
			String catalog = labels.get(LABLE_CATALOG);
			String db = labels.get(LABLE_DB);
			if (catalogtodb.containsKey(catalog)) {
				catalogtodb.get(catalog).add(db);
			} else {
				List<String> dbs = new ArrayList<String>();
				dbs.add(db);
				catalogtodb.put(catalog, dbs);
			}
		}
		if (!catalogtodb.containsKey(this.getName())) {
			String catalogs = CollectionUtil.listJoin(catalogtodb.keySet(), ",");
			throw new CatalogException("不存在这个catalog,你可以选择使用如下catalog：" + catalogs);
		} else {
			dbs.addAll(catalogtodb.get(this.getName()));
		}
	}

	@Override
	public void close() throws CatalogException {
		try {
			if (this.connection != null && !this.connection.isClosed()) {
				this.connection.close();
			}
		} catch (SQLException e) {
			log.error("关闭连接失败", e);
		}
	}

	@Override
	public List<String> listDatabases() throws CatalogException {
		return new ArrayList<>(dbs);
	}

	@Override
	public CatalogDatabase getDatabase(String databaseName) throws DatabaseNotExistException, CatalogException {
		if (!dbs.contains(databaseName)) {
			throw new DatabaseNotExistException(this.getName(), databaseName);
		}
		return new CatalogDatabaseImpl(new HashMap<>(), "k8s库");
	}

	@Override
	public List<String> listTables(String databaseName) throws DatabaseNotExistException, CatalogException {
		List<GenericKubernetesResource> selCusObjectByLabel = connection.getClientTams().selCusObjectByLabel(
				CrdVesion.fctb, this.namespace, LABLE_CATALOG, this.getName(), LABLE_DB, databaseName);
		Set<String> retset = new TreeSet<>();
		for (GenericKubernetesResource rs : selCusObjectByLabel) {
			retset.add(rs.getMetadata().getLabels().get(LABLE_TB));
		}
		return new ArrayList<>(retset);
	}

	@Override
	public List<String> listViews(String databaseName) throws DatabaseNotExistException, CatalogException {
		List<GenericKubernetesResource> selCusObjectByLabel = connection.getClientTams().selCusObjectByLabel(
				CrdVesion.fcview, this.namespace, LABLE_CATALOG, this.getName(), LABLE_DB, databaseName);
		Set<String> retset = new TreeSet<>();
		for (GenericKubernetesResource rs : selCusObjectByLabel) {
			retset.add(rs.getMetadata().getLabels().get(LABLE_TB));
		}
		return new ArrayList<>(retset);
	}

	/***
	 * 得到表或视图信息，注意v_开头为视图，t_为表。
	 */
	@SuppressWarnings("unchecked")
	@Override
	public CatalogBaseTable getTable(ObjectPath tablePath) throws TableNotExistException, CatalogException {
		String objectName = tablePath.getObjectName();
		try {
			if (objectName.startsWith(PRE_TB)) {
				List<GenericKubernetesResource> rslist = connection.getClientTams().selCusObjectByLabel(CrdVesion.fctb,
						this.namespace, LABLE_CATALOG, this.getName(), LABLE_DB, tablePath.getDatabaseName(), LABLE_TB,
						tablePath.getObjectName());
				if (rslist.isEmpty()) {
					throw new TableNotExistException(this.getName(), tablePath);
				}
				Map<String, Object> spec = (Map<String, Object>) rslist.get(0).getAdditionalProperties().get("spec");
				List<Map<String, Object>> def = (List<Map<String, Object>>) spec.get("col_def");
				ObjectMapper objmap = new ObjectMapper();
				List<MySqlColBean> collist = def.stream().map(e -> objmap.convertValue(e, MySqlColBean.class))
						.collect(Collectors.toList());
				Builder schemaBuilder = Schema.newBuilder();
				String comment = String.valueOf(spec.get("remark"));// 表的备注
				List<String> priList = new ArrayList<>();
				for (MySqlColBean mySqlColBean : collist) {
					String colname = mySqlColBean.getColumnNameStandard(FieldFormart.dbstandard);
					if (mySqlColBean.isPri()) {
						priList.add(colname);
					}
					// 需要加这个，否则在聚合时会出现：Exception in thread "main"
					// org.apache.flink.table.api.TableException: Window aggregate can only be
					// defined over a time attribute column, but TIMESTAMP(3) encountered.
					if (StringUtil.isNotNull(mySqlColBean.getWatermarkExpression())) {
						schemaBuilder.watermark(colname, mySqlColBean.getWatermarkExpression());
					}
					if (StringUtil.isNotNull(mySqlColBean.getSqlExpression())) {
						schemaBuilder.columnByExpression(colname, mySqlColBean.getSqlExpression());
					} else if (mySqlColBean.isNotNull() && mySqlColBean.isMetadata()) {// 是Metadata

						String metadataName = StringUtil.isNull(mySqlColBean.getMetadataName()) ? null
								: mySqlColBean.getMetadataName();
						DataType dataType = CatalogAssit.findColType(mySqlColBean);
						schemaBuilder.columnByMetadata(colname, dataType, metadataName, mySqlColBean.isVirtual());
					} else {
						DataType dataType = CatalogAssit.findColType(mySqlColBean);
						schemaBuilder.column(colname, dataType);
					}
					// Method 'withComment(...)' must be called after a column definition, but there
					// is no preceding column defined.
					if (StringUtil.isNotNull(mySqlColBean.getColumnComment())) {// 备注
						schemaBuilder.withComment(mySqlColBean.getColumnComment());
					}
				}

				CatalogTable.Builder tbBuilder = CatalogTable.newBuilder();
				tbBuilder.partitionKeys(priList);
				if (spec.containsKey("withOptions")) {
					tbBuilder.options((Map<String, String>) spec.get("withOptions"));
				}
				tbBuilder.comment(comment);
				tbBuilder.schema(schemaBuilder.build());
				return tbBuilder.build();

			} else if (objectName.startsWith(PRE_VIEW)) {// 查询视图

			} else {
				// 20220623 andy 这里必须抛出此异常，在CatalogManager.getPermanentTable
				// 方法只会忽略此异常。其它异常会阻碍运行下去，那个tablePath如果在db时会先尝试把db名当表名去查一次就会有问题，就不能支持db.tb模式的使用了。
				throw new TableNotExistException(super.getName(), tablePath);
			}
		} catch (TableNotExistException e) {
			// 20220623 andy 这里必须抛出此异常，在CatalogManager.getPermanentTable
			// 方法只会忽略此异常。其它异常会阻碍运行下去，那个tablePath如果在db时会先尝试把db名当表名去查一次就会有问题，就不能支持db._tb模式的使用了。
			throw e;
		} catch (Exception e) {
			throw new CatalogException("查询table失败", e);
		}
		return null;
	}

	@Override
	public boolean tableExists(ObjectPath tablePath) throws CatalogException {
		try {
			CatalogBaseTable baseTable = getTable(tablePath);
			return baseTable != null;
		} catch (TableNotExistException e) {
			return false;
		}
	}

	@Override
	public void dropTable(ObjectPath tablePath, boolean ignoreIfNotExists)
			throws TableNotExistException, CatalogException {
		try {
			validDb(tablePath);
		} catch (DatabaseNotExistException e1) {
			if (!ignoreIfNotExists) {
				throw new TableNotExistException(this.getName(), tablePath);
			}
		}
		CatalogBaseTable baseTable;
		try {
			baseTable = getTable(tablePath);
		} catch (TableNotExistException e) {
			baseTable = null;// 正常情况，其它异常不处理
		}
		if (!ignoreIfNotExists && baseTable == null) {
			throw new TableNotExistException(this.getName(), tablePath);
		}
		if (baseTable != null) {
			String objectName = tablePath.getObjectName();
			if (objectName.startsWith(PRE_TB) && baseTable.getTableKind() == TableKind.TABLE) {// 表
				this.connection.getClientTams().deleteCusObjectByLabel(CrdVesion.fctb, this.namespace, LABLE_CATALOG,
						this.getName(), LABLE_DB, tablePath.getDatabaseName(), LABLE_TB, tablePath.getObjectName());
			} else {// 视图

			}
		}
	}

	// 只支持重命名表
	@SuppressWarnings("unchecked")
	@Override
	public void renameTable(ObjectPath tablePath, String newTableName, boolean ignoreIfNotExists)
			throws TableNotExistException, TableAlreadyExistException, CatalogException {
		try {
			validDb(tablePath);
		} catch (DatabaseNotExistException e1) {
			if (!ignoreIfNotExists) {
				throw new TableNotExistException(this.getName(), tablePath);
			}
		}
		CatalogBaseTable baseTable;
		try {
			baseTable = getTable(tablePath);
		} catch (TableNotExistException e) {
			baseTable = null;// 正常情况，其它异常不处理
		}
		if (!ignoreIfNotExists && baseTable == null) {
			throw new TableNotExistException(this.getName(), tablePath);
		}
		if (baseTable != null) {
			String objectName = tablePath.getObjectName();
			if (objectName.startsWith(PRE_TB) && baseTable.getTableKind() == TableKind.TABLE) {// 表
				List<GenericKubernetesResource> crdList = this.connection.getClientTams().selCusObjectByLabel(
						CrdVesion.fctb, this.namespace, LABLE_CATALOG, this.getName(), LABLE_DB,
						tablePath.getDatabaseName(), LABLE_TB, tablePath.getObjectName());
				this.connection.getClientTams().deleteCusObjectByLabel(CrdVesion.fctb, this.namespace, LABLE_CATALOG,
						this.getName(), LABLE_DB, tablePath.getDatabaseName(), LABLE_TB, tablePath.getObjectName());
				// 设置新的表
				Map<String, String> labels = CollectionUtil.newMapStr(new String[] { LABLE_CATALOG, this.getName(),
						LABLE_DB, tablePath.getDatabaseName(), LABLE_TB, newTableName });
				Map<String, Object> spec = (Map<String, Object>) crdList.get(0).getAdditionalProperties().get("spec");
				this.connection.getClientTams().createCusObject(CrdVesion.fctb, this.namespace,
						getCrdName(tablePath.getDatabaseName(), newTableName), null, labels, spec);
			} else {

			}
		}
	}

	@Override
	public void createTable(ObjectPath tablePath, CatalogBaseTable table, boolean ignoreIfExists)
			throws TableAlreadyExistException, DatabaseNotExistException, CatalogException {
		validDb(tablePath);
		CatalogBaseTable baseTable;
		try {
			baseTable = getTable(tablePath);
		} catch (TableNotExistException e) {
			baseTable = null;// 正常情况，其它异常不处理
		}
		if (!ignoreIfExists && baseTable != null) {
			throw new TableAlreadyExistException(this.getName(), tablePath);
		}
		if (baseTable == null) {
			String objectName = tablePath.getObjectName();
			if (objectName.startsWith(PRE_TB) && table.getTableKind() == TableKind.TABLE) {// 表
				try {
					List<MySqlColBean> mysqlColList = CatalogAssit.packageColBeans(table, objectName);
					// 分区
					String partitionkeys = "";
					CatalogTable tableTrue = (CatalogTable) table;
					if (CollectionUtils.isNotEmpty(tableTrue.getPartitionKeys())) {
						partitionkeys = CollectionUtil.listJoin(tableTrue.getPartitionKeys(), ",");
					}
					// 转换为统一的配置
					Map<MiddlewareOption, String> allOptmap = new HashMap<>();
					Map<String, String> options = table.getOptions();
					if (MapUtils.isNotEmpty(options)) {
						for (String optionstr : options.keySet()) {
							MiddlewareOption findByNameUse = MiddlewareOption.findByNameUse(optionstr);
							if (findByNameUse != null) {
								allOptmap.put(findByNameUse, options.get(optionstr));
							} else {
								throw new CatalogException(
										"创建表[" + objectName + "]时检查异常，原因:catalog还不支持此option[" + optionstr + "].");
							}
						}
					}
					// 得到option相关配置，全部做为可配置项处理。
					Pair<String, String> proOptStr = Middleware.proOptStr(allOptmap,
							allOptmap.keySet().toArray(new MiddlewareOption[allOptmap.size()]));
					MySqlTbBean catalogTbInfo = MySqlTbBean.builder().db(tablePath.getDatabaseName()).tb(objectName)
							.tbComment("通过sql创建").middleware(Middleware.no).partitionkeys(partitionkeys)
							.opt(proOptStr.getRight()).catalogName(super.getName()).withOptions(proOptStr.getLeft())
							.build();
					this.createTable(mysqlColList, catalogTbInfo);
				} catch (ProjectException e) {
					throw new CatalogException("创建表[" + objectName + "]时SQL异常，原因：" + e.getMessage(), e);
				}
			} else if (objectName.startsWith(PRE_VIEW) && table.getTableKind() == TableKind.VIEW) {// 创建视图

			} else {
				throw new CatalogException(String.format("表以t_开头，视图以v_开头，不存在此类型的对象[%s]", tablePath));
			}
		}
	}

	private void createTable(List<MySqlColBean> mysqlColList, MySqlTbBean catalogTbInfo) throws ProjectException {
		try {
			Map<String, String> lables = new HashMap<>();
			lables.put(LABLE_CATALOG, this.getName());
			lables.put(LABLE_DB, catalogTbInfo.getDb());
			lables.put(LABLE_TB, catalogTbInfo.getTb());
			Map<String, Object> spec = new HashMap<>();
			spec.put("remark", catalogTbInfo.getTbComment());
			ObjectMapper objmap = new ObjectMapper();
			ArrayNode fieldNode = objmap.valueToTree(mysqlColList);
			spec.put("col_def", fieldNode);
			String crName = getCrdName(catalogTbInfo.getDb(), catalogTbInfo.getTb());
			this.connection.getClientTams().createCusObject(CrdVesion.fctb, this.namespace, crName, null, lables, spec);
		} catch (Exception e) {
			throw new ProjectException(ExceptAll.flink_catalog_table,
					"创建表[" + catalogTbInfo.getDb() + "." + catalogTbInfo.getTb() + "]时SQL异常", e);
		}
	}

	private String getCrdName(String db, String tb) {
		String crName = String.format("%s.%s.%s", StringUtil.formatRfc1123(this.getName()),
				StringUtil.formatRfc1123(db), StringUtil.formatRfc1123(tb));
		crName = crName.length() > 63 ? crName.substring(0, 63) : crName;
		return crName;
	}

	private void validDb(ObjectPath tablePath) throws DatabaseNotExistException {
		CatalogDatabase database;
		try {
			database = getDatabase(tablePath.getDatabaseName());
		} catch (DatabaseNotExistException e) {
			database = null;// 正常情况，其它异常不处理
		}
		if (database == null) {
			throw new DatabaseNotExistException(this.getName(), tablePath.getDatabaseName());
		}
	}

	@Override
	public void alterTable(ObjectPath tablePath, CatalogBaseTable newTable, boolean ignoreIfNotExists)
			throws TableNotExistException, CatalogException {
		dropTable(tablePath, true);
		try {
			createTable(tablePath, newTable, false);
		} catch (TableAlreadyExistException | DatabaseNotExistException e) {
			throw new CatalogException(e);
		}
	}

	@Override
	public List<CatalogPartitionSpec> listPartitions(ObjectPath tablePath)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public List<CatalogPartitionSpec> listPartitions(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public List<CatalogPartitionSpec> listPartitionsByFilter(ObjectPath tablePath, List<Expression> filters)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public CatalogPartition getPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public boolean partitionExists(ObjectPath tablePath, CatalogPartitionSpec partitionSpec) throws CatalogException {
		return false;
	}

	@Override
	public void createPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, CatalogPartition partition,
			boolean ignoreIfExists) throws TableNotExistException, TableNotPartitionedException,
			PartitionSpecInvalidException, PartitionAlreadyExistsException, CatalogException {

	}

	@Override
	public void dropPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public void alterPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, CatalogPartition newPartition,
			boolean ignoreIfNotExists) throws PartitionNotExistException, CatalogException {

	}

	@Override
	public List<String> listFunctions(String dbName) throws DatabaseNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogFunction getFunction(ObjectPath functionPath) throws FunctionNotExistException, CatalogException {
		return null;
	}

	@Override
	public boolean functionExists(ObjectPath functionPath) throws CatalogException {
		try {
			CatalogFunction function = getFunction(functionPath);
			return function != null;
		} catch (FunctionNotExistException e) {
			return false;
		}
	}

	@Override
	public void createFunction(ObjectPath functionPath, CatalogFunction function, boolean ignoreIfExists)
			throws FunctionAlreadyExistException, DatabaseNotExistException, CatalogException {

	}

	@Override
	public void alterFunction(ObjectPath functionPath, CatalogFunction newFunction, boolean ignoreIfNotExists)
			throws FunctionNotExistException, CatalogException {
		// 先删除再插入达到修改
		dropFunction(functionPath, ignoreIfNotExists);
		try {
			createFunction(functionPath, newFunction, ignoreIfNotExists);
		} catch (FunctionAlreadyExistException e) {
			throw new CatalogException(e);
		} catch (DatabaseNotExistException e) {
			throw new CatalogException(e);
		}
	}

	@Override
	public void dropFunction(ObjectPath functionPath, boolean ignoreIfNotExists)
			throws FunctionNotExistException, CatalogException {

	}

	@Override
	public CatalogTableStatistics getTableStatistics(ObjectPath tablePath)
			throws TableNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogColumnStatistics getTableColumnStatistics(ObjectPath tablePath)
			throws TableNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogTableStatistics getPartitionStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogColumnStatistics getPartitionColumnStatistics(ObjectPath tablePath,
			CatalogPartitionSpec partitionSpec) throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public void alterTableStatistics(ObjectPath tablePath, CatalogTableStatistics tableStatistics,
			boolean ignoreIfNotExists) throws TableNotExistException, CatalogException {

	}

	@Override
	public void alterTableColumnStatistics(ObjectPath tablePath, CatalogColumnStatistics columnStatistics,
			boolean ignoreIfNotExists) throws TableNotExistException, CatalogException, TablePartitionedException {

	}

	@Override
	public void alterPartitionStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogTableStatistics partitionStatistics, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public void alterPartitionColumnStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogColumnStatistics columnStatistics, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public boolean databaseExists(String databaseName) throws CatalogException {
		return dbs.contains(databaseName);
	}

	@Override
	public void createDatabase(String name, CatalogDatabase database, boolean ignoreIfExists)
			throws DatabaseAlreadyExistException, CatalogException {
		if (this.dbs.contains(name)) {
			if (ignoreIfExists) {
				return;
			} else {
				throw new DatabaseAlreadyExistException(this.getName(), name);
			}
		}
		Map<String, String> labels = new HashMap<>();
		labels.put(LABLE_CATALOG, this.getName());
		labels.put(LABLE_DB, name);
		Map<String, Object> spec = new HashMap<>();
		String remark = database == null || StringUtil.isNull(database.getComment())
				? DateFormatCase.YYYY_MM_DD.getInstanc().format(System.currentTimeMillis())
				: database.getComment();
		spec.put("remark", remark);
		this.connection.getClientTams().createCusObject(CrdVesion.fcdb, this.namespace,
				String.format("%s.%s", this.getName(), name), null, labels, spec);
		this.dbs.add(name);
	}

	@Override
	public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade)
			throws DatabaseNotExistException, DatabaseNotEmptyException, CatalogException {
		if (!this.dbs.contains(name)) {
			if (ignoreIfNotExists) {
				return;
			} else {
				throw new DatabaseNotExistException(this.getName(), name);
			}
		}
		List<GenericKubernetesResource> labels = this.connection.getClientTams().selCusObjectByLabel(CrdVesion.fcdb,
				this.namespace, LABLE_CATALOG, this.getName(), LABLE_DB, name);
		this.connection.getClientTams().deleteCusObject(CrdVesion.fcdb, this.namespace,
				labels.get(0).getMetadata().getName());
		this.dbs.remove(name);
	}

	@Override
	public void alterDatabase(String name, CatalogDatabase newDatabase, boolean ignoreIfNotExists)
			throws DatabaseNotExistException, CatalogException {
		throw new CatalogException("不支持修改db");
	}

}
