/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.flink.connector.redis.connector;

import net.wicp.tams.common.flink.common.constant.FlinkTypeEnum;
import net.wicp.tams.common.flink.connector.redis.config.FlinkJedisConfigBase;
import net.wicp.tams.common.flink.connector.redis.container.RedisCommandsContainer;
import net.wicp.tams.common.flink.connector.redis.container.RedisCommandsContainerBuilder;
import net.wicp.tams.common.flink.connector.redis.mapper.LookupRedisMapper;
import net.wicp.tams.common.flink.connector.redis.mapper.RedisCommand;
import net.wicp.tams.common.flink.connector.redis.mapper.RedisCommandDescription;
import net.wicp.tams.common.flink.connector.redis.options.RedisLookupOptions;
import net.wicp.tams.common.flink.connector.redis.options.RedisSourceOptions;
import org.apache.flink.annotation.Internal;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.metrics.Gauge;
import org.apache.flink.shaded.guava31.com.google.common.cache.Cache;
import org.apache.flink.shaded.guava31.com.google.common.cache.CacheBuilder;
import org.apache.flink.table.data.GenericRowData;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.functions.FunctionContext;
import org.apache.flink.table.functions.TableFunction;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.types.RowKind;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * The RedisRowDataLookupFunction is a standard user-defined table function, it
 * can be used in tableAPI and also useful for temporal table join plan in SQL.
 * It looks up the result as {@link RowData}.
 */
@Internal
public class RedisRowDataLookupFunction extends TableFunction<RowData> {

    private static final Logger LOG = LoggerFactory.getLogger(RedisRowDataLookupFunction.class);
    private static final long serialVersionUID = 1L;

    private String additionalKey;
    private LookupRedisMapper lookupRedisMapper;
    private RedisCommand redisCommand;

    protected final RedisLookupOptions redisLookupOptions;

    private FlinkJedisConfigBase flinkJedisConfigBase;
    private RedisCommandsContainer redisCommandsContainer;

    private final long cacheMaxSize;
    private final long cacheExpireMs;
    private final int maxRetryTimes;

    private final boolean isBatchMode;

    private final int batchSize;

    private final int batchMinTriggerDelayMs;

    private transient Cache<Object, RowData> cache;

    private transient Consumer<Object[]> evaler;

    private final DataType physicalDataType;
    private List<RowType.RowField> fields = null;
    private final Configuration optionsWith;
    private String[] keyNames;

    public RedisRowDataLookupFunction(FlinkJedisConfigBase flinkJedisConfigBase, LookupRedisMapper lookupRedisMapper,
                                      RedisLookupOptions redisLookupOptions, DataType physicalDataType, Configuration optionsWith, String[] keyNames) {
        this.physicalDataType = physicalDataType;
        this.flinkJedisConfigBase = flinkJedisConfigBase;

        this.lookupRedisMapper = lookupRedisMapper;
        this.redisLookupOptions = redisLookupOptions;
        RedisCommandDescription redisCommandDescription = lookupRedisMapper.getCommandDescription();
        this.redisCommand = redisCommandDescription.getRedisCommand();
        this.additionalKey = redisCommandDescription.getAdditionalKey();

        this.cacheMaxSize = this.redisLookupOptions.getCacheMaxSize();
        this.cacheExpireMs = this.redisLookupOptions.getCacheExpireMs();
        this.maxRetryTimes = this.redisLookupOptions.getMaxRetryTimes();

        this.isBatchMode = this.redisLookupOptions.isBatchMode();

        this.batchSize = this.redisLookupOptions.getBatchSize();

        this.batchMinTriggerDelayMs = this.redisLookupOptions.getBatchMinTriggerDelayMs();
        this.optionsWith = optionsWith;
        this.keyNames = keyNames;
    }

    /**
     * The invoke entry point of lookup function. only support one object now
     *
     * @param objects the lookup key. Currently only support single rowkey.
     */
    public void eval(Object... objects) throws IOException {

        for (int retry = 0; retry <= maxRetryTimes; retry++) {
            try {
                // fetch result
                this.evaler.accept(objects);
                break;
            } catch (Exception e) {
                LOG.error(String.format("redis lookup error, retry times = %d", retry), e);
                if (retry >= maxRetryTimes) {
                    throw new RuntimeException("Execution of Redis lookup failed.", e);
                }
                try {
                    Thread.sleep(1000 * retry);
                } catch (InterruptedException e1) {
                    throw new RuntimeException(e1);
                }
            }
        }
    }

    @Override
    public void open(FunctionContext context) {
        LOG.info("start open ...");
        RedisSourceOptions.packageOptions(optionsWith);
        LOG.info("redislookupconnector配置信息===>" + redisLookupOptions.toString());
        try {
            this.redisCommandsContainer = RedisCommandsContainerBuilder.build(redisLookupOptions);
            // synchronized (o){
            this.redisCommandsContainer.open();
            // }
        } catch (Exception e) {
            LOG.error("Redis has not been properly initialized: ", e);
            throw new RuntimeException(e);
        }

        this.cache = cacheMaxSize <= 0 || cacheExpireMs <= 0 ? null
                : CacheBuilder.newBuilder().recordStats().expireAfterWrite(cacheExpireMs, TimeUnit.MILLISECONDS)
                .maximumSize(cacheMaxSize).build();

        if (cache != null) {
            context.getMetricGroup().gauge("lookupCacheHitRate", (Gauge<Double>) () -> cache.stats().hitRate());

            this.evaler = in -> {
				Map<String, String> kv = new HashMap<>();
				String searchkeysuffix = handlekeysuffix(in,kv);
                RowData cacheRowData = cache.getIfPresent(searchkeysuffix);
                if (cacheRowData != null) {
                    collect(cacheRowData);
                } else {
                    // fetch result
//                    byte[] key = lookupRedisMapper.serialize(in);
//                    byte[] value = null;

                    switch (redisCommand) {
//                        case GET:
//                            value = this.redisCommandsContainer.get(key);
//                            RowData rowData = this.lookupRedisMapper.deserialize(value);
//                            break;
                        case HGET:
                            getAndCache(searchkeysuffix,kv);
                            break;
                        default:
                            throw new IllegalArgumentException("Cannot process such data type: " + redisCommand);
                    }

                }
            };

        } else {
            this.evaler = in -> {
                // fetch result
				Map<String, String> kv = new HashMap<>();
				String searchkeysuffix = handlekeysuffix(in,kv);

                switch (redisCommand) {
//                    case GET:
//                        value = this.redisCommandsContainer.get(key);
//                        break;
                    case HGET:
                        getAndCache(searchkeysuffix, kv);
                        break;
                    default:
                        throw new IllegalArgumentException("Cannot process such data type: " + redisCommand);
                }
            };
        }

        LOG.info("end open.");
    }

    private String handlekeysuffix(Object[] in, Map<String, String> kv) {
        if (keyNames.length == 1) {
			kv.put(keyNames[0],String.valueOf(in[0]));
            return String.valueOf(in[0]);
        } else if (keyNames.length == 2) {
			kv.put(keyNames[0],String.valueOf(in[0]));
			kv.put(keyNames[1],String.valueOf(in[1]));
			StringJoiner stringJoiner = new StringJoiner(":");
			if (keyNames[0].equals(optionsWith.get(RedisSourceOptions.routeColName))) {
                return stringJoiner.add(String.valueOf(in[0])).add(String.valueOf(in[1])).toString();
            } else if (keyNames[1].equals(optionsWith.get(RedisSourceOptions.routeColName))) {
                return stringJoiner.add(String.valueOf(in[1])).add(String.valueOf(in[0])).toString();
            }else {
				throw new RuntimeException("路由参数不在表字段定义范围内，或关联条件中不包含路由参数！");
			}
        }else{
			throw new RuntimeException("运行时接受到了不合法的参数，当前最多支持包含路由参数内的两个参数！");
		}
    }

    private void getAndCache(String searchkeysuffix, Map<String, String> kv) {
        Map<String, String> allvalue;
        final RowType rowType = (RowType) physicalDataType.getLogicalType();
        this.fields = rowType.getFields();
        allvalue = this.redisCommandsContainer
                .hgetAll(redisLookupOptions.getSearchkeyprefix().concat(":").concat(searchkeysuffix));
		for(String key:kv.keySet()){
			allvalue.put(key,kv.get(key));
		}
        GenericRowData row = new GenericRowData(fields.size());
        row.setRowKind(RowKind.INSERT);
        packRow(allvalue, row);
        collect(row);
        if (cache != null && null != row) {
            cache.put(searchkeysuffix, row);
        }
    }

    private void packRow(Map<String, String> allValue, GenericRowData rowData) {
        for (int i = 0; i < fields.size(); i++) {
            RowType.RowField rowField = fields.get(i);
            FlinkTypeEnum flinkTypeEnum = FlinkTypeEnum.findByFlinkRowType(rowField.getType().getTypeRoot().toString());
            Object value = allValue.containsKey(rowField.getName())
                    ? FlinkTypeEnum.getValue(flinkTypeEnum,allValue.get(rowField.getName()), rowField.getType())
                    : null;
            rowData.setField(i, value);
        }
    }

    @Override
    public void close() {
        LOG.info("start close ...");
        if (redisCommandsContainer != null) {
            try {
                redisCommandsContainer.close();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        LOG.info("end close.");
    }
}
