
package net.wicp.tams.common.flink.connector.redis.connector;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.flink.api.common.serialization.SerializationSchema;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.streaming.api.functions.sink.RichSinkFunction;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.table.types.logical.RowType.RowField;
import org.apache.flink.types.RowKind;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.flink.common.constant.FlinkTypeEnum;
import net.wicp.tams.common.flink.connector.redis.options.RedisSourceOptions;
import net.wicp.tams.common.redis.RedisAssit;
import net.wicp.tams.common.redis.pool.AbsPool;
import redis.clients.jedis.Jedis;

/**
 * 自定义 sink 写入 redis，只保存传过来的数据，不会全量删除（后面加参数扩展）
 **/
@Slf4j
public class RedisRichSinkFunction extends RichSinkFunction<RowData> {
	private static final long serialVersionUID = 1L;
	private final int[] keyColIndex;// 主键,需要index
	private final List<RowField> fields;
	private final SerializationSchema<RowData> serialization;
	private final Configuration optionsWith;
	private AbsPool standalone;

	public RedisRichSinkFunction(ResolvedSchema schema, SerializationSchema<RowData> serialization,
			Configuration optionsWith) {
		if (schema.getPrimaryKey().isPresent()) {// 有主键
			List<String> keys = schema.getPrimaryKey().get().getColumns();
			keyColIndex = new int[keys.size()];
			int index = 0;
			for (int i = 0; i < schema.getColumns().size(); i++) {
				if (keys.contains(schema.getColumns().get(i).getName())) {
					keyColIndex[index++] = i;
				}
			}
		} else {
			keyColIndex = new int[0];
		}

		// toSourceRowDataType是全部的字段，toSinkRowDataType是select后面的字段
		final RowType rowType = (RowType) schema.toSinkRowDataType().getLogicalType();
		this.fields = rowType.getFields();
//		this.colNames = schema.getColumnNames().toArray(new String[schema.getColumnNames().size()]);
		this.serialization = serialization;
		this.optionsWith = optionsWith;
	}

	@Override
	public void open(Configuration parameters) throws Exception {
		super.open(parameters);
		RedisSourceOptions.packageOptions(optionsWith);
		String serverName = optionsWith.get(RedisSourceOptions.groupid);// 用groupid做服务名
		standalone = RedisAssit.standalone(serverName);
	}

	@Override
	public void invoke(RowData value, Context context) throws Exception {
		RedisSourceOptions.packageOptions(this.optionsWith);
		String serverName = optionsWith.get(RedisSourceOptions.groupid);// 用groupid做服务名
		String streamkey = Conf.get(String.format("common.redis.redisserver.%s.streamkey", serverName));
		String groupkey = Conf.get(String.format("common.redis.redisserver.%s.groupkey", serverName));

		if (this.standalone == null || !this.standalone.isInit()) {
			this.standalone = RedisAssit.standalone(serverName);
		}
		// kafka传过来的RowData是binRowData
//		GenericRowData tempvalue = (GenericRowData) value;

		Jedis jedis = null;
		try {
			jedis = standalone.getResource();
			// 1、更新具体的key
			List<String> keyValues = new ArrayList<String>();
			for (int keyIndex : keyColIndex) {
				String valueOf = FlinkTypeEnum.getStr(fields.get(keyIndex), value, keyIndex);
				keyValues.add(valueOf);
			}
			String rowKeyValue = CollectionUtil.listJoin(keyValues, "`");
			String key = String.format("%s:%s", optionsWith.get(RedisSourceOptions.searchkeyprefix), rowKeyValue);
			Map<String, String> data = new HashMap<String, String>();
			for (int i = 0; i < fields.size(); i++) {
				String colValue = FlinkTypeEnum.getStr(fields.get(i), value, i);// getStr(fields.get(i),
																				// tempvalue.getField(i));
				data.put(fields.get(i).getName(), colValue);
			}
			jedis.hmset(key, data);
			// 2、需要插入到全量的key array
			RowKind rowKind = value.getRowKind();
			switch (rowKind) {
			case INSERT:
				jedis.sadd(groupkey, key);// 添加元素
				break;
			case DELETE:
				jedis.del(key);
				jedis.srem(groupkey, key);// 移除元素
				break;
			default:
				break;
			}

			// 3、发送stream流消息
			byte[] serialize = serialization.serialize(value);// 序列化为duckula,因为消息需要
			Map<byte[], byte[]> content = new HashMap<byte[], byte[]>();
			content.put("data".getBytes(), serialize);
			content.put("time".getBytes(), String.valueOf(System.currentTimeMillis()).getBytes());
			this.standalone.putStreamByte(jedis, streamkey, content);
		} catch (Exception e) {
			log.error("sink redis失败", e);
		} finally {
			standalone.returnResource(jedis);
		}
	}

	@Override
	public void close() throws Exception {
		if (this.standalone != null || this.standalone.isInit()) {
			this.standalone.destroy();
		}
	}
}