/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.flink.connector.redis.source;

import static net.wicp.tams.common.flink.connector.redis.connector.RedisScanFunction.createValueFormatProjection;

import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import org.apache.flink.api.common.serialization.DeserializationSchema;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.streaming.api.functions.source.SourceFunction;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.catalog.UniqueConstraint;
import org.apache.flink.table.connector.ChangelogMode;
import org.apache.flink.table.connector.format.DecodingFormat;
import org.apache.flink.table.connector.source.DynamicTableSource;
import org.apache.flink.table.connector.source.LookupTableSource;
import org.apache.flink.table.connector.source.ScanTableSource;
import org.apache.flink.table.connector.source.SourceFunctionProvider;
import org.apache.flink.table.connector.source.TableFunctionProvider;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.table.types.utils.DataTypeUtils;
import org.apache.flink.types.RowKind;
import org.apache.flink.util.Preconditions;

import net.wicp.tams.common.flink.connector.redis.config.FlinkJedisConfigBase;
import net.wicp.tams.common.flink.connector.redis.config.FlinkJedisPoolConfig;
import net.wicp.tams.common.flink.connector.redis.connector.RedisRowDataLookupFunction;
import net.wicp.tams.common.flink.connector.redis.connector.RedisScanFunction;
import net.wicp.tams.common.flink.connector.redis.mapper.LookupRedisMapper;
import net.wicp.tams.common.flink.connector.redis.options.RedisLookupOptions;
import net.wicp.tams.common.flink.connector.redis.options.RedisSourceOptions;

public class RedisDynamicTableSource implements LookupTableSource, ScanTableSource {
	public static final String IDENTIFIER = "redis";
	/**
	 * Data type to configure the formats.
	 */
	protected final ResolvedSchema schema;
	private List<RowType.RowField> fields;
	/**
	 * Optional format for decoding keys from Kafka.
	 */
	protected final @Nullable DecodingFormat<DeserializationSchema<RowData>> decodingFormat;

	protected final RedisLookupOptions redisLookupOptions;

	private Configuration optionsWith;

	public RedisDynamicTableSource( ResolvedSchema schema,
			DecodingFormat<DeserializationSchema<RowData>> decodingFormat, RedisLookupOptions redisLookupOptions,
			 Configuration c) {
        this.schema=schema;
		// Format attributes
		Preconditions.checkNotNull(schema.toPhysicalRowDataType(), "Physical data type must not be null.");
		this.decodingFormat = decodingFormat;
		this.redisLookupOptions = redisLookupOptions;
		this.optionsWith = c;
	}

	@Override
	public LookupRuntimeProvider getLookupRuntimeProvider(LookupContext context) {

		String[] keyNames = new String[context.getKeys().length];

		for (int i = 0; i < keyNames.length; i++) {
			int[] innerKeyArr = context.getKeys()[i];
			Preconditions.checkArgument(
					innerKeyArr.length == 1, "redis only support non-nested look up keys");
			keyNames[i] = schema.getColumnNames().get(innerKeyArr[0]);
		}

		FlinkJedisConfigBase flinkJedisConfigBase = new FlinkJedisPoolConfig.Builder()
				.setHost(this.redisLookupOptions.getHostname()).setPort(this.redisLookupOptions.getPort()).build();

		LookupRedisMapper lookupRedisMapper = new LookupRedisMapper(this.createDeserialization(context,
				this.decodingFormat, createValueFormatProjection(this.schema.toPhysicalRowDataType())));

		return TableFunctionProvider.of(new RedisRowDataLookupFunction(flinkJedisConfigBase, lookupRedisMapper,
				this.redisLookupOptions, this.schema.toPhysicalRowDataType(), this.optionsWith,keyNames));
//        }
	}

	private @Nullable DeserializationSchema<RowData> createDeserialization(Context context,
			@Nullable DecodingFormat<DeserializationSchema<RowData>> format, int[] projection) {
		if (format == null) {
			return null;
		}
		DataType physicalFormatDataType = DataTypeUtils.projectRow(this.schema.toPhysicalRowDataType(), projection);
		return format.createRuntimeDecoder(context, physicalFormatDataType);
	}

	@Override
	public DynamicTableSource copy() {
		return new RedisDynamicTableSource(this.schema, decodingFormat, redisLookupOptions, 
				optionsWith);
	}

	@Override
	public String asSummaryString() {
		return IDENTIFIER;
	}

	@Override
	public ChangelogMode getChangelogMode() {
		boolean append = optionsWith.get(RedisSourceOptions.append).booleanValue();
		return append ? ChangelogMode.newBuilder().addContainedKind(RowKind.INSERT).build()
				: ChangelogMode.newBuilder().addContainedKind(RowKind.INSERT).addContainedKind(RowKind.UPDATE_BEFORE)
						.addContainedKind(RowKind.UPDATE_AFTER).addContainedKind(RowKind.DELETE).build();
	}

	@Override
	public ScanRuntimeProvider getScanRuntimeProvider(ScanContext runtimeProviderContext) {
		final RowType rowType = (RowType) schema.toPhysicalRowDataType().getLogicalType();
		this.fields = rowType.getFields();
		Optional<UniqueConstraint> primaryKeyopt = schema.getPrimaryKey();
		UniqueConstraint primaryKey = primaryKeyopt.isPresent()?primaryKeyopt.get():null;
		final SourceFunction<RowData> sourceFunction = new RedisScanFunction(optionsWith, fields,primaryKey);
		return SourceFunctionProvider.of(sourceFunction, false);
	}
}
