/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.flink.connector.redis.options;

import net.wicp.tams.common.Conf;
import net.wicp.tams.common.constant.ods.AddColName;

import org.apache.flink.configuration.ConfigOption;
import org.apache.flink.configuration.ConfigOptions;
import org.apache.flink.configuration.ReadableConfig;

import java.time.Duration;
import java.util.Properties;

/***
 * 规则：在controlkey下面有2个key，control:group 用于放该表下所有的键 ,control:stream
 * 用于放该表下所有的增量消息，消息示例：{"db":"demo","id":"101","keyformart":"dim","opttype":"update","tb":"user_info"}
 * 
 * @author Andy
 *
 */
public abstract class RedisSourceOptions {
	public static final ConfigOption<String> host = ConfigOptions.key("host").stringType().noDefaultValue()
			.withDescription("主机");

	public static final ConfigOption<String> controlkey = ConfigOptions.key("controlkey").stringType().noDefaultValue()
			.withDescription("组的key，用于存放此表中的所有id,相关于表的索引");

	public static final ConfigOption<String> searchkeyprefix = ConfigOptions.key("searchkeyprefix").stringType()
			.noDefaultValue().withDescription("redis_key的前缀，通常以dim:db:table表示");

	public static final ConfigOption<String> groupid = ConfigOptions.key("groupid").stringType().noDefaultValue()
			.withDescription("stream的消费groupid");

	/////////////////////////////////////////////////// 下面参数可选//////////////////////////////////////////////////////////////////////////
	// 并行度（槽）redis6才支持多线程，线程数要小于ECS的CPU数量。暂不支持
//	public static final ConfigOption<Integer> parallelism = ConfigOptions.key("parallelism").intType().defaultValue(1)
//			.withDescription("并行度");
	// 用于做cdc的字段名
	public static final ConfigOption<String> updateColName = ConfigOptions.key("updateColName").stringType()
			.defaultValue(AddColName.commitTime.getColNameTrue()).withDescription("用于分隔全量和增量的时间字段");

	// 如tenantid做为原分库分表的路由。
	public static final ConfigOption<String> routeColName = ConfigOptions.key("routeColName").stringType()
			.noDefaultValue().withDescription("路由字段");

	public static final ConfigOption<String> fieldFormart = ConfigOptions.key("fieldFormart").stringType()
			.defaultValue("ori").withDescription("所有字段的格式");// 默认是原始字段，不做处理

	// 是否为append模式
	public static final ConfigOption<Boolean> append = ConfigOptions.key("append").booleanType().defaultValue(false)
			.withDescription("是否为append模式");

	public static final ConfigOption<Integer> port = ConfigOptions.key("port").intType().defaultValue(6379)
			.withDescription("端口");

	public static final ConfigOption<Integer> db = ConfigOptions.key("db").intType().defaultValue(0)
			.withDescription("数据库");

	public static final ConfigOption<String> password = ConfigOptions.key("password").stringType().noDefaultValue()
			.withDescription("密码");

	public static final ConfigOption<String> streamkey = ConfigOptions.key("streamkey").stringType().noDefaultValue()
			.withDescription("pub的key");

	public static final ConfigOption<String> groupkey = ConfigOptions.key("groupkey").stringType().noDefaultValue()
			.withDescription("存放所有记录的key集合");

	public static final ConfigOption<Long> lookupcachemaxrows = ConfigOptions.key("lookup.cache.max-rows").longType()
			.defaultValue(100000L).withDescription("缓存最大行数");

	public static final ConfigOption<Duration> lookupcachettlsec = ConfigOptions.key("lookup.cache.ttl-sec")
			.durationType().defaultValue(Duration.ofSeconds(600)).withDescription("缓存超期时间");

	public static final ConfigOption<Boolean> lookupasync = ConfigOptions.key("lookup.async").booleanType()
			.defaultValue(false).withDescription("whether to set async lookup.");

	public static final ConfigOption<Integer> lookupmaxretries = ConfigOptions.key("lookup.max-retries").intType()
			.defaultValue(1).withDescription("the max retry times if lookup database failed.");

	public static void packageOptions(ReadableConfig options) {
		Properties props = new Properties();
		String serverName = options.get(RedisSourceOptions.groupid);// 用groupid做服务名，groupid每个表都必须惟一
		props.put(String.format("common.redis.redisserver.%s.defaultDb", serverName),
				options.get(RedisSourceOptions.db));
		if (options.getOptional(RedisSourceOptions.password).isPresent()) {
			props.put(String.format("common.redis.redisserver.%s.password", serverName),
					options.get(RedisSourceOptions.password));
		}
		props.put(String.format("common.redis.redisserver.%s.host", serverName), options.get(RedisSourceOptions.host));
		props.put(String.format("common.redis.redisserver.%s.port", serverName),
				options.get(RedisSourceOptions.port).toString());

		/// 用于传递数据，common-redis用不到
		String streamkey = null;
		if (options.getOptional(RedisSourceOptions.streamkey).isPresent()) {
			streamkey = options.getOptional(RedisSourceOptions.streamkey).get();
		} else {// 默认为groupkey下面的stream目录
			streamkey = String.format("%s:%s:%s", options.get(RedisSourceOptions.searchkeyprefix), "control", "stream");
		}
		//
		String groupkey = null;
		if (options.getOptional(RedisSourceOptions.groupkey).isPresent()) {
			groupkey = options.getOptional(RedisSourceOptions.groupkey).get();
		} else {// 默认为groupkey下面的stream目录
			groupkey = String.format("%s:%s:%s", options.get(RedisSourceOptions.searchkeyprefix), "control", "group");
		}
		props.put(String.format("common.redis.redisserver.%s.streamkey", serverName), streamkey);
		props.put(String.format("common.redis.redisserver.%s.groupkey", serverName), groupkey);
		// props.put(String.format("common.redis.redisserver.%s.groupid", serverName),
		// options.get(RedisSourceOptions.groupid));
		Conf.overProp(props);
	}

	public static RedisLookupOptions getRedisLookupOptions(ReadableConfig tableOptions) {
		return (RedisLookupOptions) RedisLookupOptions.builder().setLookupAsync(tableOptions.get(lookupasync))
				.setMaxRetryTimes(tableOptions.get(lookupmaxretries))
				.setCacheExpireMs(tableOptions.get(lookupcachettlsec).toMillis())
				.setCacheMaxSize(tableOptions.get(lookupcachemaxrows)).setHostname(tableOptions.get(host))
				.setPort(tableOptions.get(port)).setSearchkeyprefix(tableOptions.get(searchkeyprefix))
				.setGroupid(tableOptions.get(groupid)).build();
	}
}
