/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.flink.connector.redis.connector;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.stream.IntStream;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.flink.api.common.functions.RuntimeContext;
import org.apache.flink.api.common.typeinfo.TypeInformation;
import org.apache.flink.api.java.typeutils.ResultTypeQueryable;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.streaming.api.functions.source.RichSourceFunction;
import org.apache.flink.table.catalog.UniqueConstraint;
import org.apache.flink.table.data.GenericRowData;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.LogicalTypeRoot;
import org.apache.flink.table.types.logical.RowType.RowField;
import org.apache.flink.table.types.logical.utils.LogicalTypeChecks;
import org.apache.flink.types.RowKind;
import org.apache.flink.util.Preconditions;

import com.google.protobuf.InvalidProtocolBufferException;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.apiext.DateUtil;
import net.wicp.tams.common.apiext.LoggerUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.TimeAssist;
import net.wicp.tams.common.binlog.alone.DuckulaAssit;
import net.wicp.tams.common.binlog.alone.ListenerConf.DuckulaEvent;
import net.wicp.tams.common.binlog.alone.ListenerConf.DuckulaEventItem;
import net.wicp.tams.common.constant.DateFormatCase;
import net.wicp.tams.common.constant.FieldFormart;
import net.wicp.tams.common.constant.JvmStatus;
import net.wicp.tams.common.constant.ods.AddColName;
import net.wicp.tams.common.constant.ods.AddColNameType;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.flink.common.constant.FlinkTypeEnum;
import net.wicp.tams.common.flink.connector.redis.options.RedisSourceOptions;
import net.wicp.tams.common.redis.RedisAssit;
import net.wicp.tams.common.redis.pool.AbsPool;
import redis.clients.jedis.Jedis;

/***
 * 不需要CheckpointedFunction
 * 
 * @author Andy.zhou
 *
 */
@Slf4j
public class RedisScanFunction extends RichSourceFunction<RowData> implements ResultTypeQueryable<RowData> {
	private static final long serialVersionUID = 1L;

	private final Configuration optionsWith;

	private final List<RowField> rowTypeFields;

	private final List<String> keys;

	private final String routeColName;

	private final List<String> addColNames = new ArrayList<String>();// 附加字段

	private final boolean append;// 流模式

	private AbsPool standalone;

	// net.wicp.tams.common.constant.FieldFormart$1@350f18a6 is not serializable.
	// The object probably contains or references non serializable fields.
	private transient FieldFormart fieldFormart;// 列格式

	private final long beginTime;// 当前时间为一个分界

	/***
	 * 需要在配置文件中存在相关的配置
	 * 
	 * @param optionsWith
	 * @param rowTypeFields
	 */
	public RedisScanFunction(Configuration optionsWith, List<RowField> rowTypeFields, UniqueConstraint primaryKey) {
		this.optionsWith = optionsWith;
		this.rowTypeFields = rowTypeFields;
		this.keys = primaryKey == null ? null : primaryKey.getColumns();
		this.append = optionsWith.get(RedisSourceOptions.append);
		this.routeColName = optionsWith.get(RedisSourceOptions.routeColName);
		this.fieldFormart = FieldFormart.valueOf(optionsWith.get(RedisSourceOptions.fieldFormart));
		// 取
		this.beginTime = System.currentTimeMillis();
		// 检查类型
		List<String> allColList = AddColName.getAllColNameTrue(this.fieldFormart);
		for (RowField rowField : rowTypeFields) {
			if (allColList.contains(rowField.getName())) {
				this.addColNames.add(rowField.getName());
			}
			FlinkTypeEnum flinkTypeEnum = FlinkTypeEnum.findByFlinkRowType(rowField.getType().getTypeRoot().toString());
			if (flinkTypeEnum == null) {
				throw new ProjectExceptionRuntime(ExceptAll.param_notfit,
						"列：【" + rowField.getName() + "】不支持的类型【" + rowField.getType().getTypeRoot().toString() + "】");
			}
		}
	}

	@Override
	public void run(SourceContext<RowData> ctx) throws Exception {
		TimeZone.setDefault(TimeZone.getTimeZone("GMT+8"));// TODO写死时区，临时方案，否则下面的curTime.getTime()会有问题比
		RedisSourceOptions.packageOptions(this.optionsWith);
		String serverName = optionsWith.get(RedisSourceOptions.groupid);// 用groupid做服务名
		String streamkey = Conf.get(String.format("common.redis.redisserver.%s.streamkey", serverName));
		String groupkey = Conf.get(String.format("common.redis.redisserver.%s.groupkey", serverName));
		if (this.standalone == null || !this.standalone.isInit()) {
			this.standalone = RedisAssit.standalone(serverName);
		}
		// 做全量
		Jedis jedis = standalone.getResource();
		// log.info("==========groupkey=" + groupkey);
		Set<String> smembers = jedis.smembers(groupkey);
		// log.info("====================smembers size=" + smembers.size());
		if (this.fieldFormart == null) {
			this.fieldFormart = FieldFormart.valueOf(optionsWith.get(RedisSourceOptions.fieldFormart));
		}
		// log.info("=======fieldFormart=" + fieldFormart.name() + "=====append=" +
		// append);
		for (String smember : smembers) {
			Map<String, String> allValue = jedis.hgetAll(smember);
			// 20220724 andy.zhou 全量需要考虑没有配置id和routecol的情况
			if (StringUtil.isNotNull(routeColName) && !allValue.containsKey(routeColName)) {
				String[] keyAry = smember.split(":");
				allValue.put(routeColName, keyAry[keyAry.length - 2]);
			}
			if (CollectionUtils.isNotEmpty(keys) && keys.size() == 1 && !allValue.containsKey(keys.get(0))) {// 暂只支持一个key的情况
				String[] keyAry = smember.split(":");
				allValue.put(keys.get(0), keyAry[keyAry.length - 1]);
			}
			// 20220724 end andy.zhou 全量需要考虑没有配置id和routecol的情况
			if (CollectionUtils.isNotEmpty(this.addColNames)) {
				for (AddColName addColName : AddColName.values()) {
					if (!allValue.containsKey(addColName.getColNameTrue(this.fieldFormart))) {// 如果redis有值就不需要自己来产生了。
						switch (addColName) {
						case commitTime:// 全量会把当前时间做为binlog时间，为的是保证rowdate不为空（如果它做为watermark时）
							allValue.put(addColName.getColNameTrue(this.fieldFormart),
									DateFormatCase.YYYY_MM_DD_hhmmss.getInstanc().format(System.currentTimeMillis()));
							break;
						case dumpTime:
							allValue.put(addColName.getColNameTrue(this.fieldFormart),
									DateFormatCase.YYYY_MM_DD_hhmmss.getInstanc().format(System.currentTimeMillis()));
							break;
						case isDelete:
							allValue.put(addColName.getColNameTrue(this.fieldFormart), "0");
							break;
						case lastOpttype:
							allValue.put(addColName.getColNameTrue(this.fieldFormart), "insert");
							break;
						case oriDb:
							allValue.put(addColName.getColNameTrue(this.fieldFormart), "");// 到了redis已分不清元始的库名
							break;
						case oriTb:
							allValue.put(addColName.getColNameTrue(this.fieldFormart), "");// 到了redis已分不清元始的表名
							break;
						default:
							break;
						}
					}
				}
			}
			// log.info("=========id=" + allValue.get("id"));
			if (!append) {// 非append不需要对比时间
				String colName = this.fieldFormart.getColName(this.optionsWith.get(RedisSourceOptions.updateColName));
				// log.info("curtime====="+allValue.get(colName));
				Date curTime = DateUtil.objToDate(allValue.get(colName));
				// log.info("curtime2="+DateFormatCase.YYYY_MM_DD_hhmmss.getInstanc().format(curTime));
				if (curTime == null) {
					log.warn("对比字段:[{}]存在空值，数据：[{}]，不能保证 exactly once语句", colName, allValue);// 不能抛异常，否则正常的select会阻塞
				} else {
					if (beginTime <= curTime.getTime()) {// 更新时间大于当前时间不要纳入
						log.info("失效：====beginTime=" + beginTime + "    curTime=" + curTime.getTime());
						continue;
					}
				}
			}
			GenericRowData rowData = new GenericRowData(rowTypeFields.size());
			rowData.setRowKind(RowKind.INSERT);
			// addValueMap与allValue合并了
			packRow(null, allValue, rowData);
			ctx.collect(rowData);
			// ctx.collect(rowData);
			// log.info("=========getRowKind=" + rowData.getRowKind());
		}
		this.standalone.returnResource(jedis);
		log.info("======================全量已做完，开始做增量=======================");
		// 拿增量
		while (true) {
			try {
				sendFlink(ctx, serverName, streamkey, beginTime);
				TimeAssist.reDoWaitInit("need-init");// 重新等待
			} catch (Throwable e) {// 防止网络闪动，断网重试
				boolean reDoWait = TimeAssist.reDoWait("need-init", 8);
				log.error("readdo error", e);
				if (reDoWait) {// 256秒即4分种*2=8分钟，8分钟就不是网络抖动问题了
					log.error("已重试8次，退出系统");
					LoggerUtil.exit(JvmStatus.s15);
				}
			}
		}
	}

	private void sendFlink(SourceContext<RowData> ctx, String serverName, String streamkey, long minTime)
			throws InvalidProtocolBufferException {
		Pair<Long, byte[]> msg = standalone.getStreamBlockOneDuckulaEvent(streamkey, serverName);// 使用serverName做为stream的groupid
		// msg==null可能出现异常。
		if (msg == null || msg.getLeft() < minTime) {// 小于指定时间的消息需要过滤掉，不处理
			return;
		}
		DuckulaEvent duckulaEvent = DuckulaEvent.parseFrom(msg.getRight());
		for (DuckulaEventItem item : duckulaEvent.getItemsList()) {
			GenericRowData rowDataafter = new GenericRowData(rowTypeFields.size());
			switch (duckulaEvent.getOptType()) {
			case insert:
				rowDataafter.setRowKind(RowKind.INSERT);
				break;
			case update:
				if (append) {
					return;
				}
				GenericRowData rowDataBefore = new GenericRowData(rowTypeFields.size());
				Map<String, String> addValueMap = CollectionUtils.isEmpty(this.addColNames)
						? new HashMap<String, String>()
						: DuckulaAssit.getAddColValuesStr(duckulaEvent, AddColNameType.all_ori, this.fieldFormart);
				packRow(addValueMap, item.getBeforeMap(), rowDataBefore);
				rowDataBefore.setRowKind(RowKind.UPDATE_BEFORE);
				ctx.collect(rowDataBefore);
				rowDataafter.setRowKind(RowKind.UPDATE_AFTER);
				break;
			case delete:
				if (append) {
					return;
				}
				rowDataafter.setRowKind(RowKind.DELETE);
				break;
			default:
				throw new ProjectExceptionRuntime(ExceptAll.duckula_datanofit,
						"redis不支持此类型" + duckulaEvent.getOptType().name());
			}
			Map<String, String> itemmap = item.getAfterMap();
			Map<String, String> addValueMap = CollectionUtils.isEmpty(this.addColNames) ? new HashMap<String, String>()
					: DuckulaAssit.getAddColValuesStr(duckulaEvent, AddColNameType.all_ori, this.fieldFormart);
			packRow(addValueMap, itemmap, rowDataafter);
			ctx.collect(rowDataafter);
		}
	}

	private void packRow(Map<String, String> addValueMap, Map<String, String> allValue, GenericRowData rowData) {
		for (int i = 0; i < rowTypeFields.size(); i++) {
			RowField rowField = rowTypeFields.get(i);
			FlinkTypeEnum flinkTypeEnum = FlinkTypeEnum.findByFlinkRowType(rowField.getType().getTypeRoot().toString());
			Object value = null;
			if (MapUtils.isNotEmpty(addValueMap) && addValueMap.containsKey(rowField.getName())) {// 附加字段
				value = FlinkTypeEnum.getValue(flinkTypeEnum,addValueMap.get(rowField.getName()), rowField.getType());
			} else {
				value = allValue.containsKey(rowField.getName())
						? FlinkTypeEnum.getValue(flinkTypeEnum,allValue.get(rowField.getName()), rowField.getType())
						: null;
			}
			if (value != null) {
				rowData.setField(i, value);
			}
		}
	}

	public static int[] createValueFormatProjection(DataType physicalDataType) {
		final LogicalType physicalType = physicalDataType.getLogicalType();
		Preconditions.checkArgument(physicalType.getTypeRoot() == LogicalTypeRoot.ROW, "Row data type expected.");
		final int physicalFieldCount = LogicalTypeChecks.getFieldCount(physicalType);
		final IntStream physicalFields = IntStream.range(0, physicalFieldCount);

		return physicalFields.toArray();
	}

	@Override
	public void cancel() {
		if (this.standalone != null || this.standalone.isInit()) {
			this.standalone.destroy();
		}
	}

	@Override
	public TypeInformation<RowData> getProducedType() {
		return null;
	}

	@Override
	public void setRuntimeContext(RuntimeContext t) {
		super.setRuntimeContext(t);
	}

	@Override
	public void open(Configuration parameters) throws Exception {
		super.open(parameters);
		RedisSourceOptions.packageOptions(optionsWith);
		String serverName = optionsWith.get(RedisSourceOptions.groupid);// 用groupid做服务名
		standalone = RedisAssit.standalone(serverName);
	}

}
