/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.flink.connector.kafka.source;

import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import org.apache.flink.api.common.serialization.DeserializationSchema;
import org.apache.flink.api.common.serialization.SerializationSchema;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.streaming.api.functions.source.SourceFunction;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.catalog.UniqueConstraint;
import org.apache.flink.table.connector.ChangelogMode;
import org.apache.flink.table.connector.format.DecodingFormat;
import org.apache.flink.table.connector.source.DynamicTableSource;
import org.apache.flink.table.connector.source.ScanTableSource;
import org.apache.flink.table.connector.source.SourceFunctionProvider;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.table.types.utils.DataTypeUtils;
import org.apache.flink.util.Preconditions;

import net.wicp.tams.common.constant.Middleware;
import net.wicp.tams.common.flink.connector.kafka.connector.KafkaScanFunction;

public class KafkaDynamicTableSource implements ScanTableSource {

	/**
	 * Data type to configure the formats.
	 */
	protected final ResolvedSchema schema;
	private List<RowType.RowField> fields;
	private final String tableName;
	/**
	 * Optional format for decoding keys from Kafka.
	 */
	protected final @Nullable DecodingFormat<DeserializationSchema<RowData>> decodingFormat;

	private Configuration optionsWith;

	public KafkaDynamicTableSource(ResolvedSchema schema, DecodingFormat<DeserializationSchema<RowData>> decodingFormat,
			Configuration c, String tableName) {
		this.schema = schema;
		this.tableName = tableName;
		// Format attributes
		Preconditions.checkNotNull(schema.toPhysicalRowDataType(), "Physical data type must not be null.");
		this.decodingFormat = decodingFormat;
		this.optionsWith = c;
	}

	private @Nullable DeserializationSchema<RowData> createDeserialization(Context context,
			@Nullable DecodingFormat<DeserializationSchema<RowData>> format, int[] projection) {
		if (format == null) {
			return null;
		}
		DataType physicalFormatDataType = DataTypeUtils.projectRow(this.schema.toPhysicalRowDataType(), projection);
		return format.createRuntimeDecoder(context, physicalFormatDataType);
	}

	@Override
	public DynamicTableSource copy() {
		return new KafkaDynamicTableSource(this.schema, decodingFormat, optionsWith, this.tableName);
	}

	@Override
	public String asSummaryString() {
		return Middleware.tamskafka.getDesc();
	}

	@Override
	public ChangelogMode getChangelogMode() {
		return ChangelogMode.insertOnly();
	}

	@Override
	public ScanRuntimeProvider getScanRuntimeProvider(ScanContext runtimeProviderContext) {
		final RowType rowType = (RowType) schema.toPhysicalRowDataType().getLogicalType();
		this.fields = rowType.getFields();
		Optional<UniqueConstraint> primaryKeyopt = schema.getPrimaryKey();
		UniqueConstraint primaryKey = primaryKeyopt.isPresent() ? primaryKeyopt.get() : null;
		
		DeserializationSchema<RowData> deserializationSchema = this.decodingFormat.createRuntimeDecoder(runtimeProviderContext, schema.toSinkRowDataType());
		
		final SourceFunction<RowData> sourceFunction = new KafkaScanFunction(optionsWith, fields, primaryKey,
				this.tableName,deserializationSchema);
		return SourceFunctionProvider.of(sourceFunction, false);
	}
}
