
package net.wicp.tams.common.flink.connector.kafka.connector;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.flink.api.common.serialization.SerializationSchema;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.streaming.api.functions.sink.RichSinkFunction;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.table.types.logical.RowType.RowField;
import org.apache.flink.types.RowKind;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.PartitionInfo;

import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;

import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.binlog.alone.DuckulaAssit;
import net.wicp.tams.common.binlog.alone.ListenerConf.DuckulaEvent;
import net.wicp.tams.common.flink.common.constant.FlinkTypeEnum;
import net.wicp.tams.common.flink.common.schema.DuckulaDeserializationSchema;
import net.wicp.tams.common.flink.common.schema.DuckulaSerializationSchema;
import net.wicp.tams.common.flink.connector.kafka.KafkaOptions;
import net.wicp.tams.common.kafka.KafkaAssitInst;

/**
 * 自定义 sink 写入 kafka。 duckula的formate会把
 * opt_type字段设置为整个记录的opttype.这样后续就可以做为upsetkafka或appendkafka继续使用。 而如果是json
 * formate，则不能做为upsetkafka做为后流使用。只能做为appendkafka供后流使用。
 **/
public class KafkaRichSinkFunction extends RichSinkFunction<RowData> {
	private static final long serialVersionUID = 1L;
	private final int[] keyColIndex;// 主键,需要index
	private final List<String> keys;// 主键的值
	private final int[] partitionColIndex;// 分区键的index
	private final int changedIndex;// 修改前的值
	private final List<RowField> fields;
	private final SerializationSchema<RowData> serialization;
	private final Configuration optionsWith;
	private final String topic;
	private final int partitions;

	public KafkaRichSinkFunction(ResolvedSchema schema, SerializationSchema<RowData> serialization,
			List<String> partitionKeys, Configuration optionsWith) {
		if (schema.getPrimaryKey().isPresent()) {// 有主键
			this.keys = schema.getPrimaryKey().get().getColumns();
			keyColIndex = new int[this.keys.size()];
			int index = 0;
			for (int i = 0; i < schema.getColumns().size(); i++) {
				if (this.keys.contains(schema.getColumns().get(i).getName())) {
					keyColIndex[index++] = i;
				}
			}
		} else {
			keyColIndex = new int[0];
			this.keys = new ArrayList<String>();
		}

		if (CollectionUtils.isNotEmpty(partitionKeys)) {// 有分区情况
			partitionColIndex = new int[partitionKeys.size()];
			int index = 0;
			for (int i = 0; i < schema.getColumns().size(); i++) {
				if (partitionKeys.contains(schema.getColumns().get(i).getName())) {
					partitionColIndex[index++] = i;
				}
			}
		} else {// 无分区情况
			partitionColIndex = new int[0];
		}
		// toSourceRowDataType是全部的字段，toSinkRowDataType是select后面的字段
		final RowType rowType = (RowType) schema.toSinkRowDataType().getLogicalType();
		this.fields = rowType.getFields();

		// 找到before键,只能从真实sink的字段里去找，而不应该从schema的fields里去找
		int beforValueIndexTemp = -1;
		for (int i = 0; i < this.fields.size(); i++) {
			if (this.fields.get(i).getName().equals(DuckulaDeserializationSchema.changed)) {
				beforValueIndexTemp = i;
				break;
			}
//			sink不能采用meta字段拿数据。
//			if (column instanceof MetadataColumn) {
//				MetadataColumn temp = (MetadataColumn) column;
//				if (temp.getMetadataKey().get().equals("value."+DuckulaDeserializationSchema.changed)) {
//					beforValueIndexTemp = i;
//					break;
//				}
//			}
		}
		this.changedIndex = beforValueIndexTemp;

//		this.colNames = schema.getColumnNames().toArray(new String[schema.getColumnNames().size()]);
		this.serialization = serialization;
		this.optionsWith = optionsWith;
		this.topic = optionsWith.getString(KafkaOptions.topic);
		KafkaProducer<String, byte[]> kafkaProducer = KafkaAssitInst.getInst().getKafkaProducer(byte[].class);
		List<PartitionInfo> partiList = kafkaProducer.partitionsFor(topic);
		this.partitions = partiList.size();
	}

	@Override
	public void open(Configuration parameters) throws Exception {
		super.open(parameters);
		//20230425 修改bug.后面想办法拿到SerializationSchema.InitializationContext
		this.serialization.open(null);
		KafkaOptions.packageOptionsSink(optionsWith);
	}

	@Override
	public void invoke(RowData value, Context context) throws Exception {
		RowKind rowKind = value.getRowKind();
		if (rowKind == RowKind.UPDATE_BEFORE) {// 忽略UPDATE_BEFORE,在updatekafka是一个墓碑，在kafka是一个delete事件，但json
												// formate不支持sink
			return;
		}
		byte[] serialize = serialization.serialize(value);// 序列化为duckula,因为消息需要
		// 组装好key的json
		ObjectNode keyjson = JsonNodeFactory.instance.objectNode();
		for (int keyIndex : keyColIndex) {
			FlinkTypeEnum flinkTypeEnum = FlinkTypeEnum
					.findByFlinkRowType(fields.get(keyIndex).getType().getTypeRoot().toString());
//			LogicalType logicalType = rowType.getFields().get(i).getType();
			Object rowDataValue = flinkTypeEnum.getRowDataValue(value, keyIndex, fields.get(keyIndex).getType());
			keyjson.putPOJO(fields.get(keyIndex).getName(), rowDataValue);
		}

		KafkaProducer<String, byte[]> kafkaProducer = KafkaAssitInst.getInst().getKafkaProducer(byte[].class);
		ProducerRecord<String, byte[]> message = null;
		ProducerRecord<String, byte[]> messageBefore = null;
		String hedgeValue = optionsWith.getString(KafkaOptions.hedge);
		boolean needHedge = "logic".equals(hedgeValue) || "physics".equals(hedgeValue);// 是否需要对冲
		if (partitions == 1 || (ArrayUtils.isEmpty(partitionColIndex) && ArrayUtils.isEmpty(keyColIndex))) {// 没设置分区且没有主键
			message = new ProducerRecord<String, byte[]>(this.topic, keyjson.toString(), serialize);

			if (needHedge && keyColIndex.length > 0 && changedIndex >= 0
					&& (serialization instanceof DuckulaSerializationSchema)) {// 需要对冲,有配置_changed值, 有主键。
																				// 且是duckula的formate时
				Pair<ObjectNode, DuckulaEvent> parseHedge = DuckulaAssit.parseHedge(serialize,
						fields.get(changedIndex).getName(), "logic".equals(hedgeValue));
				if (parseHedge != null) {// 有改变
					messageBefore = new ProducerRecord<String, byte[]>(this.topic, parseHedge.getLeft().toString(),
							parseHedge.getRight().toByteArray());
				}
			}
		} else {
			StringBuffer buff = new StringBuffer();
			if (ArrayUtils.isNotEmpty(partitionColIndex)) {
				for (int partitionKeyIndex : partitionColIndex) {// 如何分区
					buff.append(FlinkTypeEnum.getStr(fields.get(partitionKeyIndex), value, partitionKeyIndex) + "`");
				}
			} else {
				for (int keyIndex : keyColIndex) {
					buff.append(FlinkTypeEnum.getStr(fields.get(keyIndex), value, keyIndex) + "`");
				}
			}
			if (needHedge && keyColIndex.length > 0 && changedIndex >= 0
					&& (serialization instanceof DuckulaSerializationSchema)) {// 需要对冲,有配置_changed值,且是duckula的formate时
				Pair<ObjectNode, DuckulaEvent> parseHedge = DuckulaAssit.parseHedge(serialize,
						fields.get(changedIndex).getName(), "logic".equals(hedgeValue));
				if (parseHedge != null) {// 有改变
					StringBuffer buffBefore = new StringBuffer();
					if (ArrayUtils.isNotEmpty(partitionColIndex)) {
						for (int partitionKeyIndex : partitionColIndex) {// 如何分区
							buffBefore.append(DuckulaAssit.getValueStr(parseHedge.getRight(), 0,
									fields.get(partitionKeyIndex).getName()) + "`");
						}
					} else {
						for (int keyIndex : keyColIndex) {
							buffBefore.append(
									DuckulaAssit.getValueStr(parseHedge.getRight(), 0, fields.get(keyIndex).getName())
											+ "`");
						}
					}
					messageBefore = new ProducerRecord<String, byte[]>(this.topic,
							StringUtil.partition(buffBefore.toString(), partitions),
							parseHedge.getLeft().toString(), parseHedge.getRight().toByteArray());
				}
			}

			message = new ProducerRecord<String, byte[]>(this.topic, StringUtil.partition(buff.toString(), partitions),
					keyjson.toString(), serialize);
		}
		if (messageBefore != null) {// 是否有对冲数据
			kafkaProducer.send(messageBefore);
		}
		kafkaProducer.send(message);
	}

	@Override
	public void close() throws Exception {

	}
}