/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.flink.connector.kafka.sink;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.flink.api.common.serialization.SerializationSchema;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.connector.ChangelogMode;
import org.apache.flink.table.connector.format.EncodingFormat;
import org.apache.flink.table.connector.sink.DynamicTableSink;
import org.apache.flink.table.connector.sink.SinkFunctionProvider;
import org.apache.flink.table.connector.sink.abilities.SupportsPartitioning;
import org.apache.flink.table.data.RowData;
import org.apache.flink.types.RowKind;

import net.wicp.tams.common.constant.Middleware;
import net.wicp.tams.common.flink.connector.kafka.connector.KafkaRichSinkFunction;

/**
 * 自定义 sink 写入 redis
 **/
public class KafkaDynamicTableSink implements DynamicTableSink, SupportsPartitioning {
	private final ResolvedSchema schema;
	private final EncodingFormat<SerializationSchema<RowData>> encodingFormat;
	private final Configuration options;
	private final List<String> partitionKeys;
	private Map<String, String> staticPartitions = new LinkedHashMap<>();

	public KafkaDynamicTableSink(ResolvedSchema schema, EncodingFormat<SerializationSchema<RowData>> encodingFormat,
			Configuration options,List<String> partitionKeys) {
		this.schema = schema;
		this.encodingFormat = encodingFormat;
		this.options = options;
		this.partitionKeys=partitionKeys;
	}

	@Override
	public ChangelogMode getChangelogMode(ChangelogMode requestedMode) {
		return ChangelogMode.newBuilder().addContainedKind(RowKind.INSERT).addContainedKind(RowKind.UPDATE_BEFORE)
				.addContainedKind(RowKind.UPDATE_AFTER).addContainedKind(RowKind.DELETE).build();
	}

	@Override
	public SinkRuntimeProvider getSinkRuntimeProvider(Context context) {
		SerializationSchema<RowData> serializationSchema = encodingFormat.createRuntimeEncoder(context,
				schema.toSinkRowDataType());
		KafkaRichSinkFunction redisRichSinkFunction = new KafkaRichSinkFunction(schema, serializationSchema,
				partitionKeys, options);
		return SinkFunctionProvider.of(redisRichSinkFunction);
	}

	@Override
	public DynamicTableSink copy() {
		return new KafkaDynamicTableSink(this.schema, this.encodingFormat, this.options,this.partitionKeys);
	}

	@Override
	public String asSummaryString() {
		return Middleware.tamskafka.getDesc();
	}

	// 需要实现可分区，否则通过 insert into会不成功。
	// Table 'mycatalog.mall_rebate_dev.t_test' is a partitioned table, but the
	// underlying DynamicTableSink doesn't implement the SupportsPartitioning
	// interface.
//	@Override
//	public int partition(Object key, int numPartitions) {
//		return 0;
//	}

	@Override
	public void applyStaticPartition(Map<String, String> partition) {
		// make it a LinkedHashMap to maintain partition column order
        staticPartitions = new LinkedHashMap<>();
        for (String partitionCol : partitionKeys) {
            if (partition.containsKey(partitionCol)) {
                staticPartitions.put(partitionCol, partition.get(partitionCol));
            }
        }
	}
}