/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.flink.connector.binlog.connector;

import java.util.Iterator;
import java.util.List;

import org.apache.flink.api.common.ExecutionConfig;
import org.apache.flink.api.common.functions.RuntimeContext;
import org.apache.flink.api.common.state.ListState;
import org.apache.flink.api.common.state.ListStateDescriptor;
import org.apache.flink.api.common.typeinfo.TypeHint;
import org.apache.flink.api.common.typeinfo.TypeInformation;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.runtime.state.FunctionInitializationContext;
import org.apache.flink.runtime.state.FunctionSnapshotContext;
import org.apache.flink.streaming.api.checkpoint.CheckpointedFunction;
import org.apache.flink.streaming.api.functions.source.RichSourceFunction;

import com.twitter.chill.protobuf.ProtobufSerializer;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.binlog.alone.BusiAssit;
import net.wicp.tams.common.binlog.alone.ListenerConf.CheckPoint;
import net.wicp.tams.common.binlog.alone.ListenerConf.ColHis;
import net.wicp.tams.common.binlog.alone.ListenerConf.ConnConf;
import net.wicp.tams.common.binlog.alone.ListenerConf.ConnConf.Builder;
import net.wicp.tams.common.binlog.alone.ListenerConf.DuckulaEvent;
import net.wicp.tams.common.binlog.alone.ListenerConf.Position;
import net.wicp.tams.common.binlog.alone.parser.ParseLogOnline;

@Slf4j
public class BinlogSource extends RichSourceFunction<DuckulaEvent> implements CheckpointedFunction {

	private static final long serialVersionUID = 1L;

	private ConnConf connConf;
	private ParseLogOnline logFetcher;

	private static final String listener = "net.wicp.tams.common.flink.connector.binlog.connector.FlinkBinlogListener";

	private static final String chk = "net.wicp.tams.common.binlog.alone.checkpoint.CheckPointMemory";

	public BinlogSource(ConnConf.Builder connConfBuilder, ExecutionConfig config) {
		connConfBuilder.setListener(listener);
		// common.binlog.alone.conf.default.chk
		connConfBuilder.setChk(chk);
		log.info("====设置chk:{}", chk);
		this.connConf = connConfBuilder.build();
		// 注册,这个是全范围，不是算子范围，所以不能在setRuntimeContext方法里一同配置
		if (config != null) {// 测试用
			config.registerTypeWithKryoSerializer(DuckulaEvent.class, ProtobufSerializer.class);
		}
	}

	/***
	 * 需要在配置文件中存在相关的配置
	 * 
	 * @param configKey 监听配置key
	 */
	public BinlogSource(String configKey, ExecutionConfig config) {
		ConnConf.Builder connConfBuilder = BusiAssit.configMap(configKey);
		connConfBuilder.setListener(listener);
		connConfBuilder.setChk(chk);
		log.info("====设置chk:{}", chk);
		this.connConf = connConfBuilder.build();
		if (config != null) {
			config.registerTypeWithKryoSerializer(DuckulaEvent.class, ProtobufSerializer.class);
		}
	}

	// 传入全局的config,
	public BinlogSource(ExecutionConfig config) {
		this("_global", config);
	}

	@Override
	public void setRuntimeContext(RuntimeContext t) {
		super.setRuntimeContext(t);
		t.getExecutionConfig().registerTypeWithKryoSerializer(CheckPoint.class, ProtobufSerializer.class);
		// t.getExecutionConfig().registerTypeWithKryoSerializer(DuckulaEvent.class,
		// ProtobufSerializer.class);
	}

	@Override
	public void open(Configuration parameters) throws Exception {
		super.open(parameters);
		if (parameters != null) {// 为了测试用
			getRuntimeContext().getUserCodeClassLoader().loadClass(listener);
			getRuntimeContext().getUserCodeClassLoader().loadClass(chk);
		}
		/*
		 * getRuntimeContext().getUserCodeClassLoader().loadClass("org.h2.Driver");
		 * getRuntimeContext().getUserCodeClassLoader().loadClass(
		 * "org.h2.mvstore.db.TransactionStore");
		 * getRuntimeContext().getUserCodeClassLoader().loadClass(
		 * "org.h2.mvstore.db.TransactionStore$1");
		 */
		this.logFetcher = new ParseLogOnline(connConf.toBuilder());
	}

	private List<ColHis> colsList = null;

	@Override
	public void run(SourceContext<DuckulaEvent> ctx) throws Exception {
		FlinkBinlogListener binlogListener = (FlinkBinlogListener) this.logFetcher.getBuffType()
				.getBinlogListenerProxy().getIBinlogListener(this.connConf.getHost());
		binlogListener.setCtx(ctx);
		this.logFetcher.setColHis(this.colsList);
		this.logFetcher.read();
	}

	@Override
	public void cancel() {
		if (this.logFetcher != null) {
			log.info("============cancel the logFetcher");
			this.logFetcher.close();
		}
	}

	private transient ListState<CheckPoint> checkpointedState;

	@Override
	public void snapshotState(FunctionSnapshotContext context) throws Exception {
		checkpointedState.clear();
		CheckPoint checkPoint = logFetcher.getCheckPointCur();// .getCheckPoint(timestamp);
		checkpointedState.add(checkPoint);
	}

	@Override
	public void initializeState(FunctionInitializationContext context) throws Exception {
		ListStateDescriptor<CheckPoint> descriptor = new ListStateDescriptor<>("duckula-checkPoint",
				TypeInformation.of(new TypeHint<CheckPoint>() {
				}));
		checkpointedState = context.getOperatorStateStore().getListState(descriptor);
		if (context.isRestored()) {
			Iterator<CheckPoint> iterator = checkpointedState.get().iterator();
			if (iterator.hasNext()) {
				Builder newBuilder = connConf.toBuilder();
				CheckPoint checkPoint = iterator.next();
				Position pos = checkPoint.getPos();
				log.info("the binlog begin from:{}", pos.getGtids());
				newBuilder.setPos(pos);
				this.connConf = newBuilder.build();
				this.colsList = checkPoint.getColsList();
			}
		}
	}
}
