package net.wicp.tams.common.flink.catalog;

import com.alibaba.fastjson.JSONObject;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.jdbc.MySqlColBean;
import net.wicp.tams.common.apiext.jdbc.MySqlTbBean;
import net.wicp.tams.common.apiext.json.JSONUtil;
import net.wicp.tams.common.constant.Middleware;
import net.wicp.tams.common.constant.MiddlewareOption;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.flink.common.CatalogAssit;
import net.wicp.tams.common.flink.common.constant.DelLevel;
import net.wicp.tams.common.flink.common.constant.FlinkTypeEnum;
import net.wicp.tams.common.flink.common.constant.db.*;
import net.wicp.tams.common.flink.common.module.IPackageEnum;
import net.wicp.tams.common.flink.common.module.IUdfEnum;
import net.wicp.tams.common.jdbc.DruidAssit;
import net.wicp.tams.common.jdbc.MySqlAssitExt;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.flink.table.api.Schema;
import org.apache.flink.table.api.Schema.Builder;
import org.apache.flink.table.api.Schema.UnresolvedColumn;
import org.apache.flink.table.api.Schema.UnresolvedPhysicalColumn;
import org.apache.flink.table.api.Schema.UnresolvedPrimaryKey;
import org.apache.flink.table.api.TableNotExistException;
import org.apache.flink.table.catalog.*;
import org.apache.flink.table.catalog.CatalogBaseTable.TableKind;
import org.apache.flink.table.catalog.exceptions.*;
import org.apache.flink.table.catalog.stats.CatalogColumnStatistics;
import org.apache.flink.table.catalog.stats.CatalogTableStatistics;
import org.apache.flink.table.expressions.Expression;
import org.apache.flink.table.factories.Factory;
import org.apache.flink.table.functions.FunctionKind;
import org.apache.flink.table.types.AtomicDataType;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.logical.LogicalTypeRoot;

import java.sql.*;
import java.util.*;

public class MysqlCatalog extends AbstractCatalog {

	private final String tbPre = "t_";
	private final String viewPre = "v_";
	private final String errorMsg = "表以t_开头，视图以v_开头，不存在此类型的对象[%s]";
	private final long tenantId;// 租户
	private final long operate;

	private Connection connection;

	public MysqlCatalog(String catalogName, String defaultDatabase, long tenantId, long operate) {
		super(catalogName, defaultDatabase);
		this.tenantId = tenantId;
		this.operate = operate;
	}

	@SuppressWarnings("rawtypes")
	@Override
	public void open() throws CatalogException {
		this.connection = DruidAssit.getConnection(MysqlCatalogFactoryOptions.IDENTIFIER);
		// 初始化数据库脚本
		try {
			MySqlAssitExt.execSql(this.connection, InitSql.metaColSql);
			MySqlAssitExt.execSql(this.connection, InitSql.metaDbSql);
			MySqlAssitExt.execSql(this.connection, InitSql.metaFunSql);
			MySqlAssitExt.execSql(this.connection, InitSql.metaTbSql);
		} catch (Exception e1) {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_exec_fail, "初始化相关元数据表失败", e1);
		}

		try {
			this.createDatabase(this.getDefaultDatabase(), null, true);
		} catch (CatalogException | DatabaseAlreadyExistException e1) {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_exec_fail, "初始化default库失败", e1);
		}
		List<String> listFunctions = CatalogAssit.listFunctions(this.connection, this.tenantId);
		// 得到IUdfEnum的构造器，一个个自动加载函数
		ServiceLoader<IPackageEnum> udffats = ServiceLoader.load(IPackageEnum.class);
		for (IPackageEnum udffat : udffats) {
			try {
				IUdfEnum[] udfs = udffat.queryAll();
				for (IUdfEnum udf : udfs) {
					if (!listFunctions.contains(udf.getFullName())) {//
						CatalogAssit.createFunByMoudle(this.connection, super.getDefaultDatabase(), -1l,
								udf.getDbName(), udf, this.tenantId, this.operate);// moleid为null,因为不是从模块导入的
					}
				}
			} catch (Exception e) {
				throw new ProjectExceptionRuntime(ExceptAll.flink_udf_errorformate, "需要定义实现IUdfEnum接口的枚举类", e);
			}
		}

	}

	@Override
	public void close() throws CatalogException {
		DruidAssit.close(this.connection);
	}

	@Override
	@SuppressWarnings("unchecked")
	public List<String> listDatabases() throws CatalogException {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
				"select name from meta_db where tenant_id=?", false, this.tenantId);
		List<String> dbs = (List<String>) CollectionUtil.getColFromObj(querySqlMap, "name");
		return dbs;
	}

	@Override
	public CatalogDatabase getDatabase(String databaseName) throws DatabaseNotExistException, CatalogException {
		ResultSet rs = MySqlAssitExt.querySql(this.connection,
				String.format("select * from %s where %s='%s' and  %s='%s'", ColsMetaDb.tb,
						ColsMetaDb.tenantId.getOriColName(), this.tenantId, ColsMetaDb.name.getOriColName(),
						databaseName));
		try {
			if (rs.next()) {
				Map<String, String> properties = new HashMap<String, String>();
				String propertiesStr = rs.getString(ColsMetaDb.properties.getOriColName());
				if (StringUtil.isNull(rs)) {
					properties = JSONUtil.jsonToMapStr(JSONObject.parseObject(propertiesStr));
				}
				CatalogDatabaseImpl retobj = new CatalogDatabaseImpl(properties,
						rs.getString(ColsMetaDb.comment.getOriColName()));
				return retobj;
			}
		} catch (SQLException e) {
			throw new ProjectExceptionRuntime(ExceptAll.duckula_nodata, "查数据库" + databaseName + "失败", e);
		}
		throw new DatabaseNotExistException(this.getName(), databaseName);
	}

	@Override
	public List<String> listTables(String databaseName) throws DatabaseNotExistException, CatalogException {
		List<String> colFromObj = CatalogAssit.listTableOrViewByDb(this.connection, databaseName, TableKind.TABLE,
				this.tenantId);
		return colFromObj;
	}

	@Override
	public List<String> listViews(String databaseName) throws DatabaseNotExistException, CatalogException {
		List<String> colFromObj = CatalogAssit.listTableOrViewByDb(this.connection, databaseName, TableKind.VIEW,
				this.tenantId);
		return colFromObj;
	}

	/***
	 * 得到表或视图信息，注意v_开头为视图，t_为表。
	 */
	@Override
	public CatalogBaseTable getTable(ObjectPath tablePath) throws TableNotExistException, CatalogException {
		Map<String, String> tbinfo = null;
		try {
			tbinfo = queryTable(tablePath);
		} catch (Exception e) {
			throw new CatalogException("查询table失败", e);
		}
		if (tbinfo == null) {
			throw new TableNotExistException(this.getName(), tablePath.getObjectName());
		}
		String objectName = tablePath.getObjectName();
		if (objectName.startsWith(tbPre)) {// 表
			Map<String, String> tbLogicInfo = getTableInfoLogic(
					Long.parseLong(tbinfo.get(ColsMetaTb.tbLogicId.getOriColName())), tablePath);
			Builder builder = Schema.newBuilder();
			String comment = tbLogicInfo.get(ColsMetaTbLogic.comment.getOriColName());
			String partitionkeysStr = tbLogicInfo.get(ColsMetaTbLogic.partitionkeys.getOriColName());
			List<String> partitionKeys = StringUtil.isNull(partitionkeysStr) ? new ArrayList<>()
					: Arrays.asList(partitionkeysStr.split(","));
			String optionsStr = tbinfo.get(ColsMetaTb.withOptions.getOriColName());
			Map<String, String> options = null;
			if (StringUtil.isNotNull(optionsStr)) {
				JSONObject jsonObject = JSONObject.parseObject(optionsStr);
				options = JSONUtil.jsonToMapStr(jsonObject);
			} else {
				options = new HashMap<String, String>();
			}
			String[] keys = tbLogicInfo.get(ColsMetaTbLogic.keyCol.getOriColName()).split(",");
			if(ArrayUtils.isEmpty(keys)) {
				throw new CatalogException(tablePath.getFullName()+"没有配置主键，请确认表的配置是正确的。");
			}
			builder.primaryKey(keys);

			List<Map<String, String>> queryCols = new ArrayList<Map<String, String>>();
			try {
				queryCols = queryCols(Long.parseLong(tbLogicInfo.get(ColsMetaTbLogic.id.getOriColName())));
			} catch (Exception e) {
				throw new CatalogException("查询cols失败");
			}
			if(CollectionUtils.isEmpty(queryCols)) {
				throw new CatalogException(tablePath.getFullName()+"没有配置字段，请确认表的配置是正确的。");
			}
			for (Map<String, String> col : queryCols) {
				String colName = col.get(ColsMetaCol.name.getOriColName());
				String watermarkExpression = col.get(ColsMetaCol.watermarkExpression.getOriColName());
				if (StringUtil.isNotNull(watermarkExpression)) {
					builder.watermark(colName, watermarkExpression);
				}
				// TODO 还有很多要支持
				String sqlexpression = col.get(ColsMetaCol.sqlexpression.getOriColName());
				if (StringUtil.isNotNull(sqlexpression)) {
					builder.columnByExpression(colName, sqlexpression);
				} else {
					FlinkTypeEnum flinkTypeEnum = FlinkTypeEnum
							.findByFlinkRowType(col.get(ColsMetaCol.type.getOriColName()));
					if (flinkTypeEnum == null || flinkTypeEnum.getConvert().getObj() == null) {
						throw new CatalogException(
								"此类型[" + col.get(ColsMetaCol.type.getOriColName()) + "]还不支持，需要连接相关人员。");
					}
					DataType dataType = null;
					switch (flinkTypeEnum) {
					case DECIMAL:
//						dataType = flinkTypeEnum.getConvert().getObj();// TODO 注意支持int precision, int scale
//						break;
					default:
						dataType = flinkTypeEnum.getConvert().getObj();
						break;
					}
					String notNullStr = col.get(ColsMetaCol.notNull.getOriColName());
					if ("yes".equals(StringUtil.hasNull(notNullStr, "no"))) {// 不为空
						dataType = dataType.notNull();
					} else {
						dataType = dataType.nullable();
					}
					builder.column(colName, dataType);
				}
			}
			//所有catalogtable统一加一个处理时间字段
			builder.columnByExpression("proc_time","PROCTIME()");
			// builder.watermark(tbrs.getString("watermark_row_time_attribute"),
			// tbrs.getString("watermark_expression"), DataTypes.TIMESTAMP(3));
			CatalogTable retobj = CatalogTable.of(builder.build(), comment, partitionKeys, options);
			return retobj;
		} else if (objectName.startsWith(viewPre)) {// 视图
			Builder builder = Schema.newBuilder();
			String comment = tbinfo.get(ColsMetaView.remark.getOriColName());
			String originalQuery = tbinfo.get(ColsMetaView.originalQuery.getOriColName());
			String expandedQuery = tbinfo.get(ColsMetaView.expandedQuery.getOriColName());
			String optionsStr = tbinfo.get(ColsMetaView.withOptions.getOriColName());
			Map<String, String> options = null;
			if (StringUtil.isNotNull(optionsStr)) {
				JSONObject jsonObject = JSONObject.parseObject(optionsStr);
				options = JSONUtil.jsonToMapStr(jsonObject);
			} else {
				options = new HashMap<String, String>();
			}
			CatalogView retobj = CatalogView.of(builder.build(), comment, originalQuery, expandedQuery, options);
			return retobj;
		} else {
			throw new CatalogException(String.format(errorMsg, tablePath));
		}

	}

	/***
	 * 得到逻辑表信息,不用租户限制它了。
	 * 
	 * @param logicTbId
	 * @param tablePath
	 * @return
	 */
	private Map<String, String> getTableInfoLogic(long logicTbId, ObjectPath tablePath) {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
				String.format("select * from  %s where %s=? and  %s=?", ColsMetaTbLogic.tb,
						ColsMetaTb.id.getOriColName(), ColsMetaTb.tenantId.getOriColName()),
				false, logicTbId, this.tenantId);
		if (CollectionUtils.isNotEmpty(querySqlMap)) {
			return querySqlMap.get(0);
		} else {
			throw new TableNotExistException(this.getName(), tablePath.getObjectName());
		}
	}

	@Override
	public boolean tableExists(ObjectPath tablePath) throws CatalogException {
		try {
			CatalogBaseTable baseTable = getTable(tablePath);
			return baseTable != null;
		} catch (TableNotExistException e) {
			return false;
		}
	}

	@Override
	public void dropTable(ObjectPath tablePath, boolean ignoreIfNotExists)
			throws TableNotExistException, CatalogException {
		Result dropTable;
		try {
			dropTable = CatalogAssit.dropTable(this.connection, super.getDefaultDatabase(), tablePath.getDatabaseName(),
					tablePath.getObjectName(), DelLevel.delLogicNotTb, this.tenantId);
			if (!dropTable.isSuc()) {
				if (ignoreIfNotExists && ExceptAll.flink_catalog_tablenotexit == dropTable.getExcept()) {
					throw new TableNotExistException(this.getName(), tablePath.getObjectName());
				} else {
					throw new CatalogException(dropTable.getMessage());
				}
			}
		} catch (ProjectException e) {
			throw new CatalogException(e.getMessage());
		}
	}

	@Override
	public void renameTable(ObjectPath tablePath, String newTableName, boolean ignoreIfNotExists)
			throws TableNotExistException, TableAlreadyExistException, CatalogException {

	}

	@Override
	public void createTable(ObjectPath tablePath, CatalogBaseTable table, boolean ignoreIfExists)
			throws TableAlreadyExistException, DatabaseNotExistException, CatalogException {
		validDb(tablePath);
		CatalogBaseTable baseTable;
		try {
			baseTable = getTable(tablePath);
		} catch (TableNotExistException e) {
			baseTable = null;// 正常情况，其它异常不处理
		}
		if (!ignoreIfExists && baseTable != null) {
			throw new TableAlreadyExistException(this.getName(), tablePath);
		}
		if (baseTable == null) {
			String objectName = tablePath.getObjectName();
			if (objectName.startsWith(tbPre) && table.getTableKind() == TableKind.TABLE) {// 表
				try {
					// 主键
					Optional<UnresolvedPrimaryKey> primaryKey = table.getUnresolvedSchema().getPrimaryKey();
					List<String> primaryKeys = primaryKey.isPresent() ? primaryKey.get().getColumnNames()
							: new ArrayList<String>();
					List<MySqlColBean> mysqlColList = new ArrayList<MySqlColBean>();

					List<UnresolvedColumn> columns = table.getUnresolvedSchema().getColumns();
					for (UnresolvedColumn column : columns) {
						String type = "";
//						String sqlexpression = "";
//						String watermarkRowTimeAttribute = "";
//						String watermarkExpression = "";
						if (column instanceof UnresolvedPhysicalColumn) {// TODO 细分不同的表
							UnresolvedPhysicalColumn temp = (UnresolvedPhysicalColumn) column;
							try {
								// TODO 逻辑表信息丢失了， 如何处理逻辑表？
								LogicalTypeRoot typeRoot = ((AtomicDataType) temp.getDataType()).getLogicalType()
										.getTypeRoot();
								FlinkTypeEnum flinkTypeEnum = FlinkTypeEnum.valueOf(typeRoot.name());
								type = flinkTypeEnum.name();
							} catch (Exception e) {
								throw new CatalogException("创建表[" + objectName + "]时类型检查异常，原因：还不支持此类型[" + temp.getName()
										+ "]" + e.getMessage(), e);
							}
						}
						MySqlColBean tempbean = new MySqlColBean(column.getName(), type);
//						tempbean.setDb(tablePath.getDatabaseName());
//						tempbean.setTb(objectName);
						tempbean.setPri(primaryKeys.contains(column.getName()));
						tempbean.setDataType(type);
						mysqlColList.add(tempbean);
					}

					// 分区
					String partitionkeys = "";
					if (table.getTableKind() == TableKind.TABLE) {// 是表
						CatalogTable tableTrue = (CatalogTable) table;
						if (CollectionUtils.isNotEmpty(tableTrue.getPartitionKeys())) {
							partitionkeys = CollectionUtil.listJoin(tableTrue.getPartitionKeys(), ",");
						}
					}
					// 转换为统一的配置
					Map<MiddlewareOption, String> allOptmap = new HashMap<MiddlewareOption, String>();
					Map<String, String> options = table.getOptions();
					if (MapUtils.isNotEmpty(options)) {
						for (String optionstr : options.keySet()) {
							MiddlewareOption findByNameUse = MiddlewareOption.findByNameUse(optionstr);
							if (findByNameUse != null) {
								allOptmap.put(findByNameUse, options.get(optionstr));
							} else {
								throw new CatalogException(
										"创建表[" + objectName + "]时检查异常，原因还没支持此option[" + optionstr + "].");
							}
						}
					}
					// 得到option相关配置，全部做为可配置项处理。
					Pair<String, String> proOptStr = Middleware.proOptStr(allOptmap,
							allOptmap.keySet().toArray(new MiddlewareOption[allOptmap.size()]));

					MySqlTbBean catalogTbInfo = MySqlTbBean.builder().db(tablePath.getDatabaseName()).tb(objectName)
							.tbComment("通过sql创建").middleware(Middleware.no).partitionkeys(partitionkeys)
							.opt(proOptStr.getRight()).withOptions(proOptStr.getLeft()).build();
					CatalogAssit.createTable(this.connection, this.getDefaultDatabase(), mysqlColList, catalogTbInfo,
							this.tenantId, this.operate);
				} catch (ProjectException e) {
					throw new CatalogException("创建表[" + objectName + "]时SQL异常，原因：" + e.getMessage(), e);
				}
			} else if (objectName.startsWith(viewPre) && table.getTableKind() == TableKind.VIEW) {// 查询视图
				try {
					Timestamp currentTime = new Timestamp(System.currentTimeMillis());
					CatalogView catalogView = (CatalogView) table;
					PreparedStatement insertColPre = this.connection.prepareStatement(
							String.format("insert %s(%s,%s,%s,%s,%s,%s,%s,%s,%s,%s) value(?,?,?,?,?,?,?,?,?,?)", ColsMetaView.tb,
									ColsMetaView.dbName.getOriColName(), ColsMetaView.kind.getOriColName(),
									ColsMetaView.name.getOriColName(), ColsMetaView.originalQuery.getOriColName(),
									ColsMetaView.expandedQuery.getOriColName(), ColsMetaView.createBy.getOriColName(),
									ColsMetaView.updateBy.getOriColName(), ColsMetaView.createTime.getOriColName(),
									ColsMetaView.updateTime.getOriColName(), ColsMetaView.tenantId.getOriColName()));
					MySqlAssitExt.setPreParam(insertColPre, false, tablePath.getDatabaseName(),
							catalogView.getTableKind().name(), objectName, catalogView.getOriginalQuery(),
							catalogView.getExpandedQuery(), operate, operate, currentTime, currentTime,
							tenantId);
					insertColPre.execute();
				} catch (Exception e) {
					throw new CatalogException("创建视图[" + objectName + "]时SQL异常", e);
				}
			} else {
				throw new CatalogException(String.format(errorMsg, tablePath));
			}
		}
	}

	private void validDb(ObjectPath tablePath) throws DatabaseNotExistException {
		CatalogDatabase database;
		try {
			database = getDatabase(tablePath.getDatabaseName());
		} catch (DatabaseNotExistException e) {
			database = null;// 正常情况，其它异常不处理
		}
		if (database == null) {
			throw new DatabaseNotExistException(this.getName(), tablePath.getDatabaseName());
		}
	}

	// 查询表或视图
	private Map<String, String> queryTable(ObjectPath tablePath) throws SQLException {
		String objectName = tablePath.getObjectName();
		if (objectName.startsWith(tbPre)) {
			Map<String, String> queryTable = CatalogAssit.queryTable(this.connection, super.getDefaultDatabase(),
					tablePath.getDatabaseName(), tablePath.getObjectName(), this.tenantId);
			return queryTable;
		} else if (objectName.startsWith(viewPre)) {// 查询视图
			Map<String, String> queryView = CatalogAssit.queryView(this.connection, super.getDefaultDatabase(),
					tablePath.getDatabaseName(), tablePath.getObjectName(), this.tenantId);
			return queryView;
		} else {
			throw new CatalogException(String.format(errorMsg, tablePath));
		}
	}

	private Map<String, String> queryFun(ObjectPath functionPath) throws SQLException {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
				String.format("select * from %s where %s=? and  %s=? and  %s=?", ColsMetaFun.tb,
						ColsMetaFun.tenantId.getOriColName(), ColsMetaFun.dbName.getOriColName(),
						ColsMetaFun.name.getOriColName()),
				false, this.tenantId, functionPath.getDatabaseName(), functionPath.getObjectName());
		if (CollectionUtils.isNotEmpty(querySqlMap)) {
			return querySqlMap.get(0);
		}
		return null;
	}

	private List<Map<String, String>> queryCols(Long logicTableId) throws SQLException {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
				String.format("select * from  %s where %s=? and  %s=? order by id", ColsMetaCol.tb,
						ColsMetaCol.tenantId.getOriColName(), ColsMetaCol.tbLogicId.getOriColName()),
				false, this.tenantId, logicTableId);
		return querySqlMap;
	}

	@Override
	public void alterTable(ObjectPath tablePath, CatalogBaseTable newTable, boolean ignoreIfNotExists)
			throws TableNotExistException, CatalogException {
		dropTable(tablePath, true);
		try {
			createTable(tablePath, newTable, false);
		} catch (TableAlreadyExistException | DatabaseNotExistException e) {
			throw new CatalogException(e);
		}
	}

	@Override
	public List<CatalogPartitionSpec> listPartitions(ObjectPath tablePath)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public List<CatalogPartitionSpec> listPartitions(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public List<CatalogPartitionSpec> listPartitionsByFilter(ObjectPath tablePath, List<Expression> filters)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public CatalogPartition getPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public boolean partitionExists(ObjectPath tablePath, CatalogPartitionSpec partitionSpec) throws CatalogException {
		return false;
	}

	@Override
	public void createPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, CatalogPartition partition,
			boolean ignoreIfExists) throws TableNotExistException, TableNotPartitionedException,
			PartitionSpecInvalidException, PartitionAlreadyExistsException, CatalogException {

	}

	@Override
	public void dropPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public void alterPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, CatalogPartition newPartition,
			boolean ignoreIfNotExists) throws PartitionNotExistException, CatalogException {

	}

	@Override
	public List<String> listFunctions(String dbName) throws DatabaseNotExistException, CatalogException {
		List<String> listFunctions = CatalogAssit.listFunctions(this.connection, dbName, this.tenantId);
		return listFunctions;
	}

	@Override
	public CatalogFunction getFunction(ObjectPath functionPath) throws FunctionNotExistException, CatalogException {
		Map<String, String> funinfo = null;
		try {
			funinfo = queryFun(functionPath);
		} catch (Exception e) {
			throw new CatalogException("查询fun失败");
		}
		if (funinfo == null) {
			throw new FunctionNotExistException(this.getName(), functionPath);
		}
		CatalogFunction retFun = new CatalogFunctionImpl(funinfo.get(ColsMetaFun.identifier.getOriColName()),
				FunctionLanguage.valueOf(funinfo.get(ColsMetaFun.language.getOriColName())));
		return retFun;
	}

	@Override
	public boolean functionExists(ObjectPath functionPath) throws CatalogException {
		try {
			CatalogFunction function = getFunction(functionPath);
			return function != null;
		} catch (FunctionNotExistException e) {
			return false;
		}
	}

	@Override
	public void createFunction(ObjectPath functionPath, CatalogFunction function, boolean ignoreIfExists)
			throws FunctionAlreadyExistException, DatabaseNotExistException, CatalogException {
		validDb(functionPath);
		CatalogFunction baseFun;
		try {
			baseFun = getFunction(functionPath);
		} catch (FunctionNotExistException e) {
			baseFun = null;// 正常情况，其它异常不处理
		}
		if (!ignoreIfExists && baseFun != null) {
			throw new FunctionAlreadyExistException(this.getName(), functionPath);
		}
		// 创建函数
		if (baseFun == null) {
			try {
				CatalogAssit.createFunByMoudle(this.connection, super.getDefaultDatabase(), -1l,
						functionPath.getDatabaseName(), new IUdfEnum() {

							@Override
							public String getName() {
								return functionPath.getObjectName();
							}

							@Override
							public FunctionKind getKind() {
								return FunctionKind.SCALAR;
							}

							@Override
							public String getClassFullName() {
								return function.getClassName();
							}

							@Override
							public String getDbName() {
								return functionPath.getDatabaseName();
							}

						}, this.tenantId, this.operate);
			} catch (SQLException e1) {
				throw new CatalogException("创建函数时失败", e1);
			}
		}
	}

	@Override
	public void alterFunction(ObjectPath functionPath, CatalogFunction newFunction, boolean ignoreIfNotExists)
			throws FunctionNotExistException, CatalogException {
		// 先删除再插入达到修改
		dropFunction(functionPath, ignoreIfNotExists);
		try {
			createFunction(functionPath, newFunction, ignoreIfNotExists);
		} catch (FunctionAlreadyExistException e) {
			throw new CatalogException(e);
		} catch (DatabaseNotExistException e) {
			throw new CatalogException(e);
		}
	}

	@Override
	public void dropFunction(ObjectPath functionPath, boolean ignoreIfNotExists)
			throws FunctionNotExistException, CatalogException {
		Map<String, String> funinfo = null;
		try {
			funinfo = queryFun(functionPath);
		} catch (Exception e) {
			throw new CatalogException("查询fun失败");
		}
		if (!ignoreIfNotExists && funinfo == null) {
			throw new FunctionNotExistException(this.getName(), functionPath);
		}
		if (funinfo != null) {
			try {
				MySqlAssitExt.execSqlPre(this.connection,
						String.format("delete from %s where %s=? ", ColsMetaFun.tb, ColsMetaFun.id.getOriColName()),
						Long.parseLong(funinfo.get(ColsMetaFun.id.getOriColName())));
			} catch (Exception e) {
				throw new CatalogException("删除函数时失败", e);
			}
		}
	}

	@Override
	public CatalogTableStatistics getTableStatistics(ObjectPath tablePath)
			throws TableNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogColumnStatistics getTableColumnStatistics(ObjectPath tablePath)
			throws TableNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogTableStatistics getPartitionStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogColumnStatistics getPartitionColumnStatistics(ObjectPath tablePath,
			CatalogPartitionSpec partitionSpec) throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public void alterTableStatistics(ObjectPath tablePath, CatalogTableStatistics tableStatistics,
			boolean ignoreIfNotExists) throws TableNotExistException, CatalogException {

	}

	@Override
	public void alterTableColumnStatistics(ObjectPath tablePath, CatalogColumnStatistics columnStatistics,
			boolean ignoreIfNotExists) throws TableNotExistException, CatalogException, TablePartitionedException {

	}

	@Override
	public void alterPartitionStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogTableStatistics partitionStatistics, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public void alterPartitionColumnStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogColumnStatistics columnStatistics, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public boolean databaseExists(String databaseName) throws CatalogException {
		String sql = "select count(1) from " + ColsMetaDb.tb + " where " + ColsMetaDb.tenantId.getOriColName()
				+ "=? and " + ColsMetaDb.name.getOriColName() + "=?";
		try {
			PreparedStatement ps = this.connection.prepareStatement(sql);
			ps.setLong(1, this.tenantId);
			ps.setString(2, databaseName);
			ResultSet rs = ps.executeQuery();
			while (rs.next()) {
				int count = rs.getInt(1);
				if (count == 0) {
					return false;
				} else {
					return true;
				}
			}
		} catch (SQLException e) {
			throw new CatalogException("查询表" + databaseName + "异常");
		}
		return false;
	}

	@Override
	public void createDatabase(String name, CatalogDatabase database, boolean ignoreIfExists)
			throws DatabaseAlreadyExistException, CatalogException {
		// 库名较为特殊，不能通过租户来判断有元
		int count = MySqlAssitExt.querySqlCount(this.connection,
				String.format("select count(1) from %s where %s='%s' and %s='%s'", ColsMetaDb.tb,
						ColsMetaDb.tenantId.getOriColName(), this.tenantId, ColsMetaDb.name.getOriColName(), name));
		if (!ignoreIfExists && count > 0) {
			throw new DatabaseAlreadyExistException(this.getName(), name);
		}
		if (count == 0) {
			String desc = (database != null && database.getDescription().isPresent()) ? database.getDescription().get()
					: "";
			CatalogAssit.createDb(this.connection, name,
					(database == null) ? "" : StringUtil.hasNull(database.getComment()), desc, tenantId, this.operate);
		}
	}

	@Override
	public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade)
			throws DatabaseNotExistException, DatabaseNotEmptyException, CatalogException {

	}

	@Override
	public void alterDatabase(String name, CatalogDatabase newDatabase, boolean ignoreIfNotExists)
			throws DatabaseNotExistException, CatalogException {

	}

	/**
	 * 由FactoryUtils调用，如果返回空，就根据connector字段来判断，利用Java SPI去实现工厂的获取
	 * AbstractJdbcCatalog默认会返回Jdbc动态工厂这是不对的
	 * 
	 * @return
	 */
	public Optional<Factory> getFactory() {
		return Optional.empty();
	}

}
