package net.wicp.tams.common.flink.catalog;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.flink.table.api.Schema;
import org.apache.flink.table.api.Schema.Builder;
import org.apache.flink.table.api.Schema.UnresolvedColumn;
import org.apache.flink.table.api.Schema.UnresolvedPhysicalColumn;
import org.apache.flink.table.api.Schema.UnresolvedPrimaryKey;
import org.apache.flink.table.api.TableNotExistException;
import org.apache.flink.table.catalog.AbstractCatalog;
import org.apache.flink.table.catalog.CatalogBaseTable;
import org.apache.flink.table.catalog.CatalogBaseTable.TableKind;
import org.apache.flink.table.catalog.CatalogDatabase;
import org.apache.flink.table.catalog.CatalogDatabaseImpl;
import org.apache.flink.table.catalog.CatalogFunction;
import org.apache.flink.table.catalog.CatalogFunctionImpl;
import org.apache.flink.table.catalog.CatalogPartition;
import org.apache.flink.table.catalog.CatalogPartitionSpec;
import org.apache.flink.table.catalog.CatalogTable;
import org.apache.flink.table.catalog.FunctionLanguage;
import org.apache.flink.table.catalog.ObjectPath;
import org.apache.flink.table.catalog.exceptions.CatalogException;
import org.apache.flink.table.catalog.exceptions.DatabaseAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotEmptyException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotExistException;
import org.apache.flink.table.catalog.exceptions.FunctionAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.FunctionNotExistException;
import org.apache.flink.table.catalog.exceptions.PartitionAlreadyExistsException;
import org.apache.flink.table.catalog.exceptions.PartitionNotExistException;
import org.apache.flink.table.catalog.exceptions.PartitionSpecInvalidException;
import org.apache.flink.table.catalog.exceptions.TableAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.TableNotPartitionedException;
import org.apache.flink.table.catalog.exceptions.TablePartitionedException;
import org.apache.flink.table.catalog.stats.CatalogColumnStatistics;
import org.apache.flink.table.catalog.stats.CatalogTableStatistics;
import org.apache.flink.table.expressions.Expression;
import org.apache.flink.table.factories.Factory;

import com.alibaba.fastjson.JSONObject;

import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.json.JSONUtil;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.flink.catalog.constant.ColsMetaCol;
import net.wicp.tams.common.flink.catalog.constant.ColsMetaDb;
import net.wicp.tams.common.flink.catalog.constant.ColsMetaFun;
import net.wicp.tams.common.flink.catalog.constant.ColsMetaTb;
import net.wicp.tams.common.flink.module.IUdfEnum;
import net.wicp.tams.common.flink.module.constant.DataTypeEnum;
import net.wicp.tams.common.flink.module.factory.IPackageEnum;
import net.wicp.tams.common.jdbc.DruidAssit;
import net.wicp.tams.common.jdbc.MySqlAssitExt;

public class MysqlCatalog extends AbstractCatalog {

	private Connection connection;

	public MysqlCatalog(String catalogName, String defaultDatabase) {
		super(catalogName, defaultDatabase);
	}

	@SuppressWarnings("rawtypes")
	@Override
	public void open() throws CatalogException {
		this.connection = DruidAssit.getConnection(MysqlCatalogFactoryOptions.IDENTIFIER);
		// 初始化数据库脚本
		try {
			MySqlAssitExt.execSql(this.connection, InitSql.metaColSql);
			MySqlAssitExt.execSql(this.connection, InitSql.metaDbSql);
			MySqlAssitExt.execSql(this.connection, InitSql.metaFunSql);
			MySqlAssitExt.execSql(this.connection, InitSql.metaTbSql);
		} catch (Exception e1) {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_exec_fail, "初始化相关元数据表失败", e1);
		}

		try {
			this.createDatabase(this.getDefaultDatabase(), null, true);
		} catch (CatalogException | DatabaseAlreadyExistException e1) {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_exec_fail, "初始化default库失败", e1);
		}

		// 导入模块已定义的函数
		try {
			List<String> listFunctions = listFunctions(super.getDefaultDatabase());
			// 得到IUdfEnum的构造器，一个个自动加载函数
			ServiceLoader<IPackageEnum> udffats = ServiceLoader.load(IPackageEnum.class);
			for (IPackageEnum udffat : udffats) {
				try {
					IUdfEnum[] udfs = udffat.queryAll();
					for (IUdfEnum udf : udfs) {
						if (!listFunctions.contains(udf.getFullName())) {//
							MySqlAssitExt.execSqlPre(this.connection, String.format(
									"insert into %s(%s,%s,%s,%s,%s,%s,%s) value(?,?,?,?,?,?,?)", ColsMetaFun.tb,
									ColsMetaFun.dbName.getOriColName(), ColsMetaFun.name.getOriColName(),
									ColsMetaFun.kind.getOriColName(), ColsMetaFun.language.getOriColName(),
									ColsMetaFun.identifier.getOriColName(), ColsMetaFun.description.getOriColName(),
									ColsMetaFun.detailDescription.getOriColName()), super.getDefaultDatabase(),
									udf.getFullName(), udf.getKind().name(), udf.getLanguage().name(),
									udf.getClassz().getName(), udf.getDescription(), udf.getDetailDescription());
						}
					}
				} catch (Exception e) {
					throw new ProjectExceptionRuntime(ExceptAll.flink_udf_errorformate, "需要定义实现IUdfEnum接口的枚举类", e);
				}
			}
		} catch (DatabaseNotExistException e) {
			throw new CatalogException(e);
		}
	}

	@Override
	public void close() throws CatalogException {
		DruidAssit.close(this.connection);
	}

	@Override
	@SuppressWarnings("unchecked")
	public List<String> listDatabases() throws CatalogException {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMap(this.connection, "select name from meta_db",
				false);
		List<String> dbs = (List<String>) CollectionUtil.getColFromObj(querySqlMap, "name");
		return dbs;
	}

	@Override
	public CatalogDatabase getDatabase(String databaseName) throws DatabaseNotExistException, CatalogException {
		ResultSet rs = MySqlAssitExt.querySql(this.connection, String.format("select * from %s where %s='%s'",
				ColsMetaDb.tb, ColsMetaDb.name.getOriColName(), databaseName));
		try {
			if (rs.next()) {
				Map<String, String> properties = new HashMap<String, String>();
				String propertiesStr = rs.getString(ColsMetaDb.properties.getOriColName());
				if (StringUtil.isNull(rs)) {
					properties = JSONUtil.jsonToMapStr(JSONObject.parseObject(propertiesStr));
				}
				CatalogDatabaseImpl retobj = new CatalogDatabaseImpl(properties,
						rs.getString(ColsMetaDb.comment.getOriColName()));
				return retobj;
			}
		} catch (SQLException e) {
			throw new ProjectExceptionRuntime(ExceptAll.duckula_nodata, "查数据库" + databaseName + "失败", e);
		}
		throw new DatabaseNotExistException(MysqlCatalogFactoryOptions.IDENTIFIER, databaseName);
	}

	@Override
	public List<String> listTables(String databaseName) throws DatabaseNotExistException, CatalogException {
		List<String> colFromObj = queryTableOrViewByDb(databaseName, TableKind.TABLE);
		return colFromObj;
	}

	@SuppressWarnings("unchecked")
	private List<String> queryTableOrViewByDb(String databaseName, TableKind kind) {
		List<Map<String, String>> querySqlMap = new ArrayList<Map<String, String>>();
		try {
			querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
					String.format("select %s from  %s where %s=? and %s=?", ColsMetaTb.name.getOriColName(),
							ColsMetaTb.tb, ColsMetaTb.dbName.getOriColName(), ColsMetaTb.kind.getOriColName()),
					false, databaseName, kind);
		} catch (Exception e) {
			throw new CatalogException("通过库查询表失败", e);
		}
		List<String> colFromObj = (List<String>) CollectionUtil.getColFromObj(querySqlMap,
				ColsMetaTb.name.getOriColName());
		return colFromObj;
	}

	@Override
	public List<String> listViews(String databaseName) throws DatabaseNotExistException, CatalogException {
		List<String> colFromObj = queryTableOrViewByDb(databaseName, TableKind.VIEW);
		return colFromObj;
	}

	@Override
	public CatalogBaseTable getTable(ObjectPath tablePath) throws TableNotExistException, CatalogException {
		Map<String, String> tableinfo = getTableInfo(tablePath);

		Builder builder = Schema.newBuilder();
		String comment = tableinfo.get(ColsMetaTb.comment.getOriColName());
		List<String> partitionKeys = Arrays.asList(tableinfo.get(ColsMetaTb.partitionkeys.getOriColName()).split(","));
		String optionsStr = tableinfo.get(ColsMetaTb.options.getOriColName());
		Map<String, String> options = null;
		if (StringUtil.isNotNull(optionsStr)) {
			JSONObject jsonObject = JSONObject.parseObject(optionsStr);
			options = JSONUtil.jsonToMapStr(jsonObject);
		}
		builder.primaryKey(tableinfo.get(ColsMetaTb.keyCol.getOriColName()).split(","));

		List<Map<String, String>> queryCols = new ArrayList<Map<String, String>>();
		try {
			queryCols = queryCols(Long.parseLong(tableinfo.get(ColsMetaTb.id.getOriColName())));
		} catch (Exception e) {
			throw new CatalogException("查询cols失败");
		}
		for (Map<String, String> col : queryCols) {
			String colName = col.get(ColsMetaCol.name.getOriColName());
			String watermarkExpression = col.get(ColsMetaCol.watermarkExpression.getOriColName());
			if (StringUtil.isNotNull(watermarkExpression)) {
				builder.watermark(colName, watermarkExpression);
			}
			// TODO 还有很多要支持
			String sqlexpression = col.get(ColsMetaCol.sqlexpression.getOriColName());
			if (StringUtil.isNotNull(sqlexpression)) {
				builder.columnByExpression(colName, sqlexpression);
			} else {
				DataTypeEnum dataTypeEnum = DataTypeEnum.find(col.get(ColsMetaCol.type.getOriColName()));
				if (dataTypeEnum == null || dataTypeEnum.getDataType() == null) {
					throw new CatalogException("此类型[" + col.get(ColsMetaCol.type.getOriColName()) + "]还不支持，需要连接相关人员。");
				}
				builder.column(colName, dataTypeEnum.getDataType());
			}
		}
		// builder.watermark(tbrs.getString("watermark_row_time_attribute"),
		// tbrs.getString("watermark_expression"), DataTypes.TIMESTAMP(3));
		CatalogTable retobj = CatalogTable.of(builder.build(), comment, partitionKeys, options);
		return retobj;

	}

	private Map<String, String> getTableInfo(ObjectPath tablePath) {
		Map<String, String> tableinfo = null;
		try {
			tableinfo = queryTable(tablePath);
		} catch (Exception e) {
			throw new CatalogException("查询table失败");
		}

		if (tableinfo == null) {
			throw new TableNotExistException(MysqlCatalogFactoryOptions.IDENTIFIER, tablePath.getObjectName());
		}
		return tableinfo;
	}

	@Override
	public boolean tableExists(ObjectPath tablePath) throws CatalogException {
		try {
			CatalogBaseTable baseTable = getTable(tablePath);
			return baseTable != null;
		} catch (TableNotExistException e) {
			return false;
		}
	}

	@Override
	public void dropTable(ObjectPath tablePath, boolean ignoreIfNotExists)
			throws TableNotExistException, CatalogException {
		Map<String, String> queryTable = null;
		try {
			queryTable = queryTable(tablePath);
		} catch (SQLException e) {
			throw new CatalogException("查表失败", e);
		}
		if (ignoreIfNotExists && queryTable == null) {
			throw new TableNotExistException(MysqlCatalogFactoryOptions.IDENTIFIER, tablePath.getObjectName());
		}
		if (queryTable != null) {
			long id = Long.parseLong(queryTable.get(ColsMetaTb.id.getOriColName()));
			// TODO 需要事务
			MySqlAssitExt.execSql(this.connection,
					String.format("delete from %s where %s=%s", ColsMetaTb.tb, ColsMetaTb.id.getOriColName(), id));
			MySqlAssitExt.execSql(this.connection,
					String.format("delete from %s where %s=%s", ColsMetaCol.tb, ColsMetaCol.tbId.getOriColName(), id));
		}
	}

	@Override
	public void renameTable(ObjectPath tablePath, String newTableName, boolean ignoreIfNotExists)
			throws TableNotExistException, TableAlreadyExistException, CatalogException {

	}

	@Override
	public void createTable(ObjectPath tablePath, CatalogBaseTable table, boolean ignoreIfExists)
			throws TableAlreadyExistException, DatabaseNotExistException, CatalogException {
		validDb(tablePath);

		CatalogBaseTable baseTable;
		try {
			baseTable = getTable(tablePath);
		} catch (TableNotExistException e) {
			baseTable = null;// 正常情况，其它异常不处理
		}
		if (!ignoreIfExists && baseTable != null) {
			throw new TableAlreadyExistException(MysqlCatalogFactoryOptions.IDENTIFIER, tablePath);
		}
		if (baseTable == null) {
			try {
				PreparedStatement insertTablePre = this.connection
						.prepareStatement(String.format("insert %s(%s,%s,%s,%s,%s,%s,%s) value(?,?,?,?,?,?,?)",
								ColsMetaTb.tb, ColsMetaTb.dbName.getOriColName(), ColsMetaTb.name.getOriColName(),
								ColsMetaTb.kind.getOriColName(), ColsMetaTb.keyCol.getOriColName(),
								ColsMetaTb.comment.getOriColName(), ColsMetaTb.partitionkeys.getOriColName(),
								ColsMetaTb.options.getOriColName()));
				String partitionkeys = "";
				if (table.getTableKind() == TableKind.TABLE) {// 是表
					CatalogTable tableTrue = (CatalogTable) table;
					if (CollectionUtils.isNotEmpty(tableTrue.getPartitionKeys())) {
						partitionkeys = CollectionUtil.listJoin(tableTrue.getPartitionKeys(), ",");
					}
				}
				String keyColStr = "";
				Optional<UnresolvedPrimaryKey> primaryKey = table.getUnresolvedSchema().getPrimaryKey();
				if (primaryKey.isPresent()) {
					keyColStr = CollectionUtil.listJoin(primaryKey.get().getColumnNames(), ",");
				}
				String options = JSONUtil.getJsonForMap(table.getOptions());
				MySqlAssitExt.setPreParam(insertTablePre, tablePath.getDatabaseName(), tablePath.getObjectName(),
						table.getTableKind().name(), keyColStr, StringUtil.hasNull(table.getComment()), partitionkeys,
						options);
				insertTablePre.execute();
				Map<String, String> queryTable = queryTable(tablePath);
				long tbid = Long.parseLong(queryTable.get(ColsMetaTb.id.getOriColName()));
				// 插入列值
				PreparedStatement insertColPre = this.connection
						.prepareStatement(String.format("insert %s(%s,%s,%s,%s,%s,%s,%s) value(?,?,?,?,?,?,?)",
								ColsMetaCol.tb, ColsMetaCol.dbName.getOriColName(), ColsMetaCol.tbId.getOriColName(),
								ColsMetaCol.name.getOriColName(), ColsMetaCol.type.getOriColName(),
								ColsMetaCol.sqlexpression.getOriColName(),
								ColsMetaCol.watermarkRowTimeAttribute.getOriColName(),
								ColsMetaCol.watermarkExpression.getOriColName()));
				List<UnresolvedColumn> columns = table.getUnresolvedSchema().getColumns();
				for (UnresolvedColumn column : columns) {
					String type = "";
					String sqlexpression = "";
					String watermarkRowTimeAttribute = "";
					String watermarkExpression = "";
					if (column instanceof UnresolvedPhysicalColumn) {// TODO 细分不同的表
						UnresolvedPhysicalColumn temp = (UnresolvedPhysicalColumn) column;
						type = temp.getDataType().toString();
					}
					MySqlAssitExt.setPreParam(insertColPre, tablePath.getDatabaseName(), tbid, column.getName(), type,
							sqlexpression, watermarkRowTimeAttribute, watermarkExpression);
					insertColPre.execute();
				}

			} catch (SQLException e) {
				throw new CatalogException("创建表[" + tablePath.getObjectName() + "]时SQL异常", e);
			}
		}
	}

	private void validDb(ObjectPath tablePath) throws DatabaseNotExistException {
		CatalogDatabase database;
		try {
			database = getDatabase(tablePath.getDatabaseName());
		} catch (DatabaseNotExistException e) {
			database = null;// 正常情况，其它异常不处理
		}
		if (database == null) {
			throw new DatabaseNotExistException(MysqlCatalogFactoryOptions.IDENTIFIER, tablePath.getDatabaseName());
		}
	}

	private Map<String, String> queryTable(ObjectPath tablePath) throws SQLException {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
				String.format("select * from  %s where %s=? and %s=?", ColsMetaTb.tb, ColsMetaTb.dbName.getOriColName(),
						ColsMetaTb.name.getOriColName()),
				false, tablePath.getDatabaseName(), tablePath.getObjectName());
		if (CollectionUtils.isNotEmpty(querySqlMap)) {
			return querySqlMap.get(0);
		}
		return null;
	}

	private Map<String, String> queryFun(ObjectPath functionPath) throws SQLException {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
				String.format("select * from %s where %s=? and  %s=?", ColsMetaFun.tb,
						ColsMetaFun.dbName.getOriColName(), ColsMetaFun.name.getOriColName()),
				false, functionPath.getDatabaseName(), functionPath.getObjectName());
		if (CollectionUtils.isNotEmpty(querySqlMap)) {
			return querySqlMap.get(0);
		}
		return null;
	}

	private List<Map<String, String>> queryCols(Long tableId) throws SQLException {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
				String.format("select * from  %s where %s=?", ColsMetaCol.tb, ColsMetaCol.tbId.getOriColName()), false,
				tableId);
		return querySqlMap;
	}

	@Override
	public void alterTable(ObjectPath tablePath, CatalogBaseTable newTable, boolean ignoreIfNotExists)
			throws TableNotExistException, CatalogException {
		dropTable(tablePath, true);
		try {
			createTable(tablePath, newTable, false);
		} catch (TableAlreadyExistException | DatabaseNotExistException e) {
			throw new CatalogException(e);
		}
	}

	@Override
	public List<CatalogPartitionSpec> listPartitions(ObjectPath tablePath)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public List<CatalogPartitionSpec> listPartitions(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public List<CatalogPartitionSpec> listPartitionsByFilter(ObjectPath tablePath, List<Expression> filters)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public CatalogPartition getPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public boolean partitionExists(ObjectPath tablePath, CatalogPartitionSpec partitionSpec) throws CatalogException {
		return false;
	}

	@Override
	public void createPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, CatalogPartition partition,
			boolean ignoreIfExists) throws TableNotExistException, TableNotPartitionedException,
			PartitionSpecInvalidException, PartitionAlreadyExistsException, CatalogException {

	}

	@Override
	public void dropPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public void alterPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, CatalogPartition newPartition,
			boolean ignoreIfNotExists) throws PartitionNotExistException, CatalogException {

	}

	@Override
	public List<String> listFunctions(String dbName) throws DatabaseNotExistException, CatalogException {
		try {
			List<Map<String, String>> funs = MySqlAssitExt.querySqlMapPre(this.connection,
					String.format("select %s from %s where %s=?", ColsMetaFun.name.getOriColName(), ColsMetaFun.tb,
							ColsMetaFun.dbName.getOriColName()),
					false, dbName);
			Set<String> cols = CollectionUtil.getColSetFromObj(funs, ColsMetaFun.name.getOriColName());
			return new ArrayList<String>(cols);
		} catch (SQLException e) {
			throw new CatalogException("查询函数失败", e);
		}
	}

	@Override
	public CatalogFunction getFunction(ObjectPath functionPath) throws FunctionNotExistException, CatalogException {
		Map<String, String> funinfo = null;
		try {
			funinfo = queryFun(functionPath);
		} catch (Exception e) {
			throw new CatalogException("查询fun失败");
		}
		if (funinfo == null) {
			throw new FunctionNotExistException(MysqlCatalogFactoryOptions.IDENTIFIER, functionPath);
		}
		CatalogFunction retFun = new CatalogFunctionImpl(funinfo.get(ColsMetaFun.identifier.getOriColName()),
				FunctionLanguage.valueOf(funinfo.get(ColsMetaFun.language.getOriColName())));
		return retFun;
	}

	@Override
	public boolean functionExists(ObjectPath functionPath) throws CatalogException {
		try {
			CatalogFunction function = getFunction(functionPath);
			return function != null;
		} catch (FunctionNotExistException e) {
			return false;
		}
	}

	@Override
	public void createFunction(ObjectPath functionPath, CatalogFunction function, boolean ignoreIfExists)
			throws FunctionAlreadyExistException, DatabaseNotExistException, CatalogException {
		validDb(functionPath);
		CatalogFunction baseFun;
		try {
			baseFun = getFunction(functionPath);
		} catch (FunctionNotExistException e) {
			baseFun = null;// 正常情况，其它异常不处理
		}
		if (!ignoreIfExists && baseFun != null) {
			throw new FunctionAlreadyExistException(MysqlCatalogFactoryOptions.IDENTIFIER, functionPath);
		}
		// 创建函数
		if (baseFun == null) {
			try {
				String language = function.getFunctionLanguage() == null ? FunctionLanguage.JAVA.name()
						: function.getFunctionLanguage().name();
				String desc = function.getDescription() != null && function.getDescription().isPresent()
						? function.getDescription().get()
						: "";
				String descDetail = function.getDetailedDescription() != null
						&& function.getDetailedDescription().isPresent() ? function.getDetailedDescription().get() : "";
				MySqlAssitExt.execSqlPre(this.connection,
						String.format("insert into %s(%s,%s,%s,%s,%s,%s) value(?,?,?,?,?,?)", ColsMetaFun.tb,
								ColsMetaFun.dbName.getOriColName(), ColsMetaFun.name.getOriColName(),
								ColsMetaFun.language.getOriColName(), ColsMetaFun.identifier.getOriColName(),
								ColsMetaFun.description.getOriColName(), ColsMetaFun.detailDescription.getOriColName()),
						functionPath.getDatabaseName(), functionPath.getObjectName(), language, function.getClassName(),
						desc, descDetail);
			} catch (Exception e) {
				throw new CatalogException("创建函数时失败", e);
			}
		}
	}

	@Override
	public void alterFunction(ObjectPath functionPath, CatalogFunction newFunction, boolean ignoreIfNotExists)
			throws FunctionNotExistException, CatalogException {
		// 先删除再插入达到修改
		dropFunction(functionPath, ignoreIfNotExists);
		try {
			createFunction(functionPath, newFunction, ignoreIfNotExists);
		} catch (FunctionAlreadyExistException e) {
			throw new CatalogException(e);
		} catch (DatabaseNotExistException e) {
			throw new CatalogException(e);
		}
	}

	@Override
	public void dropFunction(ObjectPath functionPath, boolean ignoreIfNotExists)
			throws FunctionNotExistException, CatalogException {
		Map<String, String> funinfo = null;
		try {
			funinfo = queryFun(functionPath);
		} catch (Exception e) {
			throw new CatalogException("查询fun失败");
		}
		if (!ignoreIfNotExists && funinfo == null) {
			throw new FunctionNotExistException(MysqlCatalogFactoryOptions.IDENTIFIER, functionPath);
		}
		if (funinfo != null) {
			try {
				MySqlAssitExt.execSqlPre(this.connection,
						String.format("delete from %s where %s=? ", ColsMetaFun.tb, ColsMetaFun.id.getOriColName()),
						Long.parseLong(funinfo.get(ColsMetaFun.id.getOriColName())));
			} catch (Exception e) {
				throw new CatalogException("删除函数时失败", e);
			}
		}
	}

	@Override
	public CatalogTableStatistics getTableStatistics(ObjectPath tablePath)
			throws TableNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogColumnStatistics getTableColumnStatistics(ObjectPath tablePath)
			throws TableNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogTableStatistics getPartitionStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogColumnStatistics getPartitionColumnStatistics(ObjectPath tablePath,
			CatalogPartitionSpec partitionSpec) throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public void alterTableStatistics(ObjectPath tablePath, CatalogTableStatistics tableStatistics,
			boolean ignoreIfNotExists) throws TableNotExistException, CatalogException {

	}

	@Override
	public void alterTableColumnStatistics(ObjectPath tablePath, CatalogColumnStatistics columnStatistics,
			boolean ignoreIfNotExists) throws TableNotExistException, CatalogException, TablePartitionedException {

	}

	@Override
	public void alterPartitionStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogTableStatistics partitionStatistics, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public void alterPartitionColumnStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogColumnStatistics columnStatistics, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public boolean databaseExists(String databaseName) throws CatalogException {
		String sql = "select count(1) from " + ColsMetaDb.tb + " where " + ColsMetaDb.name.getOriColName() + "=?";
		try {
			PreparedStatement ps = this.connection.prepareStatement(sql);
			ps.setString(1, databaseName);
			ResultSet rs = ps.executeQuery();
			while (rs.next()) {
				int count = rs.getInt(1);
				if (count == 0) {
					return false;
				} else {
					return true;
				}
			}
		} catch (SQLException e) {
			throw new CatalogException("查询表" + databaseName + "异常");
		}
		return false;
	}

	@Override
	public void createDatabase(String name, CatalogDatabase database, boolean ignoreIfExists)
			throws DatabaseAlreadyExistException, CatalogException {
		int count = MySqlAssitExt.querySqlCount(this.connection, String.format("select count(1) from %s where %s='%s'",
				ColsMetaDb.tb, ColsMetaDb.name.getOriColName(), name));
		if (!ignoreIfExists && count > 0) {
			throw new DatabaseAlreadyExistException(MysqlCatalogFactoryOptions.IDENTIFIER, name);
		}
		if (count == 0) {
			String desc = (database != null && database.getDescription().isPresent()) ? database.getDescription().get()
					: "";
			MySqlAssitExt.execSql(this.connection,
					String.format("insert into %s(name,comment,description) values('%s','%s','%s')", ColsMetaDb.tb,
							name, (database == null) ? "" : StringUtil.hasNull(database.getComment()), desc));
		}
	}

	@Override
	public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade)
			throws DatabaseNotExistException, DatabaseNotEmptyException, CatalogException {

	}

	@Override
	public void alterDatabase(String name, CatalogDatabase newDatabase, boolean ignoreIfNotExists)
			throws DatabaseNotExistException, CatalogException {

	}

	/**
	 * 由FactoryUtils调用，如果返回空，就根据connector字段来判断，利用Java SPI去实现工厂的获取
	 * AbstractJdbcCatalog默认会返回Jdbc动态工厂这是不对的
	 * 
	 * @return
	 */
	public Optional<Factory> getFactory() {
		return Optional.empty();
	}

}
