/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.flink.catalog;

import java.io.InputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.ServiceLoader;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.flink.table.api.Schema;
import org.apache.flink.table.api.Schema.Builder;
import org.apache.flink.table.api.Schema.UnresolvedColumn;
import org.apache.flink.table.api.Schema.UnresolvedComputedColumn;
import org.apache.flink.table.api.Schema.UnresolvedMetadataColumn;
import org.apache.flink.table.api.Schema.UnresolvedPhysicalColumn;
import org.apache.flink.table.api.Schema.UnresolvedPrimaryKey;
import org.apache.flink.table.api.Schema.UnresolvedWatermarkSpec;
import org.apache.flink.table.catalog.AbstractCatalog;
import org.apache.flink.table.catalog.CatalogBaseTable;
import org.apache.flink.table.catalog.CatalogBaseTable.TableKind;
import org.apache.flink.table.catalog.CatalogDatabase;
import org.apache.flink.table.catalog.CatalogDatabaseImpl;
import org.apache.flink.table.catalog.CatalogFunction;
import org.apache.flink.table.catalog.CatalogFunctionImpl;
import org.apache.flink.table.catalog.CatalogPartition;
import org.apache.flink.table.catalog.CatalogPartitionSpec;
import org.apache.flink.table.catalog.CatalogTable;
import org.apache.flink.table.catalog.CatalogView;
import org.apache.flink.table.catalog.Column;
import org.apache.flink.table.catalog.FunctionLanguage;
import org.apache.flink.table.catalog.ObjectPath;
import org.apache.flink.table.catalog.ResolvedCatalogView;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.catalog.exceptions.CatalogException;
import org.apache.flink.table.catalog.exceptions.DatabaseAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotEmptyException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotExistException;
import org.apache.flink.table.catalog.exceptions.FunctionAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.FunctionNotExistException;
import org.apache.flink.table.catalog.exceptions.PartitionAlreadyExistsException;
import org.apache.flink.table.catalog.exceptions.PartitionNotExistException;
import org.apache.flink.table.catalog.exceptions.PartitionSpecInvalidException;
import org.apache.flink.table.catalog.exceptions.TableAlreadyExistException;
//20220623 andy 确保不是org.apache.flink.table.api.TableNotExistException异常
import org.apache.flink.table.catalog.exceptions.TableNotExistException;
import org.apache.flink.table.catalog.exceptions.TableNotPartitionedException;
import org.apache.flink.table.catalog.exceptions.TablePartitionedException;
import org.apache.flink.table.catalog.stats.CatalogColumnStatistics;
import org.apache.flink.table.catalog.stats.CatalogTableStatistics;
import org.apache.flink.table.expressions.Expression;
import org.apache.flink.table.expressions.SqlCallExpression;
import org.apache.flink.table.factories.CatalogFactory;
import org.apache.flink.table.factories.Factory;
import org.apache.flink.table.functions.FunctionKind;
import org.apache.flink.table.types.AtomicDataType;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.logical.LogicalTypeRoot;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;

import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.jdbc.JdbcAssit;
import net.wicp.tams.common.apiext.jdbc.MySqlColBean;
import net.wicp.tams.common.apiext.jdbc.MySqlTbBean;
import net.wicp.tams.common.apiext.json.JSONUtil;
import net.wicp.tams.common.apiext.json.jackson.SingleQuoteStringSerializer;
import net.wicp.tams.common.constant.Middleware;
import net.wicp.tams.common.constant.MiddlewareOption;
import net.wicp.tams.common.constant.dic.YesOrNo;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.flink.common.CatalogAssit;
import net.wicp.tams.common.flink.common.SqlRunTams;
import net.wicp.tams.common.flink.common.constant.CatalogName;
import net.wicp.tams.common.flink.common.constant.DelLevel;
import net.wicp.tams.common.flink.common.constant.FlinkTypeEnum;
import net.wicp.tams.common.flink.common.constant.db.ColsMetaCol;
import net.wicp.tams.common.flink.common.constant.db.ColsMetaDb;
import net.wicp.tams.common.flink.common.constant.db.ColsMetaFun;
import net.wicp.tams.common.flink.common.constant.db.ColsMetaTb;
import net.wicp.tams.common.flink.common.constant.db.ColsMetaTbLogic;
import net.wicp.tams.common.flink.common.constant.db.ColsMetaView;
import net.wicp.tams.common.flink.common.module.IPackageEnum;
import net.wicp.tams.common.flink.common.module.IUdfEnum;
import net.wicp.tams.common.flink.module.factory.PackageTams;
import net.wicp.tams.common.flink.paimon.PaimonFlinkAssit;
import net.wicp.tams.common.jdbc.DruidAssit;
import net.wicp.tams.common.jdbc.MySqlAssitExt;

/***
 *
 * @author andy
 *
 */
public class MysqlCatalog extends AbstractCatalog {

	private final String tbPre = "t_";
	private final String viewPre = "v_";
	private final String errorMsg = "表以t_开头，视图以v_开头，不存在此类型的对象[%s]";
	private final long tenantId;// 租户
	private final long operate;
	private final String mysqlDefaultDb;

	private Connection connection;

	public MysqlCatalog(String catalogName, String mysqlDefaultDb, long tenantId, long operate) {
		super(catalogName, CatalogAssit.catalogDefaultDb);
		this.tenantId = tenantId;
		this.operate = operate;
		this.mysqlDefaultDb = mysqlDefaultDb;
	}

	@SuppressWarnings({ "rawtypes", "unused" })
	@Override
	public void open() throws CatalogException {
		this.connection = DruidAssit.getConnection(CatalogAssit.IDENTIFIER);
		// 初始化数据库脚本
		try {
			InputStream inputStream = IOUtil.fileToInputStream("/META-INF/tams/dbinit/default.sql", MysqlCatalog.class);
			JdbcAssit.exeSQLScript(connection, inputStream, true, false);
		} catch (Exception e1) {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_exec_fail, "初始化相关元数据表失败", e1);
		}

		try {// 创建默认的db和duckula，有可能这2个库是同一个，也有可能不是。
			this.createDatabase(this.getDefaultDatabase(), null, true);
		} catch (CatalogException | DatabaseAlreadyExistException e1) {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_exec_fail, "初始化default库失败", e1);
		}

		List<String> moudlePaths = CatalogAssit.listMoudlePaths(this.connection, this.mysqlDefaultDb, this.tenantId);
		for (String moudlePath : moudlePaths) {
			// TODO需要看加载磁盘路径
			String osPath = IOUtil.mergeFolderAndFilePath("/data/duckula-data/upload/modulejar", moudlePath);
			ClassLoader userClassLoader = CatalogFactory.Context.class.getClassLoader();
//			  ResourceUri resourceUri = new ResourceUri(ResourceType.JAR, osPath);
//			  EnvironmentSettings settings = EnvironmentSettings.newInstance().build();
//			  ResourceManager resourceManager =
//		                new ResourceManager((ReadableConfig)settings.getConfiguration(), (MutableURLClassLoader)userClassLoader);
//			  resourceManager.registerJarResources(Collections.singletonList(resourceUri));
		}

		// 由于页面已管理了函数，不需要再次加载（公共函数需要加载）
		List<String> listFunctions = CatalogAssit.listFunctions(this.connection, this.mysqlDefaultDb, this.tenantId);
		// 得到IUdfEnum的构造器，一个个自动加载函数
		ServiceLoader<IPackageEnum> udffats = ServiceLoader.load(IPackageEnum.class);
		for (IPackageEnum udffat : udffats) {
			if (udffat instanceof PackageTams) {// 公共函数需要加载
				try {
					IUdfEnum[] udfs = udffat.queryAll();
					for (IUdfEnum udf : udfs) {
						if (!listFunctions.contains(udf.getFullName())) {//
							CatalogAssit.createFunByMoudle(this.connection, this.mysqlDefaultDb, -1l,
									CatalogAssit.catalogDefaultDb, udf, this.tenantId, this.operate);// moleid为null,因为不是从模块导入的
						}
					}
				} catch (Exception e) {
					throw new ProjectExceptionRuntime(ExceptAll.flink_udf_errorformate, "需要定义实现IUdfEnum接口的枚举类", e);
				}
				break;
			}
		}
	}

	@Override
	public void close() throws CatalogException {
		DruidAssit.close(this.connection);
	}

	@Override
	@SuppressWarnings("unchecked")
	public List<String> listDatabases() throws CatalogException {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
				"select name from meta_db where tenant_id=?"
						+ String.format(" and %s='%s'", ColsMetaDb.catalogName.getOriColName(), super.getName()),
				false, this.tenantId);
		List<String> dbs = (List<String>) CollectionUtil.getColFromObj(querySqlMap, "name");
		return dbs;
	}

	@Override
	public CatalogDatabase getDatabase(String databaseName) throws DatabaseNotExistException, CatalogException {
		ResultSet rs = MySqlAssitExt.querySql(this.connection, String.format(
				"select * from %s.%s where %s='%s' and  %s='%s'"
						+ String.format(" and %s='%s'", ColsMetaDb.catalogName.getOriColName(), super.getName()),
				this.mysqlDefaultDb, ColsMetaDb._tb, ColsMetaDb.tenantId.getOriColName(), this.tenantId,
				ColsMetaDb.name.getOriColName(), databaseName));
		try {
			if (rs.next()) {
				Map<String, String> properties = new HashMap<String, String>();
				String propertiesStr = rs.getString(ColsMetaDb.properties.getOriColName());
				if (StringUtil.isNull(rs)) {
					properties = JSONUtil.jsonToMapStr((ObjectNode) JSONUtil.parserStr(new ObjectMapper(), propertiesStr));
				}
				CatalogDatabaseImpl retobj = new CatalogDatabaseImpl(properties,
						rs.getString(ColsMetaDb.comment.getOriColName()));
				return retobj;
			}
		} catch (SQLException e) {
			throw new ProjectExceptionRuntime(ExceptAll.duckula_nodata, "查数据库" + databaseName + "失败", e);
		}
		throw new DatabaseNotExistException(this.getName(), databaseName);
	}

	@Override
	public List<String> listTables(String databaseName) throws DatabaseNotExistException, CatalogException {
		List<String> colFromObj = CatalogAssit.listTableOrViewByDb(this.connection, databaseName, TableKind.TABLE,
				this.tenantId);
		return colFromObj;
	}

	@Override
	public List<String> listViews(String databaseName) throws DatabaseNotExistException, CatalogException {
		List<String> colFromObj = CatalogAssit.listTableOrViewByDb(this.connection, databaseName, TableKind.VIEW,
				this.tenantId);
		return colFromObj;
	}

	/***
	 * 得到表或视图信息，注意v_开头为视图，t_为表。
	 */
	@Override
	public CatalogBaseTable getTable(ObjectPath tablePath) throws TableNotExistException, CatalogException {
		Map<String, String> tbinfo = null;
		try {
			tbinfo = queryTable(tablePath);
		} catch (TableNotExistException e) {
			// 20220623 andy 这里必须抛出此异常，在CatalogManager.getPermanentTable
			// 方法只会忽略此异常。其它异常会阻碍运行下去，那个tablePath如果在db时会先尝试把db名当表名去查一次就会有问题，就不能支持db._tb模式的使用了。
			throw e;
		} catch (Exception e) {
			throw new CatalogException("查询table失败", e);
		}
		if (tbinfo == null) {
			throw new TableNotExistException(this.getName(), tablePath);
		}
		String objectName = tablePath.getObjectName();
		if (objectName.startsWith(tbPre)) {// 表
			Map<String, String> tbLogicInfo = getTableInfoLogic(
					Long.parseLong(tbinfo.get(ColsMetaTb.tbLogicId.getOriColName())), tablePath);
			Builder builder = Schema.newBuilder();
			String comment = tbLogicInfo.get(ColsMetaTbLogic.comment.getOriColName());
			// 分区键以物理表定义为主，逻辑表定义为辅。
			String partitionkeysStr = StringUtil.hasNull(tbinfo.get(ColsMetaTb.partitionkeys.getOriColName()),
					tbLogicInfo.get(ColsMetaTbLogic.partitionkeys.getOriColName()));
			List<String> partitionKeys = StringUtil.isNull(partitionkeysStr) ? new ArrayList<>()
					: Arrays.asList(partitionkeysStr.split(","));
			String optionsStr = tbinfo.get(ColsMetaTb.withOptions.getOriColName());
			Map<String, String> options = null;
			if (StringUtil.isNotNull(optionsStr)) {
				ObjectNode jsonObject =  (ObjectNode)JSONUtil.parserStr(new ObjectMapper(), optionsStr);
				options = JSONUtil.jsonToMapStr(jsonObject);
			} else {
				options = new HashMap<String, String>();
			}

			String keysStr = StringUtil.hasNull(tbinfo.get(ColsMetaTb.keyCol.getOriColName()),
					tbLogicInfo.get(ColsMetaTbLogic.keyCol.getOriColName()));
			String[] keys = null;// 20230329 andyzhou主键
			if (StringUtil.isNotNull(keysStr)) {// 20220623
				// 无主键也需要支持，不能报错。append模式
				keys = keysStr.split(",");
				builder.primaryKey(keys);
//				throw new CatalogException(tablePath.getFullName() + "没有配置主键，请确认表的配置是正确的。");
			}

			List<Map<String, String>> queryCols = new ArrayList<Map<String, String>>();
			try {
				queryCols = CatalogAssit.queryCols(this.connection,
						Long.parseLong(tbLogicInfo.get(ColsMetaTbLogic.id.getOriColName())), this.tenantId);
			} catch (Exception e) {
				throw new CatalogException("查询cols失败");
			}
			if (CollectionUtils.isEmpty(queryCols)) {
				throw new CatalogException(tablePath.getFullName() + "没有配置字段，请确认表的配置是正确的。");
			}
			for (Map<String, String> col : queryCols) {
				String colName = col.get(ColsMetaCol.name.getOriColName());
				// 需要加这个，否则在聚合时会出现：Exception in thread "main"
				// org.apache.flink.table.api.TableException: Window aggregate can only be
				// defined over a time attribute column, but TIMESTAMP(3) encountered.
				String watermarkExpression = col.get(ColsMetaCol.watermarkExpression.getOriColName());
				if (StringUtil.isNotNull(watermarkExpression)) {
					builder.watermark(colName, watermarkExpression);
				}

				String sqlexpression = col.get(ColsMetaCol.sqlexpression.getOriColName());
				String isMetadataStr = col.get(ColsMetaCol.isMetadata.getOriColName());// 是否Metadata，yes
				if (StringUtil.isNotNull(sqlexpression)) {
					builder.columnByExpression(colName, sqlexpression);
				} else if (StringUtil.isNotNull(isMetadataStr) && YesOrNo.yes.name().equalsIgnoreCase(isMetadataStr)) {// 是Metadata
					YesOrNo isVirtual = YesOrNo
							.valueOf(StringUtil.hasNull(col.get(ColsMetaCol.isVirtual.getOriColName()), "no"));
					String metadataName = StringUtil.isNull(col.get(ColsMetaCol.metadataName.getOriColName())) ? null
							: col.get(ColsMetaCol.metadataName.getOriColName());
					DataType dataType = PaimonFlinkAssit.findColType(col, ArrayUtils.contains(keys, colName));
					builder.columnByMetadata(colName, dataType, metadataName, isVirtual == YesOrNo.yes);
				} else {
					DataType dataType = PaimonFlinkAssit.findColType(col, ArrayUtils.contains(keys, colName));
					builder.column(colName, dataType);
				}
				// Method 'withComment(...)' must be called after a column definition, but there
				// is no preceding column defined.
				if (col.containsKey(ColsMetaCol.comment.getOriColName())) {// 备注
					builder.withComment(col.get(ColsMetaCol.comment.getOriColName()));
				}
			}
			// 所有catalogtable统一加一个处理时间字段
			// 由界面生成，不能偷偷加一个proc_time字段。
//			builder.columnByExpression("proc_time", "PROCTIME()");
			// builder.watermark(tbrs.getString("watermark_row_time_attribute"),
			// tbrs.getString("watermark_expression"), DataTypes.TIMESTAMP(3));

			CatalogTable retobj = CatalogTable.of(builder.build(), comment, partitionKeys, options);
			return retobj;
		} else if (objectName.startsWith(viewPre)) {// 视图
			Builder builder = Schema.newBuilder();
			String columnsJsonStr = tbinfo.get(ColsMetaView.columns.getOriColName());
			if (StringUtil.isNotNull(columnsJsonStr)) {
				ArrayNode colsAry = (ArrayNode)JSONUtil.parserStr(new ObjectMapper(), columnsJsonStr);
				for (Object object : colsAry) {
					ObjectNode colJson = (ObjectNode) object;
					FlinkTypeEnum flinkTypeEnum = FlinkTypeEnum
							.valueOf(colJson.get(ColsMetaCol.type.getOriColName()).asText());
					DataType dataType = null;
					switch (flinkTypeEnum) {
					case DECIMAL:
						dataType = flinkTypeEnum.getConvert().getObj(12, 2);// TODO 后面需要添加支持
						break;
					default:
						dataType = flinkTypeEnum.getConvert().getObj();
						break;
					}
					builder.column(colJson.get(ColsMetaCol.name.getOriColName()).asText(), dataType);
				}
			}
			String comment = tbinfo.get(ColsMetaView.remark.getOriColName());
			String originalQuery = tbinfo.get(ColsMetaView.originalQuery.getOriColName());
			String expandedQuery = tbinfo.get(ColsMetaView.expandedQuery.getOriColName());
			String optionsStr = tbinfo.get(ColsMetaView.withOptions.getOriColName());
			Map<String, String> options = null;
			if (StringUtil.isNotNull(optionsStr)) {
				ObjectNode jsonObject = (ObjectNode)JSONUtil.parserStr(new ObjectMapper(), optionsStr);
				options = JSONUtil.jsonToMapStr(jsonObject);
			} else {
				options = new HashMap<String, String>();
			}
			CatalogView retobj = CatalogView.of(builder.build(), comment, originalQuery, expandedQuery, options);
			return retobj;
		} else {
			throw new CatalogException(String.format(errorMsg, tablePath));
		}

	}

	

	/***
	 * 得到逻辑表信息,不用租户限制它了。
	 *
	 * @param logicTbId
	 * @param tablePath
	 * @return
	 * @throws TableNotExistException
	 */
	private Map<String, String> getTableInfoLogic(long logicTbId, ObjectPath tablePath) throws TableNotExistException {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
				String.format("select * from  %s where %s=? and  %s=?", ColsMetaTbLogic._tb,
						ColsMetaTb.id.getOriColName(), ColsMetaTb.tenantId.getOriColName()),
				false, logicTbId, this.tenantId);
		if (CollectionUtils.isNotEmpty(querySqlMap)) {
			return querySqlMap.get(0);
		} else {
			throw new TableNotExistException(this.getName(), tablePath);
		}
	}

	@Override
	public boolean tableExists(ObjectPath tablePath) throws CatalogException {
		try {
			CatalogBaseTable baseTable = getTable(tablePath);
			return baseTable != null;
		} catch (TableNotExistException e) {
			return false;
		}
	}

	@Override
	public void dropTable(ObjectPath tablePath, boolean ignoreIfNotExists)
			throws TableNotExistException, CatalogException {
		try {
			validDb(tablePath);
		} catch (DatabaseNotExistException e1) {
			if (!ignoreIfNotExists) {
				throw new TableNotExistException(this.getName(), tablePath);
			}
		}
		CatalogBaseTable baseTable;
		try {
			baseTable = getTable(tablePath);
		} catch (TableNotExistException e) {
			baseTable = null;// 正常情况，其它异常不处理
		}
		if (!ignoreIfNotExists && baseTable == null) {
			throw new TableNotExistException(this.getName(), tablePath);
		}

		if (baseTable != null) {
			String objectName = tablePath.getObjectName();
			if (objectName.startsWith(tbPre) && baseTable.getTableKind() == TableKind.TABLE) {// 表
				Result dropTable;
				try {
					dropTable = CatalogAssit.dropTable(this.connection, this.mysqlDefaultDb,
							tablePath.getDatabaseName(), tablePath.getObjectName(), DelLevel.delLogicNotTb,
							this.tenantId, true);
					if (!dropTable.isSuc()) {
						if (ignoreIfNotExists && ExceptAll.flink_catalog_tablenotexit == dropTable.getExcept()) {
							throw new TableNotExistException(this.getName(), tablePath);
						} else {
							throw new CatalogException(dropTable.getMessage());
						}
					}
				} catch (ProjectException e) {
					throw new CatalogException(e.getMessage());
				}
			} else {// 视图
				Result dropTable;
				try {
					dropTable = CatalogAssit.dropView(this.connection, this.mysqlDefaultDb, tablePath.getDatabaseName(),
							tablePath.getObjectName(), this.tenantId);
					if (!dropTable.isSuc()) {
						if (ignoreIfNotExists && ExceptAll.flink_catalog_tablenotexit == dropTable.getExcept()) {
							throw new TableNotExistException(this.getName(), tablePath);
						} else {
							throw new CatalogException(dropTable.getMessage());
						}
					}
				} catch (ProjectException e) {
					throw new CatalogException(e.getMessage());
				}
			}
		}
	}

	@Override
	public void renameTable(ObjectPath tablePath, String newTableName, boolean ignoreIfNotExists)
			throws TableNotExistException, TableAlreadyExistException, CatalogException {

	}

	@Override
	public void createTable(ObjectPath tablePath, CatalogBaseTable table, boolean ignoreIfExists)
			throws TableAlreadyExistException, DatabaseNotExistException, CatalogException {
		validDb(tablePath);
		CatalogBaseTable baseTable;
		try {
			baseTable = getTable(tablePath);
		} catch (TableNotExistException e) {
			baseTable = null;// 正常情况，其它异常不处理
		}
		if (!ignoreIfExists && baseTable != null) {
			throw new TableAlreadyExistException(this.getName(), tablePath);
		}
		if (baseTable == null) {
			String objectName = tablePath.getObjectName();
			if (objectName.startsWith(tbPre) && table.getTableKind() == TableKind.TABLE) {// 表
				try {
					List<MySqlColBean> mysqlColList = CatalogAssit.packageColBeans(table, objectName);
					// 分区
					String partitionkeys = "";
					if (table.getTableKind() == TableKind.TABLE) {// 是表
						CatalogTable tableTrue = (CatalogTable) table;
						if (CollectionUtils.isNotEmpty(tableTrue.getPartitionKeys())) {
							partitionkeys = CollectionUtil.listJoin(tableTrue.getPartitionKeys(), ",");
						}
					}
					// 转换为统一的配置
					Map<MiddlewareOption, String> allOptmap = new HashMap<MiddlewareOption, String>();
					Map<String, String> options = table.getOptions();
					if (MapUtils.isNotEmpty(options)) {
						for (String optionstr : options.keySet()) {
							MiddlewareOption findByNameUse = MiddlewareOption.findByNameUse(optionstr);
							if (findByNameUse != null) {
								allOptmap.put(findByNameUse, options.get(optionstr));
							} else {
								throw new CatalogException(
										"创建表[" + objectName + "]时检查异常，原因:catalog还不支持此option[" + optionstr + "].");
							}
						}
					}
					// 得到option相关配置，全部做为可配置项处理。
					Pair<String, String> proOptStr = Middleware.proOptStr(allOptmap,
							allOptmap.keySet().toArray(new MiddlewareOption[allOptmap.size()]));
					MySqlTbBean catalogTbInfo = MySqlTbBean.builder().db(tablePath.getDatabaseName()).tb(objectName)
							.tbComment("通过sql创建").middleware(Middleware.no).partitionkeys(partitionkeys)
							.opt(proOptStr.getRight()).catalogName(super.getName()).withOptions(proOptStr.getLeft())
							.build();
					CatalogAssit.createTable(this.connection, this.mysqlDefaultDb, mysqlColList, catalogTbInfo,
							this.tenantId, this.operate, true);
				} catch (ProjectException e) {
					throw new CatalogException("创建表[" + objectName + "]时SQL异常，原因：" + e.getMessage(), e);
				}
			} else if (objectName.startsWith(viewPre) && table.getTableKind() == TableKind.VIEW) {// 创建视图
				try {
					// 没有用父类：CatalogView
					ResolvedCatalogView catalogView = (ResolvedCatalogView) table;
					ResolvedSchema resolvedSchema = catalogView.getResolvedSchema();
					ArrayNode cols = SingleQuoteStringSerializer.getInst().createArrayNode();
					for (Column colEle : resolvedSchema.getColumns()) {
						ObjectNode jsonobj = JsonNodeFactory.instance.objectNode();
						jsonobj.put(ColsMetaCol.name.getOriColName(), colEle.getName());
						FlinkTypeEnum flinkTypeEnum = FlinkTypeEnum
								.valueOf(colEle.getDataType().getLogicalType().getTypeRoot().name());
						jsonobj.put(ColsMetaCol.type.getOriColName(), flinkTypeEnum.name());
						cols.add(jsonobj);
					}					
					String colsStr = cols.toString();
					SqlRunTams sqlRunTams=new SqlRunTams();//添加血缘关系
					sqlRunTams.addCatalog(this);
					CatalogAssit.createView(this.connection, this.mysqlDefaultDb, tablePath.getDatabaseName(),
							objectName, colsStr, catalogView.getOriginalQuery(), catalogView.getExpandedQuery(),
							catalogView.getComment(), tenantId, operate,true,sqlRunTams);
				} catch (Exception e) {
					throw new CatalogException("创建视图[" + objectName + "]时SQL异常", e);
				}
			} else {
				throw new CatalogException(String.format(errorMsg, tablePath));
			}
		}
	}

	

	private void validDb(ObjectPath tablePath) throws DatabaseNotExistException {
		CatalogDatabase database;
		try {
			database = getDatabase(tablePath.getDatabaseName());
		} catch (DatabaseNotExistException e) {
			database = null;// 正常情况，其它异常不处理
		}
		if (database == null) {
			throw new DatabaseNotExistException(this.getName(), tablePath.getDatabaseName());
		}
	}

	// 查询表或视图
	private Map<String, String> queryTable(ObjectPath tablePath) throws SQLException, TableNotExistException {
		String objectName = tablePath.getObjectName();
		if (objectName.startsWith(tbPre)) {
			Map<String, String> queryTable = CatalogAssit.queryTable(this.connection, this.mysqlDefaultDb,
					tablePath.getDatabaseName(), tablePath.getObjectName(), this.tenantId);
			return queryTable;
		} else if (objectName.startsWith(viewPre)) {// 查询视图
			Map<String, String> queryView = CatalogAssit.queryView(this.connection, this.mysqlDefaultDb,
					tablePath.getDatabaseName(), tablePath.getObjectName(), this.tenantId);
			return queryView;
		} else {
			// 20220623 andy 这里必须抛出此异常，在CatalogManager.getPermanentTable
			// 方法只会忽略此异常。其它异常会阻碍运行下去，那个tablePath如果在db时会先尝试把db名当表名去查一次就会有问题，就不能支持db.tb模式的使用了。
			throw new TableNotExistException(super.getName(), tablePath);
//			throw new CatalogException(String.format(errorMsg, tablePath));
		}
	}

	private Map<String, String> queryFun(ObjectPath functionPath) throws SQLException {
		List<Map<String, String>> querySqlMap = MySqlAssitExt.querySqlMapPre(this.connection,
				String.format("select * from %s.%s where %s=? and  %s=? and  %s=?", this.mysqlDefaultDb,
						ColsMetaFun._tb, ColsMetaFun.tenantId.getOriColName(), ColsMetaFun.dbName.getOriColName(),
						ColsMetaFun.name.getOriColName()),
				false, this.tenantId, functionPath.getDatabaseName(), functionPath.getObjectName());
		if (CollectionUtils.isNotEmpty(querySqlMap)) {
			return querySqlMap.get(0);
		}
		return null;
	}

	@Override
	public void alterTable(ObjectPath tablePath, CatalogBaseTable newTable, boolean ignoreIfNotExists)
			throws TableNotExistException, CatalogException {
		dropTable(tablePath, true);
		try {
			createTable(tablePath, newTable, false);
		} catch (TableAlreadyExistException | DatabaseNotExistException e) {
			throw new CatalogException(e);
		}
	}

	@Override
	public List<CatalogPartitionSpec> listPartitions(ObjectPath tablePath)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public List<CatalogPartitionSpec> listPartitions(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public List<CatalogPartitionSpec> listPartitionsByFilter(ObjectPath tablePath, List<Expression> filters)
			throws TableNotExistException, TableNotPartitionedException, CatalogException {
		return null;
	}

	@Override
	public CatalogPartition getPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public boolean partitionExists(ObjectPath tablePath, CatalogPartitionSpec partitionSpec) throws CatalogException {
		return false;
	}

	@Override
	public void createPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, CatalogPartition partition,
			boolean ignoreIfExists) throws TableNotExistException, TableNotPartitionedException,
			PartitionSpecInvalidException, PartitionAlreadyExistsException, CatalogException {

	}

	@Override
	public void dropPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public void alterPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, CatalogPartition newPartition,
			boolean ignoreIfNotExists) throws PartitionNotExistException, CatalogException {

	}

	@Override
	public List<String> listFunctions(String dbName) throws DatabaseNotExistException, CatalogException {
		List<String> listFunctions = CatalogAssit.listFunctions(this.connection, mysqlDefaultDb, this.tenantId);
		return listFunctions;
	}

	@Override
	public CatalogFunction getFunction(ObjectPath functionPath) throws FunctionNotExistException, CatalogException {
		Map<String, String> funinfo = null;
		try {
			funinfo = queryFun(functionPath);
		} catch (Exception e) {
			throw new CatalogException("查询fun失败");
		}
		if (funinfo == null) {
			throw new FunctionNotExistException(this.getName(), functionPath);
		}
		CatalogFunction retFun = new CatalogFunctionImpl(funinfo.get(ColsMetaFun.identifier.getOriColName()),
				FunctionLanguage.valueOf(funinfo.get(ColsMetaFun.language.getOriColName())));
		return retFun;
	}

	@Override
	public boolean functionExists(ObjectPath functionPath) throws CatalogException {
		try {
			CatalogFunction function = getFunction(functionPath);
			return function != null;
		} catch (FunctionNotExistException e) {
			return false;
		}
	}

	@Override
	public void createFunction(ObjectPath functionPath, CatalogFunction function, boolean ignoreIfExists)
			throws FunctionAlreadyExistException, DatabaseNotExistException, CatalogException {
		validDb(functionPath);
		CatalogFunction baseFun;
		try {
			baseFun = getFunction(functionPath);
		} catch (FunctionNotExistException e) {
			baseFun = null;// 正常情况，其它异常不处理
		}
		if (!ignoreIfExists && baseFun != null) {
			throw new FunctionAlreadyExistException(this.getName(), functionPath);
		}
		// 创建函数
		if (baseFun == null) {
			try {
				CatalogAssit.createFunByMoudle(this.connection, this.mysqlDefaultDb, -1l,
						functionPath.getDatabaseName(), new IUdfEnum() {

							@Override
							public String getName() {
								return functionPath.getObjectName();
							}

							@Override
							public FunctionKind getKind() {
								return FunctionKind.SCALAR;
							}

							@Override
							public String getClassFullName() {
								return function.getClassName();
							}

//							@Override
//							public String getDbName() {
//								return functionPath.getDatabaseName();
//							}

						}, this.tenantId, this.operate);
			} catch (SQLException e1) {
				throw new CatalogException("创建函数时失败", e1);
			}
		}
	}

	@Override
	public void alterFunction(ObjectPath functionPath, CatalogFunction newFunction, boolean ignoreIfNotExists)
			throws FunctionNotExistException, CatalogException {
		// 先删除再插入达到修改
		dropFunction(functionPath, ignoreIfNotExists);
		try {
			createFunction(functionPath, newFunction, ignoreIfNotExists);
		} catch (FunctionAlreadyExistException e) {
			throw new CatalogException(e);
		} catch (DatabaseNotExistException e) {
			throw new CatalogException(e);
		}
	}

	@Override
	public void dropFunction(ObjectPath functionPath, boolean ignoreIfNotExists)
			throws FunctionNotExistException, CatalogException {
		Map<String, String> funinfo = null;
		try {
			funinfo = queryFun(functionPath);
		} catch (Exception e) {
			throw new CatalogException("查询fun失败");
		}
		if (!ignoreIfNotExists && funinfo == null) {
			throw new FunctionNotExistException(this.getName(), functionPath);
		}
		if (funinfo != null) {
			try {
				MySqlAssitExt.execSqlPre(this.connection,
						String.format("delete from %s.%s where %s=? ", this.mysqlDefaultDb, ColsMetaFun._tb,
								ColsMetaFun.id.getOriColName()),
						Long.parseLong(funinfo.get(ColsMetaFun.id.getOriColName())));
			} catch (Exception e) {
				throw new CatalogException("删除函数时失败", e);
			}
		}
	}

	@Override
	public CatalogTableStatistics getTableStatistics(ObjectPath tablePath)
			throws TableNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogColumnStatistics getTableColumnStatistics(ObjectPath tablePath)
			throws TableNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogTableStatistics getPartitionStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public CatalogColumnStatistics getPartitionColumnStatistics(ObjectPath tablePath,
			CatalogPartitionSpec partitionSpec) throws PartitionNotExistException, CatalogException {
		return null;
	}

	@Override
	public void alterTableStatistics(ObjectPath tablePath, CatalogTableStatistics tableStatistics,
			boolean ignoreIfNotExists) throws TableNotExistException, CatalogException {

	}

	@Override
	public void alterTableColumnStatistics(ObjectPath tablePath, CatalogColumnStatistics columnStatistics,
			boolean ignoreIfNotExists) throws TableNotExistException, CatalogException, TablePartitionedException {

	}

	@Override
	public void alterPartitionStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogTableStatistics partitionStatistics, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public void alterPartitionColumnStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogColumnStatistics columnStatistics, boolean ignoreIfNotExists)
			throws PartitionNotExistException, CatalogException {

	}

	@Override
	public boolean databaseExists(String databaseName) throws CatalogException {
		String sql = "select count(1) from " + ColsMetaDb._tb + " where " + ColsMetaDb.tenantId.getOriColName()
				+ "=? and " + ColsMetaDb.name.getOriColName() + "=?";
		try {
			PreparedStatement ps = this.connection.prepareStatement(sql);
			ps.setLong(1, this.tenantId);
			ps.setString(2, databaseName);
			ResultSet rs = ps.executeQuery();
			while (rs.next()) {
				int count = rs.getInt(1);
				if (count == 0) {
					return false;
				} else {
					return true;
				}
			}
		} catch (SQLException e) {
			throw new CatalogException("查询表" + databaseName + "异常");
		}
		return false;
	}

	@Override
	public void createDatabase(String name, CatalogDatabase database, boolean ignoreIfExists)
			throws DatabaseAlreadyExistException, CatalogException {
		// 库名较为特殊，不能通过租户来判断有元
		int count = MySqlAssitExt.querySqlCount(this.connection,
				String.format("select count(1) from %s.%s where %s='%s' and %s='%s' and %s='%s'", this.mysqlDefaultDb,
						ColsMetaDb._tb, ColsMetaDb.tenantId.getOriColName(), this.tenantId,
						ColsMetaDb.name.getOriColName(), name, ColsMetaDb.catalogName.getOriColName(),
						CatalogName.mycatalog.name()));
		if (!ignoreIfExists && count > 0) {
			throw new DatabaseAlreadyExistException(this.getName(), name);
		}
		if (count == 0) {
			String desc = (database != null && database.getDescription().isPresent()) ? database.getDescription().get()
					: "";
			CatalogAssit.createDb(this.connection, this.mysqlDefaultDb, name, CatalogName.find(super.getName()),
					(database == null) ? "" : StringUtil.hasNull(database.getComment()), desc, tenantId, this.operate);
		}
	}

	@Override
	public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade)
			throws DatabaseNotExistException, DatabaseNotEmptyException, CatalogException {

	}

	@Override
	public void alterDatabase(String name, CatalogDatabase newDatabase, boolean ignoreIfNotExists)
			throws DatabaseNotExistException, CatalogException {

	}

	/**
	 * 由FactoryUtils调用，如果返回空，就根据connector字段来判断，利用Java SPI去实现工厂的获取
	 * AbstractJdbcCatalog默认会返回Jdbc动态工厂这是不对的
	 *
	 * @return
	 */
	public Optional<Factory> getFactory() {
		return Optional.empty();
	}

}
