/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.wicp.tams.common.flink.paimon;

import static org.apache.flink.table.descriptors.DescriptorProperties.NAME;
import static org.apache.flink.table.descriptors.DescriptorProperties.WATERMARK;
import static org.apache.flink.table.descriptors.DescriptorProperties.WATERMARK_ROWTIME;
import static org.apache.flink.table.descriptors.DescriptorProperties.WATERMARK_STRATEGY_DATA_TYPE;
import static org.apache.flink.table.descriptors.DescriptorProperties.WATERMARK_STRATEGY_EXPR;
import static org.apache.flink.table.descriptors.Schema.SCHEMA;
import static org.apache.flink.table.factories.FactoryUtil.CONNECTOR;
import static org.apache.flink.table.types.utils.TypeConversions.fromLogicalToDataType;
import static org.apache.paimon.CoreOptions.PATH;
import static org.apache.paimon.flink.FlinkCatalogOptions.LOG_SYSTEM_AUTO_REGISTER;
import static org.apache.paimon.flink.FlinkCatalogOptions.REGISTER_TIMEOUT;
import static org.apache.paimon.flink.LogicalTypeConversion.toDataType;
import static org.apache.paimon.flink.LogicalTypeConversion.toLogicalType;
import static org.apache.paimon.flink.log.LogStoreRegister.registerLogSystem;
import static org.apache.paimon.flink.log.LogStoreRegister.unRegisterLogSystem;
import static org.apache.paimon.flink.utils.FlinkCatalogPropertiesUtil.compoundKey;
import static org.apache.paimon.flink.utils.FlinkCatalogPropertiesUtil.deserializeNonPhysicalColumn;
import static org.apache.paimon.flink.utils.FlinkCatalogPropertiesUtil.deserializeWatermarkSpec;
import static org.apache.paimon.flink.utils.FlinkCatalogPropertiesUtil.nonPhysicalColumnsCount;
import static org.apache.paimon.flink.utils.FlinkCatalogPropertiesUtil.serializeNonPhysicalColumns;
import static org.apache.paimon.flink.utils.FlinkCatalogPropertiesUtil.serializeWatermarkSpec;
import static org.apache.paimon.utils.Preconditions.checkArgument;

import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.flink.table.api.Schema.UnresolvedColumn;
import org.apache.flink.table.api.Schema.UnresolvedPhysicalColumn;
import org.apache.flink.table.api.TableColumn;
import org.apache.flink.table.api.TableSchema;
import org.apache.flink.table.api.WatermarkSpec;
import org.apache.flink.table.catalog.AbstractCatalog;
import org.apache.flink.table.catalog.CatalogBaseTable;
import org.apache.flink.table.catalog.CatalogDatabase;
import org.apache.flink.table.catalog.CatalogDatabaseImpl;
import org.apache.flink.table.catalog.CatalogFunction;
import org.apache.flink.table.catalog.CatalogPartition;
import org.apache.flink.table.catalog.CatalogPartitionSpec;
import org.apache.flink.table.catalog.CatalogTable;
import org.apache.flink.table.catalog.CatalogTableImpl;
import org.apache.flink.table.catalog.ObjectPath;
import org.apache.flink.table.catalog.TableChange;
import org.apache.flink.table.catalog.TableChange.AddColumn;
import org.apache.flink.table.catalog.TableChange.AddWatermark;
import org.apache.flink.table.catalog.TableChange.After;
import org.apache.flink.table.catalog.TableChange.ColumnPosition;
import org.apache.flink.table.catalog.TableChange.DropColumn;
import org.apache.flink.table.catalog.TableChange.DropWatermark;
import org.apache.flink.table.catalog.TableChange.First;
import org.apache.flink.table.catalog.TableChange.ModifyColumnComment;
import org.apache.flink.table.catalog.TableChange.ModifyColumnName;
import org.apache.flink.table.catalog.TableChange.ModifyColumnPosition;
import org.apache.flink.table.catalog.TableChange.ModifyPhysicalColumnType;
import org.apache.flink.table.catalog.TableChange.ModifyWatermark;
import org.apache.flink.table.catalog.TableChange.ResetOption;
import org.apache.flink.table.catalog.TableChange.SetOption;
import org.apache.flink.table.catalog.exceptions.CatalogException;
import org.apache.flink.table.catalog.exceptions.DatabaseAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotEmptyException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotExistException;
import org.apache.flink.table.catalog.exceptions.FunctionNotExistException;
import org.apache.flink.table.catalog.exceptions.PartitionNotExistException;
import org.apache.flink.table.catalog.exceptions.TableAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.TableNotExistException;
import org.apache.flink.table.catalog.stats.CatalogColumnStatistics;
import org.apache.flink.table.catalog.stats.CatalogTableStatistics;
import org.apache.flink.table.descriptors.DescriptorProperties;
import org.apache.flink.table.expressions.Expression;
import org.apache.flink.table.factories.Factory;
import org.apache.flink.table.types.AbstractDataType;
import org.apache.flink.table.types.AtomicDataType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.LogicalTypeRoot;
import org.apache.flink.table.types.logical.RowType;
import org.apache.paimon.catalog.Catalog;
import org.apache.paimon.catalog.Identifier;
import org.apache.paimon.flink.DataCatalogTable;
import org.apache.paimon.flink.FlinkTableFactory;
import org.apache.paimon.flink.LogicalTypeConversion;
import org.apache.paimon.flink.SystemCatalogTable;
import org.apache.paimon.options.Options;
import org.apache.paimon.schema.Schema;
import org.apache.paimon.schema.SchemaChange;
import org.apache.paimon.schema.SchemaManager;
import org.apache.paimon.table.FileStoreTable;
import org.apache.paimon.table.Table;
import org.apache.paimon.utils.StringUtils;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;

import net.wicp.tams.common.Conf;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.jdbc.MySqlColBean;
import net.wicp.tams.common.apiext.jdbc.MySqlTbBean;
import net.wicp.tams.common.constant.Middleware;
import net.wicp.tams.common.constant.MiddlewareOption;
import net.wicp.tams.common.flink.catalog.MysqlCatalogFactoryOptions;
import net.wicp.tams.common.flink.common.CatalogAssit;
import net.wicp.tams.common.flink.common.constant.CatalogName;
import net.wicp.tams.common.flink.common.constant.DelLevel;
import net.wicp.tams.common.flink.common.constant.FlinkTypeEnum;
import net.wicp.tams.common.flink.common.constant.db.ColsMetaDb;
import net.wicp.tams.common.jdbc.DruidAssit;
import net.wicp.tams.common.jdbc.MySqlAssitExt;

/** Catalog for paimon. */
public class FlinkCatalog extends AbstractCatalog {
	private final ClassLoader classLoader;

	private final Catalog catalog;
	private final boolean logStoreAutoRegister;

	private final Duration logStoreAutoRegisterTimeout;

	private final long tenantId;// andy.zhou 20230630 增加租户
	private final long operate;// andy.zhou 20230630 增加操作者（暂不启用）
	private final String mysqlDefaultDb;// andy.zhou 20230630 mysqlcatalog本身的默认库名
	private final String warehouse;// 原代码没有，但示例sql却需要？后面查原因'warehouse'='file:/tmp/paimon'

	public String getWarehouse() {
		return warehouse;
	}

	public FlinkCatalog(Catalog catalog, String name, String defaultDatabase, ClassLoader classLoader, Options options,
			long tenantId, long operate, String mysqlDefaultDb, String warehouse) {
		super(name, defaultDatabase);
		this.catalog = catalog;
		this.classLoader = classLoader;
		this.logStoreAutoRegister = options.get(LOG_SYSTEM_AUTO_REGISTER);
		this.logStoreAutoRegisterTimeout = options.get(REGISTER_TIMEOUT);
		this.tenantId = tenantId;
		this.operate = operate;
		this.mysqlDefaultDb = mysqlDefaultDb;
		this.warehouse = warehouse;
		// 初始化数据
		Properties props = new Properties();
		String propPre = String.format("common.jdbc.datasource.%s.", FlinkCatalogFactory.IDENTIFIER);
		props.put(propPre + MysqlCatalogFactoryOptions.host.key(), options.get(MysqlCatalogFactoryOptions.host.key()));
		props.put(propPre + MysqlCatalogFactoryOptions.username.key(),
				options.get(MysqlCatalogFactoryOptions.username.key()));
		props.put(propPre + MysqlCatalogFactoryOptions.password.key(),
				options.get(MysqlCatalogFactoryOptions.password.key()));
		props.put(propPre + MysqlCatalogFactoryOptions.port.key(), options.get(MysqlCatalogFactoryOptions.port.key()));
		props.put(propPre + MysqlCatalogFactoryOptions.defaultdb.key(),
				options.get(MysqlCatalogFactoryOptions.defaultdb.key()));
		if (StringUtil.isNotNull(options.get(MysqlCatalogFactoryOptions.url.key()))) {
			props.put(propPre + MysqlCatalogFactoryOptions.url.key(),
					options.get(MysqlCatalogFactoryOptions.url.key()));
		}
		if (StringUtil.isNotNull(options.get(MysqlCatalogFactoryOptions.urlparam.key()))) {
			props.put(propPre + MysqlCatalogFactoryOptions.urlparam.key(),
					options.get(MysqlCatalogFactoryOptions.urlparam.key()));
		}
		// 函数或其它地方需要,
		props.put(CatalogAssit.keyForTenantId, options.get(MysqlCatalogFactoryOptions.tenantId.key()));
		props.put(CatalogAssit.keyForOperate, options.get(MysqlCatalogFactoryOptions.operate.key()));
		Conf.overProp(props);
		// end 初始化数据

		try {
			this.catalog.createDatabase(defaultDatabase, true);
		} catch (Catalog.DatabaseAlreadyExistException ignore) {
		}
	}

	public Catalog catalog() {
		return catalog;
	}

	@Override
	public Optional<Factory> getFactory() {
		return Optional.of(new FlinkTableFactory());
	}

	@Override
	public List<String> listDatabases() throws CatalogException {
		return catalog.listDatabases();
	}

	@Override
	public boolean databaseExists(String databaseName) throws CatalogException {
		return catalog.databaseExists(databaseName);
	}

	@Override
	public CatalogDatabase getDatabase(String databaseName) throws CatalogException, DatabaseNotExistException {
		if (databaseExists(databaseName)) {
			return new CatalogDatabaseImpl(Collections.emptyMap(), null);
		}
		throw new DatabaseNotExistException(getName(), databaseName);
	}

	@Override
	public void createDatabase(String name, CatalogDatabase database, boolean ignoreIfExists)
			throws DatabaseAlreadyExistException, CatalogException {
		Connection connection = DruidAssit.getConnection(FlinkCatalogFactory.IDENTIFIER);
		try {
			connection.setAutoCommit(false);
			// 库名较为特殊，不能通过租户来判断有元
			int count = MySqlAssitExt.querySqlCount(connection,
					String.format("select count(1) from %s.%s where %s='%s' and %s='%s' and %s='%s'",
							this.mysqlDefaultDb, ColsMetaDb._tb, ColsMetaDb.tenantId.getOriColName(), this.tenantId,
							ColsMetaDb.name.getOriColName(), name, ColsMetaDb.catalogName.getOriColName(),
							CatalogName.tscatalog.name()));
			if (!ignoreIfExists && count > 0) {
				throw new DatabaseAlreadyExistException(this.getName(), name);
			}
			if (count == 0) {
				String comment = (database != null) ? (database.getComment() == null ? "" : database.getComment()) : "";
				CatalogAssit.createDb(connection, this.mysqlDefaultDb, name, CatalogName.find(super.getName()), comment,
						(database == null) ? "" : StringUtil.hasNull(this.warehouse), tenantId, this.operate);
			}
			// tablestore 目前不支持带描述的建库，所以需要重置为空
			// database = null;
			createDatabaseOri(name, database, ignoreIfExists);
			connection.commit();
			connection.setAutoCommit(true);
		} catch (Throwable e) {
			try {
				connection.rollback();
			} catch (Exception e2) {
				e2.printStackTrace();
			}
			if (e instanceof DatabaseAlreadyExistException) {
				throw (DatabaseAlreadyExistException) e;
			} else if (e instanceof CatalogException) {
				throw (CatalogException) e;
			} else {
				throw new CatalogException("未知异常:" + e.getMessage());
			}
		} finally {
			DruidAssit.close(connection);
		}

	}

	private void createDatabaseOri(String name, CatalogDatabase database, boolean ignoreIfExists)
			throws DatabaseAlreadyExistException {
		if (database != null) {
			if (database.getProperties().size() > 0) {
				throw new UnsupportedOperationException("Create database with properties is unsupported.");
			}

			if (database.getDescription().isPresent() && !database.getDescription().get().equals("")) {
				throw new UnsupportedOperationException("Create database with description is unsupported.");
			}
		}

		try {
			catalog.createDatabase(name, ignoreIfExists);
		} catch (Catalog.DatabaseAlreadyExistException e) {
			throw new DatabaseAlreadyExistException(getName(), e.database());
		}
	}

	@Override
	public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade)
			throws DatabaseNotEmptyException, DatabaseNotExistException, CatalogException {
		// andy.zhou 20230316 不支持删除库，因为有函数、视图、表等各种数据要删除，较较危险
		throw new UnsupportedOperationException();
//    			try {
//    				if (this.listTables(name).size() <= 0) {
//    					this.connection.setAutoCommit(false);
//    					// 如果filesystem相关库下已经不存在表信息，但mysql里还是存在关联的表数据，说明mysql和filesystem元数据信息已不一致，应以filesystem为准，将mysql相关信息清掉
//    					CatalogAssit.dropCascadeTable(name, this.tenantId, this.operate, this.connection, super.getName(),
//    							mysqlDefaultDb);
//    					CatalogAssit.dropDatabase(name, this.tenantId, this.operate, this.connection, super.getName(),
//    							mysqlDefaultDb);
//    					catalog.dropDatabase(name, ignoreIfNotExists, cascade);
//    					this.connection.commit();
//    				} else {
//    					throw new Catalog.DatabaseNotEmptyException(name);
//    				}
//    			} catch (Throwable e) {
//    				e.printStackTrace();
//    				try {
//    					this.connection.rollback();
//    				} catch (SQLException ex) {
//    					ex.printStackTrace();
//    				}
//    				throw new RuntimeException(e.getMessage());
//    			} finally {
//    				try {
//    					this.connection.setAutoCommit(true);
//    				} catch (SQLException e) {
//    					e.printStackTrace();
//    				}
//    			}
	}

	@Override
	public List<String> listTables(String databaseName) throws DatabaseNotExistException, CatalogException {
		try {
			return catalog.listTables(databaseName);
		} catch (Catalog.DatabaseNotExistException e) {
			throw new DatabaseNotExistException(getName(), e.database());
		}
	}

	@Override
	public CatalogTable getTable(ObjectPath tablePath) throws TableNotExistException, CatalogException {
		Table table;
		try {
			table = catalog.getTable(toIdentifier(tablePath));
		} catch (Catalog.TableNotExistException e) {
			throw new TableNotExistException(getName(), tablePath);
		}

		if (table instanceof FileStoreTable) {
			return toCatalogTable(table);
		} else {
			return new SystemCatalogTable(table);
		}
	}

	@Override
	public boolean tableExists(ObjectPath tablePath) throws CatalogException {
		return catalog.tableExists(toIdentifier(tablePath));
	}

	@Override
	public void dropTable(ObjectPath tablePath, boolean ignoreIfNotExists)
			throws TableNotExistException, CatalogException {
		Connection connection = DruidAssit.getConnection(FlinkCatalogFactory.IDENTIFIER);
		try {
			connection.setAutoCommit(false);
			if (!this.tableExists(tablePath)) {
				// 如果filesystem相关库下已经不存在表信息，但mysql里还是存在关联的表数据，说明mysql和filesystem元数据信息已不一致，应以filesystem为准，将mysql相关信息清掉
				CatalogAssit.dropTable(connection, this.mysqlDefaultDb, tablePath.getDatabaseName(),
						tablePath.getObjectName(), DelLevel.delLogicForce, this.tenantId, false);
				return;
			}
			CatalogAssit.dropTable(connection, this.mysqlDefaultDb, tablePath.getDatabaseName(),
					tablePath.getObjectName(), DelLevel.delLogicForce, this.tenantId, false);
			dropTableOri(tablePath, ignoreIfNotExists);
			connection.commit();
		} catch (Throwable e) {
			e.printStackTrace();
			try {
				connection.rollback();
			} catch (SQLException ex) {
				ex.printStackTrace();
			}
			throw new RuntimeException(e.getMessage());
		} finally {
			DruidAssit.close(connection);
		}
	}

	private void dropTableOri(ObjectPath tablePath, boolean ignoreIfNotExists) throws TableNotExistException {
		Identifier identifier = toIdentifier(tablePath);
		Table table = null;
		try {
			if (logStoreAutoRegister && catalog.tableExists(identifier)) {
				table = catalog.getTable(identifier);
			}
			catalog.dropTable(toIdentifier(tablePath), ignoreIfNotExists);
			if (logStoreAutoRegister && table != null) {
				unRegisterLogSystem(identifier, table.options(), classLoader);
			}
		} catch (Catalog.TableNotExistException e) {
			throw new TableNotExistException(getName(), tablePath);
		}
	}

	@Override
	public void createTable(ObjectPath tablePath, CatalogBaseTable table, boolean ignoreIfExists)
			throws TableAlreadyExistException, DatabaseNotExistException, CatalogException {
		// createTableOri(tablePath, table, ignoreIfExists);

		boolean dbexist = databaseExists(tablePath.getDatabaseName());
		if (!dbexist) {
			throw new DatabaseNotExistException(super.getName(), tablePath.getDatabaseName());
		}

		boolean tableExists = tableExists(tablePath);
		if (ignoreIfExists && tableExists) {// 如果存在表时忽略
			return;
		}

//		String filepath = new Path(new Path(warehouse, tablePath.getDatabaseName().concat(".db")),
//				tablePath.getObjectName()).toString();
		Connection connection = DruidAssit.getConnection(FlinkCatalogFactory.IDENTIFIER);
		try {
			connection.setAutoCommit(false);
			Map<String, String> queryTable = CatalogAssit.queryTable(connection, this.mysqlDefaultDb,
					tablePath.getDatabaseName(), tablePath.getObjectName(), this.tenantId);
			if (MapUtils.isNotEmpty(queryTable)) {// 数据库已存在此表需要删除，以保持同步
				CatalogAssit.dropTable(connection, this.mysqlDefaultDb, tablePath.getDatabaseName(),
						tablePath.getObjectName(), DelLevel.delLogicForce, this.tenantId, false);
			}
			List<MySqlColBean> colList = new ArrayList<>();
			Map<MiddlewareOption, String> allOptmap = new HashMap<>();
//			allOptmap.put(MiddlewareOption.path, filepath);
			Pair<String, String> proOptStr = Middleware.proOptStr(allOptmap,
					allOptmap.keySet().toArray(new MiddlewareOption[allOptmap.size()]));

			MySqlTbBean catalogTbInfo = MySqlTbBean.builder().db(tablePath.getDatabaseName())
					.tb(tablePath.getObjectName()).tbComment("tablestore表").middleware(Middleware.no)
					.partitionkeys(null).opt(proOptStr.getRight()).catalogName(super.getName())
					.withOptions(proOptStr.getLeft()).build();

			List<org.apache.flink.table.api.Schema.UnresolvedColumn> columns = table.getUnresolvedSchema().getColumns();
			for (org.apache.flink.table.api.Schema.UnresolvedColumn col : columns) {
				MySqlColBean tempbean = new MySqlColBean(col.getName(), "");
				// 不支持元数据列
				tempbean.setMetadata(false);
				tempbean.setMetadataName("");
				tempbean.setVirtual(false);
				Optional<org.apache.flink.table.api.Schema.UnresolvedPrimaryKey> primaryKey = table
						.getUnresolvedSchema().getPrimaryKey();
				List<String> primaryKeys = primaryKey.isPresent() ? primaryKey.get().getColumnNames()
						: new ArrayList<>();
				boolean isKey = primaryKeys.contains(col.getName());
				tempbean.setPri(isKey);
				tempbean.setDataType("");
				if (col instanceof org.apache.flink.table.api.Schema.UnresolvedPhysicalColumn) {
					org.apache.flink.table.api.Schema.UnresolvedPhysicalColumn temp = (org.apache.flink.table.api.Schema.UnresolvedPhysicalColumn) col;
					LogicalTypeRoot typeRoot = ((AtomicDataType) temp.getDataType()).getLogicalType().getTypeRoot();
					try {
						FlinkTypeEnum flinkTypeEnum = FlinkTypeEnum.valueOf(typeRoot.name());
						tempbean.setFlinkTypeEnum(flinkTypeEnum.name());
					} catch (Exception e) {
						throw new CatalogException("创建表[" + tablePath.getFullName() + "]时类型检查异常，原因：还不支持此类型["
								+ temp.getName() + "]" + e.getMessage(), e);
					}
				} else {
					// todo
					throw new RuntimeException("暂时只支持物理列，不支持元数据列、计算列等其他列");
				}
				tempbean.setDataType("");
				tempbean.setSqlExpression("");
				tempbean.setWatermarkExpression("");
				colList.add(tempbean);
			}
			CatalogAssit.createTable(connection, mysqlDefaultDb, colList, catalogTbInfo, tenantId, operate, false);
			createTableOri(tablePath, table, ignoreIfExists);
			connection.commit();
			// log.info("已创建表：{} 对应的逻辑表{}", physicalId, logicId);
		} catch (Throwable e) {
			e.printStackTrace();
			try {
				connection.rollback();
			} catch (SQLException ex) {
				ex.printStackTrace();
			}
			throw new RuntimeException(e.getMessage());
		} finally {
			DruidAssit.close(connection);
		}
	}

	private void createTableOri(ObjectPath tablePath, CatalogBaseTable table, boolean ignoreIfExists)
			throws TableAlreadyExistException, DatabaseNotExistException {
		if (!(table instanceof CatalogTable)) {
			throw new UnsupportedOperationException("Only support CatalogTable, but is: " + table.getClass());
		}
		CatalogTable catalogTable = (CatalogTable) table;
		Map<String, String> options = table.getOptions();
		String connector = options.get(CONNECTOR.key());
		options.remove(CONNECTOR.key());
		if (!StringUtils.isNullOrWhitespaceOnly(connector) && !FlinkCatalogFactory.IDENTIFIER.equals(connector)) {
			throw new CatalogException("Paimon Catalog only supports paimon tables,"
					+ " but you specify  'connector'= '" + connector + "' when using Paimon Catalog\n"
					+ " You can create TEMPORARY table instead if you want to create the table of other connector.");
		}

		Identifier identifier = toIdentifier(tablePath);
		if (logStoreAutoRegister) {
			// Although catalog.createTable will copy the default options, but we need this
			// info
			// here before create table, such as table-default.kafka.bootstrap.servers
			// defined in
			// catalog options. Temporarily, we copy the default options here.
			if (catalog instanceof org.apache.paimon.catalog.AbstractCatalog) {
				((org.apache.paimon.catalog.AbstractCatalog) catalog).copyTableDefaultOptions(options);
			}
			options.put(REGISTER_TIMEOUT.key(), logStoreAutoRegisterTimeout.toString());
			registerLogSystem(catalog, identifier, options, classLoader);
		}
		// remove table path
		String specific = options.remove(PATH.key());
		if (specific != null || logStoreAutoRegister) {
			catalogTable = catalogTable.copy(options);
		}

		boolean unRegisterLogSystem = false;
		try {
			// andy.zhou 20230908
			catalog.createTable(identifier, PaimonFlinkAssit.fromCatalogTable(catalogTable), ignoreIfExists);
		} catch (Catalog.TableAlreadyExistException e) {
			unRegisterLogSystem = true;
			throw new TableAlreadyExistException(getName(), tablePath);
		} catch (Catalog.DatabaseNotExistException e) {
			unRegisterLogSystem = true;
			throw new DatabaseNotExistException(getName(), e.database());
		} finally {
			if (logStoreAutoRegister && unRegisterLogSystem) {
				unRegisterLogSystem(identifier, options, classLoader);
			}
		}
	}

	private List<SchemaChange> toSchemaChange(TableChange change) {
		List<SchemaChange> schemaChanges = new ArrayList<>();
		if (change instanceof AddColumn) {
			AddColumn add = (AddColumn) change;
			String comment = add.getColumn().getComment().orElse(null);
			SchemaChange.Move move = getMove(add.getPosition(), add.getColumn().getName());
			schemaChanges.add(SchemaChange.addColumn(add.getColumn().getName(),
					LogicalTypeConversion.toDataType(add.getColumn().getDataType().getLogicalType()), comment, move));
			return schemaChanges;
		} else if (change instanceof AddWatermark) {
			AddWatermark add = (AddWatermark) change;
			setWatermarkOptions(add.getWatermark(), schemaChanges);
			return schemaChanges;
		} else if (change instanceof DropColumn) {
			DropColumn drop = (DropColumn) change;
			schemaChanges.add(SchemaChange.dropColumn(drop.getColumnName()));
			return schemaChanges;
		} else if (change instanceof DropWatermark) {
			String watermarkPrefix = compoundKey(SCHEMA, WATERMARK, 0);
			schemaChanges.add(SchemaChange.removeOption(compoundKey(watermarkPrefix, WATERMARK_ROWTIME)));
			schemaChanges.add(SchemaChange.removeOption(compoundKey(watermarkPrefix, WATERMARK_STRATEGY_EXPR)));
			schemaChanges.add(SchemaChange.removeOption(compoundKey(watermarkPrefix, WATERMARK_STRATEGY_DATA_TYPE)));
			return schemaChanges;
		} else if (change instanceof ModifyColumnName) {
			ModifyColumnName modify = (ModifyColumnName) change;
			schemaChanges.add(SchemaChange.renameColumn(modify.getOldColumnName(), modify.getNewColumnName()));
			return schemaChanges;
		} else if (change instanceof ModifyPhysicalColumnType) {
			ModifyPhysicalColumnType modify = (ModifyPhysicalColumnType) change;
			LogicalType newColumnType = modify.getNewType().getLogicalType();
			LogicalType oldColumnType = modify.getOldColumn().getDataType().getLogicalType();
			if (newColumnType.isNullable() != oldColumnType.isNullable()) {
				schemaChanges.add(SchemaChange.updateColumnNullability(modify.getNewColumn().getName(),
						newColumnType.isNullable()));
			}
			schemaChanges.add(SchemaChange.updateColumnType(modify.getOldColumn().getName(),
					LogicalTypeConversion.toDataType(newColumnType)));
			return schemaChanges;
		} else if (change instanceof ModifyColumnPosition) {
			ModifyColumnPosition modify = (ModifyColumnPosition) change;
			SchemaChange.Move move = getMove(modify.getNewPosition(), modify.getNewColumn().getName());
			schemaChanges.add(SchemaChange.updateColumnPosition(move));
			return schemaChanges;
		} else if (change instanceof TableChange.ModifyColumnComment) {
			ModifyColumnComment modify = (ModifyColumnComment) change;
			schemaChanges
					.add(SchemaChange.updateColumnComment(modify.getNewColumn().getName(), modify.getNewComment()));
			return schemaChanges;
		} else if (change instanceof ModifyWatermark) {
			ModifyWatermark modify = (ModifyWatermark) change;
			setWatermarkOptions(modify.getNewWatermark(), schemaChanges);
			return schemaChanges;
		} else if (change instanceof SetOption) {
			SetOption setOption = (SetOption) change;
			String key = setOption.getKey();
			String value = setOption.getValue();

			SchemaManager.checkAlterTablePath(key);

			schemaChanges.add(SchemaChange.setOption(key, value));
			return schemaChanges;
		} else if (change instanceof ResetOption) {
			ResetOption resetOption = (ResetOption) change;
			schemaChanges.add(SchemaChange.removeOption(resetOption.getKey()));
			return schemaChanges;
		} else {
			throw new UnsupportedOperationException("Change is not supported: " + change.getClass());
		}
	}

	@Override
	public void alterTable(ObjectPath tablePath, CatalogBaseTable newTable, boolean ignoreIfNotExists)
			throws TableNotExistException, CatalogException {
		Connection connection = DruidAssit.getConnection(FlinkCatalogFactory.IDENTIFIER);
		try {
			connection.setAutoCommit(false);
			if (!this.tableExists(tablePath)) {
				// 如果filesystem相关库下已经不存在表信息，但mysql里还是存在关联的表数据，说明mysql和filesystem元数据信息已不一致，应以filesystem为准，将mysql相关信息清掉
				CatalogAssit.dropTable(connection, this.mysqlDefaultDb, tablePath.getDatabaseName(),
						tablePath.getObjectName(), DelLevel.delLogicForce, this.tenantId, false);
				connection.commit();
				return;
			}
			CatalogAssit.dropTable(connection, this.mysqlDefaultDb, tablePath.getDatabaseName(),
					tablePath.getObjectName(), DelLevel.delLogicForce, this.tenantId, false);
			Pair<MySqlTbBean, List<MySqlColBean>> tableInfo = getCreateTableInfo(newTable, tablePath);
			CatalogAssit.createTable(connection, this.mysqlDefaultDb, tableInfo.getRight(), tableInfo.getLeft(),
					this.tenantId, this.operate, false);
			alterTableOri(tablePath, newTable, ignoreIfNotExists);
			connection.commit();
		} catch (Throwable e) {
			e.printStackTrace();
			try {
				connection.rollback();
			} catch (SQLException ex) {
				ex.printStackTrace();
			}
			throw new RuntimeException(e.getMessage());
		} finally {
			DruidAssit.close(connection);
		}
	}

	private Pair<MySqlTbBean, List<MySqlColBean>> getCreateTableInfo(CatalogBaseTable newTable, ObjectPath tablePath) {
		List<MySqlColBean> mysqlColList = CatalogAssit.packageColBeans(newTable, tablePath.getObjectName());
		// 分区
		String partitionkeys = "";
		CatalogTable tableTrue = (CatalogTable) newTable;
		if (CollectionUtils.isNotEmpty(tableTrue.getPartitionKeys())) {
			partitionkeys = CollectionUtil.listJoin(tableTrue.getPartitionKeys(), ",");
		}
		// tablestore不做校验，因为与普通表不通，那些枚举不合适
		String options = JSONObject.toJSONString(newTable.getOptions(), SerializerFeature.UseSingleQuotes);
		MySqlTbBean catalogTbInfo = MySqlTbBean.builder().db(tablePath.getDatabaseName()).tb(tablePath.getObjectName())
				.tbComment("通过sql创建").middleware(Middleware.no).partitionkeys(partitionkeys).opt(options)
				.catalogName(super.getName()).withOptions("{'total':0,'rows':[]}").build();
		return Pair.of(catalogTbInfo, mysqlColList);
	}

	private void alterTableOri(ObjectPath tablePath, CatalogBaseTable newTable, boolean ignoreIfNotExists)
			throws TableNotExistException {
		if (ignoreIfNotExists && !tableExists(tablePath)) {
			return;
		}

		CatalogTable table = getTable(tablePath);

		// Currently, Flink SQL only support altering table properties.
		validateAlterTable(table, (CatalogTable) newTable);

		List<SchemaChange> changes = new ArrayList<>();
		Map<String, String> oldProperties = table.getOptions();
		for (Map.Entry<String, String> entry : newTable.getOptions().entrySet()) {
			String key = entry.getKey();
			String value = entry.getValue();

			if (Objects.equals(value, oldProperties.get(key))) {
				continue;
			}

			if (PATH.key().equalsIgnoreCase(key)) {
				throw new IllegalArgumentException("Illegal table path in table options: " + value);
			}

			changes.add(SchemaChange.setOption(key, value));
		}

		oldProperties.keySet().forEach(k -> {
			if (!newTable.getOptions().containsKey(k)) {
				changes.add(SchemaChange.removeOption(k));
			}
		});

		try {
			catalog.alterTable(toIdentifier(tablePath), changes, ignoreIfNotExists);
		} catch (Catalog.TableNotExistException e) {
			throw new TableNotExistException(getName(), tablePath);
		} catch (Catalog.ColumnAlreadyExistException | Catalog.ColumnNotExistException e) {
			throw new CatalogException(e);
		}
	}

	@Override
	public void alterTable(ObjectPath tablePath, CatalogBaseTable newTable, List<TableChange> tableChanges,
			boolean ignoreIfNotExists) throws TableNotExistException, CatalogException {
		Connection connection = DruidAssit.getConnection(FlinkCatalogFactory.IDENTIFIER);
		try {
			connection.setAutoCommit(false);
			if (!this.tableExists(tablePath)) {
				// 如果filesystem相关库下已经不存在表信息，但mysql里还是存在关联的表数据，说明mysql和filesystem元数据信息已不一致，应以filesystem为准，将mysql相关信息清掉
				CatalogAssit.dropTable(connection, this.mysqlDefaultDb, tablePath.getDatabaseName(),
						tablePath.getObjectName(), DelLevel.delLogicForce, this.tenantId, false);
				connection.commit();
				return;
			}
			CatalogAssit.dropTable(connection, this.mysqlDefaultDb, tablePath.getDatabaseName(),
					tablePath.getObjectName(), DelLevel.delLogicForce, this.tenantId, false);
			Pair<MySqlTbBean, List<MySqlColBean>> tableInfo = getCreateTableInfo(newTable, tablePath);
			CatalogAssit.createTable(connection, this.mysqlDefaultDb, tableInfo.getRight(), tableInfo.getLeft(),
					this.tenantId, this.operate, false);
			alterTableOri(tablePath, newTable, tableChanges, ignoreIfNotExists);
			connection.commit();
		} catch (Throwable e) {
			e.printStackTrace();
			try {
				connection.rollback();
			} catch (SQLException ex) {
				ex.printStackTrace();
			}
			throw new RuntimeException(e.getMessage());
		} finally {
			DruidAssit.close(connection);
		}

	}

	private void alterTableOri(ObjectPath tablePath, CatalogBaseTable newTable, List<TableChange> tableChanges,
			boolean ignoreIfNotExists) throws TableNotExistException {
		if (ignoreIfNotExists && !tableExists(tablePath)) {
			return;
		}

		CatalogTable table = getTable(tablePath);

		validateAlterTable(table, (CatalogTable) newTable);

		List<SchemaChange> changes = new ArrayList<>();
		if (null != tableChanges) {
			List<SchemaChange> schemaChanges = tableChanges.stream()
					.flatMap(tableChange -> toSchemaChange(tableChange).stream()).collect(Collectors.toList());
			changes.addAll(schemaChanges);
		}

		try {
			catalog.alterTable(toIdentifier(tablePath), changes, ignoreIfNotExists);
		} catch (Catalog.TableNotExistException | Catalog.ColumnAlreadyExistException
				| Catalog.ColumnNotExistException e) {
			throw new TableNotExistException(getName(), tablePath);
		}
	}

	private SchemaChange.Move getMove(ColumnPosition columnPosition, String fieldName) {
		SchemaChange.Move move = null;
		if (columnPosition instanceof First) {
			move = SchemaChange.Move.first(fieldName);
		} else if (columnPosition instanceof After) {
			move = SchemaChange.Move.after(fieldName, ((After) columnPosition).column());
		}
		return move;
	}

	private String getWatermarkKeyPrefix() {
		return compoundKey(SCHEMA, WATERMARK, 0);
	}

	private String getWatermarkRowTimeKey(String watermarkPrefix) {
		return compoundKey(watermarkPrefix, WATERMARK_ROWTIME);
	}

	private String getWatermarkExprKey(String watermarkPrefix) {
		return compoundKey(watermarkPrefix, WATERMARK_STRATEGY_EXPR);
	}

	private String getWatermarkExprDataTypeKey(String watermarkPrefix) {
		return compoundKey(watermarkPrefix, WATERMARK_STRATEGY_DATA_TYPE);
	}

	private void setWatermarkOptions(org.apache.flink.table.catalog.WatermarkSpec wms,
			List<SchemaChange> schemaChanges) {
		String watermarkPrefix = getWatermarkKeyPrefix();
		schemaChanges.add(SchemaChange.setOption(getWatermarkRowTimeKey(watermarkPrefix), wms.getRowtimeAttribute()));
		schemaChanges.add(SchemaChange.setOption(getWatermarkExprKey(watermarkPrefix),
				wms.getWatermarkExpression().asSerializableString()));
		schemaChanges.add(SchemaChange.setOption(getWatermarkExprDataTypeKey(watermarkPrefix),
				wms.getWatermarkExpression().getOutputDataType().getLogicalType().asSerializableString()));
	}

	// andy.zhou 20230908
	private static void validateAlterTable(CatalogTable ct1, CatalogTable ct2) {
		if (ct1 instanceof SystemCatalogTable) {
			throw new UnsupportedOperationException("Can't alter system table.");
		}
		org.apache.flink.table.api.Schema ts1 = ct1.getUnresolvedSchema();
		org.apache.flink.table.api.Schema ts2 = ct2.getUnresolvedSchema();

		if (ts1.getPrimaryKey().isPresent() != ts2.getPrimaryKey().isPresent()) {
			throw new UnsupportedOperationException("Altering primary key is not supported yet.");
		}
		if (ts1.getPrimaryKey().isPresent() && ts2.getPrimaryKey().isPresent()) {
			List<String> key1s = ts1.getPrimaryKey().get().getColumnNames();
			List<String> key2s = ts2.getPrimaryKey().get().getColumnNames();
			if (key1s.size() != key2s.size()) {
				throw new UnsupportedOperationException("Altering primary key is not supported yet.");
			}
			for (int i = 0; i < key1s.size(); i++) {
				if (!key1s.get(i).equals(key1s.get(i))) {
					throw new UnsupportedOperationException("Altering primary key is not supported yet.");
				}
				AbstractDataType<?> dataType1 = null;
				AbstractDataType<?> dataType2 = null;
				for (int j = 0; j < ts1.getColumns().size(); j++) {
					UnresolvedPhysicalColumn unresolvedColumn1 = (UnresolvedPhysicalColumn) ts1.getColumns().get(j);
					if (unresolvedColumn1.getName().equals(key1s.get(i))) {
						dataType1 = unresolvedColumn1.getDataType();
						break;
					}
				}

				for (int j = 0; j < ts2.getColumns().size(); j++) {
					UnresolvedPhysicalColumn unresolvedColumn2 = (UnresolvedPhysicalColumn) ts2.getColumns().get(j);
					if (unresolvedColumn2.getName().equals(key1s.get(i))) {
						dataType2 = unresolvedColumn2.getDataType();
						break;
					}
				}
				if (!Objects.equals(dataType1, dataType2)) {
					throw new UnsupportedOperationException("Altering primary key is not supported yet.");
				}
			}
		}

		if (CollectionUtils.isNotEmpty(ct1.getPartitionKeys()) != CollectionUtils.isNotEmpty(ct2.getPartitionKeys())) {
			throw new UnsupportedOperationException("Altering partition keys is not supported yet.");
		}

		if (CollectionUtils.isNotEmpty(ct1.getPartitionKeys()) && CollectionUtils.isNotEmpty(ct2.getPartitionKeys())) {
			if (ct1.getPartitionKeys().size() != ct2.getPartitionKeys().size()) {
				throw new UnsupportedOperationException("Altering partition keys is not supported yet.");
			}
			for (int i = 0; i < ct1.getPartitionKeys().size(); i++) {
				if (!ct1.getPartitionKeys().get(i).equals(ct2.getPartitionKeys().get(i))) {
					throw new UnsupportedOperationException("Altering partition keys is not supported yet.");
				}
			}
		}
	}

	@Override
	public final void open() throws CatalogException {
	}

	@Override
	public final void close() throws CatalogException {
		try {
			catalog.close();
		} catch (Exception e) {
			throw new CatalogException("Failed to close catalog " + catalog.toString(), e);
		}
	}

	private CatalogTableImpl toCatalogTable(Table table) {
		Map<String, String> newOptions = new HashMap<>(table.options());

		TableSchema.Builder builder = TableSchema.builder();

		// add columns
		List<RowType.RowField> physicalRowFields = toLogicalType(table.rowType()).getFields();
		List<String> physicalColumns = table.rowType().getFieldNames();
		int columnCount = physicalRowFields.size() + nonPhysicalColumnsCount(newOptions, physicalColumns);
		int physicalColumnIndex = 0;
		for (int i = 0; i < columnCount; i++) {
			String optionalName = newOptions.get(compoundKey(SCHEMA, i, NAME));
			// to check old style physical column option
			if (optionalName == null || physicalColumns.contains(optionalName)) {
				// build physical column from table row field
				RowType.RowField field = physicalRowFields.get(physicalColumnIndex++);
				builder.field(field.getName(), fromLogicalToDataType(field.getType()));
			} else {
				// build non-physical column from options
				builder.add(deserializeNonPhysicalColumn(newOptions, i));
			}
		}

		// extract watermark information
		if (newOptions.keySet().stream().anyMatch(key -> key.startsWith(compoundKey(SCHEMA, WATERMARK)))) {
			builder.watermark(deserializeWatermarkSpec(newOptions));
		}

		// add primary keys
		if (table.primaryKeys().size() > 0) {
			builder.primaryKey(table.primaryKeys().toArray(new String[0]));
		}

		TableSchema schema = builder.build();

		// remove schema from options
		DescriptorProperties removeProperties = new DescriptorProperties(false);
		removeProperties.putTableSchema(SCHEMA, schema);
		removeProperties.asMap().keySet().forEach(newOptions::remove);

		return new DataCatalogTable(table, schema, table.partitionKeys(), newOptions, table.comment().orElse(""));
	}

	public static Schema fromCatalogTable(CatalogTable catalogTable) {
		TableSchema schema = catalogTable.getSchema();
		RowType rowType = (RowType) schema.toPhysicalRowDataType().getLogicalType();

		Map<String, String> options = new HashMap<>(catalogTable.getOptions());
		// Serialize virtual columns and watermark to the options
		// This is what Flink SQL needs, the storage itself does not need them
		options.putAll(columnOptions(schema));

		Schema.Builder schemaBuilder = Schema.newBuilder().comment(catalogTable.getComment()).options(options)
				.primaryKey(schema.getPrimaryKey().map(pk -> pk.getColumns()).orElse(Collections.emptyList()))
				.partitionKeys(catalogTable.getPartitionKeys());
		Map<String, String> columnComments = getColumnComments(catalogTable);
		rowType.getFields().forEach(field -> schemaBuilder.column(field.getName(), toDataType(field.getType()),
				columnComments.get(field.getName())));

		return schemaBuilder.build();
	}

	private static Map<String, String> getColumnComments(CatalogTable catalogTable) {
		return catalogTable.getUnresolvedSchema().getColumns().stream().filter(c -> c.getComment().isPresent())
				.collect(Collectors.toMap(org.apache.flink.table.api.Schema.UnresolvedColumn::getName,
						c -> c.getComment().get()));
	}

	/** Only reserve necessary options. */
	private static Map<String, String> columnOptions(TableSchema schema) {
		Map<String, String> columnOptions = new HashMap<>();
		// field name -> index
		final Map<String, Integer> indexMap = new HashMap<>();
		List<TableColumn> tableColumns = schema.getTableColumns();
		for (int i = 0; i < tableColumns.size(); i++) {
			indexMap.put(tableColumns.get(i).getName(), i);
		}

		// non-physical columns
		List<TableColumn> nonPhysicalColumns = tableColumns.stream().filter(c -> !c.isPhysical())
				.collect(Collectors.toList());
		if (!nonPhysicalColumns.isEmpty()) {
			columnOptions.putAll(serializeNonPhysicalColumns(indexMap, nonPhysicalColumns));
		}

		// watermark
		List<WatermarkSpec> watermarkSpecs = schema.getWatermarkSpecs();
		if (!watermarkSpecs.isEmpty()) {
			checkArgument(watermarkSpecs.size() == 1);
			columnOptions.putAll(serializeWatermarkSpec(watermarkSpecs.get(0)));
		}

		return columnOptions;
	}

	public static Identifier toIdentifier(ObjectPath path) {
		return new Identifier(path.getDatabaseName(), path.getObjectName());
	}

	// --------------------- unsupported methods ----------------------------

	@Override
	public final void alterDatabase(String name, CatalogDatabase newDatabase, boolean ignoreIfNotExists)
			throws CatalogException {
		throw new UnsupportedOperationException();
	}

	@Override
	public final void renameTable(ObjectPath tablePath, String newTableName, boolean ignoreIfNotExists)
			throws CatalogException, TableNotExistException, TableAlreadyExistException {
		ObjectPath toTable = new ObjectPath(tablePath.getDatabaseName(), newTableName);
		try {
			catalog.renameTable(toIdentifier(tablePath), toIdentifier(toTable), ignoreIfNotExists);
		} catch (Catalog.TableNotExistException e) {
			throw new TableNotExistException(getName(), tablePath);
		} catch (Catalog.TableAlreadyExistException e) {
			throw new TableAlreadyExistException(getName(), toTable);
		}
	}

	@Override
	public final List<String> listViews(String databaseName) throws CatalogException {
		return Collections.emptyList();
	}

	@Override
	public final List<CatalogPartitionSpec> listPartitions(ObjectPath tablePath) throws CatalogException {
		return Collections.emptyList();
	}

	@Override
	public final List<CatalogPartitionSpec> listPartitions(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws CatalogException {
		return Collections.emptyList();
	}

	@Override
	public final List<CatalogPartitionSpec> listPartitionsByFilter(ObjectPath tablePath, List<Expression> filters)
			throws CatalogException {
		return Collections.emptyList();
	}

	@Override
	public final CatalogPartition getPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws PartitionNotExistException, CatalogException {
		throw new PartitionNotExistException(getName(), tablePath, partitionSpec);
	}

	@Override
	public final boolean partitionExists(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws CatalogException {
		return false;
	}

	@Override
	public final void createPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogPartition partition, boolean ignoreIfExists) throws CatalogException {
		throw new UnsupportedOperationException();
	}

	@Override
	public final void dropPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec, boolean ignoreIfNotExists)
			throws CatalogException {
		throw new UnsupportedOperationException();
	}

	@Override
	public final void alterPartition(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogPartition newPartition, boolean ignoreIfNotExists) throws CatalogException {
		throw new UnsupportedOperationException();
	}

	@Override
	public final List<String> listFunctions(String dbName) throws CatalogException {
		return Collections.emptyList();
	}

	@Override
	public final CatalogFunction getFunction(ObjectPath functionPath)
			throws FunctionNotExistException, CatalogException {
		throw new FunctionNotExistException(getName(), functionPath);
	}

	@Override
	public final boolean functionExists(ObjectPath functionPath) throws CatalogException {
		return false;
	}

	@Override
	public final void createFunction(ObjectPath functionPath, CatalogFunction function, boolean ignoreIfExists)
			throws CatalogException {
		throw new UnsupportedOperationException(
				"Create function is not supported," + " maybe you can use 'CREATE TEMPORARY FUNCTION' instead.");
	}

	@Override
	public final void alterFunction(ObjectPath functionPath, CatalogFunction newFunction, boolean ignoreIfNotExists)
			throws CatalogException {
		throw new UnsupportedOperationException();
	}

	@Override
	public final void dropFunction(ObjectPath functionPath, boolean ignoreIfNotExists) throws CatalogException {
		throw new UnsupportedOperationException();
	}

	@Override
	public final CatalogTableStatistics getTableStatistics(ObjectPath tablePath) throws CatalogException {
		return CatalogTableStatistics.UNKNOWN;
	}

	@Override
	public final CatalogColumnStatistics getTableColumnStatistics(ObjectPath tablePath) throws CatalogException {
		return CatalogColumnStatistics.UNKNOWN;
	}

	@Override
	public final CatalogTableStatistics getPartitionStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec)
			throws CatalogException {
		return CatalogTableStatistics.UNKNOWN;
	}

	@Override
	public final CatalogColumnStatistics getPartitionColumnStatistics(ObjectPath tablePath,
			CatalogPartitionSpec partitionSpec) throws CatalogException {
		return CatalogColumnStatistics.UNKNOWN;
	}

	@Override
	public final void alterTableStatistics(ObjectPath tablePath, CatalogTableStatistics tableStatistics,
			boolean ignoreIfNotExists) throws CatalogException {
		throw new UnsupportedOperationException();
	}

	@Override
	public final void alterTableColumnStatistics(ObjectPath tablePath, CatalogColumnStatistics columnStatistics,
			boolean ignoreIfNotExists) throws CatalogException {
		throw new UnsupportedOperationException();
	}

	@Override
	public final void alterPartitionStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogTableStatistics partitionStatistics, boolean ignoreIfNotExists) throws CatalogException {
		throw new UnsupportedOperationException();
	}

	@Override
	public final void alterPartitionColumnStatistics(ObjectPath tablePath, CatalogPartitionSpec partitionSpec,
			CatalogColumnStatistics columnStatistics, boolean ignoreIfNotExists) throws CatalogException {
		throw new UnsupportedOperationException();
	}
}
