/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
package net.wicp.tams.common.doris.constant;

import com.fasterxml.jackson.databind.ObjectMapper;
import net.wicp.tams.common.doris.bean.RespContent;
import net.wicp.tams.common.doris.exception.StreamLoadException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * DorisStreamLoad
 **/
public class DorisStreamLoadBe1 implements Serializable{

    private static final Logger LOG = LoggerFactory.getLogger(DorisStreamLoad.class);

    private final static List<String> DORIS_SUCCESS_STATUS = new ArrayList<>(Arrays.asList("Success", "Publish Timeout"));
    private static final String loadUrlPattern = "http://%s/api/%s/%s/_stream_load?";
    private final String user;
    private final String passwd;
    private String loadUrlStr;
    private String hostPort;
    private final String db;
    private final String tbl;
    private final String authEncoding;
    private final String jobName;

//    public static void main(String[] args) {
//        String datas = "[{\"birthday\":\"2020-12-31\",\"matched_status\":\"0\",\"matched_status1\":\"0\",\"isdeleted\":\"0\",\"feed\":\"user_info\",\"update_time\":\"2022-01-04 00:00:00\",\"money\":\"123.0\",\"_last_opttype\":\"insert\",\"name\":\"00t424401\",\"field-abc\":\"dd\",\"systemdbname\":\"demo\",\"id\":\"210\",\"add1\":\"1\",\"age\":\"5556\",\"lastupdateddt\":\"2022-01-08 13:18:10\"}]";
//        DorisStreamLoadBe1 dorisStreamLoadBe = new DorisStreamLoadBe1("172.31.43.250:8040","test","user_info","root","YjAzMzIyMTg5MWUw","testjobname");
//        dorisStreamLoadBe.loadBatch(datas);
//    }
    public DorisStreamLoadBe1(String hostPort, String db, String tbl, String user, String passwd, String jobName) {
        this.hostPort = hostPort;
        this.db = db;
        this.tbl = tbl;
        this.user = user;
        this.passwd = passwd;
        this.jobName = jobName;
        this.loadUrlStr = String.format(loadUrlPattern, hostPort, db, tbl);
        this.authEncoding = Base64.getEncoder().encodeToString(String.format("%s:%s", user, passwd).getBytes(StandardCharsets.UTF_8));
    }

    public String getLoadUrlStr() {
        return loadUrlStr;
    }

    public String getHostPort() {
        return hostPort;
    }

    public String getJobName() {
        return jobName;
    }

    public void setHostPort(String hostPort) {
        this.hostPort = hostPort;
        this.loadUrlStr = String.format(loadUrlPattern, hostPort, this.db, this.tbl);
    }


    private HttpURLConnection getConnection(String urlStr, String label) throws IOException {
        URL url = new URL(urlStr);
        HttpURLConnection conn = (HttpURLConnection) url.openConnection();
        conn.setInstanceFollowRedirects(false);
        conn.setRequestMethod("PUT");
        String authEncoding = Base64.getEncoder().encodeToString(String.format("%s:%s", user, passwd).getBytes(StandardCharsets.UTF_8));
        conn.setRequestProperty("Authorization", "Basic " + authEncoding);
        conn.addRequestProperty("Expect", "100-continue");
        conn.addRequestProperty("Content-Type", "text/plain; charset=UTF-8");
        conn.addRequestProperty("label", label);
        conn.addRequestProperty("column_separator", ",");
        conn.addRequestProperty("format", "json");
        conn.addRequestProperty("strip_outer_array", "true");
        conn.addRequestProperty("merge_type", "APPEND");
        conn.setDoOutput(true);
        conn.setDoInput(true);
        return conn;
    }

    public static class LoadResponse {
        public int status;
        public String respMsg;
        public String respContent;

        public LoadResponse(int status, String respMsg, String respContent) {
            this.status = status;
            this.respMsg = respMsg;
            this.respContent = respContent;
        }
        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("status: ").append(status);
            sb.append(", resp msg: ").append(respMsg);
            sb.append(", resp content: ").append(respContent);
            return sb.toString();
        }
    }

    public void load(String value) throws StreamLoadException {
        LoadResponse loadResponse = loadBatch(value);
//        LOG.info("Streamload Response:{}",loadResponse);
        if(loadResponse.status != 200){
            System.out.println(getJobName()+" streamload response1 hostport:"+this.hostPort+" "+db+"."+tbl+" "+loadResponse);
            throw new StreamLoadException("stream load error: " + loadResponse.respContent);
        }else{
            ObjectMapper obj = new ObjectMapper();
            try {
                RespContent respContent = obj.readValue(loadResponse.respContent, RespContent.class);
                if(!DORIS_SUCCESS_STATUS.contains(respContent.getStatus())){
                    System.out.println(getJobName()+" streamload response2 hostport:"+this.hostPort+" "+db+"."+tbl+" "+loadResponse);
                    throw new StreamLoadException("stream load error: " + respContent.getMessage());
                }
            } catch (IOException e) {
                System.out.println(getJobName()+" streamload response3 hostport:"+this.hostPort+" "+db+"."+tbl+" "+loadResponse);
                throw new StreamLoadException(e);
            }
        }
    }

    private LoadResponse loadBatch(String value) {
        Calendar calendar = Calendar.getInstance();
        String label = String.format("audit_%s%02d%02d_%02d%02d%02d_%s",
                calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH) + 1, calendar.get(Calendar.DAY_OF_MONTH),
                calendar.get(Calendar.HOUR_OF_DAY), calendar.get(Calendar.MINUTE), calendar.get(Calendar.SECOND),
                UUID.randomUUID().toString().replaceAll("-", ""));

        HttpURLConnection feConn = null;
        HttpURLConnection beConn = null;
        try {
            // build request and send to new be location
            beConn = getConnection(loadUrlStr, label);
            // send data to be
            BufferedOutputStream bos = new BufferedOutputStream(beConn.getOutputStream());
            bos.write(value.getBytes());
            bos.close();

            // get respond
            int status = beConn.getResponseCode();
            String respMsg = beConn.getResponseMessage();
            InputStream stream = (InputStream) beConn.getContent();
            BufferedReader br = new BufferedReader(new InputStreamReader(stream));
            StringBuilder response = new StringBuilder();
            String line;
            while ((line = br.readLine()) != null) {
                response.append(line);
            }
//            log.info("AuditLoader plugin load with label: {}, response code: {}, msg: {}, content: {}",label, status, respMsg, response.toString());
            return new LoadResponse(status, respMsg, response.toString());

        } catch (Exception e) {
            e.printStackTrace();
            String err = "failed to load audit via AuditLoader plugin with label: " + label;
            LOG.warn(err, e);
            return new LoadResponse(-1, e.getMessage(), err);
        } finally {
            if (feConn != null) {
                feConn.disconnect();
            }
            if (beConn != null) {
                beConn.disconnect();
            }
        }
    }
}
