/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
package net.wicp.tams.common.doris.constant;

import java.io.IOException;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.UUID;

import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultRedirectStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.springframework.util.StopWatch;

import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.databind.ObjectMapper;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.doris.bean.DorisConfig;
import net.wicp.tams.common.doris.bean.RespContent;
import net.wicp.tams.common.doris.exception.StreamLoadException;

/**
 * DorisStreamLoad
 **/
@Slf4j
public class DorisStreamLoad implements Serializable{

    /**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	private final List<String> DORIS_SUCCESS_STATUS = new ArrayList<>(Arrays.asList("Success", "Publish Timeout"));
    private final String loadUrlPattern = "http://%s/api/%s/%s/_stream_load?";
    private final String charset = "UTF-8";
    private final DorisConfig dorisConfig;
    private final String authEncoding;

    public DorisStreamLoad(DorisConfig dorisConfig) {
        this.dorisConfig = dorisConfig;
//        this.loadUrlStr = String.format(loadUrlPattern, hostPort, db, tb);
        this.authEncoding = Base64.getEncoder().encodeToString(String.format("%s:%s", dorisConfig.getUsername(), dorisConfig.getPassword()).getBytes(StandardCharsets.UTF_8));
    }


    private String getLoadUrlStr() {
        String loadUrl = String.format(loadUrlPattern, dorisConfig.getHost()+":"+dorisConfig.getHttpPort(), dorisConfig.getDb(), dorisConfig.getTb());
        return loadUrl;
    }

    private String getLoadUrlStr(String db , String tb) {
        this.dorisConfig.setDb(db);
        this.dorisConfig.setTb(tb);
        String loadUrl = String.format(this.loadUrlPattern, dorisConfig.getHost()+":"+dorisConfig.getHttpPort(), db, tb);
        return loadUrl;
    }

    /**
     * 插入或者根据key修改数据
     * @param jsonData
     * @throws Exception
     */
    public void loadJsonArrayAppend(String jsonData) throws Exception {
        loadJsonArray(jsonData,getHttpPutJsonArrayAppend());
    }

    /**
     * 插入或者根据key修改数据
     * @param jsonData
     * @throws Exception
     */
    public void loadJsonArrayAppend(String jsonData , String db ,String tb) throws Exception {
        loadJsonArray(jsonData,getHttpPutJsonArrayAppend(db,tb));
    }

    /**
     * 批量删除数据
     * @param jsonData
     * @throws Exception
     */
    public void loadJsonArrayDelete(String jsonData) throws Exception {
        loadJsonArray(jsonData,getHttpPutJsonArrayDelete());
    }

    /**
     * 批量删除数据
     * @param jsonData
     * @throws Exception
     */
    public void loadJsonArrayDelete(String jsonData , String db ,String tb) throws Exception {
        loadJsonArray(jsonData,getHttpPutJsonArrayDelete(db,tb));
    }

    /**
     * JSON import
     * @param jsonData
     * @throws Exception
     */
    private void loadJsonArray(String jsonData , HttpPut put) throws Exception {
            CloseableHttpClient client = httpClientBuilder.build();
            StringEntity entity = new StringEntity(jsonData,charset);
            put.setEntity(entity);
            try{
                log.info("streamTimeCountStart");
                StopWatch sw = new StopWatch();
                sw.start("streamTimeCount");
                CloseableHttpResponse response = client.execute(put);
                final int statusCode = response.getStatusLine().getStatusCode();
                if (statusCode != 200) {
                    throw new IOException(String.format("Stream load failed. status: %s load result: %s", statusCode,response.getStatusLine().getReasonPhrase()));
                }
                sw.stop();
                log.info(sw.getLastTaskName()+":"+sw.getTotalTimeMillis());
                log.info(sw.getLastTaskName()+":"+sw.getTotalTimeSeconds());
                generateRespContent(response);
            }catch (Exception e) {
                throw new StreamLoadException(e);
            }finally {
                client.close();
            }

    }

    private RespContent generateRespContent(CloseableHttpResponse response) throws StreamLoadException {
        ObjectMapper obj = new ObjectMapper();
        String loadResponse = "";
        RespContent respContent = new RespContent();
        try {
            loadResponse = EntityUtils.toString(response.getEntity());
            respContent = obj.readValue(loadResponse, RespContent.class);
            if(!DORIS_SUCCESS_STATUS.contains(respContent.getStatus())){
                log.error("stream load error: "+respContent.getMessage());
                throw new StreamLoadException("stream load error: " + respContent.getMessage());

            }
        } catch (IOException | StreamLoadException e) {
            log.error(" streamload response3 hostport:"+this.dorisConfig.getHost()+":"+this.dorisConfig.getHttpPort()+" "+dorisConfig.getDb()+"."+dorisConfig.getTb()+" "+loadResponse,e);
            throw new StreamLoadException(e);
        }
        log.info("写入成功,result{}", JSON.toJSONString(respContent.toString()));
        return respContent;
    }
    /**
     * Construct authentication information, the authentication method used by doris here is Basic Auth
     * @param username
     * @param password
     * @return
     */
    private String basicAuthHeader(String username, String password) {
        final String tobeEncode = username + ":" + password;
        byte[] encoded = org.apache.commons.codec.binary.Base64.encodeBase64(tobeEncode.getBytes(StandardCharsets.UTF_8));
        return "Basic " + new String(encoded);
    }

    private final static HttpClientBuilder httpClientBuilder = HttpClients
            .custom()
            .setRedirectStrategy(new DefaultRedirectStrategy() {
                @Override
                protected boolean isRedirectable(String method) {
                    // If the connection target is FE, you need to deal with 307 redirect。
                    return true;
                }
            });

    private HttpPut getHttpPutJsonArrayAppend(){
        HttpPut put = new HttpPut(this.getLoadUrlStr());
        put.removeHeaders(HttpHeaders.CONTENT_LENGTH);
        put.removeHeaders(HttpHeaders.TRANSFER_ENCODING);
        put.setHeader(HttpHeaders.EXPECT, "100-continue");
        put.setHeader(HttpHeaders.AUTHORIZATION, basicAuthHeader(this.dorisConfig.getUsername(), this.dorisConfig.getPassword()));
        put.setHeader("label", UUID.randomUUID().toString());
        put.setHeader("column_separator", ",");
        put.setHeader("format", "json");
        put.setHeader("strip_outer_array", "true");
        put.setHeader("merge_type", "APPEND");
        put.setHeader("timeout",this.dorisConfig.getTimeout());
        return put;
    }

    private HttpPut getHttpPutJsonArrayAppend(String db , String tb){
        HttpPut put = new HttpPut(this.getLoadUrlStr(db,tb));
        put.removeHeaders(HttpHeaders.CONTENT_LENGTH);
        put.removeHeaders(HttpHeaders.TRANSFER_ENCODING);
        put.setHeader(HttpHeaders.EXPECT, "100-continue");
        put.setHeader(HttpHeaders.AUTHORIZATION, basicAuthHeader(this.dorisConfig.getUsername(), this.dorisConfig.getPassword()));
        put.setHeader("label", UUID.randomUUID().toString());
        put.setHeader("column_separator", ",");
        put.setHeader("format", "json");
        put.setHeader("strip_outer_array", "true");
        put.setHeader("merge_type", "APPEND");
        put.setHeader("timeout",this.dorisConfig.getTimeout());
        return put;
    }

    private HttpPut getHttpPutJsonArrayDelete(String db , String tb){
        HttpPut put = new HttpPut(this.getLoadUrlStr(db,tb));
        put.removeHeaders(HttpHeaders.CONTENT_LENGTH);
        put.removeHeaders(HttpHeaders.TRANSFER_ENCODING);
        put.setHeader(HttpHeaders.EXPECT, "100-continue");
        put.setHeader(HttpHeaders.AUTHORIZATION, basicAuthHeader(this.dorisConfig.getUsername(), this.dorisConfig.getPassword()));
        put.setHeader("label", UUID.randomUUID().toString());
        put.setHeader("column_separator", ",");
        put.setHeader("format", "json");
        put.setHeader("strip_outer_array", "true");
        put.setHeader("merge_type", "DELETE");
        put.setHeader("timeout",this.dorisConfig.getTimeout());
        return put;
    }

    private HttpPut getHttpPutJsonArrayDelete(){
        HttpPut put = new HttpPut(this.getLoadUrlStr());
        put.removeHeaders(HttpHeaders.CONTENT_LENGTH);
        put.removeHeaders(HttpHeaders.TRANSFER_ENCODING);
        put.setHeader(HttpHeaders.EXPECT, "100-continue");
        put.setHeader(HttpHeaders.AUTHORIZATION, basicAuthHeader(this.dorisConfig.getUsername(), this.dorisConfig.getPassword()));
        put.setHeader("label", UUID.randomUUID().toString());
        put.setHeader("column_separator", ",");
        put.setHeader("format", "json");
        put.setHeader("strip_outer_array", "true");
        put.setHeader("merge_type", "DELETE");
        put.setHeader("timeout",this.dorisConfig.getTimeout());
        return put;
    }


}
