/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.doris.constant;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.tuple.Pair;

import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.LoggerUtil;
import net.wicp.tams.common.binlog.alone.DuckulaAssit;
import net.wicp.tams.common.binlog.alone.ListenerConf.DuckulaEvent;
import net.wicp.tams.common.binlog.alone.PluginAssit;
import net.wicp.tams.common.binlog.alone.binlog.bean.Rule;
import net.wicp.tams.common.constant.JvmStatus;
import net.wicp.tams.common.constant.ods.AddColName;
import net.wicp.tams.common.doris.bean.CheckPointConfig;
import net.wicp.tams.common.doris.bean.DorisConfig;

/**
 * 写入doris，攒数据批量写入，定时写入
 */
@Slf4j
public class DorisSinkV1 {
	private static final String tbFullNameFormate = "%s`%s";
	private static final String tbDataKeyFormate = "%s`%s`%s";
	private Long lastFlushTime = 0l;
	// 默认180秒刷缓存
	private final Integer flushCacheSeconds = 300;
	private final Integer flushCacheSize = 10000;
	private final ConcurrentHashMap<String, Map<String, Object>> cacheMap = new ConcurrentHashMap<>();
	private final DorisConfig dorisConfig;
	private final CheckPointConfig checkPointConfig;
	private final CheckpointService checkpointService;
	private final DorisStreamLoadBe dorisStreamLoadBe;
	private final ReadWriteLock rwl = new ReentrantReadWriteLock(true);
	private final Lock wLock = rwl.writeLock();
	private final long tryLockWaiteTime = 60;

	public DorisSinkV1(DorisConfig dorisConfig, CheckPointConfig checkPointConfig) {
		this.dorisConfig = dorisConfig;
		this.checkPointConfig = checkPointConfig;
		this.checkpointService = new CheckpointService(checkPointConfig);
		this.dorisStreamLoadBe = new DorisStreamLoadBe(dorisConfig);
		this.lastFlushTime = System.currentTimeMillis();
//        if(dorisConfig.getFlushCacheSecond() != null){
//            flushCacheSeconds = dorisConfig.getFlushCacheSecond();
//        }
//        if(dorisConfig.getFlushCacheSize() != null){
//            flushCacheSize = dorisConfig.getFlushCacheSize();
//        }
		addTimerFlush();
		addShutdownHook();
	}

	public void sink(String db, String tb, String primaryKeyVales, Map<String, String> data) throws Exception {
		wLock.tryLock(tryLockWaiteTime, TimeUnit.SECONDS);
		try {
			String tbFullName = String.format(tbFullNameFormate, db, tb);
			String dataKey = String.format(tbDataKeyFormate, db, tb, primaryKeyVales);
			Map<String, Object> tbData = cacheMap.get(tbFullName);
			if (tbData == null) {
				Map<String, Object> tbDataNew = Maps.newHashMap();
				tbDataNew.put(dataKey, data);
				cacheMap.put(tbFullName, tbDataNew);
			} else {
				tbData.put(dataKey, data);
				if (tbData.size() >= flushCacheSize) {
					log.info("表满写入开始 ,tbFullName={},tbSize={}", tbFullName, tbData.size());
					flush(tbFullName, db, tb);
					log.info("表满写入完成 ,tbFullName={}", tbFullName);
				}
			}
		} catch (Exception e) {
			log.error("消费binlog异常", e);
			rmChkAndExitJvm(e);
			throw e;
		} finally {
			wLock.unlock();
		}

	}

	public void doBusiAsyncTrue(boolean isSplit, boolean logicDel,
			Map<Rule, List<Pair<DuckulaEvent, Map<AddColName, Serializable>>>> sendDataCase) {
		if (MapUtils.isEmpty(sendDataCase)) {
			return;
		}
		Map<Pair<String, String>, ArrayNode> datas = new HashMap<>();
		for (Rule rule : sendDataCase.keySet()) {
			List<Pair<DuckulaEvent, Map<AddColName, Serializable>>> list = sendDataCase.get(rule);
			for (Pair<DuckulaEvent, Map<AddColName, Serializable>> data : list) {
				DuckulaEvent event = data.getLeft();
				Pair<String, String> ele = PluginAssit.getNewDbTb(rule, event);// Pair.of(event.getDb(), event.getTb());
				ArrayNode json = datas.get(ele);
				json = json == null ? JsonNodeFactory.instance.arrayNode(): json;
//				json.add(e);
				for (int i = 0; i < event.getItemsCount(); i++) {
					Map<String, String> valueMap = new HashMap<String, String>();
					valueMap.putAll(DuckulaAssit.getValueMap(event, i));
					if (MapUtils.isNotEmpty(data.getRight())) {
						for (AddColName addColName : data.getRight().keySet()) {
							String colNameTrue = addColName.getColNameTrue();
							if(!valueMap.containsKey(colNameTrue)) {//andy.zhou 20221110 如果列中已存在附加字段时，不要再添加了。列中的值是最正确的，推出来的附加字段准确度不高。
								valueMap.put(colNameTrue,   String.valueOf(data.getRight().get(addColName)));
							}
						}
					}
					CollectionUtil.filterNull(valueMap, 4);//20221111 andy.zhou, 去除空值保存证doris没有"null"					
					json.addPOJO(valueMap);
				}
				datas.put(ele, json);
			}
		}
		for (Pair<String, String> dbTb : datas.keySet()) {
			try {
				dorisStreamLoadBe.flushAndRetry(dbTb.getLeft(), dbTb.getRight(),
						JSONObject.toJSONString(datas.get(dbTb)));
			} catch (Exception e) {
				throw new RuntimeException("批量导入数据有问题", e);
			}
		}

	}

	private void flush(String tbKey, String db, String tb) throws Exception {
		ObjectMapper objmpa=new ObjectMapper();
		String data = objmpa.valueToTree(cacheMap.get(tbKey).values()).toString();
		log.info("写入数据开始,tb={},size={}", tb, cacheMap.get(tbKey).values().size());
		try {
			dorisStreamLoadBe.flushAndRetry(db, tb, data);
			cacheMap.remove(tbKey);
			log.info("写入数据结束,tb={}", tb);
		} catch (Exception e) {
			throw e;
		}
	}

	/**
	 * 清空缓存
	 */
	public void flushCache() throws Exception {
		wLock.tryLock(tryLockWaiteTime, TimeUnit.SECONDS);
		try {
			log.info("flushCache-start,size={}", cacheMap.size());
			for (Map.Entry<String, Map<String, Object>> entry : cacheMap.entrySet()) {
				if (entry.getValue() != null) {
					flush(entry.getKey(), entry.getKey().split("`")[0], entry.getKey().split("`")[1]);
				}
			}
		} catch (Exception e) {
			log.error("缓存写doris异常", e);
			rmChkAndExitJvm(e);
		} finally {
			wLock.unlock();
		}
	}

	private void addTimerFlush() {
		ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(10);
		scheduledExecutorService.scheduleAtFixedRate(() -> {
			try {
				log.info("定时stream load, size={}", cacheMap.size());
				flushCache();
				lastFlushTime = System.currentTimeMillis();
			} catch (Exception e) {
				// 重置点位，重启
				log.error("写doris异常重启：", e);
				rmChkAndExitJvm(e);
			}
		}, flushCacheSeconds, flushCacheSeconds, TimeUnit.SECONDS);
	}

	private void rmChkAndExitJvm(Exception e) {
		AlterMessage.alterMsgDataError(checkPointConfig.getSourceHost(), e.getMessage().toLowerCase());
		checkpointService.deleteChk(lastFlushTime);
		LoggerUtil.exit(JvmStatus.s15);// 关机
	}

	private void addShutdownHook() {
		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() {

				log.info("----------------------执行关闭进程 钩子开始-------------------------------------");
				checkpointService.deleteChk(lastFlushTime);
				log.info("----------------------执行关闭进程 钩子完成-------------------------------------");
			}
		});
	}
}
