/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.doris.constant;

import cn.hutool.json.JSONUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.doris.bean.DorisConfig;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 写入doris，攒数据批量写入，定时写入
 */
@Slf4j
public class DorisSink {
    private static final String tbFullNameFormate = "%s`%s";
    private static final String tbDataKeyFormate = "%s`%s`%s";
    private Long lastFlushTime = 0l;
    //默认180秒刷缓存
    private Long flushCacheMillis = 180000L;
    // <db`tb key,<db`tb`primKey,data>>
    private final ConcurrentHashMap<String, Map<String,Object>> cacheMap = new ConcurrentHashMap<>();
    private final DorisConfig dorisConfig;
    private final DorisStreamLoadBe dorisStreamLoadBe;

    public DorisSink(DorisConfig dorisConfig){
        this.dorisConfig = dorisConfig;
        this.dorisStreamLoadBe = new DorisStreamLoadBe(dorisConfig);
        this.flushCacheMillis = dorisConfig.getFlushCacheSecond().longValue()*1000;
        this.lastFlushTime = System.currentTimeMillis();
    }

    public void sink(String db , String tb , String primaryKeyVales , Map<String,String> data){
        String tbFullName = String.format(tbFullNameFormate,db,tb);
        String dataKey = String.format(tbDataKeyFormate,db,tb,primaryKeyVales);
        Map<String,Object> tbData = cacheMap.get(tbFullName);
        if(tbData == null){
            Map<String,Object> tbDataNew = Maps.newHashMap();
            tbDataNew.put(dataKey,data);
            cacheMap.put(tbFullName,tbDataNew);
        }else{
            tbData.put(dataKey,data);
            if(tbData.size() >= dorisConfig.getFlushCacheSize()){
                log.info("表满写入开始 time to flush cache,tbFullName={},tbSize={}",tbFullName,tbData.size());
                flush(tbFullName,db,tb);
                log.info("表满写入完成 time to flush cache,tbFullName={},tbSize={}",tbFullName,tbData.size());
            }
        }
        long currentTime = System.currentTimeMillis();
        if((currentTime - lastFlushTime) >= flushCacheMillis){
            log.info("超时刷缓存 time to flush cache,表数量={}",cacheMap.size());
            flushCache();
            lastFlushTime = currentTime;
        }
    }

    private void flush(String tbKey , String db , String tb){
        String data = JSONUtil.toJsonStr(cacheMap.get(tbKey).values());
        log.info("写入数据,tb={},size={}",tb,cacheMap.get(tbKey).values().size());
        try {
            dorisStreamLoadBe.loadJsonArrayAppend(data,db,tb);
            cacheMap.remove(tbKey);
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    /**
     * 清空缓存
     */
    public void flushCache(){
        for (Map.Entry<String, Map<String,Object>> entry : cacheMap.entrySet()) {
            if(entry.getValue() != null){
                log.info("flushCache,tb={},size={}",entry.getKey(),entry.getValue().size());
                flush(entry.getKey(),entry.getKey().split("`")[0],entry.getKey().split("`")[1]);
            }
        }
    }
}
