// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
package net.wicp.tams.common.doris.constant;

import cn.hutool.json.JSONUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.doris.bean.DorisConfig;
import net.wicp.tams.common.doris.bean.RespContent;
import net.wicp.tams.common.doris.exception.StreamLoadException;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * DorisStreamLoad
 **/
@Slf4j
public class DorisStreamLoadBe implements Serializable{


    private final static List<String> DORIS_SUCCESS_STATUS = new ArrayList<>(Arrays.asList("Success", "Publish Timeout"));
    private static final String loadUrlPattern = "http://%s/api/%s/%s/_stream_load?";

    private final String authEncoding;
    private final DorisConfig config;
    private int beNodeIndex = 0;
    private final List<String> hostsBe;
//    public static void main(String[] args) {
//        DorisConfig configt =  new DorisConfig();
//        configt.setHttpPortBe("8040");
//        configt.setHostsBe("172.31.43.250");
//        configt.setUsername("root");
//        configt.setPassword("YjAzMzIyMTg5MWUw");
//        String datas = "[{\"birthday\":\"2020-12-31\",\"matched_status\":\"0\",\"matched_status1\":\"0\",\"isdeleted\":\"0\",\"feed\":\"user_info\",\"update_time\":\"2022-01-04 00:00:00\",\"money\":\"123.0\",\"_last_opttype\":\"insert\",\"name\":\"00t424401\",\"field-abc\":\"dd\",\"systemdbname\":\"demo\",\"id\":\"210\",\"add1\":\"1\",\"age\":\"5556\",\"lastupdateddt\":\"2022-01-08 13:18:10\"}]";
//        DorisStreamLoadBe dorisStreamLoadBe = new DorisStreamLoadBe(configt);
//        dorisStreamLoadBe.loadJsonArrayDelete(datas,"test","user_info");
//    }

    public DorisStreamLoadBe(DorisConfig config) {
        this.config = config;
        this.authEncoding = Base64.getEncoder().encodeToString(String.format("%s:%s", config.getUsername(), config.getPassword()).getBytes(StandardCharsets.UTF_8));
        this.hostsBe = Lists.newArrayList(config.getHostsBe().split(","));
    }

    private HttpURLConnection getConnection(String urlStr, String label) throws IOException {
        URL url = new URL(urlStr);
        HttpURLConnection conn = (HttpURLConnection) url.openConnection();
        conn.setInstanceFollowRedirects(false);
        conn.setRequestMethod("PUT");
        String authEncoding = Base64.getEncoder().encodeToString(String.format("%s:%s", this.config.getUsername(), this.config.getPassword()).getBytes(StandardCharsets.UTF_8));
        conn.setRequestProperty("Authorization", "Basic " + authEncoding);
        conn.addRequestProperty("Expect", "100-continue");
        conn.addRequestProperty("Content-Type", "text/plain; charset=UTF-8");
        conn.addRequestProperty("label", label);
        conn.addRequestProperty("column_separator", ",");
        conn.addRequestProperty("format", "json");
        conn.addRequestProperty("strip_outer_array", "true");
        conn.addRequestProperty("merge_type", "APPEND");
        conn.setDoOutput(true);
        conn.setDoInput(true);
        return conn;
    }

    private HttpURLConnection getConnectionDelete(String urlStr, String label) throws IOException {
        URL url = new URL(urlStr);
        HttpURLConnection conn = (HttpURLConnection) url.openConnection();
        conn.setInstanceFollowRedirects(false);
        conn.setRequestMethod("PUT");
        String authEncoding = Base64.getEncoder().encodeToString(String.format("%s:%s", this.config.getUsername(), this.config.getPassword()).getBytes(StandardCharsets.UTF_8));
        conn.setRequestProperty("Authorization", "Basic " + authEncoding);
        conn.addRequestProperty("Expect", "100-continue");
        conn.addRequestProperty("Content-Type", "text/plain; charset=UTF-8");
        conn.addRequestProperty("label", label);
        conn.addRequestProperty("column_separator", ",");
        conn.addRequestProperty("format", "json");
        conn.addRequestProperty("strip_outer_array", "true");
        conn.addRequestProperty("merge_type", "DELETE");
        conn.setDoOutput(true);
        conn.setDoInput(true);
        return conn;
    }

    public static class LoadResponse {
        public int status;
        public String respMsg;
        public String respContent;

        public LoadResponse(int status, String respMsg, String respContent) {
            this.status = status;
            this.respMsg = respMsg;
            this.respContent = respContent;
        }
        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("status: ").append(status);
            sb.append(", resp msg: ").append(respMsg);
            sb.append(", resp content: ").append(respContent);
            return sb.toString();
        }
    }

    public void flushAndRetry( String db , String tb , String data) throws Exception {
        for (int i = 0; i <= 5; i++) {
            try {
                loadJsonArrayAppend(data,db,tb);
                break;
            } catch (Exception e) {
                log.info(" stream load retry times :"+i+" "+e.getMessage());
                if (i >= 5) {
                    throw e;
                }
                try {
                    Thread.sleep(1000);
                } catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                    throw new IOException("unable to flush; interrupted while doing another attempt", e);
                }
            }
        }
    }

    public LoadResponse loadJsonArrayAppend(String value , String db ,String tb) throws Exception {
        Calendar calendar = Calendar.getInstance();
        String label = String.format("audit_%s%02d%02d_%02d%02d%02d_%s",
                calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH) + 1, calendar.get(Calendar.DAY_OF_MONTH),
                calendar.get(Calendar.HOUR_OF_DAY), calendar.get(Calendar.MINUTE), calendar.get(Calendar.SECOND),
                UUID.randomUUID().toString().replaceAll("-", ""));

        HttpURLConnection feConn = null;
        HttpURLConnection beConn = null;
        try {
            String url = getLoadUrlStr(db,tb);
            beConn = getConnection(url, label);
            // send data to be
            BufferedOutputStream bos = new BufferedOutputStream(beConn.getOutputStream());
            bos.write(value.getBytes());
            bos.close();

            // get respond
            int status = beConn.getResponseCode();
            String respMsg = beConn.getResponseMessage();
            InputStream stream = (InputStream) beConn.getContent();
            BufferedReader br = new BufferedReader(new InputStreamReader(stream));
            StringBuilder response = new StringBuilder();
            String line;
            while ((line = br.readLine()) != null) {
                response.append(line);
            }
            if(status != 200){
                log.error("stream load error={},url={}",respMsg,url);
                throw new StreamLoadException(respMsg);
            }else{

                ObjectMapper obj = new ObjectMapper();
                RespContent respContent = new RespContent();
                try {
                    respContent = obj.readValue(response.toString(), RespContent.class);
                    if(!DORIS_SUCCESS_STATUS.contains(respContent.getStatus())){
                        log.error("stream load error={},url={}",respContent.getMessage(),url);
                        throw new StreamLoadException("stream load error:" + respContent.getMessage()+",errorUrl:"+respContent.getErrorURL());

                    }
                } catch (IOException | StreamLoadException e) {
                    throw new StreamLoadException(e);
                }
            }
            return new LoadResponse(status, respMsg, response.toString());
        } catch (Exception e) {
            e.printStackTrace();
            String err = "failed to load audit via AuditLoader plugin with label: " + label;
            log.error(err, e);
            throw e;
        } finally {
            if (feConn != null) {
                feConn.disconnect();
            }
            if (beConn != null) {
                beConn.disconnect();
            }
        }
    }


    public LoadResponse loadJsonArrayDelete(String value , String db ,String tb) {
        Calendar calendar = Calendar.getInstance();
        String label = String.format("audit_%s%02d%02d_%02d%02d%02d_%s",
                calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH) + 1, calendar.get(Calendar.DAY_OF_MONTH),
                calendar.get(Calendar.HOUR_OF_DAY), calendar.get(Calendar.MINUTE), calendar.get(Calendar.SECOND),
                UUID.randomUUID().toString().replaceAll("-", ""));

        HttpURLConnection feConn = null;
        HttpURLConnection beConn = null;
        try {
            // build request and send to new be location
            beConn = getConnectionDelete(getLoadUrlStr(db,tb), label);
            // send data to be
            BufferedOutputStream bos = new BufferedOutputStream(beConn.getOutputStream());
            bos.write(value.getBytes());
            bos.close();

            // get respond
            int status = beConn.getResponseCode();
            String respMsg = beConn.getResponseMessage();
            InputStream stream = (InputStream) beConn.getContent();
            BufferedReader br = new BufferedReader(new InputStreamReader(stream));
            StringBuilder response = new StringBuilder();
            String line;
            while ((line = br.readLine()) != null) {
                response.append(line);
            }
//            log.info("AuditLoader plugin load with label: {}, response code: {}, msg: {}, content: {}",label, status, respMsg, response.toString());
            return new LoadResponse(status, respMsg, response.toString());

        } catch (Exception e) {
            e.printStackTrace();
            String err = "failed to load audit via AuditLoader plugin with label: " + label;
            log.error(err, e);
            return new LoadResponse(-1, e.getMessage(), err);
        } finally {
            if (feConn != null) {
                feConn.disconnect();
            }
            if (beConn != null) {
                beConn.disconnect();
            }
        }
    }

    private String getLoadUrlStr (String db ,String tb){
        return  String.format(loadUrlPattern, getBeHostPortRotation(), db, tb);
    }

    private String getBeHostPortRotation(){
        beNodeIndex++;
        if (beNodeIndex >= hostsBe.size()){
            beNodeIndex = 0;
        }
        return hostsBe.get(beNodeIndex)+":"+this.config.getHttpPortBe();
    }

}
