package net.wicp.tams.common.doris.constant;

import com.alibaba.druid.sql.SQLUtils;
import com.alibaba.druid.sql.ast.SQLDataType;
import com.alibaba.druid.sql.ast.SQLStatement;
import com.alibaba.druid.sql.ast.statement.SQLAlterTableAddColumn;
import com.alibaba.druid.sql.ast.statement.SQLAlterTableItem;
import com.alibaba.druid.sql.ast.statement.SQLAlterTableStatement;
import com.alibaba.druid.sql.ast.statement.SQLColumnDefinition;
import com.alibaba.druid.sql.dialect.mysql.ast.statement.MySqlAlterTableChangeColumn;
import com.alibaba.druid.util.JdbcConstants;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.mysql.jdbc.CallableStatement;
import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.apiext.TimeAssist;
import net.wicp.tams.common.binlog.alone.ListenerConf;
import net.wicp.tams.common.binlog.alone.binlog.bean.Rule;
import net.wicp.tams.common.binlog.alone.binlog.bean.RuleItem;
import net.wicp.tams.common.doris.bean.CheckPointConfig;
import net.wicp.tams.common.doris.bean.DorisConfig;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.jdbc.DataSourceBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCallback;
import org.springframework.jdbc.core.StatementCallback;
import org.springframework.jdbc.datasource.SimpleDriverDataSource;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * doris jdbc load
 */
@Slf4j
public class DorisJdbcLoad {

    private final String fieldDelimiter = ",";
    private static final String DESC_DB_TB = "DESC %s.%s";
    private static final String COLUMN_TYPE_FORMAT = "%s(%s)";
    private static DorisConfig dorisConfig;
    private static final String checkBatchDeleteColumn = "__DORIS_DELETE_SIGN__";
    private static final Map<String,Boolean> dbtbCheckMap = Maps.newHashMap();
    private final DataSource dorisDataSource;
    private final JdbcTemplate dorisJdbcTemplate;
    private DorisSink dorisSink;
    private final CheckPointConfig checkPointConfig;

    public DorisJdbcLoad(DorisConfig dorisConfig,CheckPointConfig checkPointConfig) {
        DorisJdbcLoad.dorisConfig = dorisConfig;
        this.dorisDataSource = getDorisDataSource(dorisConfig.getDb());
        this.dorisJdbcTemplate = new JdbcTemplate(this.dorisDataSource);
        this.checkPointConfig = checkPointConfig;

    }

    public DataSource getDorisDataSource(String db) {
        StringBuilder url = new StringBuilder();
        url.append("jdbc:mysql://");
        url.append(dorisConfig.getHost()+":"+dorisConfig.getJdbcPort());
        url.append("/");
        url.append(db);
        return DataSourceBuilder.create().url(url.toString())
                .username(dorisConfig.getUsername())
                .password(dorisConfig.getPassword())
                .driverClassName("com.mysql.jdbc.Driver")
                .type(HikariDataSource.class)
                .build();
    }


    public Boolean alterTable(Rule rule, ListenerConf.ColHis colHis,String[] addColNames , String sql,String db , String tb){
        Boolean alterResult = true;
        executeAddColumn(rule,colHis,addColNames,sql,db,tb);
        executeModifyColumn(rule,colHis,addColNames,sql,db,tb);
        return alterResult;
    }

    private boolean executeAddColumn(Rule rule, ListenerConf.ColHis colHis,String[] addColNames , String sql,String db , String tb){
        Boolean alterResult = false;
        String alterSql = null;
        try{
            alterSql = generateAddColumnSql(rule,colHis,addColNames,sql,db,tb);
            if(!StringUtils.isEmpty(alterSql)){
                dorisJdbcTemplate.execute(alterSql);
            }else{
                return true;
            }
        }catch (Exception e){
            String msg = e.getCause().getMessage().toLowerCase();
            if (msg.contains("add column which already exists")){
                log.info("添加字段成功: "+alterSql);
            }else if (msg.contains("can not change default value")){
                System.out.println("不可修改默认值: "+alterSql);
            } else {
                log.error("修改表结构失败！",e);
                alterResult = false;
                AlterMessage.alterMsgDDL(checkPointConfig.getSourceHost(),alterSql,msg);
            }
            return alterResult;
        }
        String sqlCheck=String.format(DESC_DB_TB,db,tb);
        while (true) {
            try {
                List<Map<String,Object>> list =dorisJdbcTemplate.queryForList(sqlCheck);
                List<String> tableColumns = Lists.newArrayList();
                list.stream().forEach(stringObjectMap -> {
                    tableColumns.add(stringObjectMap.get("Field").toString());
                });
                log.info("尝试获取字段！");
                if(tableColumns.containsAll(Lists.newArrayList(addColNames))){
                    alterResult = true;
                    log.info("新增字段成功！");
                }
                if(!alterResult){
                    throw new ProjectException(ExceptAll.Project_default,"未获取到修改表结构字段结果！");
                }
                break;
            } catch (Throwable e) {
                boolean reDoWait = TimeAssist.reDoWait("doris-alterTable", dorisConfig.getSqlRetryTimes());
                if (reDoWait) {// 达到最大值就出
                    log.error("重试{}次,未获取到修改表结构字段结果！",dorisConfig.getSqlRetryTimes());
                    AlterMessage.alterMsgDDL(checkPointConfig.getSourceHost(),alterSql,"重试多次,未获取到修改表结构字段结果！");
                    break;
                } else {
                    continue;
                }
            }
        }
        return alterResult;
    }

    private Boolean executeModifyColumn(Rule rule, ListenerConf.ColHis colHis,String[] addColNames , String sql,String db , String tb){

        Boolean alterResult = false;
        String alterSql = null;
        try{
            alterSql = generateModifyColumnSql(rule,colHis,addColNames,sql,db,tb);
            if(!StringUtils.isEmpty(alterSql)){
                dorisJdbcTemplate.execute(alterSql);
            }else{
                return true;
            }
        }catch (Exception e){
            String msg = e.getCause().getMessage().toLowerCase();
            if (msg.contains("add column which already exists")){
                log.info("添加字段成功: "+alterSql);
            }else if (msg.contains("can not change default value")){
                System.out.println("不可修改默认值: "+alterSql);
            } else {
                log.error("修改表结构失败！",e);
                alterResult = false;
                AlterMessage.alterMsgDDL(checkPointConfig.getSourceHost(),alterSql,msg);
            }
            return alterResult;
        }
        String sqlCheck=String.format(DESC_DB_TB,db,tb);
        while (true) {
            try {
                List<Map<String,Object>> list =dorisJdbcTemplate.queryForList(sqlCheck);
                List<String> tableColumns = Lists.newArrayList();
                list.stream().forEach(stringObjectMap -> {
                    tableColumns.add(stringObjectMap.get("Field").toString());
                });
                log.info("尝试获取字段！");
                if(tableColumns.containsAll(Lists.newArrayList(addColNames))){
                    alterResult = true;
                    log.info("新增字段成功！");
                }

                if(!alterResult){
                    throw new ProjectException(ExceptAll.Project_default,"未获取到修改表结构字段结果！");
                }
                break;
            } catch (Throwable e) {
                boolean reDoWait = TimeAssist.reDoWait("doris-alterTable", dorisConfig.getSqlRetryTimes());
                if (reDoWait) {// 达到最大值就出
                    log.error("重试{}次,未获取到修改表结构字段结果！",dorisConfig.getSqlRetryTimes());
                    AlterMessage.alterMsgDDL(checkPointConfig.getSourceHost(),alterSql,"重试多次,未获取到修改表结构字段结果！");
                    break;
                } else {
                    continue;
                }
            }
        }
        return alterResult;
    }

    private String generateAddColumnSql(Rule rule, ListenerConf.ColHis colHis, String[] addColNames , String sql,String db,String tb){
        Map<String,SQLColumnDefinition> columnDefinitionMap = SqlUtil.getColumnDefinitionMap(sql);
        if(CollectionUtils.isEmpty(columnDefinitionMap)){
            return null;
        }
        StringBuilder alterSql = new StringBuilder();
        alterSql.append("alter table ").append(db).append(".").append(tb).append(" ");
        for(String column : addColNames){
            column = column.replaceAll("`","");
            alterSql.append("add column ").append(column).append(" ");
            //类型
            if(columnDefinitionMap.get(column) != null){
                alterSql.append(convertColumnType(columnDefinitionMap.get(column)));
            }else{
                alterSql.append(colHis.getColTypes2(colHis.getColsList().indexOf(column)));
            }
            alterSql.append(" REPLACE ");//agreegatekey replace 属性
            if(columnDefinitionMap.get(column) != null && columnDefinitionMap.get(column).getDefaultExpr() != null){
                alterSql.append(" DEFAULT '").append(columnDefinitionMap.get(column).getDefaultExpr().toString()).append("'");//默认值
            }
            if(columnDefinitionMap.get(column) != null && columnDefinitionMap.get(column).getComment() != null){
                alterSql.append(" COMMENT ").append(columnDefinitionMap.get(column).getComment().toString());//注释
            }
            alterSql.append(" ,");
        }
        alterSql.deleteCharAt(alterSql.length()-1);
        alterSql.append(";");
        return alterSql.toString();
    }

    private String generateModifyColumnSql(Rule rule, ListenerConf.ColHis colHis, String[] addColNames , String sql,String db,String tb){
        Map<String,SQLColumnDefinition> modifyDefinitionMap = SqlUtil.getModifyColumnDefinitionMap(sql);
        if(CollectionUtils.isEmpty(modifyDefinitionMap)){
            return null;
        }
         StringBuilder alterSql = new StringBuilder();
        String fullTableName = String.format("%s.%s ", db,tb);
        alterSql.append(" ALTER TABLE ").append(fullTableName);

        if(modifyDefinitionMap != null && modifyDefinitionMap.size() > 0){
            modifyDefinitionMap.forEach((k, value) -> {
                alterSql.append(" MODIFY COLUMN ").append("`").append(k).append("` ").append(convertColumnType(value)).append(" REPLACE NULL");
                if(modifyDefinitionMap.get(k)!= null && modifyDefinitionMap.get(k).getDefaultExpr()  != null){
                    String defaultValue = modifyDefinitionMap.get(k).getDefaultExpr().toString().replaceAll("'", "");
                    alterSql.append(" DEFAULT '").append(defaultValue).append("' ");
                }
                alterSql.append(fieldDelimiter);
            });
        }

        if(alterSql.length() > 0){
            alterSql.deleteCharAt(alterSql.length()-1);
        }

        return alterSql.toString();
    }




    private String convertColumnType(SQLColumnDefinition definition){
        String type = "varchar(255)";
        String mysqlType = definition.getDataType().getName().toLowerCase();

        switch (mysqlType){
            case "boolean":
            case "date" :
            case "float" :
            case "tinyint" :
            case "smallint" :
            case "int" :
            case "bigint" :
            case "double" :
            case "datetime" :
            case "timestamp" :
            case "decimal" :
//                和业务字段类型保持一致
//                type = definition.getDataType().toString();
                type = "varchar(50)";
                break;
            case "varchar":
            case "char":
                int length = definition.getDataType().getArguments() == null ? 100 : Integer.parseInt(definition.getDataType().getArguments().get(0).toString());
                type = String.format(COLUMN_TYPE_FORMAT,definition.getDataType().getName(),length*4);
                break;
            case "text" :
            case "longtext" :
            case "mediumtext" :
            case "tinytext" :
            case "json" :
            case "blob" :
                type = "string";
                break;
            default:
                type = "string";
                break;

        }
        return type;
    }


    public Boolean checkBatchDelete(String db,String tb){
        Boolean isSupported = false;
        if(dbtbCheckMap.get(String.format(DESC_DB_TB,db,tb)) != null && dbtbCheckMap.get(db+tb)){
            isSupported = true;
        }else{
            isSupported = checkBatchDeleteJdbc(db,tb);
            dbtbCheckMap.put(String.format(DESC_DB_TB,db,tb),isSupported);
        }
        return isSupported;
    }

    public Boolean checkBatchDeleteJdbc(String db , String tb){
        Boolean isSupported = false;
        StringBuilder alterTableEnableBatchDelete = new StringBuilder();
        alterTableEnableBatchDelete.append("ALTER TABLE ");
        alterTableEnableBatchDelete.append(tb).append(" ");
        alterTableEnableBatchDelete.append("ENABLE FEATURE 'BATCH_DELETE';");
        String sqlCheckBefore="SET show_hidden_columns=true;";
        String sqlCheck=String.format(DESC_DB_TB,db,tb);
        try{
            dorisJdbcTemplate.execute(alterTableEnableBatchDelete.toString());
        }catch (Exception e){
            log.info("添加批量删除权限失败,可能已有权限!",e);
        }
        while (true) {
            try {
                dorisJdbcTemplate.execute(sqlCheckBefore);
                List<Map<String,Object>> list =dorisJdbcTemplate.queryForList(sqlCheck);
                //删除标记总是在最后，所以倒叙遍历
                Collections.reverse(list);
                for(Map<String,Object> map :list){
                    if(checkBatchDeleteColumn.equals(map.get("Field"))){
                        isSupported = true;
                        break;
                    }
                }
                if(!isSupported){
                    throw new ProjectException(ExceptAll.Project_default,"未获取到批量删除标记！");
                }
                break;
            } catch (Throwable e) {
                boolean reDoWait = TimeAssist.reDoWait("doris-batchDelete", dorisConfig.getSqlRetryTimes());
                if (reDoWait) {// 达到最大值就出
                    log.error("重试3次,未获取到批量删除标记！");
                    break;
                } else {
                    continue;
                }
            }
        }
        return isSupported;
    }
    //    public static void main(String[] args) {
//        JdbcTemplate jdbcTemplate = dorisJdbcTemplate(prestoDataSource());
//
//        String sql="ALTER TABLE user ENABLE FEATURE 'BATCH_DELETE'";
//        String sqlCheckBefore="SET show_hidden_columns=true;";
//        String sqlCheck="desc user";
//        jdbcTemplate.execute(sqlCheckBefore);
//        List<Map<String,Object>> list =jdbcTemplate.queryForList(sqlCheck);
//        System.out.println(list);
//    }

}
