package net.wicp.tams.common.doris.constant;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.mysql.jdbc.CallableStatement;
import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.apiext.TimeAssist;
import net.wicp.tams.common.binlog.alone.ListenerConf;
import net.wicp.tams.common.binlog.alone.binlog.bean.Rule;
import net.wicp.tams.common.binlog.alone.binlog.bean.RuleItem;
import net.wicp.tams.common.doris.bean.DorisConfig;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.jdbc.DataSourceBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCallback;
import org.springframework.jdbc.core.StatementCallback;
import org.springframework.jdbc.datasource.SimpleDriverDataSource;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * doris jdbc load
 */
@Slf4j
public class DorisJdbcLoad {

    private static String DESC_DB_TB = "DESC %s.%s";
    private static DorisConfig dorisConfig;
    private static final String checkBatchDeleteColumn = "__DORIS_DELETE_SIGN__";
    private static final Map<String,Boolean> dbtbCheckMap = Maps.newHashMap();
    private DataSource dorisDataSource;
    private JdbcTemplate dorisJdbcTemplate;

    public DorisJdbcLoad(DorisConfig dorisConfig) {
        this.dorisConfig = dorisConfig;
        this.dorisDataSource = getDorisDataSource(dorisConfig.getDb());
        this.dorisJdbcTemplate = new JdbcTemplate(this.dorisDataSource);
    }

    public DataSource getDorisDataSource(String db) {
        StringBuilder url = new StringBuilder("");
        url.append("jdbc:mysql://");
        url.append(dorisConfig.getHost()+":"+dorisConfig.getJdbcPort());
        url.append("/");
        url.append(db);
        return DataSourceBuilder.create().url(url.toString())
                .username(dorisConfig.getUsername())
                .password(dorisConfig.getPassword())
                .driverClassName("com.mysql.jdbc.Driver")
                .type(HikariDataSource.class)
                .build();
    }

    public Boolean checkBatchDelete(String db,String tb){
        Boolean isSupported = false;
        if(dbtbCheckMap.get(String.format(DESC_DB_TB,db,tb)) != null && dbtbCheckMap.get(db+tb)){
            isSupported = true;
        }else{
            isSupported = checkBatchDeleteJdbc(db,tb);
            dbtbCheckMap.put(String.format(DESC_DB_TB,db,tb),isSupported);
        }
        return isSupported;
    }

    public Boolean checkBatchDeleteJdbc(String db , String tb){
        Boolean isSupported = false;
        StringBuilder alterTableEnableBatchDelete = new StringBuilder();
        alterTableEnableBatchDelete.append("ALTER TABLE ");
        alterTableEnableBatchDelete.append(tb).append(" ");
        alterTableEnableBatchDelete.append("ENABLE FEATURE 'BATCH_DELETE';");
        String sqlCheckBefore="SET show_hidden_columns=true;";
        String sqlCheck=String.format(DESC_DB_TB,db,tb);
        try{
            dorisJdbcTemplate.execute(alterTableEnableBatchDelete.toString());
        }catch (Exception e){
            log.info("添加批量删除权限失败,可能已有权限!",e);
        }
        while (true) {
            try {
                dorisJdbcTemplate.execute(sqlCheckBefore);
                List<Map<String,Object>> list =dorisJdbcTemplate.queryForList(sqlCheck);
                //删除标记总是在最后，所以倒叙遍历
                Collections.reverse(list);
                for(Map<String,Object> map :list){
                    if(checkBatchDeleteColumn.equals(map.get("Field"))){
                        isSupported = true;
                        break;
                    }
                }
                if(!isSupported){
                    throw new ProjectException(ExceptAll.Project_default,"未获取到批量删除标记！");
                }
                break;
            } catch (Throwable e) {
                boolean reDoWait = TimeAssist.reDoWait("doris-batchDelete", this.dorisConfig.getSqlRetryTimes());
                if (reDoWait) {// 达到最大值就出
                    log.error("重试3次,未获取到批量删除标记！");
                    break;
                } else {
                    continue;
                }
            }
        }
        return isSupported;
    }

    public Boolean alterTable(Rule rule, ListenerConf.ColHis colHis,String[] addColNames){
        Boolean alterResult = false;
        String[] dbtb = rule.getItems().get(RuleItem.dbtb).split("\\.");
        String db = dbtb[0];
        String tb = dbtb[1];
        String sql = generateAddColumnSql(rule,colHis,addColNames);
        try{
            dorisJdbcTemplate.execute(sql);
        }catch (Exception e){
            log.error("修改表结构失败！",e);
            alterResult = false;
            return alterResult;
        }
        String sqlCheck=String.format(DESC_DB_TB,db,tb);
        while (true) {
            try {
                List<Map<String,Object>> list =dorisJdbcTemplate.queryForList(sqlCheck);
                List<String> tableColumns = Lists.newArrayList();
                list.stream().forEach(stringObjectMap -> {
                    tableColumns.add(stringObjectMap.get("Field").toString());
                });
                List<String> cols = colHis.getColsList();
                if(tableColumns.containsAll(cols)){
                    alterResult = true;
                }

                if(!alterResult){
                    throw new ProjectException(ExceptAll.Project_default,"未获取到修改表结构字段结果！");
                }
                break;
            } catch (Throwable e) {
                boolean reDoWait = TimeAssist.reDoWait("doris-alterTable", this.dorisConfig.getSqlRetryTimes());
                if (reDoWait) {// 达到最大值就出
                    log.error("重试3次,未获取到修改表结构字段结果！");
                    break;
                } else {
                    continue;
                }
            }
        }
        return alterResult;
    }

    private String generateAddColumnSql(Rule rule, ListenerConf.ColHis colHis, String[] addColNames){
        String[] dbtb = rule.getItems().get(RuleItem.dbtb).split("\\.");
        String db = dbtb[0];
        String tb = dbtb[1];
        StringBuilder sql = new StringBuilder("");
        sql.append("alter table ").append(db).append(".").append(tb).append(" ");
        for(String column : addColNames){
            column = column.replaceAll("`","");
            sql.append("add column ").append(column).append(" ");
            sql.append(colHis.getColTypes2(colHis.getColsList().indexOf(column))).append(" null ,");
        }
        sql.deleteCharAt(sql.length()-1);
        sql.append(";");
        return sql.toString();
    }


    //    public static void main(String[] args) {
//        JdbcTemplate jdbcTemplate = dorisJdbcTemplate(prestoDataSource());
//
//        String sql="ALTER TABLE user ENABLE FEATURE 'BATCH_DELETE'";
//        String sqlCheckBefore="SET show_hidden_columns=true;";
//        String sqlCheck="desc user";
//        jdbcTemplate.execute(sqlCheckBefore);
//        List<Map<String,Object>> list =jdbcTemplate.queryForList(sqlCheck);
//        System.out.println(list);
//    }
}
