/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.connector;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.XMLConfiguration;
import org.apache.commons.configuration.tree.ConfigurationNode;
import org.apache.commons.configuration.tree.DefaultConfigurationNode;
import org.apache.commons.jxpath.JXPathContext;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.ReflectAssist;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.connector.constant.ColGType;
import net.wicp.tams.common.connector.constant.ColProperty;
import net.wicp.tams.common.connector.constant.ColType;
import net.wicp.tams.common.constant.PathType;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;

@Slf4j
public abstract class XmlUtil {

	private static SAXParserFactory saxFactory;

	static {
		saxFactory = SAXParserFactory.newInstance();
		saxFactory.setNamespaceAware(true);
		saxFactory.setValidating(false);
	}

	public static synchronized XMLReader getXMLReader() {
		try {
			SAXParser parser = saxFactory.newSAXParser();
			XMLReader reader = parser.getXMLReader();
			reader.setFeature("http://xml.org/sax/features/namespace-prefixes", false);
			return parser.getXMLReader();
		} catch (ParserConfigurationException e) {
			return null;
		} catch (SAXException se) {
			return null;
		}
	}

	/**
	 * 解析XML文档
	 * 
	 * @param src 要解析的字符串
	 * @return 解析后文档
	 * @throws ProjectException 解析错误
	 */
	public static final Document parserDocment(String src) throws ProjectException {
		try {
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			DocumentBuilder db = dbf.newDocumentBuilder();
			return db.parse(new InputSource(new StringReader(src)));
		} catch (Exception e) {
			throw new ProjectException(ExceptAll.Project_default, "解析XML文档错误");
		}
	}

	/**
	 * 通过标签名得到第一个元素
	 * 
	 * @param doc     文档
	 * @param tagName 标签名
	 * @return 元素
	 */
	public static Element getElementByTagName(Document doc, String tagName) {
		NodeList nodes = doc.getElementsByTagName(tagName);
		if (nodes == null || nodes.getLength() == 0)
			return null;
		else
			return (Element) nodes.item(0);
	}

	/****
	 * 通过属性名得到属性的值
	 * 
	 * @param node     含有属性的节点
	 * @param attrName 属性名
	 * @return 属性值
	 */
	@SuppressWarnings("rawtypes")
	public static final String findValueByAttrName(ConfigurationNode node, String attrName) {
		List typeAry = node.getAttributes(attrName);
		if (CollectionUtils.isEmpty(typeAry)) {
			return null;
		}
		return String.valueOf(((DefaultConfigurationNode) typeAry.get(0)).getValue());
	}

	/**
	 * 得到元素的值
	 * 
	 * @param element 文档元素
	 * @return String 元素的值
	 */
	public static String getElementValue(Element element) {
		if (element.hasChildNodes())
			return element.getFirstChild().getNodeValue();
		else
			return null;
	}

	/*****
	 * 通过名称得到指定节点下第一个元素
	 * 
	 * @param node 指定节点
	 * @param name 子节点名称
	 * @return 节点
	 */
	public static final ConfigurationNode getFirstNodeByNodeName(ConfigurationNode node, String name) {
		List<ConfigurationNode> nodes = node.getChildren(name);
		if (CollectionUtils.isEmpty(nodes)) {
			return null;
		}
		return nodes.get(0);
	}

	/**
	 * 通过xpath方式得到对象中的值
	 * 
	 * @param beanObj 要取值的对象
	 * @param xpath   取值 xpath 路径
	 * @return 返回的值
	 */
	public static Object getValueByXpath(Object beanObj, String xpath) {
		JXPathContext context = JXPathContext.newContext(beanObj);
		return context.getValue(xpath);
	}

	public static List<String> getChildValues(ConfigurationNode superNode, String subNodeName) {
		List<String> retlist = new ArrayList<>();
		final List<ConfigurationNode> includeList = superNode.getChildren(subNodeName);
		if (CollectionUtils.isEmpty(includeList)) {
			return retlist;
		}
		for (ConfigurationNode configurationNode : includeList) {
			retlist.add(String.valueOf(configurationNode.getValue()));
		}
		return retlist;
	}

	/***
	 * 把xml字符加到 目标xml文件的指定的节点
	 * 
	 * @param xml
	 * @param path
	 * @param addStr
	 * @throws ConfigurationException
	 */
	public static void mergexml(XMLConfiguration xml, String path, String addStr) throws ConfigurationException {
		InputStream addStrIn = new ByteArrayInputStream(addStr.getBytes());
		XMLConfiguration addConfig = new XMLConfiguration();
		addConfig.load(addStrIn);
		int startint = StringUtil.isNotNull(path) ? path.lastIndexOf(".") : -1;
		String key = path.substring(startint + 1);
		xml.addNodes(key, addConfig.getRoot().getChildren());
	}

	/***
	 * 通过类型。创建xml文件
	 * 
	 * @param classz
	 */
	public static Document createXmlDoc(Class<?> classz) {
		Map<String, Class[]> contextType = ReflectAssist.getAllType(classz);
		try {
			Document docment = parserDocment("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\r\n" + "<InterFaceMapping>\r\n"
					+ "	<PropertyIn>\r\n" + "	</PropertyIn>\r\n" + "	<PropertyOut>\r\n" + "	</PropertyOut>\r\n"
					+ "</InterFaceMapping>");
			if (MapUtils.isEmpty(contextType)) {
				return docment;
			}
			Element propertyInEle = getElementByTagName(docment, "PropertyIn");
			for (String key : contextType.keySet()) {
				Element tempnode = docment.createElement("COL");
				tempnode.setAttribute("name", key);
				tempnode.setAttribute("alias", key);
				tempnode.setNodeValue(key);
				Class[] classtype = contextType.get(key);
				if (classtype[0].isPrimitive()) {
					String name = classtype[0].getTypeName();
					ColType colType = ColType.string;
					if ("void".equals(name))
						continue;
					else if ("boolean".equals(name))
						colType = ColType.string;
					else if ("byte".equals(name))
						colType = ColType.bytes;
					else if ("char".equals(name))
						colType = ColType.string;
					else if ("double".equals(name))
						colType = ColType.doubler;
					else if ("float".equals(name))
						colType = ColType.doubler;
					else if ("int".equals(name))
						colType = ColType.integer;
					else if ("long".equals(name))
						colType = ColType.string;
					else if ("short".equals(name))
						colType = ColType.integer;
					tempnode.setAttribute("type", colType.name());
				} else if (classtype[0].getName().startsWith("java.lang")) {
					String name = classtype[0].getTypeName();
					ColType colType = typestrToDynabean(name);
					tempnode.setAttribute("type", colType.name());
				} else if (classtype[0].isAssignableFrom(List.class)) {
					tempnode.setAttribute("gtype", ColGType.list.name());
					ColType colType = typestrToDynabean(classtype[1].getName());
					tempnode.setAttribute("type", colType.name());
				} else if (classtype[0].isAssignableFrom(Map.class)) {
					tempnode.setAttribute("gtype", ColGType.map.name());
					tempnode.setAttribute("type", ColType.javaBean.name());
					tempnode.setAttribute("className", classtype[2].getName());
				} else if (classtype[0].isArray()) {
					tempnode.setAttribute("gtype", ColGType.array.name());
					ColType colType = typestrToDynabean(classtype[1].getName());
					tempnode.setAttribute("type", colType.name());
				} else if ("java.util.Date".equals(classtype[0].getName())) {
					tempnode.setAttribute("type", ColType.datetime.name());
				} else {// 全部设置为javabean类型
					ColType colType = typestrToDynabean(classtype[0].getName());
					tempnode.setAttribute("type", colType.name());
					tempnode.setAttribute(ColProperty.className.name(), classtype[0].getName());
				}
				propertyInEle.appendChild(tempnode);
			}
			return docment;

		} catch (ProjectException e) {
			log.error("文档出错", e);
			return null;
		} catch (Exception e) {
			log.error("文档出错", e);
			return null;
		}
	}

	private static ColType typestrToDynabean(String name) {
		ColType retColType = ColType.javaBean;
		if (name.startsWith("java.lang")) {
			// ColType colType = ColType.string;// 默认是String
			if ("java.lang.Boolean".equals(name))
				retColType = ColType.string;
			else if ("java.lang.Byte".equals(name))
				retColType = ColType.bytes;
			else if ("java.lang.Character".equals(name))
				retColType = ColType.string;
			else if ("java.lang.Double".equals(name))
				retColType = ColType.doubler;
			else if ("java.lang.Float".equals(name))
				retColType = ColType.doubler;
			else if ("java.lang.Integer".equals(name))
				retColType = ColType.integer;
			else if ("java.lang.Long".equals(name))
				retColType = ColType.string;
			else if ("java.lang.Short".equals(name))
				retColType = ColType.integer;
			else
				retColType = ColType.string;
		}
		return retColType;
	}

	/***
	 * 写xml
	 * 
	 * @param document 要写的文档
	 * @param filepath 文件存储的路径
	 * @return 写入是否成功
	 */
	public static Result writeXmlFile(Document document, String filepath) {
		try {
			// 创建TransformerFactory对象
			TransformerFactory tff = TransformerFactory.newInstance();
			// 创建 Transformer对象
			Transformer tf = tff.newTransformer();
			// 输出内容是否使用换行
			tf.setOutputProperty(OutputKeys.INDENT, "yes");
			// 创建xml文件并写入内容
			String path = PathType.getPath(filepath, false);
			tf.transform(new DOMSource(document), new StreamResult(new File(path)));
			return Result.getSuc();
		} catch (Exception e) {
			log.error("写xml失败", e);
			return Result.getError(e.getMessage());
		}
	}

}
