/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.cassandra;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.Cluster.Builder;
import com.datastax.driver.core.CodecRegistry;
import com.datastax.driver.core.ConsistencyLevel;
import com.datastax.driver.core.HostDistance;
import com.datastax.driver.core.PoolingOptions;
import com.datastax.driver.core.QueryOptions;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.TypeCodec;
import com.datastax.driver.mapping.MappingManager;
import com.datastax.driver.mapping.Result;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.cassandra.bean.Columns;
import net.wicp.tams.common.cassandra.jdbc.CassandraData;
import net.wicp.tams.common.cassandra.jdbc.CassandraDatas;
import net.wicp.tams.common.cassandra.jdbc.OptType;
import net.wicp.tams.common.cassandra.typecodec.DateCodec;

@Slf4j
public abstract class CassandraAssit {
	private static final java.util.Map<String, Cluster> instmap = new java.util.HashMap<String, Cluster>();
	// 会话,单例，只要有一个session就可以了。session有池
	private static final java.util.Map<String, Session> sessionmap = new HashMap<String, Session>();

	private static final java.util.Map<String, MappingManager> mappingManagermap = new HashMap<String, MappingManager>();

	/***
	 * 数据源关闭
	 */
	public static void shutdown() {
		for (String poolname : instmap.keySet()) {
			((Cluster) instmap.get(poolname)).close();
		}
		instmap.clear();
	}

	private static final Cluster getCluster(String name) {
		if (instmap.get(name) == null) {
			synchronized (CassandraAssit.class) {
				if (instmap.get(name) == null) {
					Properties props = Conf.getPreToProp("common.cassandra.pool.default", true);
					if (!"default".equals(name)) {
						Map<String, String> map = Conf.getPre("common.cassandra.pool." + name, true);
						for (String key : map.keySet()) {
							props.put(key, map.get(key));
						}
					}
					Conf.overProp(props);
					Cluster cluster = initCluster(props);
					instmap.put(name, cluster);
					Session session = cluster.connect();
					sessionmap.put(name, session);
					mappingManagermap.put(name, new MappingManager(session));
				}
			}
		}
		return instmap.get(name);
	}

	public static void optDatas(CassandraDatas datas) {
		if (datas == null || datas.getDatasCount() == 0) {
			return;
		}
		for (CassandraData data : datas.getDatasList()) {
			switch (data.getOptType()) {
			case insert:
			case delete:
				String delsql = String.format("DELETE FROM  %s.%s WHERE %s=%s;", datas.getKs(), datas.getTb(),
						datas.getKey(),
						CassandrType.formatValue(CassandrType.get(datas.getTypeOrDefault(datas.getKey(), "text")),
								data.getValueMap().get(datas.getKey())));
				ResultSet rs = getSession().execute(delsql);
				System.out.println("rs=" + rs);
				if (data.getOptType() == OptType.delete) {
					break;
				}
			case update:
				Set<String> valueKeys = data.getValueMap().keySet();
				String[] cols = new String[valueKeys.size()];
				String[] values = new String[valueKeys.size()];
				int i = 0;
				for (String col : valueKeys) {
					int j = i++;
					cols[j] = col;
					values[j] = CassandrType.formatValue(CassandrType.get(datas.getTypeOrDefault(col, "text")),
							data.getValueMap().get(col));
				}
				String sql = String.format("INSERT INTO %s.%s (%s) VALUES (%s);", datas.getKs(), datas.getTb(),
						CollectionUtil.arrayJoin(cols, ","), CollectionUtil.arrayJoin(values, ","));
				ResultSet rs2 = getSession().execute(sql);
				System.out.println("rs=" + rs2);
			default:
				break;
			}
		}
	}

	public static List<Columns> queryCols(String ks, String table) {
		String sql = String.format("select * from system_schema.columns where  keyspace_name='%s' and table_name='%s';",
				ks, table);
		ResultSet rs = getSession().execute(sql);
		Result<Columns> map = getMappingManager().mapper(Columns.class).map(rs);
		return map.all();
	}

	public static void addCols(String ks, String table, Map<String, String> addCols) {
		// alter table key_space_1.stu add stu_age_1 text
		for (String key : addCols.keySet()) {
			String sql = String.format("alter table %s.%s add %s %s;", ks, table, key, addCols.get(key));
			ResultSet execute = getSession().execute(sql);
		}
	}

	public static Session getSessionNoConf(String name, Properties props) {
		if (sessionmap.containsKey(name)) {
			log.warn("已存在此配置,可以调用getDataSource拿到数据源");
			return sessionmap.get(name);
		}
		Properties propsDefault = Conf.getPreToProp("common.jdbc.datasource.default", true);// 拿到默认配置源
		for (Object key : props.keySet()) {
			propsDefault.put(String.valueOf(key), props.get(key));
		}
		Properties overprops = new Properties();
		for (Object keyobj : propsDefault.keySet()) {
			overprops.put(String.format("common.jdbc.datasource.%s.%s", name, keyobj), propsDefault.get(keyobj));
		}
		// 先放到内存中
		Conf.overProp(overprops);
		getCluster(name);
		return sessionmap.get(name);
	}

	public static boolean existSession(String name) {
		return sessionmap.containsKey(name);
	}

	public static final Session getSession() {
		getCluster("default");
		return sessionmap.get("default");
	}

	public static Session getSession(String name) {
		getCluster(name);
		return sessionmap.get(name);
	}

	public static final MappingManager getMappingManager(String name) {
		getCluster(name);
		return mappingManagermap.get(name);
	}

	public static final MappingManager getMappingManager() {
		return getMappingManager("default");
	}

	private static Cluster initCluster(Properties props) {
		QueryOptions queryOptions = new QueryOptions();
		queryOptions.setConsistencyLevel(ConsistencyLevel.valueOf(props.getProperty("query.consistencyLevel")));
		queryOptions.setFetchSize(Integer.parseInt(props.getProperty("query.fetchSize")));// 默认5,000，注意，并不会返回准确的数据，可能稍微多或少一些
		// 池
		PoolingOptions poolingOptions = new PoolingOptions();
		poolingOptions.setMaxRequestsPerConnection(HostDistance.LOCAL,
				Integer.parseInt(props.getProperty("pooling.maxRequestsPerConnection")));

		poolingOptions.setConnectionsPerHost(HostDistance.LOCAL,
				Integer.parseInt(props.getProperty("pooling.coreConnectionsPerHost")),
				Integer.parseInt(props.getProperty("pooling.maxConnectionsPerHost")));

		Builder builder = Cluster.builder().addContactPoint(props.getProperty("contactpoint"))
				.withPoolingOptions(poolingOptions).withPort(Integer.parseInt(props.getProperty("port")))
				.withQueryOptions(queryOptions);
        //注册日期转换器
		CodecRegistry codecRegistry = new CodecRegistry();
		codecRegistry.register(new DateCodec(TypeCodec.date(), Date.class));
		builder.withCodecRegistry(codecRegistry).build();

		if (StringUtil.isNotNull(props.getProperty("username"))
				&& StringUtil.isNotNull(props.getProperty("password"))) {
			builder.withCredentials(props.getProperty("username"), props.getProperty("password"));
		}
		Cluster cluster = builder.build();
		// Metadata metadata = cluster.getMetadata();
		// ConsistencyLevel lever =
		// cluster.getConfiguration().getQueryOptions().getConsistencyLevel();
		// System.out.println("ConsistencyLevel=" + lever);
		// System.out.printf("Connected to cluster: %s\n", metadata.getClusterName());
		// for (Host host : metadata.getAllHosts()) {
		// System.out.printf("Datacenter: %s; Host: %s; Rack: %s\n",
		// host.getDatacenter(), host.getAddress(),
		// host.getRack());
		// }
		return cluster;
	}
}
