package net.wicp.tams.common.binlog.alone.parser;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.LoggerUtil;
import net.wicp.tams.common.apiext.PwdUtil;
import net.wicp.tams.common.apiext.ReflectAssist;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.UUIDGenerator;
import net.wicp.tams.common.apiext.jdbc.MySqlAssit;
import net.wicp.tams.common.binlog.alone.ListenerConf.ColHis;
import net.wicp.tams.common.binlog.alone.ListenerConf.ConnConf;
import net.wicp.tams.common.binlog.alone.ListenerConf.DuckulaEvent;
import net.wicp.tams.common.binlog.alone.ListenerConf.DuckulaEventItem;
import net.wicp.tams.common.binlog.alone.PluginAssit;
import net.wicp.tams.common.binlog.alone.binlog.bean.Rule;
import net.wicp.tams.common.binlog.alone.binlog.bean.RuleManager;
import net.wicp.tams.common.binlog.alone.binlog.listener.IBinlogListener;
import net.wicp.tams.common.binlog.alone.binlog.listener.ISaveCheckPoint;
import net.wicp.tams.common.binlog.alone.constant.BuffType;
import net.wicp.tams.common.binlog.alone.constant.BusiAdapt;
import net.wicp.tams.common.binlog.parser.LogEvent;
import net.wicp.tams.common.binlog.parser.event.GtidLogEvent;
import net.wicp.tams.common.binlog.parser.event.QueryLogEvent;
import net.wicp.tams.common.binlog.parser.event.RowsLogBuffer;
import net.wicp.tams.common.binlog.parser.event.RowsLogEvent;
import net.wicp.tams.common.binlog.parser.event.TableMapLogEvent;
import net.wicp.tams.common.binlog.parser.event.TableMapLogEvent.ColumnInfo;
import net.wicp.tams.common.binlog.parser.event.XidLogEvent;
import net.wicp.tams.common.constant.DateFormatCase;
import net.wicp.tams.common.constant.JvmStatus;
import net.wicp.tams.common.constant.OptType;
import net.wicp.tams.common.constant.dic.YesOrNo;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.jdbc.DruidAssit;
import net.wicp.tams.common.jdbc.MySqlAssitExt;
import net.wicp.tams.common.jdbc.beans.AlterDbInfo;

@Slf4j
public abstract class BaseLogFetcher {
	protected String fileName = "mysql-bin.000001";
	protected Charset charset = Charset.forName("utf-8");
	protected String gtids;
	protected final ConnConf connConf;
	protected final RuleManager ruleManager;
	protected final BinlogMetricGroup metric;
	// private final IBinlogListener binlogListener;// 业务处理逻辑

	protected final BuffType buffType;

	/*
	 * public IBinlogListener getBinlogListener() { return binlogListener; }
	 */

	protected ISaveCheckPoint saveCheckPoint;
	// 存放cols的map
	protected final Map<String, List<ColHis>> colsMap = new HashMap<String, List<ColHis>>();

	protected BaseLogFetcher(ConnConf.Builder connConfBuilder) {
		Validate.notBlank(connConfBuilder.getConfName(), "需要配置文件名，如：default");
		Validate.notBlank(connConfBuilder.getHost(), "需要ip地址");
		Validate.notBlank(connConfBuilder.getUsername(), "需要用户名");
		Validate.notBlank(connConfBuilder.getPassword(), "需要密码");
		Validate.notBlank(connConfBuilder.getListener(), "需要处理的监听类");
		Validate.notBlank(connConfBuilder.getRule(), "需要监听规则");
		if (connConfBuilder.getPort() <= 0) {
			connConfBuilder.setPort(3306);
		}
		if (connConfBuilder.getClientId() <= 0) {
			connConfBuilder.setClientId(StringUtil.buildPort(new UUIDGenerator().generate()));
		}

		// 20200809 设置连接池属性
		Properties props = new Properties();
		props.put(String.format("common.jdbc.datasource.%s.host", connConfBuilder.getConfName()),
				connConfBuilder.getHost());
		props.put(String.format("common.jdbc.datasource.%s.port", connConfBuilder.getConfName()),
				connConfBuilder.getPort());
		props.put(String.format("common.jdbc.datasource.%s.username", connConfBuilder.getConfName()),
				connConfBuilder.getUsername());
		props.put(String.format("common.jdbc.datasource.%s.password", connConfBuilder.getConfName()),
				connConfBuilder.getPassword());
		Conf.overProp(props);

		this.ruleManager = new RuleManager(connConfBuilder.getRule());

		try {
			// net.wicp.tams.common.binlog.alone.checkpoint.CheckPointH2db
			saveCheckPoint = (ISaveCheckPoint) Class.forName(connConfBuilder.getChk()).newInstance();
			saveCheckPoint.init(connConfBuilder);
		} catch (Exception e1) {
			log.error("创建checkpoint实例失败", e1);
			throw new ProjectExceptionRuntime(ExceptAll.project_other, "创建checkpoint实例失败");
		}
		// 需要依赖saveCheckPoint
		init(connConfBuilder);
		// 先获得分布式锁
		YesOrNo acquireLock = saveCheckPoint.acquireLock();
		if (acquireLock == null || acquireLock == YesOrNo.no) {
			throw new ProjectExceptionRuntime(ExceptAll.project_other, "不能获得分布式锁");
		}

		this.connConf = connConfBuilder.build();

		this.metric = new BinlogMetricGroup(String.format("%s:%s", connConf.getHost(), connConf.getPort()));

		IBinlogListener binlogListener = null;
		Class<?> busierClass = null;
		try {
			ClassLoader classloadertrue = Conf.pluginClassLoader("common.binlog.alone.binlog.global.busiPluginDir");
			Class<?> binlogListenerClass = classloadertrue.loadClass(connConf.getListener());
			if (!ReflectAssist.isInterface(binlogListenerClass,
					"net.wicp.tams.common.binlog.alone.binlog.listener.IBinlogListener")) {
				log.error("Listener需要net.wicp.tams.common.binlog.alone.binlog.listener.IBinlogListener类型");
				throw new ProjectExceptionRuntime(ExceptAll.Param_typenofit,
						"Listener需要net.wicp.tams.common.binlog.alone.binlog.listener.IBinlogListener类型");
			}
			try {
				binlogListener = (IBinlogListener) ReflectAssist.newInst(binlogListenerClass);
				binlogListener.init(this.connConf);
			} catch (Exception e1) {
				log.error("Listener实例化失败", e1);
				throw new ProjectExceptionRuntime(ExceptAll.Param_typenofit, "Listener实例化失败");
			}
			if (StringUtil.isNotNull(connConf.getBusier())) {
				// eg:net.wicp.tams.common.binlog.alone.parser.test.TestBusiDoWith|2
				String[] ary = connConf.getBusier().split("\\|");
				// isAssignableFrom判断失败，多classload。去掉类型检查。
				busierClass = classloadertrue.loadClass(ary[0]);
				if (ary.length > 1) {// 有顺
					BusiAdapt.setCusClassName(busierClass, Integer.parseInt(ary[1]));
				} else {
					BusiAdapt.setCusClassName(busierClass);// 设置业务处理类，顺序先不处理。
				}
			}
		} catch (ClassNotFoundException e) {
			log.error("没有指定的Listener");
			LoggerUtil.exit(JvmStatus.s15);
		} catch (Throwable e) {
			log.error("加载listener错误", e);
			LoggerUtil.exit(JvmStatus.s15);
		}

		// 加速器
		String bufferstr = Conf.get("common.binlog.alone.binlog.global.bufferType");
		this.buffType = BuffType.get(StringUtil.trimSpace(bufferstr));
		this.buffType.getBinlogListenerProxy().putIBinlogListener(this.connConf.getHost(), binlogListener);
		this.buffType.getBinlogListenerProxy().putBusiDowithAdapt(connConf);
	}

	protected void parseQueryEvent(QueryLogEvent event) {
		String db = event.getDbName();
		String sql = event.getQuery().toLowerCase();
		if (sql.startsWith("alter")) {
			try {

				AlterDbInfo info = MySqlAssitExt.parseAlterSql(sql);
				Rule findRule = this.ruleManager.findRule(db, info.getTb());
				if (findRule == null) {
					return;
				}
				ColHis findCols = findCols(db, info.getTb(), event.getWhen());
				Result tableCallBack = this.buffType.getBinlogListenerProxy().getIBinlogListener(connConf.getHost())
						.doAlterTableCallBack(findRule, findCols, sql, info.getAddColNames(), info.getUpdateColName(),
								info.getDeleteColNames());
				if (tableCallBack != null && !tableCallBack.isSuc()) {
					String msg = "插件回调服务失败:" + tableCallBack.getMessage();
					log.error(msg);
					throw new RuntimeException(msg);
				}

				/*
				 * String key = String.format("%s|%s", db, tb).toLowerCase(); if
				 * (colsMap.containsKey(key)) { colsMap.get(key).add(findCols);
				 * log.warn("db:{},tb:{},cols is altered", db, tb); } else { List<ColHis>
				 * templist = new ArrayList<ColHis>(); templist.add(findCols);// 最开始那个col为默认值
				 * colsMap.put(key, templist); log.info("db:{},tb:{},cols add in map", db, tb);
				 * }
				 */
			} catch (Exception e) {
				log.error("get tb from sql error", e);
			}
		}
	}

	protected void parseGtidLogEvent(GtidLogEvent event) throws Exception {
		this.gtids = event.getGtid();
		parseGtidLogEventSub(event);
	}

	protected abstract void parseGtidLogEventSub(GtidLogEvent event);// gtid事件子实现

	/****
	 * 初始化一些变量
	 * 
	 * @param connConfBuilder
	 */
	protected abstract void init(ConnConf.Builder connConfBuilder);

	// 关闭相关资源
	public abstract void close();

	/***
	 * 开始读binlog
	 */
	public abstract void read();

	protected long xid;

	// gtid事件结束
	protected void parseXidEvent(XidLogEvent event) {
		this.xid = event.getXid();
	}

	protected boolean isChkDb = false;

	protected boolean parseRowsEvent(RowsLogEvent event, OptType optType) {
		if (log.isDebugEnabled()) {
			Date d = new Date(event.getHeader().getWhen() * 1000);
			String datestr = DateFormatCase.YYYY_MM_DD_hhmmss.getInstanc().format(d);
			log.debug("db:{},tb:{},time:{}", event.getTable().getDbName(), event.getTable().getTableName(), datestr);
		}
		Rule findRule = this.ruleManager.findRule(event.getTable().getDbName(), event.getTable().getTableName());
		if (findRule == null) {// 没有匹配规则，直接跳过
			return false;
		}
		metric.meter_parser_pack_row.mark();
		ColHis colhis = ValidKey(event.getTable().getDbName(), event.getTable().getTableName(), event.getWhen());
		List<String[]> rowListBefore = new ArrayList<>();
		List<String[]> rowListAfter = new ArrayList<>();
		int rows = 0;
		try {
			RowsLogBuffer buffer = event.getRowsBuf(charset.name());
			BitSet columns = event.getColumns();
			BitSet changeColumns = event.getChangeColumns();
			while (buffer.nextOneRow(columns)) {
				// 处理row记录
				int type = event.getHeader().getType();
				if (LogEvent.WRITE_ROWS_EVENT_V1 == type || LogEvent.WRITE_ROWS_EVENT == type) {
					// insert的记录放在before字段中
					Pair<String[], Boolean> parseOneRow = parseOneRow(event, buffer, columns, colhis.getColsCount(),
							true);
					rowListAfter.add(parseOneRow.getLeft());
					if (parseOneRow.getRight()) {
						colhis = ValidKey(event.getTable().getDbName(), event.getTable().getTableName(),
								event.getWhen());
					}
				} else if (LogEvent.DELETE_ROWS_EVENT_V1 == type || LogEvent.DELETE_ROWS_EVENT == type) {
					// delete的记录放在before字段中
					Pair<String[], Boolean> parseOneRow = parseOneRow(event, buffer, columns, colhis.getColsCount(),
							false);
					rowListBefore.add(parseOneRow.getLeft());
					if (parseOneRow.getRight()) {
						colhis = ValidKey(event.getTable().getDbName(), event.getTable().getTableName(),
								event.getWhen());
					}
				} else {
					// update需要处理before/after
					// System.out.println("-------> before");
					Pair<String[], Boolean> parseOneRow = parseOneRow(event, buffer, columns, colhis.getColsCount(),
							true);
					rowListBefore.add(parseOneRow.getLeft());
					if (parseOneRow.getRight()) {
						colhis = ValidKey(event.getTable().getDbName(), event.getTable().getTableName(),
								event.getWhen());
					}
					if (!buffer.nextOneRow(changeColumns)) {
						break;
					}
					// System.out.println("-------> after");
					rowListAfter.add(parseOneRow(event, buffer, changeColumns, colhis.getColsCount(), true).getLeft());
				}
				rows++;
			}
		} catch (Exception e) {
			throw new RuntimeException("parse row data failed.", e);
		}

		String[][] afterArray = new String[rowListAfter.size()][];
		for (int i = 0; i < rowListAfter.size(); i++) {
			afterArray[i] = rowListAfter.get(i);
		}

		String[][] beforeArray = new String[rowListBefore.size()][];
		for (int i = 0; i < rowListBefore.size(); i++) {
			beforeArray[i] = rowListBefore.get(i);
		}

		DuckulaEvent.Builder duckulaEventBuilder = DuckulaEvent.newBuilder();
		duckulaEventBuilder.setColNum(ArrayUtils.isNotEmpty(afterArray) ? afterArray[0].length : beforeArray[0].length);

		duckulaEventBuilder.addAllCols(colhis.getColsList());
		ColumnInfo[] colsColumnInfo = event.getTable().getColumnInfo();
		for (int i = 0; i < colsColumnInfo.length; i++) {
			duckulaEventBuilder.addColsTypeValue(colsColumnInfo[i].type);
		}
		duckulaEventBuilder.setDb(event.getTable().getDbName());
		duckulaEventBuilder.setTb(event.getTable().getTableName());
		duckulaEventBuilder.setGtid(this.gtids);
		duckulaEventBuilder.setIsError(false);
		duckulaEventBuilder
				.setOptType(net.wicp.tams.common.binlog.alone.ListenerConf.OptType.forNumber(optType.getValue()));
		duckulaEventBuilder.setCommitTime(event.getWhen() * 1000);
		// DuckulaEvent[] retobjs = new DuckulaEvent[rows];

		for (int i = 0; i < rows; i++) {
			net.wicp.tams.common.binlog.alone.ListenerConf.DuckulaEventItem.Builder newBuilder = DuckulaEventItem
					.newBuilder();
			initData(newBuilder, rowListBefore, colhis.getColsList(), i, false);
			initData(newBuilder, rowListAfter, colhis.getColsList(), i, true);
			duckulaEventBuilder.addItems(newBuilder);
		}
		this.buffType.getBinlogListenerProxy().sendmsg(this.connConf.getHost(), findRule, duckulaEventBuilder);// 一批批发送
		metric.meter_parser_event.mark(rows);
		metric.counter_ringbuff_pack.inc();// 包数
		metric.counter_ringbuff_event.inc(rows);// 记录数
		return true;
	}

	/***
	 * 
	 * @param event
	 * @param buffer
	 * @param cols
	 * @param colNum
	 * @param isAfter
	 * @return String[]:反回的结果 Boolean:是否要重新拉ColName
	 * @throws UnsupportedEncodingException
	 */
	protected Pair<String[], Boolean> parseOneRow(RowsLogEvent event, RowsLogBuffer buffer, BitSet cols, int colNum,
			boolean isAfter) throws UnsupportedEncodingException {
		TableMapLogEvent map = event.getTable();
		if (map == null) {
			throw new RuntimeException("not found TableMap with tid=" + event.getTableId());
		}
		boolean needReloadColName = false;
		if (colNum != event.getTable().getColumnCnt()) {// 如果中途有修改字段，没有监听到，会发生这种情况，
			// ColHis colhis是传值的，不是传引用的，所以不能修改他的值
			ColHis colhis = findCols(event.getTable().getDbName(), event.getTable().getTableName(), event.getWhen());// 补列字段，可能会导致相同列出现多个时间的问题，不影响结果
			if (colhis.getColsCount() != event.getTable().getColumnCnt()) {
				throw new RuntimeException("TableMap:" + event.getTableId() + " the colsname is:"
						+ colhis.getColsCount() + " the value size is:" + event.getTable().getColumnCnt());
			} else {
				String key = String.format("%s|%s", event.getTable().getDbName(), event.getTable().getTableName())
						.toLowerCase();
				colsMap.remove(key);// 去除缓存，下次就可以再查数据库而命中。
				needReloadColName = true;
			}
		}
		String[] values = new String[event.getTable().getColumnCnt()];
		final int columnCnt = map.getColumnCnt();
		final ColumnInfo[] columnInfo = map.getColumnInfo();
		for (int i = 0; i < columnCnt; i++) {
			if (!cols.get(i)) {
				continue;
			}
			ColumnInfo info = columnInfo[i];
			buffer.nextValue(info.type, info.meta);

			if (buffer.isNull()) {
				//
			} else {
				final Serializable value = buffer.getValue();
				if (value instanceof byte[]) {
					values[i] = PwdUtil.base64FromBin((byte[]) value);// new
																		// String((byte[])
																		// value);
				} else {
					values[i] = String.valueOf(value);
				}
			}
		}
		return Pair.of(values, needReloadColName);
	}

	public ColHis findCols(String db, String tb, long time) {
		String key = String.format("%s|%s", db, tb).toLowerCase();
		java.sql.Connection conn = null;
		try {
			conn = DruidAssit.getConnection(connConf.getConfName());
			List<Triple<String, String, String>> colList = MySqlAssit.getColsNew(conn, db, tb,
					connConf.getRds() ? YesOrNo.yes : YesOrNo.no);

			// String[] primary = MySqlAssit.getPrimary(conn, db, tb);
			ColHis retobj = PluginAssit.convertCosHis(connConf.getHost(), db, tb, time, colList);

			// 保存
			saveCheckPoint.saveColName(retobj);
			// 放内存
			List<ColHis> findColsList = saveCheckPoint.findColsList(db, tb);
			// 预防没有排序
			Collections.sort(findColsList, new Comparator<ColHis>() {
				@Override
				public int compare(ColHis o1, ColHis o2) {
					long def = o2.getTime() - o1.getTime();
					return def > 0 ? 1 : (def < 0 ? -1 : 0);
				}
			});
			colsMap.put(key, findColsList);
			return retobj;
		} catch (Exception e) {
			log.error("获取cols错误", e);
			throw new RuntimeException("获取cols错误");
		} finally {
			if (conn != null) {
				try {
					conn.close();
				} catch (SQLException e) {
					log.error("关闭连接失败", e);
				}
			}
		}
	}

	/***
	 * 验证col是否正确，如果map没有代表是新的，需要读数据库获得
	 * 
	 * @param db
	 * @param tb
	 * @param time
	 * @return
	 */
	public ColHis ValidKey(String db, String tb, long time) {
		String key = String.format("%s|%s", db, tb).toLowerCase();
		if (colsMap.containsKey(key)) {
			List<ColHis> list = colsMap.get(key);
			ColHis selFitColName = selFitColName(time, list);
			if (selFitColName == null) {
				throw new RuntimeException("没有可用的col信息。");// 由于第1条时间设置为-1,所以它一般不会被执行
			}
			return selFitColName;
		} else {
			List<ColHis> findColsList = saveCheckPoint.findColsList(db, tb);// 需要排好序
			if (CollectionUtils.isEmpty(findColsList)) {
				ColHis retobj = findCols(db, tb, -1);// 没有列名就去查一次
				List<ColHis> retlist = new ArrayList<>();
				retlist.add(retobj);
				colsMap.put(key, retlist);
				return retobj;
			} else {
				// 预防没有排序
				Collections.sort(findColsList, new Comparator<ColHis>() {
					@Override
					public int compare(ColHis o1, ColHis o2) {
						long def = o2.getTime() - o1.getTime();
						return def > 0 ? 1 : (def < 0 ? -1 : 0);
					}
				});
				colsMap.put(key, findColsList);
				ColHis selFitColName = selFitColName(time, findColsList);
				if (selFitColName == null) {
					throw new RuntimeException("没有可用的col信息。");// 由于第1条时间设置为-1,所以它一般不会被执行
				}
				return selFitColName;
			}

		}
	}

	private ColHis selFitColName(long time, List<ColHis> list) {
		for (int i = 0; i < list.size(); i++) {
			ColHis colHis = list.get(i);
			if (colHis.getTime() <= time) {// 如果先修改表结果，再启动监听，这里会把colname的时候设置为第一条记录的生成时间，所以用等号
				return colHis;
			}
		}
		return null;
	}

	private void initData(DuckulaEventItem.Builder rowbuilder, List<String[]> datas, List<String> cols, int rowNo,
			boolean isAfter) {
		if (CollectionUtils.isEmpty(datas)) {
			return;
		}
		String[] tempary = datas.get(rowNo);
		for (int i = 0; i < tempary.length; i++) {
			if (tempary[i] != null) {
				if (isAfter) {
					rowbuilder.putAfter(cols.get(i), tempary[i]);
				} else {
					rowbuilder.putBefore(cols.get(i), tempary[i]);
				}
			}
		}
	}

	public BuffType getBuffType() {
		return buffType;
	}
}
