/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.binlog.alone.binlog.bean;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.json.EasyUiAssist;
import net.wicp.tams.common.apiext.json.easyuibean.EasyUINode;
import net.wicp.tams.common.apiext.json.easyuibean.EasyUINodeConf;
import net.wicp.tams.common.binlog.alone.constant.FilterPattern;
import net.wicp.tams.common.constant.DrdsPattern;
import net.wicp.tams.common.constant.StrPattern;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

@Data
@Slf4j
public class RuleManager {

	private final List<Rule> rules;


	private Pattern filterDbPattern;


	private Pattern filterTbPattern;

	public void setFilterDbPattern(String filterDb) {
		this.filterDbPattern = Pattern.compile(filterDb);
	}

	public void setFilterTbPattern(String filterTb) {
		this.filterTbPattern = Pattern.compile(filterTb);
	}

	public RuleManager(String rulesstr) {
		this.rules = buildRules(rulesstr);
		// 处理parent的关系
		for (Rule rule : this.rules) {
			if (rule.getItems().containsKey(RuleItem.parent)
					&& StringUtil.isNotNull(rule.getItems().get(RuleItem.parent))) {
				String[] dbtbary = StringUtil.trimSpace(rule.getItems().get(RuleItem.parent)).split(":");
				Rule parent = this.findRuleByPattern(dbtbary[0], dbtbary[1]);
				if (parent == null || parent.equals(rule)) {// 自己是自己的父，那么这个节点就是顶级节点
					rule.setParent(null);
				} else {
					rule.setParent(parent);
					parent.getSubRules().add(rule);
				}
			}
		}
	}

	// TODO RuleManagerg与RuleManager(String rulesstr)合并
	public RuleManager(JSONArray rulesJsonAry) {
		this.rules = new ArrayList<Rule>();
		if (rulesJsonAry == null || rulesJsonAry.size() == 0) {
			return;
		}
		for (int i = 0; i < rulesJsonAry.size(); i++) {
			Rule rule = new Rule();
			JSONObject tempObj = rulesJsonAry.getJSONObject(i);
			String drds = tempObj.containsKey("drds") ? StringUtil.trimSpace(tempObj.getString("drds")) : "";
			String dbPattern = tempObj.containsKey("dbPattern") ? StringUtil.trimSpace(tempObj.getString("dbPattern"))
					: "";
			String tbPattern = tempObj.containsKey("tbPattern") ? StringUtil.trimSpace(tempObj.getString("tbPattern"))
					: "";
			rule.setDbPattern(dbPattern);
			rule.setTbPattern(tbPattern);
			rule.setDrds(DrdsPattern.find(drds));
			for (RuleItem ruleItem : RuleItem.values()) {
				if (tempObj.containsKey(ruleItem.name())) {
					String value = tempObj.getString(ruleItem.name());
					if (StringUtil.isNotNull(value)) {
						rule.getItems().put(ruleItem, value);
					}
				}
			}
			// 20201226 处理过滤规则,它会是一个数组
			if (tempObj.containsKey("filter")) {
				try {
					JSONArray parseArray = tempObj.getJSONArray("filter");
					rule.putRuleFilter(parseArray);
				} catch (Exception e) {
					throw new ProjectExceptionRuntime(ExceptAll.project_undefined, "过滤规则错误，请从“过滤器”按钮进行编辑。");
				}
			}
			this.rules.add(rule);
		}
		// 处理parent的关系
		for (Rule rule : this.rules) {
			if (rule.getItems().containsKey(RuleItem.parent)
					&& StringUtil.isNotNull(rule.getItems().get(RuleItem.parent))) {
				String[] dbtbary = StringUtil.trimSpace(rule.getItems().get(RuleItem.parent)).split(":");
				Rule parent = this.findRuleByPattern(dbtbary[0], dbtbary[1]);
				if (parent == null || parent.equals(rule)) {// 自己是自己的父，那么这个节点就是顶级节点
					rule.setParent(null);
				} else {
					rule.setParent(parent);
					parent.getSubRules().add(rule);
				}
			}
		}
	}

	private List<Rule> buildRules(String rules) {
		List<Rule> ruleList = new ArrayList<>();
		if (StringUtil.isNull(rules)) {
			return ruleList;
		}
		String[] ruleAry = rules.split("&");
		List<String[]> ruleDoAfter = new ArrayList<String[]>();
		for (String rule : ruleAry) {// 把|模式打成 & 的分组模式
			// 只需取5个就可以了，多了不需要
			String[] ruleValues = rule.split("`", 5);
			if (ruleValues.length == 0 || ruleValues.length != 5) {
				throw new IllegalArgumentException("规则长度只能为5!");
			}
			if (ruleValues[0].equals("^*$") || ruleValues[1].equals("^*$")) {
				ruleDoAfter.add(ruleValues);
			} else {
				String[] dbs = ruleValues[0].split("\\|");
				String[] tbs = ruleValues[1].split("\\|");
				for (int i = 0; i < dbs.length; i++) {
					for (int j = 0; j < tbs.length; j++) {
						String[] tempAry = ruleValues.clone();
						tempAry[0] = dbs[i].replaceAll("^", "").replaceAll("$", "");
						tempAry[1] = tbs[j].replaceAll("^", "").replaceAll("$", "");
						ruleDoAfter.add(tempAry);
					}
				}
			}
		}
		for (String[] ruleValues : ruleDoAfter) {
			Rule rule = new Rule();
			String itemStr = null;
			String filterStr = null;
			if (StringUtil.isNotNull(ruleValues[2])) {
				rule.setDrds(DrdsPattern.find(ruleValues[2]));
			}
			itemStr = ruleValues[3];
			if (ruleValues.length == 5) {
				filterStr = ruleValues[4];
			}
			if (rule.getDrds() == null) {// 设置默认值，兼容之前没有设置drds值的情况
				rule.setDrds(DrdsPattern.no);
			}
			if (DrdsPattern.no != rule.getDrds()) {
				rule.setDbPattern(String.format(rule.getDrds().getDbPatternFormat(), ruleValues[0]));// drds支持
			} else {
				rule.setDbPattern(StringUtil.buildRule(ruleValues[0]));
			}
			rule.setDbLength(ruleValues[0].length());

			String tbpatternstr = StringUtil.buildRule(ruleValues[1]);
			if (DrdsPattern.no != rule.getDrds()) {
				tbpatternstr = String.format(rule.getDrds().getTbPatternFormat(), ruleValues[1]);
			}
			rule.setTbPattern(tbpatternstr);
			rule.setTbLength(ruleValues[1].length());

			JSONObject json = JSON.parseObject(itemStr);
			for (String key : json.keySet()) {
				RuleItem tempItem = RuleItem.get(key);
				if (tempItem == null) {
					log.error("规则设置出错，请检查key与发送者!");
					throw new IllegalArgumentException("规则设置出错，请检查key与发送者!");
				} else {
					rule.getItems().put(tempItem, json.getString(key));
				}
			}

			// 20201226 处理过滤规则,它会是一个数组
			if (StringUtil.isNotNull(filterStr)) {
				JSONArray parseArray = JSON.parseArray(filterStr);
				for (int j = 0; j < parseArray.size(); j++) {
					JSONObject jsonObject = parseArray.getJSONObject(j);
					RuleFilter temp = new RuleFilter();
					if (jsonObject.containsKey("field")) {
						temp.setField(jsonObject.getString("field"));
					}
					if (!jsonObject.containsKey("rule") || !jsonObject.containsKey("ruleValue")) {
						log.error("过滤规则设置出错，需要设置rule和ruleValue!" + jsonObject.toJSONString());
						throw new IllegalArgumentException("过滤规则设置出错，需要设置rule和ruleValue!" + jsonObject.toJSONString());
					}
					temp.setFilterPattern(FilterPattern.valueOf(jsonObject.getString("rule")));
					temp.setRuleValue(jsonObject.getString("ruleValue"));
					rule.getFilterRules().add(temp);
				}
			}

			ruleList.add(rule);
		}
		return ruleList;
	}

	/***
	 * 找到表的匹配规则
	 *
	 * @param db
	 * @param tb
	 * @return
	 */
	public Rule findRule(String db, String tb) {
		if (db.equals(Conf.get("common.binlog.alone.binlog.global.chk.mysql.defaultdb"))) {// 去除mysql数据库"^.*$".equals(connConf.getDbPattern())，它会存在
			// ha_health_check 心跳表
			return null;
		}
		// db全局过滤
		if (this.filterDbPattern != null) {
			if (StrPattern.checkStrFormat(this.filterDbPattern, db)) {
				return null;
			}
		}
		// tb全局过滤
		if (this.filterTbPattern != null) {
			if (StrPattern.checkStrFormat(this.filterTbPattern, tb)) {
				return null;
			}
		}
		for (Rule rule : this.getRules()) {
			if (!"^*$".equals(rule.getDbPattern())) {
				boolean retdb = StrPattern.checkStrFormat(rule.getDbPattern(), db);
				if (!retdb) {
					continue;
				}
			}
			if (!"^*$".equals(rule.getTbPattern())) {
				boolean rettb = StrPattern.checkStrFormat(rule.getTbPattern(), tb);
				if (!rettb) {
					continue;
				}
			}
			return rule;
		}
		return null;
	}

	/**
	 * 通过库模式和表模式来找到rule
	 * 
	 * @param dbPattern
	 * @param tbPattern
	 * @return
	 */
	public Rule findRuleByPattern(String dbPattern, String tbPattern) {
		Rule tempobj = new Rule();
		tempobj.setDbPattern(dbPattern);
		tempobj.setTbPattern(tbPattern);
		int indexOf = this.rules.indexOf(tempobj);
		return indexOf < 0 ? null : this.rules.get(indexOf);
	}

	/***
	 * 得到rule的树
	 * 
	 * @return
	 */
	public List<EasyUINode> getRuleTreeRoot() {
		try {
			EasyUINodeConf conf = new EasyUINodeConf("id", "text", "parent");
			List<EasyUINode> roots = EasyUiAssist.getTreeRoot(this.rules, conf);
			return roots;
		} catch (Exception e) {
			throw new ProjectExceptionRuntime(ExceptAll.param_error, "得到rule的树错误");
		}
	}

	// 把rule转为String方便存储，有可能由于特殊原因会做些规整操作
	public String toString() {
		StringBuffer buff = new StringBuffer();
		for (int i = 0; i < this.rules.size(); i++) {
			buff.append("&");
			Rule rule = this.rules.get(i);
			buff.append(rule.getDbOri() + "`");
			buff.append(rule.getTbOri() + "`");
			buff.append(rule.getDrds().name() + "`");
			JSONObject jsontrue = rule.buildRuleItem();
			buff.append(JSONObject.toJSONString(jsontrue, SerializerFeature.UseSingleQuotes));
			buff.append("`");
			JSONArray ruleFilters = rule.buildRuleFilter();
			buff.append(ruleFilters.size() == 0 ? "" : ruleFilters.toString());
		}
		String retstr = buff.length() > 1 ? buff.substring(1) : "";
		return retstr;
	}

	// 添加新的规则
	public void addRule(Rule rule) {
		if (!this.rules.contains(rule)) {
			this.rules.add(rule);
		}
	}

	public JSONArray toJsonAry() {
		JSONArray retAry = new JSONArray();
		for (Rule rule : this.rules) {
			JSONObject jsontrue = rule.buildRule();
			retAry.add(jsontrue);
		}
		return retAry;
	}

	public boolean isValid(String db, String tb) {
		return findRule(db, tb) != null;
	}
}
