package net.wicp.tams.common.binlog.alone.binlog.bean;

import java.util.ArrayList;
import java.util.List;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.binlog.alone.constant.FilterPattern;
import net.wicp.tams.common.constant.StrPattern;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

@Data
@Slf4j
public class RuleManager {

	private final List<Rule> rules;

	public RuleManager(String rulesstr) {
		this.rules = buildRules(rulesstr);
	}

	public RuleManager(JSONArray rulesJsonAry) {
		this.rules = new ArrayList<Rule>();
		if (rulesJsonAry == null || rulesJsonAry.size() == 0) {
			return;
		}
		for (int i = 0; i < rulesJsonAry.size(); i++) {
			Rule rule = new Rule();
			JSONObject tempObj = rulesJsonAry.getJSONObject(i);
			String drds = tempObj.containsKey("drds") ? StringUtil.trimSpace(tempObj.getString("drds")) : "";
			String dbPattern = tempObj.containsKey("dbPattern") ? StringUtil.trimSpace(tempObj.getString("dbPattern"))
					: "";
			String tbPattern = tempObj.containsKey("tbPattern") ? StringUtil.trimSpace(tempObj.getString("tbPattern"))
					: "";
			rule.setDbPattern(dbPattern);
			rule.setTbPattern(tbPattern);
			rule.setDrds(drds);
			for (RuleItem ruleItem : RuleItem.values()) {
				if (tempObj.containsKey(ruleItem.name())) {
					String value = tempObj.getString(ruleItem.name());
					if (StringUtil.isNotNull(value)) {
						rule.getItems().put(ruleItem, value);
					}
				}
			}
			// 20201226 处理过滤规则,它会是一个数组
			if (tempObj.containsKey("filter")) {
				try {
					JSONArray parseArray = tempObj.getJSONArray("filter");
					rule.putRuleFilter(parseArray);
				} catch (Exception e) {
					throw new ProjectExceptionRuntime(ExceptAll.project_undefined,"过滤规则错误，请从“过滤器”按钮进行编辑。");
				}
			}
			this.rules.add(rule);
		}
	}

	private List<Rule> buildRules(String rules) {
		List<Rule> ruleList = new ArrayList<>();
		if (StringUtil.isNull(rules)) {
			return ruleList;
		}
		String[] ruleAry = rules.split("&");
		for (int i = 0; i < ruleAry.length; i++) {
			// 只需取5个就可以了，多了不需要
			String[] ruleValues = ruleAry[i].split("`", 5);
			if (ruleValues.length == 0 || ruleValues.length != 5) {
				throw new IllegalArgumentException("规则长度只能为5!");
			}
			Rule rule = new Rule();

			String itemStr = null;
			String filterStr = null;
			boolean isdrds = false;

			if (StringUtil.isNotNull(ruleValues[2])) {
				rule.setDrds(ruleValues[2]);
				isdrds = true;
			}
			itemStr = ruleValues[3];
			if (ruleValues.length == 5) {
				filterStr = ruleValues[4];
			}

			if (isdrds) {
				rule.setDbPattern(String.format(Rule.drdsTbPatternFormat2, ruleValues[0]));// drds支持
			} else {
				rule.setDbPattern(buildPatter(ruleValues[0]));
			}
			rule.setDbLength(ruleValues[0].length());

			String tbpatternstr = buildPatter(ruleValues[1]);
			if (isdrds) {
				if ("dbtb".equals(rule.getDrds())) {
					tbpatternstr = String.format(Rule.drdsTbPatternFormat2, ruleValues[1]);// drds支持
				} else if ("db".equals(rule.getDrds())) {
					tbpatternstr = String.format(Rule.drdsTbPatternFormat1, ruleValues[1]);// drds支持
				} else if ("no".equals(rule.getDrds())) {
					tbpatternstr = ruleValues[1];
				}
			}
			rule.setTbPattern(tbpatternstr);
			rule.setTbLength(ruleValues[1].length());

			JSONObject json = JSON.parseObject(itemStr);
			for (String key : json.keySet()) {
				RuleItem tempItem = RuleItem.get(key);
				if (tempItem == null) {
					log.error("规则设置出错，请检查key与发送者!");
					throw new IllegalArgumentException("规则设置出错，请检查key与发送者!");
				} else {
					rule.getItems().put(tempItem, json.getString(key));
				}
			}

			// 20201226 处理过滤规则,它会是一个数组
			if (StringUtil.isNotNull(filterStr)) {
				JSONArray parseArray = JSON.parseArray(filterStr);
				for (int j = 0; j < parseArray.size(); j++) {
					JSONObject jsonObject = parseArray.getJSONObject(j);
					RuleFilter temp = new RuleFilter();
					if (jsonObject.containsKey("field")) {
						temp.setField(jsonObject.getString("field"));
					}
					if (!jsonObject.containsKey("rule") || !jsonObject.containsKey("ruleValue")) {
						log.error("过滤规则设置出错，需要设置rule和ruleValue!" + jsonObject.toJSONString());
						throw new IllegalArgumentException("过滤规则设置出错，需要设置rule和ruleValue!" + jsonObject.toJSONString());
					}
					temp.setFilterPattern(FilterPattern.valueOf(jsonObject.getString("rule")));
					temp.setRuleValue(jsonObject.getString("ruleValue"));
					rule.getFilterRules().add(temp);
				}
			}

			ruleList.add(rule);
		}
		return ruleList;
	}

	private String buildPatter(String patter) {
		if (patter.endsWith("_")) {
			return String.format("^%s[0-9]*$", patter);
		} else {
			return String.format("^%s$", patter);
		}
	}

	/***
	 * 找到表的匹配规则
	 *
	 * @param db
	 * @param tb
	 * @return
	 */
	public Rule findRule(String db, String tb) {
		if (db.equals(Conf.get("common.binlog.alone.binlog.global.chk.mysql.defaultdb"))) {// 去除mysql数据库"^.*$".equals(connConf.getDbPattern())，它会存在
			// ha_health_check 心跳表
			return null;
		}
		for (Rule rule : this.getRules()) {
			if (!"^*$".equals(rule.getDbPattern())) {
				boolean retdb = StrPattern.checkStrFormat(rule.getDbPattern(), db);
				if (!retdb) {
					continue;
				}
			}
			if (!"^*$".equals(rule.getTbPattern())) {
				boolean rettb = StrPattern.checkStrFormat(rule.getTbPattern(), tb);
				if (!rettb) {
					continue;
				}
			}
			return rule;
		}
		return null;
	}

	// 把rule转为String方便存储，有可能由于特殊原因会做些规整操作
	public String toString() {
		StringBuffer buff = new StringBuffer();
		for (int i = 0; i < this.rules.size(); i++) {
			buff.append("&");
			Rule rule = this.rules.get(i);
			buff.append(rule.getDbOri() + "`");
			buff.append(rule.getTbOri() + "`");
			buff.append(StringUtil.hasNull(rule.getDrds()) + "`");
			JSONObject jsontrue = rule.buildRuleItem();
			buff.append(JSONObject.toJSONString(jsontrue, SerializerFeature.UseSingleQuotes));
			buff.append("`");
			JSONArray ruleFilters = rule.buildRuleFilter();
			buff.append(ruleFilters.size() == 0 ? "" : ruleFilters.toString());
		}
		String retstr = buff.length() > 1 ? buff.substring(1) : "";
		return retstr;
	}

	public JSONArray toJsonAry() {
		JSONArray retAry = new JSONArray();
		for (Rule rule : this.rules) {
			JSONObject jsontrue = rule.buildRule();
			retAry.add(jsontrue);
		}
		return retAry;
	}

	public boolean isValid(String db, String tb) {
		return findRule(db, tb) != null;
	}
}
