/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.aws.sqs;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.amazonaws.AmazonClientException;
import com.amazonaws.internal.SdkInternalMap;
import com.amazonaws.services.sqs.AmazonSQS;
import com.amazonaws.services.sqs.AmazonSQSClient;
import com.amazonaws.services.sqs.model.AmazonSQSException;
import com.amazonaws.services.sqs.model.DeleteMessageRequest;
import com.amazonaws.services.sqs.model.DeleteMessageResult;
import com.amazonaws.services.sqs.model.GetQueueAttributesRequest;
import com.amazonaws.services.sqs.model.GetQueueAttributesResult;
import com.amazonaws.services.sqs.model.ListQueuesRequest;
import com.amazonaws.services.sqs.model.ListQueuesResult;
import com.amazonaws.services.sqs.model.Message;
import com.amazonaws.services.sqs.model.MessageAttributeValue;
import com.amazonaws.services.sqs.model.QueueAttributeName;
import com.amazonaws.services.sqs.model.QueueDoesNotExistException;
import com.amazonaws.services.sqs.model.ReceiveMessageRequest;
import com.amazonaws.services.sqs.model.ReceiveMessageResult;
import com.amazonaws.services.sqs.model.SendMessageRequest;
import com.amazonaws.services.sqs.model.SendMessageResult;
import com.amazonaws.services.sqs.model.SetQueueAttributesRequest;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.PwdUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.TimeAssist;
import net.wicp.tams.common.apiext.json.JSONUtil;
import net.wicp.tams.common.aws.DataType;
import net.wicp.tams.common.aws.SqsData;
import net.wicp.tams.common.aws.common.AwsHelper;
import net.wicp.tams.common.aws.common.CountingOutputStream;
import net.wicp.tams.common.aws.common.TamsCredentialsProvider;
import net.wicp.tams.common.aws.s3.S3Service;
import net.wicp.tams.common.aws.s3.threadlocal.S3ClientThreadlocal;
import net.wicp.tams.common.aws.sqs.bean.MessageCommon;
import net.wicp.tams.common.aws.sqs.bean.QueueAttributes;
import net.wicp.tams.common.aws.sqs.bean.QueueAttributes.QueueAttributesBuilder;
import net.wicp.tams.common.aws.sqs.constant.QueryType;
import net.wicp.tams.common.aws.sqs.listener.AbsSQSListener;
import net.wicp.tams.common.aws.sqs.threadlocal.SqsClientThreadlocal;
import net.wicp.tams.common.aws.sqs.threadlocal.SqsContext;
import net.wicp.tams.common.thread.ThreadPool;
import net.wicp.tams.common.thread.threadlocal.PerThreadValue;
import net.wicp.tams.common.thread.threadlocal.PerthreadManager;

@Slf4j
public class SqsService {
	private final AmazonSQS sqs;

	private final static String version = "v2";// 版本，为了向下兼容
	private final static String versionJar = "1.2.18";// jar包版本

	public AmazonSQS getSqs() {
		return sqs;
	}

	public void close() {
		// TODO 占位，暂留
	}

	private SqsService(AmazonSQS sqs) {
		this.sqs = sqs;
		init();
	}

	private SqsService() {
		final AmazonSQS amazonSQS = AmazonSQSClient.builder().withRegion(Conf.get("common.aws.region"))
				.withCredentials(new TamsCredentialsProvider()).build();
		this.sqs = amazonSQS;
		init();
	}

	// 默认使用threadLocal方式
	public static SqsService getInstThread() {
		SqsService createPerThreadEsClient = SqsClientThreadlocal.createPerThreadSqsClient();
		return createPerThreadEsClient;
	}

	private volatile static boolean isInit;

	public static final Map<String, Long> doWithTime = new HashMap<>();

	/****
	 * 初始化，只能执行一次
	 */
	private void init() {
		if (!isInit) {
			synchronized (SqsService.class) {
				if (!isInit) {
					ScheduledExecutorService service = Executors.newSingleThreadScheduledExecutor();
					// 第二个参数为首次执行的延时时间，第三个参数为定时执行的间隔时间
					service.scheduleAtFixedRate(new Runnable() {
						@Override
						public void run() {
							Map<String, ExecutorService> poolmap = ThreadPool.getExecutorservicemap();
							for (String key : poolmap.keySet()) {
								ThreadPoolExecutor cur = (ThreadPoolExecutor) poolmap.get(key);
								int activeCount = cur.getActiveCount();// 活动线程大致数量
								long taskCount = cur.getTaskCount();// 曾经执行过任务总数
								log.info("the pool name={} activeCount={} maxCount={} taskCount={}", key, activeCount,
										cur.getMaximumPoolSize(), taskCount);
							}
							int doWithLongNum = 0;
							for (String msgId : doWithTime.keySet()) {
								if (System.currentTimeMillis() - doWithTime.get(msgId) > 30000) {// 存在超过30秒就打印
									doWithLongNum++;
								}
							}
							log.info("Number of records processed for timeout:{}", doWithLongNum);
						}
					}, 10, 30, TimeUnit.SECONDS);
					log.info("version========={}", versionJar);
					Runtime.getRuntime().addShutdownHook(new Thread() {
						@Override
						public void run() {
							log.info("----------------------执行关闭进程 钩子开始-------------------------------------");
							ThreadPool.shutdown();
							log.info("----------------------执行关闭进程 钩子完成-------------------------------------");
						}
					});
					isInit = true;
				}
			}
		}
	}

	/**
	 * 接收消息 - 指定消息属性
	 * 
	 * @param queueName
	 * @return
	 */
	public List<MessageCommon> queueReceiver(String queueName) {
		return queueReceiver(queueName, Conf.getInt("common.aws.sqs.receiver.maxNumberOfMessages"),
				Conf.getInt("common.aws.sqs.receiver.waitTimeSeconds"), new String[0]);
	}

	/**
	 * 接收消息 - 指定消息属性
	 * 
	 * @param queueName
	 * @param attributeNames
	 * @return
	 */
	public List<MessageCommon> queueReceiver(String queueName, String... attributeNames) {
		return queueReceiver(queueName, Conf.getInt("common.aws.sqs.receiver.maxNumberOfMessages"),
				Conf.getInt("common.aws.sqs.receiver.waitTimeSeconds"), attributeNames);
	}

	public List<MessageCommon> queueReceiverAllattr(String queueName) {
		return queueReceiver(queueName, Conf.getInt("common.aws.sqs.receiver.maxNumberOfMessages"),
				Conf.getInt("common.aws.sqs.receiver.waitTimeSeconds"), ".*");
	}

	/**
	 * 启不超时的批处理,适用于处理DLT的queue，需要异步调用它，否则阻塞主线程
	 *
	 * @param queueName           队列名
	 * @param sqsListerer         监听
	 * @param maxNumberOfMessages 最大接收消息数
	 * @param waitTimeSeconds     接收消息等待时间（s）
	 */
	public void queueReceiverListener(String queueName, String namespace, AbsSQSListener sqsListerer,
			int maxNumberOfMessages, int waitTimeSeconds, String... attributeNames) {
		Properties props = "default".equals(namespace) ? Conf.copyProperties() : getNoDefaultPoolConf(namespace);
		// 非默认线程池拉取的数据如果没有配置则使用默认的配置
		final int maxNumberOfMessagesTrue = maxNumberOfMessages > 0 ? maxNumberOfMessages
				: ("default".equals(namespace) ? maxNumberOfMessages
						: Integer.parseInt(props.getProperty(
								String.format("common.apiext.thread.pool.%s.maxNumberOfMessages", namespace))));
		log.info("the namespace:{} queueName:{} maxNumberOfMessages:{}", namespace, queueName, maxNumberOfMessagesTrue);
		ExecutorService threadPool = ThreadPool.getThreadPoolByName(namespace, props);
		threadPool.submit(new Runnable() {
			@Override
			public void run() {
				Thread.currentThread().setName(String.format("Listener---%s", queueName));
				while (true) {
					List<MessageCommon> queueRecievList = null;
					try {
						queueRecievList = queueReceiver(queueName, maxNumberOfMessagesTrue, waitTimeSeconds,
								attributeNames);
						TimeAssist.reDoWaitInit("aws-sqs");
					} catch (QueueDoesNotExistException e) {
						log.error("队列：" + queueName + "不存在", e);
						queueRecievList = new ArrayList<>();
						try {
							Thread.sleep(60000L);// 不存在此队列，60秒再试一次
						} catch (InterruptedException e1) {
						}
					} catch (AmazonSQSException e) {
						// com.amazonaws.services.sqs.model.AmazonSQSException: Your ReceiveMessage
						// request is throttled because you have too many concurrent ReceiveMessage
						// requests coming from your account. Reduce the number of clients polling your
						// queues.
						log.error("此帐号有太多的请求，队列：" + queueName, e);
						queueRecievList = new ArrayList<>();
						boolean reDoWait = TimeAssist.reDoWait("aws-sqs", 5);
						if (reDoWait) {
							try {
								Thread.sleep(32000L);// 到达了最大值,一直等待
							} catch (InterruptedException e1) {
							}
						}
					} catch (Throwable e) {// 保护拉取线程
						log.error("拉取记录时失败", e);
						queueRecievList = new ArrayList<>();
						try {
							Thread.sleep(60000L);// 不存在此队列，60秒再试一次
						} catch (InterruptedException e1) {
						}
					}
					for (MessageCommon messageTams : queueRecievList) {
						FutureTask<Boolean> futureTask = new FutureTask<Boolean>(new Callable<Boolean>() {
							public Boolean call() {
								PerThreadValue<String> createValue = PerthreadManager.getInstance()
										.createValue("tams-namespace", String.class);
								createValue.set(namespace);
								return sqsListerer.doListener(messageTams);
							}
						});
						threadPool.submit(futureTask);
					}
				}
			}
		});
	}

	private Properties getNoDefaultPoolConf(String namespace) {
		String preStr = "common.apiext.thread.pool." + namespace;
		// 默认配置
		Properties oriProps = Conf.replacePre("common.aws.sqs.receiver.pool", preStr);
		Map<String, String> overmap = Conf.getPre(preStr, false);
		oriProps.putAll(overmap);
		return oriProps;
	}

	public void queueReceiverListener(String queueName, AbsSQSListener sqsListerer, int maxNumberOfMessages,
			int waitTimeSeconds, int visibilityTimeout, String... attributeNames) {
		queueReceiverListener(queueName, "default", sqsListerer, maxNumberOfMessages, waitTimeSeconds, attributeNames);
	}

	public void queueReceiverListenerVar(String queueNameVar, AbsSQSListener sqsListerer, int maxNumberOfMessages,
			int waitTimeSeconds, int visibilityTimeout, String... attributeNames) {
		queueReceiverListener(Conf.get(queueNameVar), "default", sqsListerer, maxNumberOfMessages, waitTimeSeconds,
				attributeNames);
	}

	/**
	 * 轮询
	 *
	 * @param queueName      队列名
	 * @param attributeNames 属性名
	 * @param sqsListerer    监听
	 */
	public void queueReceiverListener(String queueName, AbsSQSListener sqsListerer, String... attributeNames) {
		queueReceiverListener(queueName, "default", sqsListerer, 0, 0, attributeNames);
	}

	public void queueReceiverListenerVar(String queueNameVar, AbsSQSListener sqsListerer, String... attributeNames) {
		queueReceiverListener(Conf.get(queueNameVar), "default", sqsListerer, 0, 0, attributeNames);
	}

	public void queueReceiverListener(String queueName, String namespace, AbsSQSListener sqsListerer,
			String... attributeNames) {
		queueReceiverListener(queueName, namespace, sqsListerer, 0, 0, attributeNames);
	}

	public void queueReceiverListenerVar(String queueNameVar, String namespace, AbsSQSListener sqsListerer,
			String... attributeNames) {
		queueReceiverListener(Conf.get(queueNameVar), namespace, sqsListerer, 0, 0, attributeNames);
	}

	/***
	 * 得到队列最原始结果
	 * 
	 * @param queueName
	 * @param maxNumberOfMessages
	 * @param waitTimeSeconds
	 * @param attributeNames
	 * @return
	 */
	public ReceiveMessageResult queueReceiverOri(String queueName, int maxNumberOfMessages, int waitTimeSeconds,
			int visibilityTimeout, String... attributeNames) {
		final ReceiveMessageRequest receiveMessageRequest = new ReceiveMessageRequest().withQueueUrl(queueName)
				.withWaitTimeSeconds(
						waitTimeSeconds <= 0 ? Conf.getInt("common.aws.sqs.receiver.waitTimeSeconds") : waitTimeSeconds)
				.withAttributeNames("SentTimestamp", "ApproximateReceiveCount").withMaxNumberOfMessages(
						maxNumberOfMessages <= 0 ? Conf.getInt("common.aws.sqs.receiver.maxNumberOfMessages")
								: maxNumberOfMessages);
		if (visibilityTimeout > 0) {
			receiveMessageRequest.withVisibilityTimeout(visibilityTimeout);
		}
		if (ArrayUtils.isNotEmpty(attributeNames)) {
			receiveMessageRequest.withMessageAttributeNames(attributeNames);// 获取消息属性
		}
		ReceiveMessageResult result = this.sqs.receiveMessage(receiveMessageRequest);
		return result;
	}

	/***
	 * 通用的查询queue方法，没有轮询,visibilityTimeout必须干掉，会覆盖掉队列的设置
	 * 
	 * @param queueName           队列名
	 * @param maxNumberOfMessages 最大接收消息数
	 * @param waitTimeSeconds     接收消息等待时间（s）
	 * @return
	 */
	public List<MessageCommon> queueReceiver(String queueName, int maxNumberOfMessages, int waitTimeSeconds,
			String... attributeNames) {
		ReceiveMessageResult result = queueReceiverOri(queueName, maxNumberOfMessages, waitTimeSeconds, 0,
				attributeNames);
		List<MessageCommon> retlist = new ArrayList<>();
		if (!result.getMessages().isEmpty()) {
			QueueAttributes queueAttributes = getQueueAttributes(queueName);
			final List<Message> messages = result.getMessages();
			for (Message message : messages) {
				MessageCommon messagecommon = new MessageCommon();
				messagecommon.setSuc(false);
				messagecommon.setReceiveTime(System.currentTimeMillis());
				messagecommon.setVisibilityTimeout(queueAttributes.getVisibilityTimeout());
				messagecommon.setMessage(message);// 把原始数据带上，发送时可以用上
				String body = message.getBody();
				SqsData.Builder sqsDataBuilder = SqsData.newBuilder();
				sqsDataBuilder.setMessageId(message.getMessageId());
				// sqsDataBuilder.setQueueName(queueName);
				sqsDataBuilder.setQueueNameReceive(queueName);// 删除的时候需要
				sqsDataBuilder.setReceiptHandle(message.getReceiptHandle());// 删除的时候需要
				// 系统属性处理
				Map<String, String> attributes = message.getAttributes();
				if (MapUtils.isNotEmpty(attributes)) {
					if (attributes.containsKey("SentTimestamp")) {
						sqsDataBuilder.setSentTimestamp(Long.parseLong(attributes.get("SentTimestamp")));
					}
					if (attributes.containsKey("ApproximateReceiveCount")) {
						sqsDataBuilder.setReceiveCount(Integer.parseInt(attributes.get("ApproximateReceiveCount")));
					}
				}
				try {
					JSONObject parseObject = JSONObject.parseObject(body);
					if (parseObject.containsKey("bucketName")) {
						sqsDataBuilder.setBucketName(parseObject.getString("bucketName"));
					}
					if (parseObject.containsKey("s3key")) {
						sqsDataBuilder.setS3Key(parseObject.getString("s3key"));
					}
					if (parseObject.containsKey("queueName")) {
						sqsDataBuilder.setQueueName(parseObject.getString("queueName"));
					}
					if (parseObject.containsKey("msg")) {
						sqsDataBuilder.setMsg(parseObject.getString("msg"));
					}
					if (parseObject.containsKey("props")) {
						sqsDataBuilder.setProps(parseObject.getString("props"));
					}
					if (parseObject.containsKey("version")) {
						sqsDataBuilder.setVersion(parseObject.getString("version"));
					}
					if (parseObject.containsKey("context")) {// context处理
						sqsDataBuilder.setContext(parseObject.getString("context"));
					}
					if (parseObject.containsKey("dataType")) {
						DataType dataType = DataType.valueOf(parseObject.getString("dataType"));
						sqsDataBuilder.setDataType(dataType);
						if (StringUtil.isNull(sqsDataBuilder.getMsg())
								&& StringUtil.isNotNull(sqsDataBuilder.getS3Key())) {
							S3Service s3 = S3ClientThreadlocal.createPerThreadS3Client();
							if (dataType == DataType.inputStream) {
								String objectForString = s3.getObjectForStringStream(sqsDataBuilder.getBucketName(),
										sqsDataBuilder.getS3Key());
								sqsDataBuilder.setMsg(objectForString);
							} else {
								String objectForString = s3.getObjectForString(sqsDataBuilder.getBucketName(),
										sqsDataBuilder.getS3Key());
								sqsDataBuilder.setMsg(objectForString);
							}
						}
					}
				} catch (Exception e) {// 普通的文本兼容
					sqsDataBuilder.setQueueName(queueName);
					sqsDataBuilder.setDataType(DataType.str);// str类型
					sqsDataBuilder.setMsg(body);
				}

				JSONObject properJson = new JSONObject();
				if (StringUtil.isNotNull(sqsDataBuilder.getProps())) {
					properJson = JSON.parseObject(sqsDataBuilder.getProps());
				}
				// prop属性优先于body的扩展属性
				if (MapUtils.isNotEmpty(message.getMessageAttributes())) {
					Map<String, MessageAttributeValue> properties = message.getMessageAttributes();
					for (String key : properties.keySet()) {
						properJson.put(key, properties.get(key).getStringValue());
					}
				}
				sqsDataBuilder.setProperties(properJson.toString());
				messagecommon.setSqsData(sqsDataBuilder.build());
				retlist.add(messagecommon);
			}
		}
		return retlist;
	}

	/***
	 * 得到队列的属性
	 * 
	 * @param queueName
	 * @return 没有配置DLQ为maxReceiveCount 其它：dlq的最大接收次数
	 */
	public QueueAttributes getQueueAttributes(String queueName) {
		GetQueueAttributesRequest quest = new GetQueueAttributesRequest(queueName);
		quest.withAttributeNames("RedrivePolicy", "VisibilityTimeout", "ApproximateNumberOfMessages",
				"ApproximateNumberOfMessagesDelayed", "ApproximateNumberOfMessagesNotVisible",
				"MessageRetentionPeriod");
		GetQueueAttributesResult queueAttributes = this.sqs.getQueueAttributes(quest);
		QueueAttributesBuilder builder = QueueAttributes.builder();
		String jsonstr = queueAttributes.getAttributes().get("RedrivePolicy");
		if (StringUtil.isNotNull(jsonstr)) {
			int intValue = JSONObject.parseObject(jsonstr).getIntValue("maxReceiveCount");
			builder.maxReceiveCount(intValue);
		}
		builder.VisibilityTimeout(new Integer(queueAttributes.getAttributes().get("VisibilityTimeout")));
		builder.approximateNumberOfMessages(
				new Integer(queueAttributes.getAttributes().get("ApproximateNumberOfMessages")));
		builder.approximateNumberOfMessagesDelayed(
				new Integer(queueAttributes.getAttributes().get("ApproximateNumberOfMessagesDelayed")));
		builder.approximateNumberOfMessagesNotVisible(
				new Integer(queueAttributes.getAttributes().get("ApproximateNumberOfMessagesNotVisible")));
		builder.MessageRetentionPeriod(new Integer(queueAttributes.getAttributes().get("MessageRetentionPeriod")));
		return builder.build();
	}

	/**
	 * 发送消息（字符串）
	 *
	 * @param queueName    队列名
	 * @param message      消息体
	 * @param properties   属性
	 * @param context      threadlocal
	 * @param delaySeconds 延迟消息
	 */
	public Result sendStrMsg(String queueName, String message, Map<String, String> properties, JSONObject context,
			int delaySeconds) {
		try {
			Result sendRequestMsg = this.sendRequestMsg(queueName, message.getBytes("utf-8"), null, DataType.str,
					properties, context, delaySeconds);
			return sendRequestMsg;
		} catch (Exception e) {
			log.error("send error", e);
			return Result.getError("send error：" + e.getMessage());
		}
	}

	public Result sendStrMsg(String queueName, String message, Map<String, String> properties) {
		return sendStrMsg(queueName, message, properties, null, 0);
	}

	/**
	 * 发送消息 - 指定桶（字符串）
	 *
	 * @param queueName 队列名
	 * @param message   消息体
	 */
	public Result sendStrMsg(String queueName, String message, String bucketName, Map<String, String> properties) {
		try {
			if (StringUtil.isNull(bucketName)) {
				return Result.getError("Failed：bucketName cannot be null");
			}
			Result result = this.sendRequestMsg(queueName, message.getBytes("utf-8"), bucketName, DataType.str,
					properties, null, 0);
			return result;
		} catch (Exception e) {
			log.error("send error", e);
			return Result.getError("send error：" + e.getMessage());
		}
	}

	/**
	 * 发送消息（base64字符串）
	 *
	 * @param queueName    队列名
	 * @param message      消息体
	 * @param properties
	 * @param context
	 * @param delaySeconds 延迟时间
	 */
	public Result sendBase64Msg(String queueName, byte[] message, Map<String, String> properties, JSONObject context,
			int delaySeconds) {
		try {
			Result sendRequestMsg = this.sendRequestMsg(queueName, message, null, DataType.base64, properties, context,
					delaySeconds);
			return sendRequestMsg;
		} catch (Exception e) {
			log.error("send error", e);
			return Result.getError("send error：" + e.getMessage());
		}
	}

	public Result sendBase64Msg(String queueName, byte[] message, Map<String, String> properties) {
		return sendBase64Msg(queueName, message, properties, null, 0);
	}

	/**
	 * 发送消息 - 指定桶（base64字符串）
	 *
	 * @param queueName  队列名
	 * @param message    消息体
	 * @param bucketName
	 * @param properties
	 */
	public Result sendBase64Msg(String queueName, byte[] message, String bucketName, Map<String, String> properties) {
		try {
			if (StringUtil.isNull(bucketName)) {
				return Result.getError("Failed：BucketName cannot be null");
			}
			Result sendRequestMsg = this.sendRequestMsg(queueName, message, bucketName, DataType.base64, properties,
					null, 0);
			return sendRequestMsg;
		} catch (Exception e) {
			log.error("send error", e);
			return Result.getError("send error：" + e.getMessage());
		}
	}

	/**
	 * 发送消息（byte数组）
	 *
	 * @param queueName    队列名
	 * @param msgBytes     消息体
	 * @param properties
	 * @param context
	 * @param delaySeconds 延迟时间
	 */
	public Result sendByteMsg(String queueName, byte[] msgBytes, Map<String, String> properties, JSONObject context,
			int delaySeconds) {
		try {
			Result sendRequestMsg = this.sendRequestMsg(queueName, msgBytes, null, DataType.inputStream, properties,
					context, delaySeconds);
			return sendRequestMsg;
		} catch (Exception e) {
			log.error("send error", e);
			return Result.getError("send error：" + e.getMessage());
		}
	}

	public Result sendByteMsg(String queueName, byte[] msgBytes, Map<String, String> properties) {
		return sendByteMsg(queueName, msgBytes, properties, null, 0);
	}

	/**
	 * 发送消息 - 指定桶（byte数组）
	 *
	 * @param queueName 队列名
	 * @param msgBytes  消息体
	 */
	public Result sendByteMsg(String queueName, byte[] msgBytes, String bucketName, Map<String, String> properties) {
		try {
			if (StringUtil.isNull(bucketName)) {
				return Result.getError("Failed：BucketName cannot be null");
			}
			Result sendRequestMsg = this.sendRequestMsg(queueName, msgBytes, bucketName, DataType.inputStream,
					properties, null, 0);
			return sendRequestMsg;
		} catch (Exception e) {
			log.error("send error", e);
			return Result.getError("send error：" + e.getMessage());
		}
	}

	/***
	 * 通过前缀查询queueName
	 * 
	 * @param queueNamePrefix 前缀名
	 * @return
	 */
	public List<String> queryQueuePrefix(String queueNamePrefix) {
		ListQueuesRequest request = new ListQueuesRequest();
		request.withQueueNamePrefix(queueNamePrefix);
		ListQueuesResult listQueues = this.sqs.listQueues(request);
		List<String> queueUrls = listQueues.getQueueUrls();
		List<String> retlist = new ArrayList<>();
		if (CollectionUtils.isNotEmpty(queueUrls)) {
			for (String queueUrl : queueUrls) {
				int lastIndexOf = queueUrl.lastIndexOf("/");
				String substring = queueUrl.substring(lastIndexOf + 1);
				retlist.add(substring);
			}
		}
		return retlist;
	}

	/***
	 * 重发消息用。主要用于队列的转存服务，业务也可以跟据自己业务需求写转存任务
	 * 
	 * @param sqsData
	 * @param queueName 需要转存的queue数据，如果没有就使用sqsData里的QueueName
	 * @return
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public Result sendSqsDataMsg(SqsData sqsData, String queueName) {
		String queueNameTrue = StringUtil.isNull(queueName) ? sqsData.getQueueName() : queueName;
		Validate.notBlank(queueNameTrue, "数据不规则，QueueName为空值");
		if (StringUtil.isNotNull(sqsData.getS3Key())) {
			SendMessageRequest messageRequest = new SendMessageRequest();
			JSONObject propsJson = JSONObject.parseObject(sqsData.getProperties());
			JSONObject bodyjson = new JSONObject();
			if (propsJson != null && propsJson.size() < 10) {
				SdkInternalMap<String, MessageAttributeValue> messageAttributes = new SdkInternalMap<>();
				for (String key : propsJson.keySet()) {
					MessageAttributeValue attributeValue = new MessageAttributeValue();
					if (StringUtil.isNotNull(propsJson.get(key))) {
						attributeValue.setStringValue(String.valueOf(propsJson.get(key)));
						attributeValue.setDataType("String");// 统一字符串
						messageAttributes.put(key, attributeValue);
					}
				}
				messageRequest.setMessageAttributes(messageAttributes);
			} else {
				if (propsJson != null) {
					bodyjson.put("props", sqsData.getProps());
				}
			}
			if (StringUtil.isNotNull(sqsData.getContext())) {
				bodyjson.put("context", sqsData.getContext());
			}
			bodyjson.put("dataType", sqsData.getDataType().name());
			bodyjson.put("bucketName", sqsData.getBucketName());
			bodyjson.put("s3key", sqsData.getS3Key());
			bodyjson.put("version", sqsData.getVersion());
			bodyjson.put("queueName", queueNameTrue);
			messageRequest.setQueueUrl(queueNameTrue);
			messageRequest.setMessageBody(bodyjson.toJSONString());
			SendMessageResult sendMessage = this.sqs.sendMessage(messageRequest);
			return Result.getSuc(sendMessage.getMessageId());
		} else {
			JSONObject contextObj = JSONObject.parseObject(sqsData.getContext());
			Map jsonToMap = JSONUtil.jsonToMap(JSONObject.parseObject(sqsData.getProperties()));
			try {
				byte[] bytes = sqsData.getMsg().getBytes("utf-8");

				return sendRequestMsg(queueNameTrue, bytes, sqsData.getBucketName(), sqsData.getDataType(), jsonToMap,
						contextObj, 0);
			} catch (UnsupportedEncodingException e) {
				throw new RuntimeException("no suppered utf-8");
			}
		}
	}

	/***
	 * 转存消息
	 * 
	 * @param messagecommon
	 * @param queueName
	 * @return
	 */
	public Result sendMessageTamsMsg(MessageCommon messagecommon, String queueName) {
		Validate.notBlank(queueName, "queueName不能为空");
		Validate.notBlank(queueName, "数据不规则，QueueName为空值");
		Validate.notNull(messagecommon.getMessage(), "需要原始信息");
		SendMessageRequest messageRequest = new SendMessageRequest();
		messageRequest.setQueueUrl(queueName);
		Map<String, MessageAttributeValue> messageAttributesOri = messagecommon.getMessage().getMessageAttributes();
		if (MapUtils.isNotEmpty(messageAttributesOri)) {
			messageRequest.setMessageAttributes(messageAttributesOri);
		}
		String body = messagecommon.getMessage().getBody();
		messageRequest.setMessageBody(body);
		SendMessageResult sendMessage = this.sqs.sendMessage(messageRequest);
		return Result.getSuc(sendMessage.getMessageId());
	}

	public Result sendSqsDataMsg(SqsData sqsData) {
		return sendSqsDataMsg(sqsData, null);
	}

	/**
	 * 发送消息
	 * 
	 * @param queueName
	 * @param msgBytes
	 * @param bucketName
	 * @param dataType
	 * @param delaySeconds 延迟时间
	 * @return
	 */
	private Result sendRequestMsg(String queueName, byte[] msgBytes, String bucketName, DataType dataType,
			Map<String, String> properties, JSONObject context, int delaySeconds) {
		if (StringUtil.isNull(queueName)) {
			return Result.getError("Failed：QueueName cannot be null");
		}
		if (ArrayUtils.isEmpty(msgBytes)) {
			return Result.getError("Failed：MsgBytes cannot be null");
		}
		if (dataType == null) {
			return Result.getError("Failed：dataType cannot be null");
		}
		SendMessageRequest messageRequest = new SendMessageRequest();
		JSONObject jsonObject = new JSONObject();
		String propsStr = "";
		int bodyMaxSize = Conf.getInt("common.aws.sqs.sender.maxSize");
		SdkInternalMap<String, MessageAttributeValue> messageAttributes = null;
		if (!MapUtils.isEmpty(properties)) {
			// hashmap的size()方法可能会莫名地不准确，在duckula里出现一回，在胡斌代码出现一回，改用keySet()的size
			// Number of message attributes [11] exceeds the allowed maximum [10]
			if (properties.keySet().size() > 10) {// if (properties.size() > 10) {// 大于10上放body
				propsStr = JSONUtil.getJsonForMap(properties);
				try {
					bodyMaxSize = bodyMaxSize - propsStr.getBytes("utf-8").length;
				} catch (UnsupportedEncodingException e) {
					log.error("Unsupported utf-8" + e);
				}
				Validate.isTrue(bodyMaxSize >= 0, "属性不能超过body最大值：" + Conf.getInt("common.aws.sqs.sender.maxSize"));
				jsonObject.put("props", JSONObject.parseObject(propsStr));
			} else {
				messageAttributes = new SdkInternalMap<>();
				for (String key : properties.keySet()) {
					MessageAttributeValue attributeValue = new MessageAttributeValue();
					if (StringUtil.isNotNull(properties.get(key))) {
						attributeValue.setStringValue(String.valueOf(properties.get(key)));
						attributeValue.setDataType("String");// 统一字符串
						messageAttributes.put(key, attributeValue);
					}
				}
				messageRequest.setMessageAttributes(messageAttributes);
			}
		}
		// 支持消息延迟,只支持<900秒
		if (delaySeconds > 0) {
			delaySeconds = delaySeconds < 900 ? delaySeconds : 900;
			messageRequest.setDelaySeconds(delaySeconds);
		}
		// 处理context
		if (context != null && context.size() > 0) {
			JSONObject createPerThreadSqsContext = SqsContext.createPerThreadSqsContext();
			createPerThreadSqsContext.putAll(JSONUtil.jsonToMap(context));
			try {
				bodyMaxSize = bodyMaxSize - createPerThreadSqsContext.toString().getBytes("utf-8").length;
			} catch (UnsupportedEncodingException e) {
				log.error("Unsupported utf-8" + e);
			}
			Validate.isTrue(bodyMaxSize >= 0,
					"属性和context之和不能超过body最大值：" + Conf.getInt("common.aws.sqs.sender.maxSize"));
			jsonObject.put("context", createPerThreadSqsContext);
		}

		// 判断消息大小，由于属性较大
		String msg = null;
		try {
			switch (dataType) {
			case base64:
				msg = PwdUtil.base64FromBin(msgBytes);
				msgBytes = msg.getBytes("utf-8");
				break;
			case str:
				msg = new String(msgBytes, "utf-8");
				break;
			case inputStream:
				msg = new String(msgBytes, "ISO-8859-1");
				break;
			default:
				break;
			}
		} catch (Exception e) {
			log.error("不支持的编码", e);
		}

		if (msgBytes.length <= bodyMaxSize) {
			jsonObject.put("msg", msg);
		} else {// inputStream不用判断大小，全走S3
			convertS3(queueName, msgBytes, bucketName, dataType, jsonObject, msg);
		}
		messageRequest.setQueueUrl(queueName);
		jsonObject.put("queueName", queueName);
		jsonObject.put("dataType", dataType);
		jsonObject.put("version", version);

		// 系统级的大小比较，不受用户配置影响
		String msgAllStr = jsonObject.toString();
		try {
			if (msgAllStr.getBytes("utf-8").length + getMsgAttributesSize(messageAttributes) >= 262144) {// 如果加上各种转义后又超过了256K,走S3
				jsonObject.remove("msg");
				convertS3(queueName, msgBytes, bucketName, dataType, jsonObject, msg);
				msgAllStr = jsonObject.toString();
			}
		} catch (UnsupportedEncodingException e) {
			log.error("Unsupported utf-8" + e);
		}
		messageRequest.setMessageBody(msgAllStr);
		SendMessageResult sendMessage = this.sqs.sendMessage(messageRequest);
		return Result.getSuc(sendMessage.getMessageId());
	}

	private int getMsgAttributesSize(Map<String, MessageAttributeValue> msgAttributes) {
		int totalMsgAttributesSize = 0;
		if (msgAttributes == null) {
			return totalMsgAttributesSize;
		}
		for (Entry<String, MessageAttributeValue> entry : msgAttributes.entrySet()) {
			totalMsgAttributesSize += getStringSizeInBytes(entry.getKey());

			MessageAttributeValue entryVal = entry.getValue();
			if (entryVal.getDataType() != null) {
				totalMsgAttributesSize += getStringSizeInBytes(entryVal.getDataType());
			}

			String stringVal = entryVal.getStringValue();
			if (stringVal != null) {
				totalMsgAttributesSize += getStringSizeInBytes(entryVal.getStringValue());
			}

			ByteBuffer binaryVal = entryVal.getBinaryValue();
			if (binaryVal != null) {
				totalMsgAttributesSize += binaryVal.array().length;
			}
		}
		return totalMsgAttributesSize;
	}

	private static long getStringSizeInBytes(String str) {
		CountingOutputStream counterOutputStream = new CountingOutputStream();
		try {
			Writer writer = new OutputStreamWriter(counterOutputStream, "UTF-8");
			writer.write(str);
			writer.flush();
			writer.close();
		} catch (IOException e) {
			String errorMessage = "Failed to calculate the size of message payload.";
			log.error(errorMessage, e);
			throw new AmazonClientException(errorMessage, e);
		}
		return counterOutputStream.getTotalSize();
	}

	private void convertS3(String queueName, byte[] msgBytes, String bucketName, DataType dataType,
			JSONObject jsonObject, String msg) {
		String relaPath = IOUtil.mergeFolderAndFilePath(queueName, UUID.randomUUID().toString());
		bucketName = AwsHelper.buildBucketName(bucketName);
		String s3key = AwsHelper.buildBucketKey(relaPath);
		jsonObject.put("bucketName", bucketName);
		jsonObject.put("s3key", s3key);
		// 将message消息存储到桶中
		S3Service s3 = S3ClientThreadlocal.createPerThreadS3Client();
		switch (dataType) {
		case str:
		case base64:
			s3.putObjectForStr(relaPath, msg);
			break;
		case inputStream:
			s3.putObjectForInputStream(relaPath, msgBytes);
			break;
		default:
			break;
		}
	}

	/**
	 * 创建queue
	 * 
	 * @param queueName 队列名
	 * @return
	 */
	public String createQueue(String queueName) {
		return createQueue(queueName, QueryType.queue, null, 0, 0, 0, 0);
	}

	/**
	 * 创建queue
	 * 
	 * @param sourceQueueName               源队列名
	 * @param queryType                     类型 query - 标准队列,fifo - fifo队列
	 * @param dlQueueName                   死信队列名
	 * @param VisibilityTimeout             可见时间
	 * @param ReceiveMessageWaitTimeSeconds 长轮询时间
	 * @param MessageRetentionPeriod        消息保存周期
	 * @param maxReceiveCount               死信队列最大重试次数，如果dlQueueName为空，此参数不生效
	 * @return
	 */

	private String createQueue(String sourceQueueName, QueryType queryType, String dlQueueName, int VisibilityTimeout,
			int ReceiveMessageWaitTimeSeconds, int MessageRetentionPeriod, int maxReceiveCount) {
		String queueUrl = "";
		try {
			if (sourceQueueName == null) {
				throw new IllegalArgumentException("SourceQueueName cannot be null.");
			} else if (queryType == null) {
				throw new IllegalArgumentException("QueryType cannot be null.");
			} else {
				SetQueueAttributesRequest queueAttributes = new SetQueueAttributesRequest();
				Map<String, String> attributes = new HashMap<>();
				if (StringUtil.isNotNull(dlQueueName)) {
					// 创建死信队列,获取queueArn
					String deadLetterQueueUrl = creteQueue(dlQueueName, true);
					GetQueueAttributesResult deadLetterQueueAttributes = sqs.getQueueAttributes(
							new GetQueueAttributesRequest(deadLetterQueueUrl).withAttributeNames("QueueArn"));
					String deadLetterQueueArn = deadLetterQueueAttributes.getAttributes().get("QueueArn");
					// 配置队列参数
					JSONObject redrivePolicy = new JSONObject();
					redrivePolicy.put("maxReceiveCount",
							maxReceiveCount <= 0 ? Conf.get("common.aws.sqs.dlq.maxReceiveCount")
									: String.valueOf(maxReceiveCount));
					redrivePolicy.put("deadLetterTargetArn", deadLetterQueueArn);
					attributes.put(QueueAttributeName.RedrivePolicy.toString(), redrivePolicy.toString());
				}
				attributes.put(QueueAttributeName.VisibilityTimeout.toString(),
						VisibilityTimeout <= 0 ? Conf.get("common.aws.sqs.queue.visibilityTimeout")
								: String.valueOf(VisibilityTimeout));
				attributes.put(QueueAttributeName.ReceiveMessageWaitTimeSeconds.toString(),
						ReceiveMessageWaitTimeSeconds <= 0 ? Conf.get("common.aws.sqs.queue.waitTimeSeconds")
								: String.valueOf(ReceiveMessageWaitTimeSeconds));
				attributes.put(QueueAttributeName.MessageRetentionPeriod.toString(),
						MessageRetentionPeriod <= 0 ? Conf.get("common.aws.sqs.queue.MessageRetentionPeriod")
								: String.valueOf(MessageRetentionPeriod));
				queueAttributes.setAttributes(attributes);
				// 创建源队列
				queueUrl = creteQueue(sourceQueueName, false);
				if (queryType == QueryType.fifo) {// TODO 中国区暂时不支持fifo sqs
					if (!sourceQueueName.endsWith(".fifo")) {
						throw new IllegalArgumentException("The FIFO queue name must end with the .fifo suffix.");
					} else {
						attributes.put("FifoQueue", "true");
						attributes.put("ContentBasedDeduplication", "true");
					}
				}
				queueAttributes.setQueueUrl(sourceQueueName);
				this.sqs.setQueueAttributes(queueAttributes);
			}
		} catch (Exception e) {
			log.error("创建[" + sourceQueueName + "]队列失败", e);
			throw new RuntimeException("创建[" + sourceQueueName + "]队列失败：" + e.getMessage());
		}
		return queueUrl;
	}

	public String createQueue(String sourceQueueName, String dlQueueName, int VisibilityTimeout,
			int ReceiveMessageWaitTimeSeconds, int MessageRetentionPeriod, int maxReceiveCount) {
		return createQueue(sourceQueueName, QueryType.queue, dlQueueName, VisibilityTimeout,
				ReceiveMessageWaitTimeSeconds, MessageRetentionPeriod, maxReceiveCount);
	}

	public String createQueue(String sourceQueueName, String dlQueueName) {
		return createQueue(sourceQueueName, QueryType.queue, dlQueueName, 0, 0, 0, 0);
	}

	/**
	 * 创建队列
	 *
	 * @param queueName
	 * @param isDlq
	 */
	private String creteQueue(String queueName, boolean isDlq) {
		if (isDlq) {
			queueName = StringUtil.hasNull(queueName, Conf.get("common.aws.sqs.dlq.name"));
			if (StringUtil.isNull(queueName)) {
				throw new IllegalArgumentException("DLQueueName cannot be null.");
			}
		}
		return this.sqs.createQueue(queueName).getQueueUrl();
	}

	/**
	 * 删除消息
	 *
	 * @param msg
	 */
	public DeleteMessageResult deleteMessage(SqsData msg) {
		final DeleteMessageRequest batchRequest = new DeleteMessageRequest().withQueueUrl(msg.getQueueNameReceive())
				.withReceiptHandle(msg.getReceiptHandle());
		DeleteMessageResult deleteMessage = sqs.deleteMessage(batchRequest);
		return deleteMessage;
	}
}
