/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.aws.s3;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.stream.Collectors;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.PwdUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.aws.DataType;
import net.wicp.tams.common.aws.common.AwsHelper;
import net.wicp.tams.common.aws.common.TamsCredentialsProvider;
import net.wicp.tams.common.aws.s3.threadlocal.S3ClientThreadlocal;

@Slf4j
public class S3Service {
	private final AmazonS3 s3;

	private S3Service(AmazonS3 s3) {
		this.s3 = s3;
	}

	private S3Service() {
		AmazonS3 s3 = AmazonS3ClientBuilder.standard().withRegion(Conf.get("common.aws.region"))
				.withCredentials(new TamsCredentialsProvider()).build();
		this.s3 = s3;
	}

	// 默认使用threadLocal方式
	public static S3Service getInstThread() {
		S3Service createPerThreadEsClient = S3ClientThreadlocal.createPerThreadS3Client();
		return createPerThreadEsClient;
	}

	/**
	 * s3存储字符串
	 * 
	 * @param fileRelaPath
	 * @param t
	 * @param <T>
	 * @return
	 */
	public <T> PutObjectResult putObjectForStr(String fileRelaPath, String t) {
		String buildBucketName = AwsHelper.buildBucketName(Conf.get("common.aws.sqs.s3.bucketName"));
		String buildBucketKey = AwsHelper.buildBucketKey(fileRelaPath);
		return putObject(buildBucketName, null, buildBucketKey, DataType.str, t.getBytes());
	}

	/***
	 * s3存储base64字符串
	 * 
	 * @param fileRelaPath
	 * @param t            需要存储的二进制
	 * @return
	 */
	public <T> PutObjectResult putObjectForBase64(String fileRelaPath, byte[] t) {
		String buildBucketName = AwsHelper.buildBucketName(Conf.get("common.aws.sqs.s3.bucketName"));
		String buildBucketKey = AwsHelper.buildBucketKey(fileRelaPath);
		return putObject(buildBucketName, null, buildBucketKey, DataType.base64, t);
	}

	/**
	 * s3存储二进制流
	 * 
	 * @param fileRelaPath
	 * @param inputStream
	 * @param <T>
	 * @return
	 */
	public <T> PutObjectResult putObjectForInputStream(String fileRelaPath, InputStream inputStream) {
		return putObjectForInputStream(fileRelaPath, AwsHelper.getBytes(inputStream));
	}

	/**
	 * s3存储byte数组
	 * 
	 * @param fileRelaPath
	 * @param byteArr
	 * @param <T>
	 * @return
	 */
	public <T> PutObjectResult putObjectForInputStream(String fileRelaPath, byte[] byteArr) {
		String buildBucketName = AwsHelper.buildBucketName(Conf.get("common.aws.sqs.s3.bucketName"));
		String buildBucketKey = AwsHelper.buildBucketKey(fileRelaPath);
		return putObject(buildBucketName, null, buildBucketKey, DataType.inputStream, byteArr);
	}

	/**
	 * s3存储文件
	 * 
	 * @param fileRelaPath
	 * @param t
	 * @param <T>
	 * @return
	 */
	public <T> PutObjectResult putObjectForInputStream(String fileRelaPath, File t) {
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(t);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		return putObjectForInputStream(fileRelaPath, fis);
	}

	/**
	 * s3存储byte数组 - 指定类型
	 * 
	 * @param fileRelaPath
	 * @param byteArr
	 * @param <T>
	 * @return
	 */
	public <T> PutObjectResult putObject(String fileRelaPath, byte[] byteArr, DataType dataType) {
		String buildBucketName = AwsHelper.buildBucketName(Conf.get("common.aws.sqs.s3.bucketName"));
		String buildBucketKey = AwsHelper.buildBucketKey(fileRelaPath);
		return putObject(buildBucketName, null, buildBucketKey, dataType, byteArr);
	}

	/**
	 * 根据类型存储
	 * 
	 * @param bucketName
	 * @param rootPath
	 * @param fileRelaPath
	 * @param dataType
	 * @param t
	 * @param <T>
	 * @return
	 */
	public <T> PutObjectResult putObject(String bucketName, String rootPath, String fileRelaPath, DataType dataType,
			byte[] t) {
		bucketName = StringUtil.hasNull(bucketName, Conf.get("common.aws.sqs.s3.bucketName"));
		String pathTrue = IOUtil.mergeFolderAndFilePath(rootPath, fileRelaPath);
		PutObjectResult putObject = null;
		if (pathTrue.startsWith("/"))
			pathTrue = pathTrue.substring(1);
		switch (dataType) {
		case base64:
			String base64FromBin = PwdUtil.base64FromBin(t);
			putObject = s3.putObject(bucketName, pathTrue, base64FromBin);
			break;
		case str:
			try {
				putObject = s3.putObject(bucketName, pathTrue, new String(t, "utf-8"));
			} catch (UnsupportedEncodingException e) {
				log.error("UnsupportedEncodingException", e);
			}
			break;
		case inputStream:
			putObject = s3.putObject(bucketName, pathTrue, new ByteArrayInputStream(t), new ObjectMetadata());
			break;
		default:
			throw new RuntimeException("不支持的类型");
		}
		return putObject;
	}

	public String getObjectForString(String bucketName, String keyName) {
		S3ObjectInputStream inputStream = null;
		try {
			S3Object o = this.s3.getObject(bucketName, keyName);
			inputStream = o.getObjectContent();
			// String slurp = slurp(inputStream,oriEncoding);
			String slurp = new BufferedReader(new InputStreamReader(inputStream)).lines()
					.collect(Collectors.joining(System.lineSeparator()));
			return slurp;
		} catch (Exception e) {
			log.error("取得s3文件失败", e);
			throw new RuntimeException("取得s3文件失败");
		} finally {
			if (inputStream != null) {
				try {
					inputStream.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	public String getObjectForStringStream(String bucketName, String keyName) {
		S3ObjectInputStream inputStream = null;
		try {
			S3Object o = this.s3.getObject(bucketName, keyName);
			inputStream = o.getObjectContent();
			String slurp = slurp(inputStream, "ISO-8859-1");
			return slurp;
		} catch (Exception e) {
			log.error("取得s3文件失败", e);
			throw new RuntimeException("取得s3文件失败");
		} finally {
			if (inputStream != null) {
				try {
					inputStream.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	private static String slurp(InputStream in, String oriEncoding) throws IOException {
		StringBuffer out = new StringBuffer();
		byte[] b = new byte[4096];
		for (int n; (n = in.read(b)) != -1;) {
			out.append(new String(b, 0, n, oriEncoding));
		}
		return out.toString();
	}

	/***
	 * 获取字符串
	 * 
	 * @param relaKeyName 相对路径
	 * @return
	 */
	public String getObjectForString(String relaKeyName) {
		String buildBucketName = AwsHelper.buildBucketName(Conf.get("common.aws.sqs.s3.bucketName"));
		String buildBucketKey = AwsHelper.buildBucketKey(relaKeyName);
		return getObjectForString(buildBucketName, buildBucketKey);

	}

	/***
	 * 得到流的bytes
	 * 
	 * @param bucketName
	 * @param keyName
	 * @return
	 */
	public byte[] getObjectForBytes(String bucketName, String keyName) {
		S3ObjectInputStream stream = getObjectForStream(bucketName, keyName);
		byte[] bytes = AwsHelper.getBytes(stream);
		try {
			if (stream != null) {
				stream.close();
			}
		} catch (IOException e) {
		}
		return bytes;
	}

	public S3ObjectInputStream getObjectForStream(String bucketName, String keyName) {
		S3ObjectInputStream s3is = null;
		try {
			S3Object o = this.s3.getObject(bucketName, keyName);
			s3is = o.getObjectContent();
			return s3is;
		} catch (Exception e) {
			log.error("取得s3文件失败", e);
			throw new RuntimeException("取得s3文件失败");
		}
	}

	public byte[] getObjectForBytes(String relaKeyName) {
		String buildBucketName = AwsHelper.buildBucketName(Conf.get("common.aws.sqs.s3.bucketName"));
		String buildBucketKey = AwsHelper.buildBucketKey(relaKeyName);
		return getObjectForBytes(buildBucketName, buildBucketKey);
	}

	/**
	 * 获取消息数据保存新的文件
	 * 
	 * @param bucketName
	 * @param keyName
	 * @param savePath
	 */
	public void getObjectForFile(String bucketName, String keyName, String savePath) {
		S3ObjectInputStream s3is = null;
		FileOutputStream fos = null;
		try {
			S3Object o = this.s3.getObject(bucketName, keyName);
			s3is = o.getObjectContent();
			fos = new FileOutputStream(new File(StringUtil.isNotNull(savePath) ? savePath : keyName));
			IOUtil.copyInToOut(s3is, fos);
		} catch (Exception e) {
			log.error("取得s3文件失败", e);
			throw new RuntimeException("取得s3文件失败");
		} finally {
			try {
				if (s3is != null) {
					s3is.close();
				}
			} catch (IOException e) {
			}
			try {
				if (fos != null) {
					fos.close();
				}
			} catch (Exception e2) {
			}
		}
	}

	/***
	 * 获取消息数据保存新的文件
	 * 
	 * @param relaKeyName 相对路径
	 * @param savePath
	 */
	public void getObjectForFile(String relaKeyName, String savePath) {
		String buildBucketName = AwsHelper.buildBucketName(Conf.get("common.aws.sqs.s3.bucketName"));
		String buildBucketKey = AwsHelper.buildBucketKey(relaKeyName);
		getObjectForFile(buildBucketName, buildBucketKey, savePath);
	}

}
