package net.wicp.tams.common.constant.dbType;

import java.io.Serializable;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.sql.Date;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.BitSet;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.ArrayUtils;

import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.DateUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.constant.PageElement;

/***
 * 来自ListenerDef.proto的ColType,不想引入binlog-api依赖，所以独立
 * @author Andy
 *
 */
public enum BinlogType {
	/**
	 * <code>DECIMAL = 0;</code>
	 */
	DECIMAL(0, Types.DECIMAL, new String[] {}, BigDecimal.class, PageElement.NumberBox, new String[] { "DECIMAL" }),
	/**
	 * <code>TINY = 1;</code>
	 */
	TINY(1, Types.TINYINT, new String[] { "tinyint" }, Short.class, PageElement.NumberIncrement,
			new String[] { "BOOLEAN", "TINYINT" }),
	/**
	 * <code>SHORT = 2;</code>
	 */
	SHORT(2, Types.SMALLINT, new String[] { "smallint" }, Short.class, PageElement.NumberIncrement,
			new String[] { "SMALLINT" }),
	/**
	 * <code>LONG = 3;</code>
	 */
	LONG(3, Types.INTEGER, new String[] { "int", "integer(int)" }, Integer.class, PageElement.NumberIncrement,
			new String[] { "INT" }),
	/**
	 * <code>FLOAT = 4;</code>
	 */
	FLOAT(4, Types.FLOAT, new String[] { "float" }, Float.class, PageElement.NumberIncrement, new String[] { "FLOAT" }),
	/**
	 * <code>DOUBLE = 5;</code>WW
	 */
	DOUBLE(5, Types.DOUBLE, new String[] { "real(double)", "double" }, Double.class, PageElement.NumberBox,
			new String[] { "DOUBLE" }),
	/**
	 * <code>NULL = 6;</code>
	 */
	NULL(6, Types.NULL, new String[] {}, String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>TIMESTAMP = 7;</code>
	 */
	TIMESTAMP(7, Types.TIMESTAMP, new String[] {}, Date.class, PageElement.Calendar, new String[] {}),
	/**
	 * <code>LONGLONG = 8;</code>
	 */
	LONGLONG(8, Types.BIGINT, new String[] { "bigint" }, Long.class, PageElement.NumberBox, new String[] { "BIGINT" }),
	/**
	 * <code>INT24 = 9;</code>
	 */
	INT24(9, Types.INTEGER, new String[] { "mediumint" }, Integer.class, PageElement.NumberBox, new String[] {}),
	/**
	 * <code>DATE = 10;</code>
	 */
	DATE(10, Types.DATE, new String[] { "date" }, Date.class, PageElement.Calendar, new String[] { "DATE" }),
	/**
	 * <code>TIME = 11;</code>
	 */
	TIME(11, Types.TIME, new String[] {}, String.class, PageElement.ValidateBox, new String[] { "TIME" }),
	/**
	 * <code>DATETIME = 12;</code>
	 */
	DATETIME(12, Types.DATE, new String[] {}, Date.class, PageElement.Calendar, new String[] {}),
	/**
	 * <code>YEAR = 13;</code>
	 */
	YEAR(13, Types.VARCHAR, new String[] { "year" }, String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>NEWDATE = 14;</code>
	 */
	NEWDATE(14, Types.VARCHAR, new String[] {}, String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>VARCHAR = 15;</code>
	 */
	VARCHAR(15, Types.VARCHAR,
			new String[] { "varchar", "varbinary", "varchar(8000)", "tinytext", "text", "mediumtext", "longtext" },
			String.class, PageElement.ValidateBox, new String[] { "CHAR", "VARCHAR", "STRING" }),
	/**
	 * <code>BIT = 16;</code>
	 */
	BIT(16, Types.BIT, new String[] { "bit" }, BitSet.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>TIMESTAMP2 = 17;</code>
	 */
	TIMESTAMP2(17, Types.TIMESTAMP, new String[] { "timestamp" }, Long.class, PageElement.NumberIncrement,
			new String[] { "TIMESTAMP", "TIMESTAMP WITH LOCAL TIME ZONE" }),
	/**
	 * <code>DATETIME2 = 18;</code>
	 */
	DATETIME2(18, Types.DATE, new String[] { "datetime" }, Date.class, PageElement.Calendar, new String[] {}),
	/**
	 * <code>TIME2 = 19;</code>
	 */
	TIME2(19, Types.TIME, new String[] { "time" }, String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>JSON = 245;</code>
	 */
	JSON(245, Types.VARCHAR, new String[] { "json" }, String.class, PageElement.ValidateBox, new String[] {}), // mysql5.7以上支持
	/**
	 * <code>NEWDECIMAL = 246;</code>
	 */
	NEWDECIMAL(246, Types.DECIMAL, new String[] { "decimal", "numeric(decimal)" }, BigDecimal.class,
			PageElement.NumberBox, new String[] {}),
	/**
	 * <code>ENUM = 247;</code>
	 */
	ENUM(247, Types.VARCHAR, new String[] {}, String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>SET = 248;</code>
	 */
	SET(248, Types.VARCHAR, new String[] {}, String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>TINY_BLOB = 249;</code>
	 */
	TINY_BLOB(249, Types.BLOB, new String[] {}, String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>MEDIUM_BLOB = 250;</code>
	 */
	MEDIUM_BLOB(250, Types.BLOB, new String[] {}, String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>LONG_BLOB = 251;</code>
	 */
	LONG_BLOB(251, Types.BLOB, new String[] {}, String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>BLOB = 252;</code>
	 */
	BLOB(252, Types.BLOB, new String[] { "tinyblob", "blob", "mediumblob", "longblob" }, String.class,
			PageElement.ValidateBox, new String[] { "BINARY", "VARBINARY", "BYTES" }),
	/**
	 * <code>VAR_STRING = 253;</code>
	 */
	VAR_STRING(253, Types.VARCHAR, new String[] {}, String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>STRING = 254;</code>
	 */
	STRING(254, Types.VARCHAR, new String[] { "char", "enum('e3','e2','e1')", "set('s1','s3','s2')", "binary" },
			String.class, PageElement.ValidateBox, new String[] {}),
	/**
	 * <code>GEOMETRY = 255;</code>
	 */
	GEOMETRY(255, Types.VARCHAR,
			new String[] { "point", "linestring", "polygon", "geometry", "multipoint", "multilinestring",
					"multipolygon", "geometrycollection" },
			String.class, PageElement.ValidateBox, new String[] {}),

	UNRECOGNIZED(-1, Types.VARCHAR, new String[] {}, String.class, PageElement.ValidateBox, new String[] {});

	private final int value;

	private final int types;

	private final String[] mysqlTypes;// mysql数据库字段类型

	private final Type javaType;// java类型

	private final PageElement pageElement;// 页面需要显示的元素，用于低代码,一般跟javaType密切相关

	/**
	 * TODO 还没有处理的字段 "INTERVAL YEAR TO MONTH","INTERVAL DAY TO MONTH","ROW /
	 * structured types ","ARRAY"," MAP / MULTISET ","RAW"
	 */
	private final String[] flinkLogicalTypes;// flink的本地类型,参考org.apache.flink.table.data.RowData

	public String[] getFlinkLogicalTypes() {
		return flinkLogicalTypes;
	}

	public PageElement getPageElement() {
		return pageElement;
	}

	public Type getJavaType() {
		return javaType;
	}

	public int getValue() {
		return value;
	}

	private BinlogType(int value, int types, String[] mysqlTypes, Type javaType, PageElement pageElement,
			String[] flinkLogicalTypes) {
		this.value = value;
		this.types = types;
		this.mysqlTypes = mysqlTypes;
		this.javaType = javaType;
		this.pageElement = pageElement;
		this.flinkLogicalTypes = flinkLogicalTypes;
	}

	public static BinlogType getByName(String mysqlType) {
		if (StringUtil.isNull(mysqlType)) {// 默认是任意匹配
			return UNRECOGNIZED;
		}
		for (BinlogType ele : BinlogType.values()) {
			if (ArrayUtils.contains(ele.getMysqlTypes(), mysqlType.toLowerCase())) {
				return ele;
			}
		}
		return UNRECOGNIZED;
	}

	public static BinlogType getByFlinkLogicalType(String flinkLogicalType) {
		if (StringUtil.isNull(flinkLogicalType)) {// 默认是任意匹配
			return UNRECOGNIZED;
		}
		for (BinlogType ele : BinlogType.values()) {
			if (ArrayUtils.contains(ele.getFlinkLogicalTypes(), flinkLogicalType)) {
				return ele;
			}
		}
		return UNRECOGNIZED;
	}

	public static BinlogType getByValue(int value) {
		if (value < 0) {// 默认是任意匹配
			return UNRECOGNIZED;
		}
		for (BinlogType ele : BinlogType.values()) {
			if (value == ele.getValue()) {
				return ele;
			}
		}
		return UNRECOGNIZED;
	}

	private String[] strType = new String[] { "java.lang.String", "java.util.Date", "java.sql.Date" };

	public String getSqlStr(String values) {
		if (StringUtil.isNull(values)) {
			return "";
		}
		if (ArrayUtils.contains(strType, this.getJavaType().getTypeName())) {
			String[] valueAry = values.split(",", 0);
			String arrayJoin = CollectionUtil.arrayJoin(valueAry, "','");
			return String.format("'%s'", arrayJoin);
		}
		return values;
	}

	@SuppressWarnings("unchecked")
	public static <T extends Serializable> T getValue(BinlogType columnType, String value) {
		Serializable retobj = null;
		switch (columnType) {
		case LONGLONG:
			retobj = Long.valueOf(value);
			break;
		case BIT:
		case TINY:
		case SHORT:
		case INT24:
		case LONG:
		case ENUM:
		case SET:
			retobj = Integer.valueOf(value);
			break;
		case FLOAT:
			retobj = Float.valueOf(value);
			break;

		case DOUBLE:
			retobj = Double.valueOf(value);
			break;
		case DECIMAL:
		case NEWDECIMAL:
			retobj = new BigDecimal(value);
			break;
		case GEOMETRY:
			try {
				retobj = Base64.decodeBase64(value);
			} catch (Exception e) {
				retobj = value;
			}
			break;
		case YEAR:
			try {
				retobj = Integer.valueOf(value);
			} catch (Exception e) {
				retobj = value;
			}
			break;
		case DATE:
			retobj = DateUtil.objToDate(value);
			break;
		// 20220121 DATETIME2对应mysql的datetime也是timestamp
		case TIMESTAMP2:
		case DATETIME2:
			retobj = Timestamp.valueOf(value);
			break;
		case BLOB:
		default:
			retobj = value;
			break;
		}
		return (T) retobj;
	}

	public static <T extends Serializable> T getValueBuildSql(BinlogType columnType, String value) {
		Serializable retobj = null;
		switch (columnType) {
			case LONGLONG:
				retobj = Long.valueOf(value);
				break;
			case BIT:
			case TINY:
			case SHORT:
			case INT24:
			case LONG:
			case ENUM:
			case SET:
				retobj = Integer.valueOf(value);
				break;
			case FLOAT:
				retobj = Float.valueOf(value);
				break;

			case DOUBLE:
				retobj = Double.valueOf(value);
				break;
			case DECIMAL:
			case NEWDECIMAL:
				retobj = new BigDecimal(value);
				break;
			case GEOMETRY:
				try {
					retobj = Base64.decodeBase64(value);
				} catch (Exception e) {
					retobj = value;
				}
				break;
			case YEAR:
				try {
					retobj = Integer.valueOf(value);
				} catch (Exception e) {
					retobj = value;
				}
				break;
//			case DATE:
//				retobj = DateUtil.objToDate(value);
//				break;
//			// 20220121 DATETIME2对应mysql的datetime也是timestamp
//			case TIMESTAMP2:
//			case DATETIME2:
//				retobj = Timestamp.valueOf(value);
//				break;
			case BLOB:
			default:
				String ret = "'"+value+"'";
				retobj = ret.replaceAll("\\\\","");
				break;
		}
		return (T) retobj;
	}

	public static <T extends Serializable> T getDateDefaultValue(BinlogType columnType) {
		Serializable retobj = "";
		switch (columnType) {
			case DATE:
			case TIMESTAMP2:
			case DATETIME2:
				retobj = "1970-01-01 00:00:00";
				break;
			default:
				break;
		}
		return (T) retobj;
	}

	public String[] getMysqlTypes() {
		return mysqlTypes;
	}
	
	
	public static BinlogType findBinlogTypeByMysql(String mysqldataType) {
		BinlogType binlogType=BinlogType.VARCHAR;
		for (BinlogType ele : BinlogType.values()) {
			if(ArrayUtils.contains(ele.getMysqlTypes(), mysqldataType)) {
				binlogType=ele;
				break;
			}
		}
		return binlogType;
	}

	public int getTypes() {
		return types;
	}
}
