
package net.wicp.tams.common.constant;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.tuple.Pair;

import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

/**
 * flink的资源配置
 * 
 * @author Andy
 *
 */
public enum FlinkResItem {
	jm_mem_heap("JM 堆内存", -1, "jobmanager.memory.heap.size", true),

	jm_mem_heapoff("JM 堆外内存", 128, "jobmanager.memory.off-heap.size", true),

	jm_mem_jvmmeta("JM JVM元数据", 256, "jobmanager.memory.jvm-metaspace.size", true),

	jm_mem_jvmoverhead("JM JVM开销", 192, "jobmanager.memory.jvm-overhead.min", true),

	jm_cpu("JM CPU(ms)", 1000, "kubernetes.jobmanager.cpu", true), // 1000ms就是1个核

	tm_mem_heap_slot("TM 单槽堆内存", -1, "taskmanager.memory.task.heap.size", true),

	tm_mem_heap_frame("TM 框架堆内存", 128, "taskmanager.memory.framework.heap.size", true),

	tm_mem_managed("TM 管理内存占比(百分比)", 40, "taskmanager.memory.managed.fraction", true), // 即0.4

	tm_mem_heapoff_frame("TM 框架堆外内存", 128, "taskmanager.memory.framework.off-heap.size", true),

	tm_mem_heapoff_task("TM 任务堆外内存", 0, "taskmanager.memory.task.off-heap.size", true),

	tm_mem_network("TM 网络内存占比(百分比)", 10, "taskmanager.memory.network.fraction", true), // 即0.1

	tm_mem_jvmoverhead("TM JVM开销", 192, "taskmanager.memory.jvm-overhead.min", true),

	tm_mem_jvmmeta("TM JVM元数据", 256, "taskmanager.memory.jvm-metaspace.size", true),

	tm_mem_proc("整个进程的内存", -1, "taskmanager.memory.process.size", false), // 因为设置为比例，那么必须要设置它或flink内存，且不要设置堆内存

	tm_cpu_slot("TM 单槽CPU(ms)", -1, "kubernetes.taskmanager.cpu", true);

	private final String desc;

	private final String key;

	private final boolean tbFieldInclude;// 数据库是否有记录

	public boolean isTbFieldInclude() {
		return tbFieldInclude;
	}

	public String getKey() {
		return key;
	}

	private final int value;// 默认值

	public final <T extends Object> int getValue(T dataObj) {
		if (dataObj == null) {
			return this.getValue();
		}
		try {
			Object property = PropertyUtils.getProperty(dataObj, StringUtil.convertStr(this.name()));// 放的是驮峰的数据。
			return Integer.parseInt(String.valueOf(property));
		} catch (Exception e) {
			return this.getValue();
		}
	}

	public int getValue() {
		return value;
	}

	public String getDesc() {
		return desc;
	}

	private FlinkResItem(String desc, int value, String key, boolean tbFieldInclude) {
		this.desc = desc;
		this.value = value;
		this.key = key;
		this.tbFieldInclude = tbFieldInclude;
	}

	// 默认配置得到JM内存
	public static int getJmMenDefault(int heap) {
		return heap + jm_mem_heapoff.getValue() + jm_mem_jvmmeta.getValue() + jm_mem_jvmoverhead.getValue();
	}
	
	public  static <T extends Object> int getJmMen(T dataObj) {
		return getJmMenDefault(jm_mem_heap.getValue(dataObj));
	}

	// 得到TM内存（全在一个TM中）
	public static Pair<Integer, Short> getTmMenDefault(short slot) {
		return getTmMen(slot, Integer.MAX_VALUE, null);
	}

	// 得到TM内存，且内存必须小于destMenPreTask数
	public static <T extends Object> Pair<Integer, Short> getTmMen(short slot, int destMenPreTask, T dataObj) {
		return getTmMen(slot, destMenPreTask, (short) 1, dataObj);
	}
	/**
	 * 得到TM内存，不限制Task的最大内存数
	 * @param <T>
	 * @param slot  每个task的最大槽数
	 * @param dataObj 模板数据
	 * @return  计算单个task的内存大小
	 */
	public static <T extends Object> int getTmMen(short slot, T dataObj) {
		return getTmMen(slot, Integer.MAX_VALUE, (short) 1, dataObj).getLeft();
	}

	// L：单个task内存数 R:多少个task
	private static <T extends Object> Pair<Integer, Short> getTmMen(short slot, int destMenPreTask, short destTaskNum,
			T dataObj) {
		int slotHeap = tm_mem_heap_slot.getValue(dataObj);// 单槽内存堆。
		short slotPreTask = (short) (slot / destTaskNum + (slot % destTaskNum > 0 ? 1 : 0));// 每task有的最大槽数
		int other = tm_mem_heap_frame.getValue(dataObj) + tm_mem_heapoff_frame.getValue(dataObj)
				+ tm_mem_heapoff_task.getValue(dataObj) + slotHeap * slotPreTask;
		int networkPer = tm_mem_network.getValue(dataObj);
		int flinkMem;
		boolean isMultTask = destTaskNum > 1;
		if (networkPer > 0 && isMultTask) {// 大于0并且有多个task时需要正常比例设置内存
			flinkMem = other * 100 / (100 - tm_mem_managed.getValue(dataObj) - networkPer);
		} else {// 不是正常比例就取最小值64M,如stream程序
			flinkMem = (other + 64) * 100 / (100 - tm_mem_managed.getValue(dataObj));
		}
		int procMem = flinkMem + tm_mem_jvmmeta.getValue(dataObj) + tm_mem_jvmoverhead.getValue(dataObj);
		if (procMem <= destMenPreTask) {
			return Pair.of(procMem, destTaskNum);
		} else {
			// 分的不对，加多一个task再试。
			return getTmMen((short) (slotPreTask - 1), destMenPreTask, (short) (destTaskNum + 1), dataObj);
		}
	}

	// 得到cpu
	public static <T extends Object> double getTmCpu(short slotPerTask, T dataObj) {
		if (dataObj == null || tm_cpu_slot.getValue(dataObj) == -1) {
			throw new ProjectExceptionRuntime(ExceptAll.Project_default, "tm需要传入每个槽要用的cpu");
		}
		return tm_cpu_slot.getValue(dataObj) * slotPerTask / 1000d;
	}

	public static <T extends Object> double getJmCpu(T dataObj) {
		return jm_cpu.getValue(dataObj) / 1000d;
	}

	/***
	 * 得到配置信息
	 * 
	 * @return
	 */
	public static <T extends Object> Map<String, String> config(short slot, int destMenPreTask, T dataObj) {
		Map<String, String> retmap = new HashMap<String, String>();
		// 得到JM配置
		retmap.put(jm_mem_heap.getKey(), String.valueOf(jm_mem_heap.getValue(dataObj)) + "m");
		retmap.put(jm_mem_heapoff.getKey(), String.valueOf(jm_mem_heapoff.getValue(dataObj)) + "m");
		retmap.put(jm_mem_jvmmeta.getKey(), String.valueOf(jm_mem_jvmmeta.getValue(dataObj)) + "m");
		// 最小和最大一样，就是size
		retmap.put("jobmanager.memory.jvm-overhead.min", String.valueOf(jm_mem_jvmoverhead.getValue(dataObj)) + "m");
		retmap.put("jobmanager.memory.jvm-overhead.max", String.valueOf(jm_mem_jvmoverhead.getValue(dataObj)) + "m");
		// retmap.put("jobmanager.memory.jvm-overhead.fraction", "0.1");
		retmap.put(jm_cpu.getKey(), String.valueOf(jm_cpu.getValue(dataObj) / 1000d));
		// 得到TM配置
		// TaskManager memory configuration failed: Either required fine-grained memory
		// (taskmanager.memory.task.heap.size and taskmanager.memory.managed.size), or
		// Total Flink Memory size (Key: 'taskmanager.memory.flink.size' , default: null
		// (fallback keys: [])), or Total Process Memory size (Key:
		// 'taskmanager.memory.process.size' , default: null (fallback keys: [])) need
		// to be configured explicitly.
		// 设置进程内存时就不能设置堆的内存了
		// retmap.put(tm_mem_heap_slot.getKey(),
		// String.valueOf(tm_mem_heap_slot.getValue(dataObj) * slot) + "m");
		Pair<Integer, Short> tmMenAndNum = getTmMen(slot, destMenPreTask, dataObj);
		retmap.put(tm_mem_proc.getKey(), String.valueOf(tmMenAndNum.getLeft().intValue()) + "m");
		retmap.put(tm_mem_heap_frame.getKey(), String.valueOf(tm_mem_heap_frame.getValue(dataObj)) + "m");
		retmap.put(tm_mem_managed.getKey(), String.valueOf(tm_mem_managed.getValue(dataObj) / 100d));// 权重暂是不设置：taskmanager.memory.managed.consumer-weights
		retmap.put(tm_mem_heapoff_frame.getKey(), String.valueOf(tm_mem_heapoff_frame.getValue(dataObj)) + "m");
		retmap.put(tm_mem_heapoff_task.getKey(), String.valueOf(tm_mem_heapoff_task.getValue(dataObj)) + "m");
		int networkPer = tm_mem_network.getValue(dataObj);
		boolean isMultTask = tmMenAndNum.getRight().intValue() > 1;
		if (networkPer > 0 && isMultTask) {// 大于0并且有多个task时需要正常比例设置内存
			retmap.put(tm_mem_network.getKey(), String.valueOf(tm_mem_network.getValue(dataObj) / 100d));
		} else {// 不是正常比例就取最小值64M,如stream程序
			retmap.put("taskmanager.memory.network.min", "64m");
			retmap.put("taskmanager.memory.network.max", "64m");
		}
		// 最小和最大一样，就是size
		retmap.put("taskmanager.memory.jvm-overhead.min", String.valueOf(tm_mem_jvmoverhead.getValue(dataObj)) + "m");
		retmap.put("taskmanager.memory.jvm-overhead.max", String.valueOf(tm_mem_jvmoverhead.getValue(dataObj)) + "m");
		// retmap.put("taskmanager.memory.jvm-overhead.fraction", "0.1");
		retmap.put(tm_mem_jvmmeta.getKey(), String.valueOf(tm_mem_jvmmeta.getValue(dataObj)) + "m");
		retmap.put(tm_cpu_slot.getKey(), String.valueOf(tm_cpu_slot.getValue(dataObj) * slot / 1000d));
		// 设置每个task拥有的槽数： TaskManagerOptions.NUM_TASK_SLOTS
		short numPerTask = (short) (slot / tmMenAndNum.getRight().shortValue()
				+ (slot % tmMenAndNum.getRight().shortValue() > 0 ? 1 : 0));
		retmap.put("taskmanager.numberOfTaskSlots", String.valueOf(numPerTask));
		return retmap;
	}

}
