/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common-apiext
 * 项目描述：工具
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.constant;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.constant.dic.intf.IEnumCombobox;

/***
 * 分库分表模式，默认为不分库不分表
 * 
 * @author andy.zhou
 * 
 */
public enum DrdsPattern implements IEnumCombobox {
	no("不分库不分表", "^%s$", "^%s$"),

	db("只分库不分表", "^%s_[0-9a-zA-Z]{4}_[0-9]{2,}$", "^%s_[0-9a-zA-Z]{4}"),

	db_equaltb("只分库不分表(表名相同库名不同)", "^%s$", "^%s$"), // 只分库不分表的变种，这种情况较为普遍

	dbtb("即分库也分表", "^%s_[0-9a-zA-Z]{4}_[0-9]{2,}$", "%s_[0-9a-zA-Z]{4}_[0-9]{2,}$");

	private final String desc;

	private final String dbPatternFormat;// 分库模式（支持rds）

	private final String tbPatternFormat;// 分表模式(支持rds)

	private DrdsPattern(String desc, String dbPatternFormat, String tbPatternFormat) {
		this.desc = desc;
		this.dbPatternFormat = dbPatternFormat;
		this.tbPatternFormat = tbPatternFormat;
	}

	public String getDesc() {
		return desc;
	}

	public String getName() {
		return this.name();
	}

	@Override
	public String getDesc_zh() {
		return this.desc;
	}

	public String getRootDb(String oriDb) {
		Pattern dbPattern = Pattern.compile(this.getDbPatternFormat());
		Matcher m = dbPattern.matcher(oriDb);
		String rootstr = m.find() ? m.group(1) : null;
		return rootstr;
	}

	public String getRootTb(String oriTb) {
		Pattern dbPattern = Pattern.compile(this.getTbPatternFormat());
		Matcher m = dbPattern.matcher(oriTb);
		String rootstr = m.find() ? m.group(1) : null;
		return rootstr;
	}

	public String getTargetTb(String pre, Pair<String, String> oriDbtb) {
		String targetTb = null;
		switch (this) {
		case no:
		case db_equaltb:
			targetTb = StringUtil.isNull(pre) ? String.format("%s_%s", oriDbtb.getLeft(), oriDbtb.getRight())
					: String.format("%s_%s_%s", pre, oriDbtb.getLeft(), oriDbtb.getRight());
			break;
		case db:		
		case dbtb:
			targetTb = StringUtil.isNull(pre) ? String.format("%s", getRootDb(oriDbtb.getRight()))
					: String.format("%s_%s", pre, getRootDb(oriDbtb.getRight()));
		default:
			break;
		}
		return targetTb;
	}

	public boolean isSameGroup(Pair<String, String> standardDbtb, Pair<String, String> contrastDbtb) {
		boolean isSame = false;
		switch (this) {
		case no:
			isSame = standardDbtb.getLeft().equals(contrastDbtb.getLeft())
					&& standardDbtb.getRight().equalsIgnoreCase(contrastDbtb.getRight());
			break;
		case db_equaltb:
			isSame = standardDbtb.getRight().equalsIgnoreCase(contrastDbtb.getRight());
			break;
		default:// 正则表达式判断表是否相同
			String rootDb1 = getRootDb(standardDbtb.getLeft());
			String rootTb1 = getRootTb(standardDbtb.getRight());

			String rootDb2 = getRootDb(contrastDbtb.getLeft());
			String rootTb2 = getRootTb(contrastDbtb.getRight());

			isSame = rootDb1 != null && rootDb2 != null && rootDb1.equals(rootDb2) && rootTb1 != null && rootTb2 != null
					&& rootTb1.equals(rootTb2);
			break;
		}
		return isSame;
	}

	@Override
	public String getDesc_en() {
		return this.name();
	}

	public static DrdsPattern find(String name) {
		if (StringUtils.isEmpty(name)) {
			return DrdsPattern.no;
		}
		for (DrdsPattern ele : DrdsPattern.values()) {
			if (name.equalsIgnoreCase(ele.name())) {
				return ele;
			}
		}
		return DrdsPattern.no;
	}

	public String getDbPatternFormat() {
		return dbPatternFormat;
	}

	public String getTbPatternFormat() {
		return tbPatternFormat;
	}

}
