/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common-apiext
 * 项目描述：工具
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.apiext;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

import net.wicp.tams.common.Conf;

public class TamsURLClassLoader extends URLClassLoader {

	private ClassLoader system;

	private String[] systemfilterary = StringUtil.isNull(Conf.get("common.apiext.classload.systemfilter"))
			? new String[0]
			: Conf.get("common.apiext.classload.systemfilter").split(","); // new String[] {
	// "net.wicp.tams.common"
	// };

	private boolean childFirst = Conf.getBoolean("common.apiext.classload.child-first");

	public TamsURLClassLoader(URL[] classpath, ClassLoader parent) {
		super(classpath, parent);
		system = getSystemClassLoader();
	}

	private boolean canFindFormSystem(String name) {
		for (String systemfilter : systemfilterary) {
			if (name.startsWith(systemfilter)) {
				return false;
			}
		}
		return true;
	}

	@Override
	protected synchronized Class<?> loadClass(String name, boolean resolve) throws ClassNotFoundException {
		// First, check if the class has already been loaded
		Class<?> c = findLoadedClass(name);
		if (c == null) {
			if (system != null && canFindFormSystem(name)) {
				try {
					// checking system: jvm classes, endorsed, cmd classpath, etc.
					c = system.loadClass(name);
				} catch (ClassNotFoundException ignored) {
				}
			}
			if (c == null) {
				if (childFirst) {
					try {
						// checking local
						c = findClass(name);
					} catch (ClassNotFoundException e) {
						// checking parent
						// This call to loadClass may eventually call findClass again, in case the
						// parent doesn't find anything.
						c = super.loadClass(name, resolve);
					}
				} else {
					try {
						c = super.loadClass(name, resolve);
					} catch (ClassNotFoundException e) {
						c = findClass(name);
					}
				}
			}
		}
		if (resolve) {
			resolveClass(c);
		}
		return c;
	}

	@Override
	public URL getResource(String name) {
		URL url = null;
		if (system != null && canFindFormSystem(name)) {
			url = system.getResource(name);
		}
		if (url == null) {
			if (childFirst) {
				url = findResource(name);
				if (url == null) {
					// This call to getResource may eventually call findResource again, in case the
					// parent doesn't find anything.
					url = super.getResource(name);
				}
			} else {
				url = super.getResource(name);
				if (url == null) {
					url = findResource(name);
				}
			}

		}
		return url;
	}

	@Override
	public Enumeration<URL> getResources(String name) throws IOException {
		/**
		 * Similar to super, but local resources are enumerated before parent resources
		 */
		Enumeration<URL> systemUrls = null;
		if (system != null && canFindFormSystem(name)) {
			systemUrls = system.getResources(name);
		}
		Enumeration<URL> localUrls = findResources(name);
		Enumeration<URL> parentUrls = null;
		if (getParent() != null) {
			parentUrls = getParent().getResources(name);
		}
		final List<URL> urls = new ArrayList<URL>();
		if (systemUrls != null) {
			while (systemUrls.hasMoreElements()) {
				urls.add(systemUrls.nextElement());
			}
		}
		if (childFirst) {
			if (localUrls != null) {
				while (localUrls.hasMoreElements()) {
					urls.add(localUrls.nextElement());
				}
			}
			if (parentUrls != null) {
				while (parentUrls.hasMoreElements()) {
					urls.add(parentUrls.nextElement());
				}
			}
		} else {
			if (parentUrls != null) {
				while (parentUrls.hasMoreElements()) {
					urls.add(parentUrls.nextElement());
				}
			}
			if (localUrls != null) {
				while (localUrls.hasMoreElements()) {
					urls.add(localUrls.nextElement());
				}
			}
		}

		return new Enumeration<URL>() {
			Iterator<URL> iter = urls.iterator();

			public boolean hasMoreElements() {
				return iter.hasNext();
			}

			public URL nextElement() {
				return iter.next();
			}
		};
	}

	@Override
	public InputStream getResourceAsStream(String name) {
		URL url = getResource(name);
		try {
			return url != null ? url.openStream() : null;
		} catch (IOException e) {
		}
		return null;
	}

}