/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common-apiext
 * 项目描述：工具
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.apiext;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.UnknownHostException;
import java.util.Enumeration;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.callback.IConvertObj;
import net.wicp.tams.common.constant.EPlatform;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.thread.ThreadPool;

@Slf4j
public class OSinfo {
	private static String OS = System.getProperty("os.name").toLowerCase();

	private static OSinfo _instance = new OSinfo();

	private EPlatform platform;

	private OSinfo() {
	}

	public static boolean isLinux() {
		return OS.indexOf("linux") >= 0;
	}

	public static boolean isMacOS() {
		return OS.indexOf("mac") >= 0 && OS.indexOf("os") > 0 && OS.indexOf("x") < 0;
	}

	public static boolean isMacOSX() {
		return OS.indexOf("mac") >= 0 && OS.indexOf("os") > 0 && OS.indexOf("x") > 0;
	}

	public static boolean isWindows() {
		return OS.indexOf("windows") >= 0;
	}

	public static boolean isOS2() {
		return OS.indexOf("os/2") >= 0;
	}

	public static boolean isSolaris() {
		return OS.indexOf("solaris") >= 0;
	}

	public static boolean isSunOS() {
		return OS.indexOf("sunos") >= 0;
	}

	public static boolean isMPEiX() {
		return OS.indexOf("mpe/ix") >= 0;
	}

	public static boolean isHPUX() {
		return OS.indexOf("hp-ux") >= 0;
	}

	public static boolean isAix() {
		return OS.indexOf("aix") >= 0;
	}

	public static boolean isOS390() {
		return OS.indexOf("os/390") >= 0;
	}

	public static boolean isFreeBSD() {
		return OS.indexOf("freebsd") >= 0;
	}

	public static boolean isIrix() {
		return OS.indexOf("irix") >= 0;
	}

	public static boolean isDigitalUnix() {
		return OS.indexOf("digital") >= 0 && OS.indexOf("unix") > 0;
	}

	public static boolean isNetWare() {
		return OS.indexOf("netware") >= 0;
	}

	public static boolean isOSF1() {
		return OS.indexOf("osf1") >= 0;
	}

	public static boolean isOpenVMS() {
		return OS.indexOf("openvms") >= 0;
	}

	/**
	 * 获取操作系统名字
	 * 
	 * @return 操作系统名
	 */
	public static EPlatform getOSname() {
		if (isAix()) {
			_instance.platform = EPlatform.AIX;
		} else if (isDigitalUnix()) {
			_instance.platform = EPlatform.Digital_Unix;
		} else if (isFreeBSD()) {
			_instance.platform = EPlatform.FreeBSD;
		} else if (isHPUX()) {
			_instance.platform = EPlatform.HP_UX;
		} else if (isIrix()) {
			_instance.platform = EPlatform.Irix;
		} else if (isLinux()) {
			_instance.platform = EPlatform.Linux;
		} else if (isMacOS()) {
			_instance.platform = EPlatform.Mac_OS;
		} else if (isMacOSX()) {
			_instance.platform = EPlatform.Mac_OS_X;
		} else if (isMPEiX()) {
			_instance.platform = EPlatform.MPEiX;
		} else if (isNetWare()) {
			_instance.platform = EPlatform.NetWare_411;
		} else if (isOpenVMS()) {
			_instance.platform = EPlatform.OpenVMS;
		} else if (isOS2()) {
			_instance.platform = EPlatform.OS2;
		} else if (isOS390()) {
			_instance.platform = EPlatform.OS390;
		} else if (isOSF1()) {
			_instance.platform = EPlatform.OSF1;
		} else if (isSolaris()) {
			_instance.platform = EPlatform.Solaris;
		} else if (isSunOS()) {
			_instance.platform = EPlatform.SunOS;
		} else if (isWindows()) {
			_instance.platform = EPlatform.Windows;
		} else {
			_instance.platform = EPlatform.Others;
		}
		return _instance.platform;
	}

	public static String findIpAddressTrue() {
		return StringUtil.isNull(Conf.get("common.apiext.os.ip")) ? findFirstNonLoopbackAddress().getHostAddress()
				: Conf.get("common.apiext.os.ip");
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	public static Result startCmd(String cmd, IConvertObj<Result> callResult) {
		try {
			Process ps = Runtime.getRuntime().exec(cmd);
			ps.waitFor();
			Future query = ThreadPool.getDefaultPool().submit(new Callable() {
				public String call() throws Exception {
					// process.getInputStream是用来读取控制台命令结果的
					// process.getOutputStream是用来往控制台写入参数的
					// 在helm waring出现在errorstream，但同时会存在inputstream，所以需要合并。
					String retstr = IOUtil.slurp(ps.getErrorStream(), Conf.getSystemEncode());
					retstr += IOUtil.slurp(ps.getInputStream(), Conf.getSystemEncode());
					return retstr;
				}
			});
			String retstr = "";
			try {
				retstr = (String) query.get(500L, TimeUnit.MILLISECONDS);
			} catch (Exception e) {
				log.error("得到返回流时超时", e);
				Result error = new Result(ExceptAll.project_timeout);// Result.getError("得到返回流时超时");
				return error;
			}
			// callResult为空表示不关心返回结果
			Result ret = callResult == null ? Result.getSuc() : callResult.getObj(retstr, null);
			return ret;
//			if (retstr.contains("Error:")) {
//				return Result.getError(retstr);
//			} else {
//				return Result.getSuc();
//			}
		} catch (IOException ioe) {
			log.error("IO异常，文件有误", ioe);
			return Result.getError("IO异常，文件有误");
		} catch (InterruptedException e) {
			log.error("中断异常", e);
			return Result.getError("中断异常");
		}
	}

	public static InetAddress findFirstNonLoopbackAddress() {
		InetAddress result = null;
		try {
			// 记录网卡最小索引
			int lowest = Integer.MAX_VALUE;
			// 获取所有网卡
			for (Enumeration<NetworkInterface> nics = NetworkInterface.getNetworkInterfaces(); nics
					.hasMoreElements();) {
				NetworkInterface ifc = nics.nextElement();
				if (ifc.isUp()) {
					log.trace("Testing interface: " + ifc.getDisplayName());
					if (ifc.getIndex() < lowest || result == null) {
						lowest = ifc.getIndex(); // 记录索引
					} else if (result != null) {
						continue;
					}

					// @formatter:off
					if (!ignoreInterface(ifc.getDisplayName())) { // 是否是被忽略的网卡
						for (Enumeration<InetAddress> addrs = ifc.getInetAddresses(); addrs.hasMoreElements();) {
							InetAddress address = addrs.nextElement();
							if (address instanceof Inet4Address && !address.isLoopbackAddress()
									&& !ignoreAddress(address)) {
								log.trace("Found non-loopback interface: " + ifc.getDisplayName());
								result = address;
							}
						}
					}
					// @formatter:on
				}
			}
		} catch (IOException ex) {
			log.error("Cannot get first non-loopback address", ex);
		}

		if (result != null) {
			return result;
		}

		try {
			return InetAddress.getLocalHost(); // 如果以上逻辑都没有找到合适的网卡，则使用JDK的InetAddress.getLocalhost()
		} catch (UnknownHostException e) {
			log.warn("Unable to retrieve localhost");
		}

		return null;
	}

	private static boolean ignoreInterface(String interfaceName) {
		String macnames = Conf.get("common.apiext.os.ignore.mac");
		if (StringUtil.isNull(macnames)) {
			return false;
		}
		String[] macnameAry = macnames.split(",");
		for (String regex : macnameAry) {
			if (interfaceName.matches(regex)) {
				log.trace("Ignoring interface: " + interfaceName);
				return true;
			}
		}
		return false;
	}

	private static boolean ignoreAddress(InetAddress address) {
		String ips = Conf.get("common.apiext.os.ignore.ip");
		if (StringUtil.isNull(ips)) {
			return false;
		}
		String[] ipAry = ips.split(",");
		for (String regex : ipAry) {
			if (!address.getHostAddress().matches(regex) && !address.getHostAddress().startsWith(regex)) {
				log.trace("Ignoring address: " + address.getHostAddress());
				return true;
			}
		}
		return false;
	}

	/***
	 * 设置环境变量 TODO: source /etc/profile不生效
	 */
	@SuppressWarnings("rawtypes")
	public static Result setEnv(String filename, String... values) {
		if (isWindows()) {
			return Result.getError("window不支持设置环境变量");
		}
		Map map = CollectionUtil.newMap(values);
		StringBuffer buff = new StringBuffer("echo -e  \"");
		StringBuffer buffkey = new StringBuffer();
		for (Object key : map.keySet()) {
			// 其它系统偿试用linux
			buff.append(String.format("%s=%s\\n", key, map.get(key)));
			buffkey.append(" " + key);
		}
		String filenameTrue = StringUtil.isNull(filename) ? String.valueOf(UUIDGenerator.getUniqueLong()) : filename;
		buff.append("export" + buffkey.toString() + "\" >  /etc/profile.d/" + filenameTrue + ".sh");
		Result exportRs = OSinfo.startCmd(buff.toString(), null);
		if (!exportRs.isSuc()) {
			return exportRs;
		}
		Result sourceRs = OSinfo.startCmd("source /etc/profile", null);
		return sourceRs;
	}

}
