/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common-apiext
 * 项目描述：工具
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.apiext;

import java.io.Serializable;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.collections.list.GrowthList;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.MutableTriple;
import org.mvel2.MVEL;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.callback.ICount;

/**
 * 集合的简便操作方法
 * 
 * @author 偏锋书生
 *
 */
@SuppressWarnings({ "rawtypes", "unchecked" })
@Slf4j
public abstract class CollectionUtil {
	public static int indexOf(Object[][] oriDatas, Object[] curData, int start) {
		if (oriDatas == null || oriDatas.length == 0 || curData == null) {
			return -1;
		}
		for (int i = start; i < oriDatas.length; i++) {
			Object[] oriData = oriDatas[i];
			if (oriData.length != curData.length) {
				continue;
			}
			boolean isEqual = true;
			for (int j = 0; j < oriData.length; j++) {
				if (!Objects.equals(oriData[j], curData[j])) {
					isEqual = false;
					break;
				}
			}
			if (isEqual) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * 把List通过分隔符进行分隔
	 * 
	 * @param fromList 要连接的List
	 * @param joinStr  连接的字符串
	 * @return String 连接后字符串
	 */

	public static <T> String listJoin(Collection<T> fromList, String joinStr) {
		if (CollectionUtils.isEmpty(fromList))
			return null;
		joinStr = StringUtils.isEmpty(joinStr) ? "," : joinStr;
		StringBuffer toList = new StringBuffer();
		Iterator iterator = fromList.iterator();
		toList.append(String.valueOf(iterator.next()));
		while (iterator.hasNext()) {
			String string = String.valueOf(iterator.next());
			toList.append(joinStr);
			toList.append(string);
		}
		return toList.toString();
	}

	/**
	 * 把Array通过分隔符进行分隔
	 * 
	 * @param fromList 要连接的数组
	 * @param joinStr  连接的字符串
	 * @return 连接后的字符串
	 */

	public static String arrayJoin(Object[] fromList, String joinStr) {
		if (ArrayUtils.isEmpty(fromList))
			return null;
		List<?> objlist = Arrays.asList(fromList);
		return listJoin(objlist, joinStr);
	}

	/****
	 * 数组的合并<br>
	 * eg: CollectionUtil.arrayMerge(String[].class, ary1,ary2)
	 * 
	 * @param clazz  数组类型
	 * @param a      合并数组一
	 * @param b      合并数组二
	 * @param noSame 去掉相同项，true：去掉,false，不去掉
	 * 
	 * @param <T>    数组里元素类型
	 * @return 全并后的数组,去除了相同的项
	 */

	public static <T> T[] arrayMerge(Class<T[]> clazz, T[] a, T[] b, boolean noSame) {
		if (ArrayUtils.isEmpty(a)) {
			return b;
		}
		if (ArrayUtils.isEmpty(b)) {
			return a;
		}
		if (noSame) {
			List<T> retlist = new ArrayList<>();
			for (T t : a) {
				if (!retlist.contains(t)) {
					retlist.add(t);
				}
			}
			for (T t : b) {
				if (!retlist.contains(t)) {
					retlist.add(t);
				}
			}
			return retlist.toArray(newArrayByArrayClass(clazz, retlist.size()));
		} else {
			T[] newArray = newArrayByArrayClass(clazz, a.length + b.length);
			System.arraycopy(a, 0, newArray, 0, a.length);
			System.arraycopy(b, 0, newArray, a.length, b.length);
			return newArray;
		}
	}

	/***
	 * 取交集
	 * 
	 * @param clazz
	 * @param a
	 * @param b
	 * @return
	 */
	public static <T> T[] arrayAnd(Class<T[]> clazz, T[] a, T[] b) {
		if (ArrayUtils.isEmpty(a) || ArrayUtils.isEmpty(b)) {
			return newArrayByArrayClass(clazz, 0);
		}
		List<T> retlist = new ArrayList<>();
		for (T t : a) {
			if (ArrayUtils.contains(b, t)) {
				retlist.add(t);
			}
		}
		return retlist.toArray(newArrayByArrayClass(clazz, retlist.size()));
	}

	/***
	 * 取交集
	 * 
	 * @param clazz
	 * @param a
	 * @param b
	 * @return
	 */
	public static <T> List<T> listAnd(Class<T> clazz, List<T> a, List<T> b, Comparator<T> comparator) {
		if (CollectionUtils.isEmpty(a) || CollectionUtils.isEmpty(b) || comparator == null) {
			return Collections.EMPTY_LIST;
		}
		List<T> retlist = new ArrayList<T>();
		for (T add : b) {
			boolean eq = false;
			for (T base : a) {
				if (comparator.compare(base, add) == 0) {// 相等
					eq = true;
					break;
				}
			}
			if (eq) {
				retlist.add(add);
			}
		}
		return retlist;
	}

	/***
	 * 取并集
	 * 
	 * @param clazz
	 * @param a
	 * @param b
	 * @return
	 */
	public static <T> List<T> listOr(Class<T> clazz, List<T> a, List<T> b, Comparator<T> comparator) {
		if (CollectionUtils.isEmpty(a)) {
			return b;
		}
		if (CollectionUtils.isEmpty(b)) {
			return a;
		}
		List<T> retlist = new ArrayList<T>();// 有些list不能添加
		if (comparator == null) {
			retlist.addAll(a);
			retlist.addAll(b);
			return retlist;
		}
		retlist.addAll(a);
		for (T add : b) {
			boolean eq = false;
			for (T base : a) {
				if (comparator.compare(base, add) == 0) {// 相等
					eq = true;
					break;
				}
			}
			if (!eq) {
				retlist.add(add);
			}
		}
		return retlist;
	}



	/***
	 * 通过类型创建数组
	 * 
	 * @param clazz  类型
	 * @param length 数组长度
	 * 
	 * @param <T>    数组里元素类型
	 * @return 数组实例对象
	 */
	public static <T> T[] newArrayByArrayClass(Class<T[]> clazz, int length) {
		return (T[]) Array.newInstance(clazz.getComponentType(), length);
	}

	/**
	 * 把list分成sumPerRow一组
	 * 
	 * @param inputList 要分隔的List
	 * @param sumPerRow 第个List的个数
	 * @param <T>       List元素类型
	 * @param countObj  自定义计数
	 * @param replenish 是否要补充满
	 * @return 二维List
	 */
	public static <T> List<List<T>> splitList(Class<T> clazz, List<T> inputList, int sumPerRow, ICount<T> countObj,
			boolean replenish) {
		List<List<T>> returnList = new ArrayList<List<T>>();
		if (CollectionUtils.isEmpty(inputList) || sumPerRow == 0) {
			return returnList;
		}
		int count = 0;
		List<T> addList = new ArrayList<T>();
		for (int i = 0; i < inputList.size(); i++) {
			T t = inputList.get(i);
			int curCount = countObj == null ? 1 : countObj.countObj(t);
			if (curCount + addList.size() > sumPerRow) {// 超过了最大行值
				if (addList.size() > 0) {// 当时已有数据
					if (replenish && count < sumPerRow) {// 补数据
						try {
							for (int j = 0; j < sumPerRow - count; j++) {
								T newInstance = clazz.newInstance();
								addList.add(newInstance);
							}

						} catch (Exception e) {
						}
					}
					returnList.add(addList);
					addList = new ArrayList<T>();
					count = 0;
				}
				addList.add(t);
				count += curCount;
			} else {
				addList.add(t);
				count += curCount;
			}
			if (count >= sumPerRow) {
				if (replenish && count < sumPerRow) {// 补数据
					try {
						for (int j = 0; j < sumPerRow - count; j++) {
							T newInstance = clazz.newInstance();
							addList.add(newInstance);
						}

					} catch (Exception e) {
					}
				}
				returnList.add(addList);
				addList = new ArrayList<T>();
				count = 0;
			} else {
				if ((i == inputList.size() - 1) && (addList.size() > 0)) {// 最后一个元素
					if (replenish && count < sumPerRow) {// 补数据
						try {
							for (int j = 0; j < sumPerRow - count; j++) {
								T newInstance = clazz.newInstance();
								addList.add(newInstance);
							}

						} catch (Exception e) {
						}
					}
					returnList.add(addList);
				}
			}
		}
		return returnList;
	}

	public static <T> List<List<T>> splitList(Class<T> clazz, List<T> inputList, int sumPerRow, boolean replenish) {
		return splitList(clazz, inputList, sumPerRow, null, replenish);// 没有特殊
	}

	public static <T> List<List<T>> splitList(Class<T> clazz, List<T> inputList, int sumPerRow) {
		return splitList(clazz, inputList, sumPerRow, null, false);// 没有特殊
	}

	/***
	 * 把list分成listnum个list
	 * 
	 * @param inputList 输入的原始list
	 * @param listnum   每几个元素为一组进行分隔
	 * @param <T>       List元素类型
	 * @return 二维List
	 */
	public static <T> List<List<T>> splitListN(Class<T> clazz, List<T> inputList, int listnum) {
		if (CollectionUtils.isEmpty(inputList) || listnum == 0) {
			return null;
		}
		int tempint = inputList.size() / listnum;
		if (tempint == 0) {
			List<List<T>> returnList = new ArrayList<List<T>>();
			returnList.add(inputList);
			return returnList;
		}
		List<List<T>> retlist = splitList(clazz, inputList.subList(0, tempint * listnum), tempint);
		int j = 0;
		for (int i = 0; i < inputList.size() % listnum; i++) {
			retlist.get(j++).add(inputList.get(tempint * listnum + i));
		}
		return retlist;
	}

	/**
	 * 通过List得到对象的单个列值
	 * 
	 * @param fromList 要操作的数据源
	 * @param colName  要提取的列名
	 * @return List 提取预定列的List
	 */
	public static List<?> getColFromObj(List<?> fromList, String colName) {
		List<Object> retList = new ArrayList<Object>();
		if (CollectionUtils.isEmpty(fromList)) {
			return retList;
		}
		for (Object object : fromList) {
			Object result = null;
			if (ReflectAssist.isInterface(object.getClass(), "java.util.Map")) {
				Map tempObjMap = (Map) object;
				result = tempObjMap.get(colName);
			} else {
				result = MVEL.eval(colName, object);
			}
			retList.add(result);
		}
		return retList;
	}

	/***
	 * 得到list的某个列
	 * 
	 * @param fromList orilist
	 * @param colName  列名
	 * @return
	 */
	public static <T> List<String> getColValStrFromObj(List<T> fromList, String colName) {
		List<String> retList = new ArrayList<String>();
		if (CollectionUtils.isEmpty(fromList)) {
			return retList;
		}
		for (T object : fromList) {
			String result = null;
			if (ReflectAssist.isInterface(object.getClass(), "java.util.Map")) {
				Map tempObjMap = (Map) object;
				result = String.valueOf(tempObjMap.get(colName));
			} else {
				result = String.valueOf(MVEL.eval(colName, object));
			}
			retList.add(result);
		}
		return retList;
	}

	public static <T extends Serializable> Set<T> getColSetFromObj(List<?> fromList, String colName) {
		Set<T> retset = new HashSet<T>();
		if (CollectionUtils.isEmpty(fromList)) {
			return retset;
		}
		for (Object object : fromList) {
			T result = null;
			if (ReflectAssist.isInterface(object.getClass(), "java.util.Map")) {
				Map tempObjMap = (Map) object;
				result = (T) tempObjMap.get(colName);
			} else {
				result = (T) MVEL.eval(colName, object);
			}
			retset.add(result);
		}
		return retset;
	}

	/**
	 * @param inputCollection 要操作的字符
	 * @param predicate       规则
	 * @return 返回符合条件的集合并把这个集合从inputCollection删除（即inputCollection只剩余不合条件的数据）
	 */
	public static Collection selectFilter(Collection inputCollection, Predicate predicate) {
		Collection retCollection = (Collection) CollectionUtils.select(inputCollection, predicate);
		CollectionUtils.filter(inputCollection, predicate);
		return retCollection;
	}

	/***
	 * 把集合去重
	 * 
	 * @param collection 要操作的集合
	 */
	public static void distinctFilter(Collection collection) {
		Set<?> temps = new HashSet<>();
		temps.addAll(collection);
		collection.clear();
		collection.addAll(temps);
	}

	/***
	 * 过滤空值
	 * 
	 * @param includeBlack 空格模式 1：null 2：null和"" 3：null和"" 和 " " 4：3and"null"
	 * 
	 * @param orimap       要处理的map
	 */
	public static void filterNull(Map orimap, int includeBlack) {
		if (orimap == null) {
			return;
		}
		List<Object> removes = new ArrayList<>();
		for (Object key : orimap.keySet()) {
			Object value = orimap.get(key);
			boolean needRemove = false;
			switch (includeBlack) {
			case 1:
				needRemove = value == null ? true : false;
				break;
			case 2:
				needRemove = StringUtils.isEmpty(String.valueOf(value));
				break;
			case 3:
				needRemove = StringUtil.isNull(false, value);
				break;
			case 4:
				needRemove = StringUtil.isNull(true, value);
				break;
			default:
				break;
			}
			if (needRemove) {
				removes.add(key);
			}
		}
		for (Object key : removes) {
			orimap.remove(key);
		}
	}

	/*****
	 * 过滤原始的 List
	 * 
	 * @param oriList 原对象列表
	 * @param colName 对象列名
	 * @param include 允许的值
	 * @param exclude 排除的值
	 * @return 过滤后的list
	 */

	public static List filter(List oriList, final String colName, String include, String exclude) {
		List retAry = new ArrayList();
		if (CollectionUtils.isEmpty(oriList)) {
			return retAry;
		}
		if (StringUtils.isNotBlank(include)) {
			String[] iAry = include.split(",");
			for (int i = 0; i < iAry.length; i++) {
				String includeValue = iAry[i];
				for (Object eleObj : oriList) {
					try {
						if (ReflectAssist.isInterface(eleObj.getClass(), "java.util.Map")) {
							Map tempObj = (Map) eleObj;
							if (includeValue.equalsIgnoreCase(String.valueOf(tempObj.get(colName)))) {
								retAry.add(eleObj);
								break;// 退出本层循环
							}
						} else {
							Object tempObj = PropertyUtils.getProperty(eleObj, colName);
							if (includeValue.equalsIgnoreCase(String.valueOf(tempObj))) {
								retAry.add(eleObj);
								break;// 退出本层循环
							}
						}
					} catch (Exception e) {
						// TODO: handle exception
					}
				}
			}
		}

		if (StringUtils.isNotBlank(exclude)) {
			final String[] eAry = exclude.split(",");
			retAry = CollectionUtils.isEmpty(retAry) ? oriList : retAry;
			CollectionUtils.filter(retAry, new Predicate() {
				@Override
				public boolean evaluate(Object object) {
					try {
						if (ReflectAssist.isInterface(object.getClass(), "java.util.Map")) {
							Map tempObj = (Map) object;
							if (ArrayUtils.contains(eAry, String.valueOf(tempObj.get(colName)))) {
								return false;
							}
						} else {
							Object tempObj = PropertyUtils.getProperty(object, colName);
							if (ArrayUtils.contains(eAry, tempObj)) {
								return false;
							}
						}
					} catch (Exception e) {
						// TODO: handle exception
					}
					return true;
				}
			});
		}

		retAry = CollectionUtils.isEmpty(retAry) ? oriList : retAry;
		return retAry;

	}

	/***
	 * int数组转为List,因为Arrays.asList只支持对象的数组转成List
	 * 
	 * @param oriAry 源数组
	 * @return List 转换后的list
	 */
	public static List<Integer> asList(int[] oriAry) {
		List<Integer> ret = new ArrayList<Integer>();
		if (org.apache.commons.lang3.ArrayUtils.isNotEmpty(oriAry)) {
			for (int integer : oriAry) {
				ret.add(integer);
			}
		}
		return ret;
	}

	public static List<String> asList(Enum[] oriAry) {
		List<String> ret = new ArrayList<String>();
		if (org.apache.commons.lang3.ArrayUtils.isNotEmpty(oriAry)) {
			for (Enum enumEle : oriAry) {
				ret.add(enumEle.name());
			}
		}
		return ret;
	}

	/***
	 * 把string数据转成整形List
	 * 
	 * @param oriAry 源数组
	 * @return List 转换后的list
	 */
	public static List<Integer> asList(String[] oriAry) {
		List<Integer> ret = new ArrayList<Integer>();
		if (org.apache.commons.lang3.ArrayUtils.isNotEmpty(oriAry)) {
			for (String ele : oriAry) {
				ret.add(Integer.parseInt(ele));
			}
		}
		return ret;
	}

	/***
	 * 把任意数组转成List
	 * 
	 * @param oriList 源数组
	 * @return List 转换后的list
	 */
	public static List<String> asList(List<?> oriList) {
		if (oriList == null) {
			return null;
		}
		List<String> ret = new ArrayList<String>();
		for (Object object : oriList) {
			ret.add(String.valueOf(object));
		}
		return ret;
	}

	/****
	 * 得到Properties中key以 keyPre+"." 开头的所有属性的集合
	 * 
	 * @param prop         源属性
	 * @param keyOriPre    开始值
	 * @param keyTargetPre 替换值
	 * @return 满足条件的属性集合
	 */
	public static Map<String, String> getPropsByKeypre(Properties prop, String keyOriPre, String keyTargetPre,
			boolean delPre) {
		if (prop == null || prop.size() == 0 || StringUtil.isNull(keyOriPre)) {
			return new HashMap<>();
		}
		// 20200512 处理多线程情况下：java.util.ConcurrentModificationException 问题
		Properties tempProp = (Properties) prop.clone();
		Set<Object> propKeys = tempProp.keySet();
		Map<String, String> retMap = new HashMap<String, String>();
		String tempStr = String.format("%s.",
				keyOriPre.endsWith(".") ? keyOriPre.substring(0, keyOriPre.length() - 1) : keyOriPre);

		for (Object object : propKeys) {
			String key = String.valueOf(object);
			if (key.startsWith(tempStr)) {
				// 不能用replaceFirst，只能用substring，防止出现rm-bp1505575w78f52d3(jdbc)事件。
				retMap.put(delPre ? key.substring(tempStr.length()) : key, String.valueOf(tempProp.get(key)));// 不能用prop.getProperty
																												// 会导致int取不了值
			}
		}
		if (StringUtil.isNotNull(keyTargetPre)) {
			String tempStr2 = String.format("%s.",
					keyTargetPre.endsWith(".") ? keyTargetPre.substring(0, keyOriPre.length() - 1) : keyTargetPre);
			for (Object object : propKeys) {
				String key = String.valueOf(object);
				if (key.startsWith(tempStr2)) {
					// 不能用replaceFirst，只能用substring，防止出现rm-bp1505575w78f52d3(jdbc)事件。
					retMap.put(delPre ? key.substring(tempStr2.length()) : key, String.valueOf(tempProp.get(key)));// 不能用prop.getProperty
																													// 会导致int取不了值
				}
			}
		}

		return retMap;
	}

	public static Map<String, String> getPropsByKeypre(Properties prop, String keyOriPre, boolean delPre) {
		return getPropsByKeypre(prop, keyOriPre, null, delPre);
	}

	public static Properties getPropsSubByKeypre(Properties prop, String keyPre, boolean delPre) {
		return convMapToProperties(getPropsByKeypre(prop, keyPre, delPre));
	}

	public static Properties convMapToProperties(Map map) {
		Properties properties = new Properties();
		if (MapUtils.isEmpty(map)) {
			return properties;
		} else {
			for (Object ele : map.keySet()) {
				properties.put(ele, map.get(ele));
			}
			return properties;
		}
	}

	/**
	 * 把map转为驮峰形式
	 * 
	 * @param map
	 */
	public static Map convertMapForKey(Map map) {
		Map retmap = new HashMap();
		for (Object key : map.keySet()) {
			String keyTrue = StringUtil.convertStr(String.valueOf(key));
			retmap.put(keyTrue, map.get(key));
		}
		return retmap;
	}

	/***
	 * 通过有序的数组快整创建map
	 * 
	 * @param input 参数，单数为key 双数为value
	 * @return 创建的map
	 */
	public  static <T> Map newMap(T... input) {
		Map ret = new HashMap<>();
		if (ArrayUtils.isEmpty(input)) {
			return ret;
		}
		for (int i = 0; i < input.length / 2; i++) {
			Object key = input[2 * i];
			Object value = (2 * i + 1) < input.length ? input[2 * i + 1] : null;
			ret.put(key, value);
		}
		return ret;
	}

	public static Map<String, String> newMapStr(String... input) {
		Map<String, String> ret = new HashMap<String, String>();
		if (ArrayUtils.isEmpty(input)) {
			return ret;
		}
		for (int i = 0; i < input.length / 2; i++) {
			String key = input[2 * i];
			String value = (2 * i + 1) < input.length ? input[2 * i + 1] : null;
			ret.put(key, value);
		}
		return ret;
	}

	/***
	 * 把map转为Properties文件格式
	 * 
	 * @param datamap  文件
	 * @param retract  缩进
	 * @param splitStr 分隔符
	 * @return
	 */
	public static <T> String toPropString(Map<String, T> datamap, int retract, String splitStr) {
		if (MapUtils.isEmpty(datamap)) {
			return "";
		}
		// 前缀的空隔
		String prenull = "";
		for (int i = 0; i < (retract < 0 ? 0 : retract); i++) {
			prenull += " ";
		}
		StringBuffer buff = new StringBuffer();
		String formatestr = "%s%s=%s" + splitStr;

		Set<String> keySet = datamap.keySet();
		String[] keyAry = keySet.toArray(new String[keySet.size()]);
		Arrays.sort(keyAry); // 排序输出
		for (String key : keyAry) {
			buff.append(
					String.format(formatestr, prenull, key, StringUtil.trimSpace(String.valueOf(datamap.get(key)))));
		}
		// 支持最后的换行
		// 在容器里面使用linux不需要删除换行符，否则多删除一个字符。
		int lastDelChartNum = OSinfo.isWindows() ? splitStr.length() : ("%n".equals(splitStr) ? 0 : splitStr.length());
		return buff.subSequence(0, buff.length() - lastDelChartNum).toString();
	}

	// 以换行方式组织字符，如Properties文件格式
	public static <T> String toPropStringLn(Map<String, T> datamap, int retract) {
		return toPropString(datamap, retract, "%n");
	}

	// 分隔符分隔map值，默认为逗号
	public static <T> String toPropString(Map<String, T> datamap, String splitStr) {
		return toPropString(datamap, 0, StringUtil.hasNull(splitStr, ","));
	}

	/**
	 * 单层map转换多层map
	 * 
	 * @param source
	 * @return
	 */
	public static Map<String, Object> flattenedMapToMultilayerMap(Map<String, Object> source) {
		Map<String, Object> rootResult = new LinkedHashMap<>();
		for (Map.Entry<String, Object> entry : source.entrySet()) {
			String key = entry.getKey();
			buildMultilayerMap(rootResult, key, entry.getValue());
		}
		return rootResult;
	}

	public static Map<String, Object> asMap(Object object) {
		Map<String, Object> result = new LinkedHashMap<>();
		if (!(object instanceof Map)) {
			result.put("document", object);
			return result;
		}

		Map<Object, Object> map = (Map<Object, Object>) object;
		for (Map.Entry<Object, Object> entry : map.entrySet()) {
			Object value = entry.getValue();
			if (value instanceof Map) {
				value = asMap(value);
			}
			Object key = entry.getKey();
			if (key instanceof CharSequence) {
				result.put(key.toString(), value);
			} else {
				result.put("[" + key.toString() + "]", value);
			}
		}
		return result;
	}

	/**
	 * 对比list
	 * 
	 * @param oldList        旧数据
	 * @param newList        新数据
	 * @param comPareColName 比较的列名
	 * @param idColName      id列名
	 * @return L：修改数据， M:删除数据 R:新增数据
	 */
	public static <T1, T2> MutableTriple<List<T2>, List<T1>, List<T2>> packageComparison(List<T1> oldList,
			List<T2> newList, String comPareColName, String idColName) {
		if (CollectionUtils.isEmpty(oldList)) {
			return MutableTriple.of(null, null, newList);
		}
		if (CollectionUtils.isEmpty(newList)) {
			return MutableTriple.of(null, oldList, null);
		}
		List<T1> deleteList = new ArrayList<T1>();
		List<T2> updateList = new ArrayList<T2>();
		List<T2> addList = new ArrayList<T2>();
		List<String> oldKeyList = getColValStrFromObj(oldList, comPareColName);
		List<String> newKeyList = getColValStrFromObj(newList, comPareColName);

		for (T2 newEle : newList) {
			try {
				String cloValue = BeanUtils.getProperty(newEle, comPareColName);
				if (oldKeyList.contains(cloValue)) {
					updateList.add(newEle);
					try {
						Object idvalue = null;
						for (T1 t1 : oldList) {
							String comPareColValue = BeanUtils.getProperty(t1, comPareColName);
							if (StringUtil.isNotNull(cloValue) && StringUtil.isNotNull(comPareColValue)
									&& cloValue.equals(comPareColValue)) {
								idvalue = PropertyUtils.getProperty(t1, idColName);
								break;
							}
						}
						if (StringUtil.isNotNull(idvalue)) {
							PropertyUtils.setProperty(newEle, idColName, idvalue);
						}
					} catch (Exception e) {
						log.error("错误设置id,可能接下来要通过id进行修改", e);
					}
				} else {
					addList.add(newEle);
				}
			} catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
				log.error("比较时错误.", e);
			}
		}
		for (T1 oldEle : oldList) {
			try {
				String cloValue = BeanUtils.getProperty(oldEle, comPareColName);
				if (!newKeyList.contains(cloValue)) {
					deleteList.add(oldEle);
				}
			} catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
				log.error("比较时错误.", e);
			}
		}
		return MutableTriple.of(updateList, deleteList, addList);
	}

	public static Map<String, Object> getFlattenedMap(Map<String, Object> source) {
		Map<String, Object> result = new LinkedHashMap<>();
		buildFlattenedMap(result, source, null);
		return result;
	}

	private static void buildFlattenedMap(Map<String, Object> result, Map<String, Object> source, String path) {
		for (Map.Entry<String, Object> entry : source.entrySet()) {
			String key = entry.getKey();
			if (!StringUtils.isBlank(path)) {
				if (key.startsWith("[")) {
					key = path + key;
				} else {
					key = path + '.' + key;
				}
			}
			Object value = entry.getValue();
			if (value instanceof String) {
				result.put(key, value);
			} else if (value instanceof Map) {
				Map<String, Object> map = (Map<String, Object>) value;
				buildFlattenedMap(result, map, key);
			} else if (value instanceof Collection) {
				Collection<Object> collection = (Collection<Object>) value;
				int count = 0;
				for (Object object : collection) {
					buildFlattenedMap(result, Collections.singletonMap("[" + (count++) + "]", object), key);
				}
			} else {
				result.put(key, (value != null ? value.toString() : ""));
			}
		}
	}

	private static void buildMultilayerMap(Map<String, Object> parent, String path, Object value) {
		String[] keys = StringUtils.split(path, ".");
		String key = keys[0];
		if (key.endsWith("]")) {
			String listKey = key.substring(0, key.indexOf("["));
			String listPath = path.substring(key.indexOf("["));
			List<Object> chlid = bulidChlidList(parent, listKey);
			buildMultilayerList(chlid, listPath, value);
		} else {
			if (keys.length == 1) {
				parent.put(key, stringToObj(value.toString()));
			} else {
				String newpath = path.substring(path.indexOf(".") + 1);
				Map<String, Object> chlid = bulidChlidMap(parent, key);
				;
				buildMultilayerMap(chlid, newpath, value);
			}
		}
	}

	private static List<Object> bulidChlidList(Map<String, Object> parent, String key) {
		if (parent.containsKey(key)) {
			return (List<Object>) parent.get(key);
		} else {
			List<Object> chlid = new GrowthList(16);
			parent.put(key, chlid);
			return chlid;
		}
	}

	private static Object stringToObj(String obj) {
		Object result = null;
		if (obj.equals("true") || obj.equals("false")) {
			result = Boolean.valueOf(obj);
		} else if (StringUtil.isBigDecimal(obj)) {
			if (obj.indexOf(".") == -1) {
				result = Long.valueOf(obj.toString());
			} else {
				result = Double.valueOf(obj.toString());
			}
		} else {
			result = obj;
		}
		return result;
	}

	private static void buildMultilayerList(List<Object> parent, String path, Object value) {
		String[] keys = StringUtils.split(path, ".");
		String key = keys[0];
		int index = Integer.valueOf(key.replace("[", "").replace("]", ""));
		if (keys.length == 1) {
			parent.add(index, stringToObj(value.toString()));
		} else {
			String newpath = path.substring(path.indexOf(".") + 1);
			Map<String, Object> chlid = bulidChlidMap(parent, index);
			;
			buildMultilayerMap(chlid, newpath, value);
		}
	}

	private static Map<String, Object> bulidChlidMap(Map<String, Object> parent, String key) {
		if (parent.containsKey(key)) {
			return (Map<String, Object>) parent.get(key);
		} else {
			Map<String, Object> chlid = new LinkedHashMap<>(16);
			parent.put(key, chlid);
			return chlid;
		}
	}

	private static Map<String, Object> bulidChlidMap(List<Object> parent, int index) {
		Map<String, Object> chlid = null;
		try {
			Object obj = parent.get(index);
			if (null != obj) {
				chlid = (Map<String, Object>) obj;
			}
		} catch (Exception e) {
			log.warn("get list error");
		}

		if (null == chlid) {
			chlid = new LinkedHashMap<>(16);
			parent.add(index, chlid);
		}
		return chlid;
	}

}
