/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common-apiext
 * 项目描述：工具
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.constant;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;

import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.constant.dic.intf.IEnumCombobox;

/***
 * 中间件类型
 * 
 * @author andy.zhou
 * 
 */
public enum Middleware implements IEnumCombobox {
	no("不设置值", new String[] {}, new MiddlewareOption[] { MiddlewareOption.connector }, ""),

	// lookupCacheTtl,lookupCacheMaxRows jdbclookup在级联的情况下不能开缓存且flink16版本也将会废弃，不使用
	mysql("mysql数据库", new String[] { "flinkcatalogmeta", "jdbctype", "duckularule" },
			new MiddlewareOption[] { MiddlewareOption.connector, MiddlewareOption.url, MiddlewareOption.tableName,
					MiddlewareOption.username, MiddlewareOption.password },
			"rds"),

	es("es搜索", new String[] { "duckularule" }, new MiddlewareOption[] {}, "es"),

	sqlserver("sqlserver数据库", new String[] { "duckularule" }, new MiddlewareOption[] {}, "ss"),

	http("http服务器", new String[] { "duckularule" }, new MiddlewareOption[] {}, "http"),

	logger("logger日志", new String[] { "duckularule" }, new MiddlewareOption[] {}, ""),

	kafka("kafka", new String[] { "flinkcatalogmeta", "duckularule" },
			new MiddlewareOption[] { MiddlewareOption.connector, MiddlewareOption.topic, MiddlewareOption.kafka_servers,
					MiddlewareOption.kafka_groupid, MiddlewareOption.format, MiddlewareOption.kafka_scan_mode,
					MiddlewareOption.kafka_sink_partitioner },
			"ak"),

	// 不需要加groupid,它会把groupid设置为null，表示每个consumer instance是广播消费，这样就可以不用再建物理表了。
	upsertkafka("upsert-kafka", new String[] { "flinkcatalogmeta" },
			new MiddlewareOption[] { MiddlewareOption.connector, MiddlewareOption.topic, MiddlewareOption.kafka_servers,
					MiddlewareOption.keyformat, MiddlewareOption.valueformat, MiddlewareOption.valuefieldsinclude },
			"uk"),

	// 自定义的kafka，可以避免开源kafka很多问题
	// 1. key为空，不方便做分区
	// 2. 开源的connector不支持分区键语句（upset支持key分区，而append支持自定义分区但它key值又为空）
	// 3. UPDATE_BEFORE需要忽略，因为它会分为2个事件发送过来，不好组装为一个完整的update binlog事件
	// 4.MiddlewareOption.offset暂不使用，因为没有提供source
	// 5. 支持rowKind值来自指定列
	tamskafka("tams-kafka", new String[] { "flinkcatalogmeta" },
			new MiddlewareOption[] { MiddlewareOption.connector, MiddlewareOption.topic, MiddlewareOption.kafka_servers,
					MiddlewareOption.format, MiddlewareOption.rowKindColName },
			"tk"),

	duckula("binlog监听", new String[] { "flinkcatalogmeta", "jdbctype" },
			new MiddlewareOption[] { MiddlewareOption.db, MiddlewareOption.tb, MiddlewareOption.host,
					MiddlewareOption.username, MiddlewareOption.password, MiddlewareOption.port,
					MiddlewareOption.fieldFormart, MiddlewareOption.cdc, MiddlewareOption.addColNameType,
					MiddlewareOption.updateColName, MiddlewareOption.append },
			"d"),

	doris("doris数据库", new String[] { "flinkcatalogmeta", "duckularule" },
			new MiddlewareOption[] { MiddlewareOption.connector, MiddlewareOption.fenodes,
					MiddlewareOption.table_identifier, MiddlewareOption.username, MiddlewareOption.password,
					MiddlewareOption.dorisbatchsize, MiddlewareOption.dorisSinkLabelPrefix,
					MiddlewareOption.dorisFormat, MiddlewareOption.dorisReadJsonByLine },
			"do"),

	redis("redis", new String[] { "flinkcatalogmeta", "duckularule" },
			new MiddlewareOption[] { MiddlewareOption.connector, MiddlewareOption.host, MiddlewareOption.port,
					MiddlewareOption.password, MiddlewareOption.groupid, MiddlewareOption.searchkeyprefix,
					MiddlewareOption.updateColName, MiddlewareOption.routeColName, MiddlewareOption.db,
					MiddlewareOption.append, MiddlewareOption.fieldFormart },
			"r"),

	jdbctams("tams-jdbc", new String[] { "flinkcatalogmeta" },
			new MiddlewareOption[] { MiddlewareOption.connector, MiddlewareOption.host, MiddlewareOption.port,
					MiddlewareOption.password, MiddlewareOption.searchkeyprefix, MiddlewareOption.db,
					MiddlewareOption.tableName, MiddlewareOption.tenantColName, MiddlewareOption.lookupCacheTtl,
					MiddlewareOption.lookupCacheMaxRows },
			"jt"),

	kudu("kudu数据库", new String[] { "flinkcatalogmeta", "duckularule" }, new MiddlewareOption[] {}, "kd"),

	// prometheus的pushgateway
	pushgateway("pushgateway", new String[] { "flinkcatalogmeta" },
			new MiddlewareOption[] { MiddlewareOption.connector, MiddlewareOption.pushgateway, MiddlewareOption.job,
					MiddlewareOption.name, MiddlewareOption.help, MiddlewareOption.type, MiddlewareOption.labels },
			"p");

	private final String desc;

	private final String tbPre;

	public String getTbPre() {
		return tbPre;
	}

	private final String[] groups;// 所属的组,flinkcatalogmeta:支持的元数据 jdbctype:用于flink的cdc配置，
									// duckularule：duckula的rule检查对应的中间件

	private final MiddlewareOption[] flinkOptions;

	public Result checkOpt(Map<MiddlewareOption, String> optValue) {
		Result retobj = Result.getSuc();
		switch (this) {
		case kafka:
			if (StringUtil.isNull(optValue.get(MiddlewareOption.kafka_groupid))) {
				retobj = Result.getError("kafka和tamskafka都必须填写" + MiddlewareOption.kafka_groupid.getDesc());
			}
			break;
		default:
			break;
		}
		return retobj;
	}

	public String buildFlinkOpt(Map<String, String> keymap) {
		if (MapUtils.isEmpty(keymap)) {
			return "";
		}
		Map<MiddlewareOption, String> keymaps = new HashMap<MiddlewareOption, String>();
		for (MiddlewareOption flinkOption : flinkOptions) {
			if (keymap.containsKey(flinkOption.name())) {
				keymaps.put(flinkOption, keymap.get(flinkOption.name()));
			}
		}
		return buildFlinkOptStr(keymaps);
	}

	// 组装flinksql的opt
	public String buildFlinkOptStr(Map<MiddlewareOption, String> keymap) {
		if (this == no || MapUtils.isEmpty(keymap)) {
			return "";
		}
		String optmain = "";
		StringBuffer buf = new StringBuffer();
		for (MiddlewareOption middlewareConfOption : keymap.keySet()) {
			if (ArrayUtils.contains(flinkOptions, middlewareConfOption)) {
				buf.append(String.format(",'%s' = '%s'", middlewareConfOption.getNameUse(),
						keymap.get(middlewareConfOption)));
			}
		}
		optmain = buf.length() > 0 ? buf.substring(1) : "";
		return String.format("WITH (%s)", optmain);
	}

	public static JSONObject convertJson(Map<MiddlewareOption, String> keymap) {
		JSONObject retobj = new JSONObject();
		if (MapUtils.isEmpty(keymap)) {
			return retobj;
		}
		for (MiddlewareOption middlewareOption : keymap.keySet()) {
			retobj.put(middlewareOption.getNameUse(), keymap.get(middlewareOption));
		}
		return retobj;
	}

	private static String convertOtherJson(Map<MiddlewareOption, String> otherOptmap) {
		// {'total':1,'rows':[{'editor':'text','name':'tableName','value':'abc','nameUse':'table-name','desc':'表名'}]}
		JSONObject retobj = new JSONObject();
		retobj.put("total", otherOptmap.size());
		JSONArray rows = new JSONArray();
		for (MiddlewareOption middlewareOption : otherOptmap.keySet()) {
			JSONObject temp = new JSONObject();
			temp.put("editor", "text");
			temp.put("name", middlewareOption.name());
			temp.put("value", otherOptmap.get(middlewareOption));
			temp.put("nameUse", middlewareOption.getNameUse());
			temp.put("desc", middlewareOption.getDesc());
			rows.add(temp);
		}
		retobj.put("rows", rows);
		String jsonString = JSONObject.toJSONString(retobj, SerializerFeature.UseSingleQuotes);
		return jsonString;
	}

	/***
	 * 产生ops和otheropt
	 * 
	 * @param allOps
	 * @param otherOpt
	 * @return L:withOption R:opt
	 */
	public static Pair<String, String> proOptStr(Map<MiddlewareOption, String> allOps, MiddlewareOption... otherOpt) {
		Map<MiddlewareOption, String> otherOpsMap = new HashMap<MiddlewareOption, String>();
		Map<MiddlewareOption, String> allOpsMap = MapUtils.isEmpty(allOps) ? new HashMap<MiddlewareOption, String>()
				: allOps;
		if (ArrayUtils.isNotEmpty(otherOpt) && MapUtils.isNotEmpty(allOpsMap)) {
			for (MiddlewareOption key : otherOpt) {
				if (allOpsMap.containsKey(key)) {
					otherOpsMap.put(key, allOpsMap.get(key));
				}
			}
		}
		String allOpt = JSONObject.toJSONString(Middleware.convertJson(allOpsMap), SerializerFeature.UseSingleQuotes);
		String otherJson = Middleware.convertOtherJson(otherOpsMap);
		return Pair.of(allOpt, otherJson);
	}

	public String[] getGroups() {
		return groups;
	}

	private Middleware(String desc, String[] groups, MiddlewareOption[] flinkOptions, String tbPre) {
		this.desc = desc;
		this.groups = groups;
		this.flinkOptions = flinkOptions;
		this.tbPre = tbPre;
	}

	public static Middleware[] getMiddleTypeByGroup(String... groups) {
		Middleware[] allTypes = Middleware.values();
		if (ArrayUtils.isEmpty(groups)) {
			return allTypes;
		}
		Set<Middleware> retset = new HashSet<Middleware>();
		for (String group : groups) {
			for (Middleware middlewareType : allTypes) {
				if (ArrayUtils.contains(middlewareType.getGroups(), group)) {
					retset.add(middlewareType);
				}
			}
		}
		return retset.toArray(new Middleware[retset.size()]);
	}

	public String getDesc() {
		return desc;
	}

	public String getName() {
		return this.name();
	}

	@Override
	public String getDesc_zh() {
		return this.desc;
	}

	@Override
	public String getDesc_en() {
		return this.name();
	}

	public MiddlewareOption[] getFlinkOptions() {
		return flinkOptions;
	}

	public static Middleware find(String name) {
		if (StringUtils.isEmpty(name)) {
			return null;
		}
		for (Middleware ele : Middleware.values()) {
			if (name.equalsIgnoreCase(ele.name())) {
				return ele;
			}
		}
		return null;
	}

}
