/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common-apiext
 * 项目描述：工具
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.apiext.jdbc;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.protobuf.ProtocolStringList;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.jdbc.JdbcDatas.Builder;
import net.wicp.tams.common.callback.ICreateDbCallBack;
import net.wicp.tams.common.constant.DbType;
import net.wicp.tams.common.constant.DrdsPattern;
import net.wicp.tams.common.constant.FieldFormart;
import net.wicp.tams.common.constant.StrPattern;
import net.wicp.tams.common.constant.dbType.BinlogType;
import net.wicp.tams.common.constant.dic.YesOrNo;
import net.wicp.tams.common.constant.ods.AddColName;
import net.wicp.tams.common.constant.ods.AddColNameType;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

@Slf4j
public abstract class MySqlAssit extends JdbcAssit {
	public static String getColsForQuery(String[] cols) {
		String colsstr = "`" + CollectionUtil.arrayJoin(cols, "`,`") + "`";
		return colsstr;
	}

	public static List<String[]> getAllTables(Connection conn) {
		return getAllTables(conn, null, null);
	}

	public static List<String> getAllDbs(Connection conn, String... excludes) {
		List<Map<String, String>> querySqlMapPre = new ArrayList<Map<String, String>>();
		querySqlMapPre = querySqlMapPre(conn, "show databases", false);
		List<String> excluesDbs = new ArrayList<String>();
		excluesDbs.add("mysql");
		if (excludes != null) {
			excluesDbs.addAll(Arrays.asList(excludes));
		}
		List<String> returlist = new ArrayList<String>();
		for (Map<String, String> map : querySqlMapPre) {
			String dbName = map.get("Database");
			if (!excluesDbs.contains(dbName)) {
				returlist.add(dbName);
			}
		}
		return returlist;
	}

	// json类型的的输入
	public static String jsonValue(JSONObject json) {
		StringBuffer buff = new StringBuffer("JSON_OBJECT(");
		for (String key : json.keySet()) {
			buff.append(String.format("'%s', '%s'", key, json.get(key)));
		}
		buff.append(")");
		return buff.toString();
	}

	// json类型的的输入
	public static String jsonValue(JSONArray jsonArray) {
		StringBuffer buff = new StringBuffer("JSON_ARRAY(");
		for (Object ele : jsonArray) {
			buff.append(String.format("'%s'", String.valueOf(ele)));
		}
		buff.append(")");
		return buff.toString();
	}

	/***
	 * 查询sql的结果，分页
	 * 
	 * @param conn
	 * @param sql
	 * @param from
	 * @param num
	 * @param isConvertKey 是否转为驼峰模式
	 * @return
	 */
	public static List<Map<String, String>> querySqlMap(Connection conn, String sql, int from, int num,
			boolean isConvertKey) {
		int indexOf = sql.indexOf("limit");
		if (indexOf >= 0) {
			sql = sql.substring(0, indexOf);
		}
		sql += String.format(" limit %s,%s", from, num);
		List<Map<String, String>> querySqlMap = JdbcAssit.querySqlMap(conn, sql, isConvertKey);
		return querySqlMap;
	}

	/***
	 * 查询指定SQL的总记录数
	 * 
	 * @param conn
	 * @param sql
	 * @return
	 */
	public static int querySqlCount(Connection conn, String sql) {
		int indexOf = sql.indexOf("limit");
		if (indexOf >= 0) {
			sql = sql.substring(0, indexOf);
		}
		sql = sql.substring(sql.indexOf("from"));
		sql = "select count(1) " + sql;
		ResultSet rs = JdbcAssit.querySql(conn, sql);
		int allnum = -1;
		try {
			allnum = rs.next() ? rs.getInt(1) : 0;
		} catch (SQLException e) {
			log.error("查询总数失败", e);
		}
		return allnum;
	}

	/***
	 * 查询所有满足条件的表名
	 * 
	 * @param conn
	 * @param dbPattern
	 * @param tbPattern
	 * @return
	 */
	public static List<String[]> getAllTables(Connection conn, String dbPattern, String tbPattern) {
		String sql = "select  TABLE_SCHEMA,TABLE_NAME from information_schema.tables where 1=1";
		ResultSet rs = JdbcAssit.querySql(conn, sql);
		List<String[]> retlist = new ArrayList<String[]>();
		String dbPatternTrue = StringUtil.buildRule(StringUtil.trimSpace(dbPattern));
		String tbPatternTrue = StringUtil.buildRule(StringUtil.trimSpace(tbPattern));
		try {
			while (rs.next()) {

				if (StringUtil.isNotNull(dbPatternTrue) && !"^*$".equals(dbPatternTrue) && !"*".equals(dbPatternTrue)) {
					boolean isChecked = StrPattern.checkStrFormat(dbPatternTrue, rs.getString("TABLE_SCHEMA"));
					if (!isChecked) {
						continue;
					}
				}
				if (StringUtil.isNotNull(tbPatternTrue) && !"^*$".equals(tbPatternTrue) && !"*".equals(tbPatternTrue)) {
					boolean isChecked = StrPattern.checkStrFormat(tbPatternTrue, rs.getString("TABLE_NAME"));
					if (!isChecked) {
						continue;
					}
				}

				retlist.add(new String[] { rs.getString("TABLE_SCHEMA"), rs.getString("TABLE_NAME") });
			}
		} catch (Exception e) {
			log.error("find tables error", e);
		} finally {
			try {
				rs.close();
			} catch (SQLException e) {
			}
		}
		return retlist;
	}

	public static List<List<Pair<String, String>>> getAllTablesSplitGroup(Connection conn, String dbPattern,
			String tbPattern, DrdsPattern splitPattern) {
		List<String[]> tables = getAllTables(conn, dbPattern, tbPattern);
		List<List<Pair<String, String>>> retlist = new ArrayList<List<Pair<String, String>>>();
		for (String[] table : tables) {
			if (CollectionUtils.isEmpty(retlist)) {
				List<Pair<String, String>> temp = new ArrayList<Pair<String, String>>();
				temp.add(Pair.of(table[0], table[1]));
				retlist.add(temp);
			} else {
				boolean hasSame = false;
				for (List<Pair<String, String>> list : retlist) {
					boolean sameGroup = splitPattern.isSameGroup(list.get(0), Pair.of(table[0], table[1]));
					if (sameGroup) {
						list.add(Pair.of(table[0], table[1]));
						hasSame = true;
						break;
					}
				}
				if (!hasSame) {
					List<Pair<String, String>> temp = new ArrayList<Pair<String, String>>();
					temp.add(Pair.of(table[0], table[1]));
					retlist.add(temp);
				}
			}

		}
		return retlist;
	}

	/***
	 * 查库中的所有表
	 * 
	 * @param conn
	 * @param dbName
	 * @return
	 */
	public static List<String[]> getAllTables(Connection conn, String dbName) {
		String sql = "select  TABLE_SCHEMA,TABLE_NAME from information_schema.tables where TABLE_SCHEMA=?";
		ResultSet rs = null;
		List<String[]> retlist = new ArrayList<String[]>();
		try {
			PreparedStatement stmt = conn.prepareStatement(sql);
			setPreParam(stmt, true, dbName);
			rs = stmt.executeQuery();
			while (rs.next()) {
				retlist.add(new String[] { rs.getString("TABLE_SCHEMA"), rs.getString("TABLE_NAME") });
			}
		} catch (Exception e) {
			log.error("find tables error", e);
		} finally {
			try {
				rs.close();
			} catch (SQLException e) {
				e.printStackTrace();
			}
		}
		return retlist;
	}

	public static List<String[]> getAllTablesPattern(Connection conn, String dbName, String tbPattern) {
		List<String[]> allTables = getAllTables(conn, dbName);
		String tbPatternTemp = StringUtil.isNull(tbPattern) ? "*" : tbPattern;
		final String tbPatternTrue = "^" + tbPatternTemp + "$";
		CollectionUtils.filter(allTables, new Predicate() {

			@Override
			public boolean evaluate(Object object) {
				if (StringUtil.isNull(tbPattern)) {// 全部命中
					return true;
				}
				String[] name = (String[]) object;
				boolean retdb = StrPattern.checkStrFormat(tbPatternTrue, name[1]);
				return retdb;
			}
		});
		return allTables;
	}

	/***
	 * 批量更新mysql数据
	 * 
	 * @param conn    连接
	 * @param db      数据库名
	 * @param tb      表名
	 * @param datas   要新增的数据
	 * @param cols    要新增的列名
	 * @param typeAry 类型，用于兼容为空的情况
	 * @return
	 */
	public static Result dataUpset(Connection conn, String db, String tb, Object[][] datas, String[] cols,
			BinlogType[] typeAry, boolean batch) {
		Object[][] priDatas = new Object[datas.length][];
		StringBuffer ids = new StringBuffer();
		try {
			String[] tempAry = new String[cols.length];
			for (int i = 0; i < tempAry.length; i++) {
				tempAry[i] = "?";
			}
			String sql = String.format("replace into %s.%s (%s) values (%s)", db, tb, MySqlAssit.getColsForQuery(cols),
					CollectionUtil.arrayJoin(tempAry, ","));

			PreparedStatement stmt = conn.prepareStatement(sql);

			if (batch) {
				conn.setAutoCommit(false);
				stmt.clearBatch();
				for (int i = 0; i < datas.length; i++) {
					if (i < datas.length - 1) {
						int indexOf = CollectionUtil.indexOf(priDatas, priDatas[i], i + 1);
						if (indexOf >= 0) {
							continue;
						}
					}
					ids.append(datas[i][0] + ",");
					JdbcAssit.setPreParam(stmt, typeAry, false, datas[i]);
					stmt.addBatch();
				}
				stmt.executeBatch();
				conn.commit();
			} else {
				conn.setAutoCommit(true);
				for (int i = 0; i < datas.length; i++) {
					if (i < datas.length - 1) {
						int indexOf = CollectionUtil.indexOf(priDatas, priDatas[i], i + 1);
						if (indexOf >= 0) {
							continue;
						}
					}
					ids.append(datas[i][0] + ",");
					JdbcAssit.setPreParam(stmt, typeAry, false, datas[i]);
					stmt.execute();
				}
			}
			stmt.close();
			return Result.getSuc();
		} catch (Exception e) {
			String errmsg = String.format("批量插入失败,db【%s】,tb【%s】,ids【%s】", db, tb, ids.toString());
			log.error(errmsg, e);
			return new Result(ExceptAll.jdbc_exec_fail);
		}
	}

	public static Result dataUpset(Connection conn, String db, String tb, List<Map<String, String>> datas) {
		if (CollectionUtils.isEmpty(datas)) {
			return Result.getSuc();
		}
		Builder dataBuilder = createJdbcDataBuilder(conn, db, tb);
		for (Map<String, String> data : datas) {
			JdbcData.Builder jdbcDataBuild = JdbcData.newBuilder();
			jdbcDataBuild.putAllValue(data);
			dataBuilder.addDatas(jdbcDataBuild);
		}
		return dataUpset(conn, dataBuilder.build(), true);
	}

	public static Builder createJdbcDataBuilder(Connection conn, String db, String tb) {
		Builder dataBuilder = JdbcDatas.newBuilder();
		dataBuilder.setDb(db);
		dataBuilder.setTb(tb);
		dataBuilder.setOptType(OptType.update);
		String[] primary = MySqlAssit.getPrimary(conn, db, tb);
		dataBuilder.addAllKeys(Arrays.asList(primary));
		String[][] cols = MySqlAssit.getCols(conn, db, tb, YesOrNo.no);
		dataBuilder.addAllCols(Arrays.asList(cols[0]));
		for (int i = 0; i < cols[0].length; i++) {
			BinlogType binlogType = BinlogType.getByName(cols[1][i]);
			dataBuilder.putType(cols[0][i], binlogType.name());
		}
		return dataBuilder;
	}

	public static Result dataUpsetV1(Connection conn, JdbcDatas datas, boolean batch) {
		String sql = packageBatchSql(datas);
		PreparedStatement stmt = null;
		try {
			stmt = conn.prepareStatement(sql.toString());
			conn.setAutoCommit(false);
			stmt.execute();
			conn.commit();
		} catch (SQLException throwables) {
			throw new ProjectExceptionRuntime(ExceptAll.param_error,
					"db:[" + datas.getDb() + "],tb:[" + datas.getTb() + "] upset数据出错:" + throwables.getMessage(),
					throwables);
		} finally {
			try {
				stmt.close();
			} catch (SQLException throwables) {
				throwables.printStackTrace();
			}
		}

		return Result.getSuc();
	}

	public static String packageBatchSql(JdbcDatas datas) {
		List<JdbcData> datasList = datas.getDatasList();
		Object[][] datasInput = new Object[datasList.size()][];
		StringBuilder sqlValues = new StringBuilder();
		for (int i = 0; i < datasList.size(); i++) {
			JdbcData jdbcData = datasList.get(i);
			datasInput[i] = new Object[datas.getColsCount()];
			StringBuilder sqlValue = new StringBuilder();
			sqlValue.append("(");
			for (int j = 0; j < datas.getColsCount(); j++) {
				String value = jdbcData.getValueMap().get(datas.getCols(j));
				if (value == null) {
					sqlValue.append(value).append(",");
				} else {
					value = value.replaceAll("'", "''");
					BinlogType binlogType = BinlogType.valueOf(datas.getTypeMap().get(datas.getCols(j)));
					Serializable value2 = BinlogType.getValueBuildSql(binlogType, value);
					if (value2 instanceof String) {// json要处理特别字段(暂不处理)
						// ((String) value2).replaceAll("", replacement)
						value2 = StringUtil.filterEmoji((String) value2);// 表情符暂时过滤，后面找解决办法，如：反馈信息测试😂
					}
					sqlValue.append(value2).append(",");
				}
			}
			if (sqlValue.length() > 1) {
				sqlValue.deleteCharAt(sqlValue.length() - 1);
				sqlValue.append(")");
			}
			sqlValues.append(sqlValue.toString()).append(",");
		}
		if (sqlValues.length() > 0) {
			sqlValues.deleteCharAt(sqlValues.length() - 1);
		}

		String sql = String.format("replace into %s.%s (%s) values %s", datas.getDb(), datas.getTb(),
				MySqlAssit.getColsForQuery(datas.getColsList().toArray(new String[datas.getColsList().size()])),
				sqlValues.toString());
		return sql;
	}

	public static Result dataUpset(Connection conn, JdbcDatas datas, boolean batch) {
		List<JdbcData> datasList = datas.getDatasList();
		Object[][] datasInput = new Object[datasList.size()][];
		for (int i = 0; i < datasList.size(); i++) {
			JdbcData jdbcData = datasList.get(i);
			datasInput[i] = new Object[datas.getColsCount()];
			for (int j = 0; j < datas.getColsCount(); j++) {
				String value = jdbcData.getValueMap().get(datas.getCols(j));
				if (value == null) {
					datasInput[i][j] = value;
				} else {
					BinlogType binlogType = BinlogType.valueOf(datas.getTypeMap().get(datas.getCols(j)));
					Serializable value2 = BinlogType.getValue(binlogType, value);
					datasInput[i][j] = value2;
				}
			}
		}
		Map<String, String> typeMap = datas.getTypeMap();
		ProtocolStringList colsList = datas.getColsList();
		BinlogType[] typeAry = new BinlogType[colsList.size()];
		for (int i = 0; i < typeAry.length; i++) {
			typeAry[i] = BinlogType.valueOf(typeMap.get(colsList.get(i)));
		}
		return dataUpset(conn, datas.getDb(), datas.getTb(), datasInput,
				datas.getColsList().toArray(new String[datas.getColsList().size()]), typeAry, batch);
	}

	public static Result dataDelete(Connection conn, String db, String tb, Object[][] datas, String[] keys) {
		if (ArrayUtils.isEmpty(keys)) {
			keys = getPrimary(conn, db, tb);
		}
		if (!ArrayUtils.isEmpty(keys)) {// 需要删除ID
			StringBuilder builder = new StringBuilder(String.format("delete from %s.%s where 1=1", db, tb));
			for (String key : keys) {
				builder.append(String.format(" and %s=?", key));
			}
			try {
				conn.setAutoCommit(false);
				PreparedStatement stmtDel = conn.prepareStatement(builder.toString());
				for (int i = 0; i < datas.length; i++) {
					JdbcAssit.setPreParam(stmtDel, null, false, datas[i]);
					stmtDel.addBatch();
				}
				stmtDel.executeBatch();
				conn.commit();
				stmtDel.close();
				return Result.getSuc();
			} catch (SQLException e) {
				log.error("通过主键删除失败", e);
				return new Result(new ProjectException(ExceptAll.jdbc_exec_fail, "通过主键删除失败"));
			}
		} else {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_exec_fail);
		}
	}

	public static Result dataDelete(Connection conn, JdbcDatas datas) {
		List<JdbcData> datasList = datas.getDatasList();
		Object[][] datasInput = new Object[datasList.size()][];
		for (int i = 0; i < datasList.size(); i++) {
			JdbcData jdbcData = datasList.get(i);
			datasInput[i] = new Object[datas.getKeysCount()];
			for (int j = 0; j < datas.getKeysCount(); j++) {
				String value = jdbcData.getValueMap().get(datas.getKeys(j));
				BinlogType binlogType = BinlogType.valueOf(datas.getTypeMap().get(datas.getKeys(j)));
				Serializable value2 = BinlogType.getValue(binlogType, value);
				datasInput[i][j] = value2;
			}
		}
		return dataDelete(conn, datas.getDb(), datas.getTb(), datasInput,
				datas.getKeysList().toArray(new String[datas.getKeysCount()]));
	}

	public static Result dataChange(Connection conn, JdbcDatas datas, boolean logicDel) {
		if (datas.getOptType() == OptType.delete && !logicDel) {// 不是逻辑删除
			return dataDelete(conn, datas);
		} else {
			return dataUpsetV1(conn, datas, true);
		}
	}

	public static String[] getPrimary(Connection conn, String db, String tb,DbType dbType) {		
		String[] keyids = null;
		if (dbType == DbType.doris) {// doris通过建表语名拿到主键
			List<Map<String, String>> querySqlMap = MySqlAssit.querySqlMap(conn,
					String.format("show create table %s.%s", db, tb), false);
			String createSql = querySqlMap.get(0).get("Create Table");
			String uk = "UNIQUE KEY";
			String ak = "AGGREGATE KEY";
			// 当前仅支持主键为uk或ak；
			int indexBegin = createSql.indexOf(uk);
			if (indexBegin == -1) {
				keyids = getkey(createSql, createSql.indexOf(ak), ak);
			} else {
				keyids = getkey(createSql, indexBegin, uk);
			}
		}else if (dbType == DbType.mysql) {
			PreparedStatement preparedStatement = null;
			try {
				preparedStatement = conn.prepareStatement(
						"SELECT k.column_name FROM information_schema.table_constraints t JOIN information_schema.key_column_usage k USING (constraint_name,table_schema,table_name) WHERE t.constraint_type='PRIMARY KEY' AND t.table_schema=? AND t.table_name=?");
				JdbcAssit.setPreParam(preparedStatement, null, true, db, tb);
				ResultSet rs2 = preparedStatement.executeQuery();
				if (!rs2.next()) {
					return new String[0];
				}
				List<String> retlist = new ArrayList<>();
				retlist.add(rs2.getString(1));
				while (rs2.next()) {
					retlist.add(rs2.getString(1));
				}
				keyids= retlist.toArray(new String[retlist.size()]);
			} catch (SQLException e) {
				log.error("查主键失败");
				throw new ProjectExceptionRuntime(ExceptAll.jdbc_query_primary);
			} finally {
				try {
					if (preparedStatement != null) {
						preparedStatement.close();
					}
				} catch (Exception e2) {
					log.error("关闭preparedStatement失败", e2);
				}

			}
		}
		return keyids;
	}
	public static String[] getPrimary(Connection conn, String db, String tb) {
		return getPrimary(conn,db,tb,DbType.mysql);
	}

	/***
	 * retAry[0] 列名 retAry[1] 列类型
	 * 
	 * @param conn
	 * @param db
	 * @param tb
	 * @param isRds
	 * @return
	 */
	public static String[][] getCols(Connection conn, String db, String tb, YesOrNo isRds) {
		try {
			PreparedStatement prepCols = conn.prepareStatement(
					"select   column_name,data_type   from  information_schema.columns  where  table_schema=? and table_name=? order by ORDINAL_POSITION");
			List<String> ret = new ArrayList<>();
			List<String> retType = new ArrayList<>();
			JdbcAssit.setPreParam(prepCols, null, true, db, tb);
			ResultSet rs = prepCols.executeQuery();
			while (rs.next()) {
				ret.add(rs.getString(1));
				retType.add(rs.getString(2));
			}
			rs.close();
			if (CollectionUtils.isEmpty(ret)) {// 库名或表名错误
				throw new RuntimeException("请确认数据为库存在库名：[" + db + "],表名：[" + tb + "]");
			}
			if (YesOrNo.yes == isRds && ArrayUtils.isEmpty(getPrimary(conn, db, tb))) {// 是rds
				ret.add("_rowkey_");
				retType.add("varchar");
			}
			String[][] retAry = new String[2][ret.size()];
			retAry[0] = ret.toArray(new String[ret.size()]);
			retAry[1] = retType.toArray(new String[retType.size()]);
			return retAry;
		} catch (Exception e) {
			log.error("获取cols错误", e);
			throw new RuntimeException("获取cols错误");
		}
	}

	// 把列类型转为BinlogType类型
	public static BinlogType[] convertColumnType(String[] cosTypes) {
		BinlogType[] rettypes = new BinlogType[cosTypes.length];
		for (int i = 0; i < rettypes.length; i++) {
			rettypes[i] = BinlogType.getByName(cosTypes[i]);
		}
		return rettypes;
	}


	/***
	 * 跟据db/tb和需要的列得到名字和类型（要包含主键）
	 * @param connection
	 * @param db
	 * @param tb
	 * @param needcols 需要的列，为空表示全部列
	 * @param routeCol 路由列，必须加入的导出列
	 * @param primarys 主键
	 * @return
	 */
	public static Pair<String[], BinlogType[]> needColAndType(Connection connection, String db, String tb,
			String[] needcols, String routeCol, String[] primarys) {
		String[][] cols = MySqlAssit.getCols(connection, db, tb, YesOrNo.yes);
		if (StringUtil.isNull(needcols)) {
			return Pair.of(cols[0], convertColumnType(cols[1]));
		} else {
			String[] retEles = CollectionUtil.arrayAnd(String[].class, cols[0], needcols);// cols[0]与configCols不能调换顺序
			retEles = CollectionUtil.arrayMerge(String[].class, retEles, primarys, true);// 加主键
			if (StringUtil.isNotNull(routeCol)) {
				retEles = CollectionUtil.arrayMerge(String[].class, retEles, new String[] { routeCol }, true);// 加路由键
			}
			String[] retTypes = new String[retEles.length];
			for (int k = 0; k < retEles.length; k++) {
				int index = ArrayUtils.indexOf(cols[0], retEles[k]);
				retTypes[k] = cols[1][index];
			}
			return Pair.of(retEles, convertColumnType(retTypes));
		}
	}


//	public static List<Triple<String, String, String>> getColsNew(Connection conn, String db, String tb,
//			YesOrNo isRds) {
//		try {
//			PreparedStatement prepCols = conn.prepareStatement(
//					"select   column_name,data_type,column_type   from  information_schema.columns  where  table_schema=? and table_name=? order by ORDINAL_POSITION");
//			List<Triple<String, String, String>> retlist = new ArrayList<Triple<String, String, String>>();
//			JdbcAssit.setPreParam(prepCols, null, true, db, tb);
//			ResultSet rs = prepCols.executeQuery();
//			while (rs.next()) {
//				Triple<String, String, String> data = Triple.of(rs.getString(1), rs.getString(2), rs.getString(3));
//				retlist.add(data);
//			}
//			rs.close();
//			if (CollectionUtils.isEmpty(retlist)) {// 库名或表名错误
//				throw new RuntimeException("请确认数据为库存在库名：[" + db + "],表名：[" + tb + "]");
//			}
//			if (YesOrNo.yes == isRds && ArrayUtils.isEmpty(getPrimary(conn, db, tb))) {// 是rds
//				Triple<String, String, String> data = Triple.of("_rowkey_", "varchar", "varchar(255)");
//				retlist.add(data);
//			}
//			return retlist;
//		} catch (Exception e) {
//			log.error("获取cols错误", e);
//			throw new RuntimeException("获取cols错误");
//		}
//	}

	public static List<MySqlColBean> getColsBean(Connection conn, String db) {
		return getColsBean(conn, db, null);
	}

	public static List<MySqlColBean> getColsBeanCommon(Connection conn, String db, String tb, DbType dbType) {
		try {
			Validate.notBlank(db, "需要传入指定的库名。");
			boolean hastb = StringUtil.isNotNull(tb);
			PreparedStatement prepCols = conn.prepareStatement(
					"select   table_schema,table_name,column_name,data_type,column_type,column_comment,numeric_precision,numeric_scale,character_maximum_length,column_key,is_nullable   from  information_schema.columns  where  1=1  and table_schema=?"
							+ (hastb ? " and table_name=?" : "") + " order by ORDINAL_POSITION");
			List<MySqlColBean> retlist = new ArrayList<MySqlColBean>();
			if (hastb) {
				JdbcAssit.setPreParam(prepCols, null, true, db, tb);
			} else {
				JdbcAssit.setPreParam(prepCols, true, db);
			}
			ResultSet rs = prepCols.executeQuery();

			String[] keyids = null;
			if (dbType == DbType.doris) {// doris通过建表语名拿到主键
				keyids=getPrimary(conn, db, tb, dbType);
			}
			while (rs.next()) {
				MySqlColBean tempobj = new MySqlColBean();
				tempobj.setDb(rs.getString("table_schema"));
				tempobj.setTb(rs.getString("table_name"));
				tempobj.setColumnComment(rs.getString("column_comment"));
				tempobj.setColumnName(rs.getString("column_name"));
				tempobj.setColumnType(rs.getString("column_type"));
				tempobj.setDataType(rs.getString("data_type"));
				tempobj.setNumericScale(rs.getInt("numeric_scale"));
				tempobj.setNotNull("NO".equalsIgnoreCase(rs.getString("is_nullable")) ? true : false);
				long maxlength = StringUtil.isNull(rs.getString("numeric_precision"))
						? rs.getLong("character_maximum_length")
						: rs.getLong("numeric_precision");
				tempobj.setMaxlength(maxlength);
				if (dbType == DbType.mysql) {
					tempobj.setPri("PRI".equals(rs.getString("column_key")));
				} else if (dbType == DbType.doris) {
					boolean iskey = ArrayUtils.contains(keyids, rs.getString("column_name")) ? true : false;
					tempobj.setPri(iskey);
				}

				retlist.add(tempobj);
			}
			rs.close();
			if (CollectionUtils.isEmpty(retlist)) {// 库名或表名错误
				throw new RuntimeException("请确认数据为库存在库名：[" + db + "],表名：[" + tb + "]");
			}
			return retlist;
		} catch (Exception e) {
			log.error("获取cols错误", e);
			throw new RuntimeException("获取cols错误");
		}
	}

	private static String[] getkey(String createSql, int indexBegin, String keytype) {
		int indexEnd = createSql.indexOf(")", indexBegin);
		String keysStr = createSql.substring(indexBegin + keytype.length() + 1, indexEnd);
		keysStr=keysStr.replace(" ", "");//20221124 去除空格，防止bug
		keysStr = keysStr.replace("`", "");
		return keysStr.split(",");
	}

	public static List<MySqlColBean> getColsBean(Connection conn, String db, String tb) {
		return getColsBeanCommon(conn, db, tb, DbType.mysql);
	}

	public static List<MySqlColBean> getColsBeanByDoris(Connection conn, String db, String tb) {
		return getColsBeanCommon(conn, db, tb, DbType.doris);
	}

	/***
	 * 转换列
	 * 
	 * @param oriList
	 * @param fieldFormart
	 * @param addColNameType
	 * @return
	 */
	public static List<MySqlColBean> convertCol(List<MySqlColBean> oriList, FieldFormart fieldFormart,
			AddColNameType addColNameType) {
		List<MySqlColBean> retlis = new ArrayList<MySqlColBean>();
		for (MySqlColBean ori : oriList) {
			MySqlColBean mySqlColBean = null;
			try {
				mySqlColBean = (MySqlColBean) ori.clone();
			} catch (CloneNotSupportedException e) {
			}
			mySqlColBean.setColumnName(ori.getColumnNameStandard(fieldFormart));
			retlis.add(mySqlColBean);
		}
		if (addColNameType != AddColNameType.no) {
			for (AddColName addColName : AddColName.values()) {
				if ((addColNameType != AddColNameType.selective && addColNameType != AddColNameType.selective_ori)
						|| addColName.isSetValue()) {
					MySqlColBean mySqlColBean = new MySqlColBean();
					mySqlColBean.setColumnComment(addColName.getDesc());
					mySqlColBean.setDataType(addColName.getBinlogType().name());
					mySqlColBean.setColumnName(addColName.getColNameTrue(fieldFormart));
					retlis.add(mySqlColBean);
				}
			}
		}
		return retlis;
	}

	/***
	 * 通过模式得到所有列
	 * 
	 * @param conn
	 * @param dbPattern
	 * @param tbPattern
	 * @return
	 */
	public static List<MySqlColBean> getColsBeanByPattern(Connection conn, String dbPattern, String tbPattern) {
		List<String> allDbs = getAllDbs(conn);
		String dbPatternTemp = StringUtil.isNull(dbPattern) ? "*" : dbPattern;
		final String dbPatternTrue = "^" + dbPatternTemp + "$";

		String tbPatternTemp = StringUtil.isNull(tbPattern) ? "*" : tbPattern;
		final String tbPatternTrue = "^" + tbPatternTemp + "$";

		CollectionUtils.filter(allDbs, new Predicate() {
			@Override
			public boolean evaluate(Object object) {
				if (StringUtil.isNull(dbPattern)) {// 所有库
					return true;
				}
				String dbname = String.valueOf(object);
				boolean retdb = StrPattern.checkStrFormat(dbPatternTrue, dbname);
				return retdb;
			}
		});
		List<MySqlColBean> retlist = new ArrayList<MySqlColBean>();
		for (String db : allDbs) {
			List<MySqlColBean> tbBeans = getColsBean(conn, db);
			CollectionUtils.filter(tbBeans, new Predicate() {
				@Override
				public boolean evaluate(Object object) {
					if (StringUtil.isNull(tbPattern)) {// 所有表
						return true;
					}
					MySqlColBean mySqlColBean = (MySqlColBean) object;
					boolean retdb = StrPattern.checkStrFormat(tbPatternTrue, mySqlColBean.getTb());
					return retdb;
				}
			});
			retlist.addAll(tbBeans);
		}

		return retlist;
	}

	/***
	 * 组装Sql语句
	 * 
	 * @param cols             mysql的列
	 * @param toDbType         目标数据库类型
	 * @param createDbCallBack 创表语句其它配置
	 * @param addCols          附加字段
	 * @param toLowerCase      是否全部改写为小写
	 * @param needTenantId     是否要支持tenantId字段
	 * @return
	 */
	public static String getTableCreateSQL(List<MySqlColBean> cols, DbType toDbType, ICreateDbCallBack createDbCallBack,
			List<MySqlColBean> addCols, boolean toLowerCase, boolean needTenantId) {
		List<String> sqlList = getTableCreateSQLList(cols, toDbType, createDbCallBack, FieldFormart.dbstandard, addCols,
				toLowerCase, needTenantId, false);
		StringBuffer allSqlBuf = new StringBuffer();
		for (String sql : sqlList) {
			allSqlBuf.append(sql);
		}
		return allSqlBuf.toString();
	}

	private static String[] tenantIdColname = new String[] { "tenantid", "tenant_id" };
	private static String[] bigColType = new String[] { "longblob", "json", "longtext", "mediumtext", "text", "blob" };

//长字符串，jdbc驱动不支持，会截断执行
	public static List<String> getTableCreateSQLList(List<MySqlColBean> cols, DbType toDbType,
			ICreateDbCallBack createDbCallBack, FieldFormart fieldFormart, List<MySqlColBean> addCols,
			boolean toLowerCase, boolean needTenantId, boolean tenantidIsnotnull) {
		// 先分组
		Map<Pair<String, String>, List<MySqlColBean>> colsBytable = splitGroupByDbTb(cols, fieldFormart);
		List<String> retlist = new ArrayList<String>();

		for (Pair<String, String> key : colsBytable.keySet()) {
			StringBuffer allSqlBuf = new StringBuffer();
			List<MySqlColBean> colObjs = colsBytable.get(key);
			Collections.sort(colObjs);
			// 库名和表名
			String dbtbname = String.format("%s.%s", key.getLeft(), key.getRight());
			if (createDbCallBack != null) {
				dbtbname = createDbCallBack.getDbAndTb(key);
			}
			allSqlBuf.append(new StringBuffer("create table IF NOT EXISTS " + dbtbname + "("));
			List<String> prikeylist = new ArrayList<String>();
			List<String> colnamelist = new ArrayList<String>();
			// doris分统字段必须放最前面,tenantid也需要放前面
			List<MySqlColBean> keyCols = new ArrayList<MySqlColBean>();
			List<MySqlColBean> nokeyCols = new ArrayList<MySqlColBean>();
			MySqlColBean tenantidCol = null;
			List<MySqlColBean> bigColNames = new ArrayList<MySqlColBean>();
			for (MySqlColBean colObj : colObjs) {
				if (needTenantId && ArrayUtils.contains(tenantIdColname, colObj.getColumnNameStandard(fieldFormart))) {
					tenantidCol = colObj;
				} else {
					if (colObj.isPri()) {
						keyCols.add(colObj);
					} else {
						nokeyCols.add(colObj);
					}
				}
				if (ArrayUtils.contains(bigColType, colObj.getDataType())) {
					bigColNames.add(colObj);
				}
				colnamelist.add(colObj.getColumnNameStandard(fieldFormart));
			}

			// tenantid
			if (tenantidCol != null) {
				allSqlBuf.append(
						String.format("`%s` %s " + (tenantidIsnotnull ? "not null" : "") + " comment \"_key__\",",
								tenantidCol.getColumnNameStandard(fieldFormart),
								createDbCallBack == null ? tenantidCol.getColumnType()
										: createDbCallBack.getColumnType(tenantidCol, bigColNames)));
			}
			// 主键
			for (MySqlColBean colObj : keyCols) {
				// 主键必须加not null,因为doris元数据跟据它来判断主键，不加程序不能判断主键来去重（一段时间内同一条数据只能记一次）
				allSqlBuf.append(String.format("`%s` %s not null comment \"_key__\",",
						colObj.getColumnNameStandard(fieldFormart), createDbCallBack == null ? colObj.getColumnType()
								: createDbCallBack.getColumnType(colObj, bigColNames)));
				prikeylist.add(colObj.getColumnNameStandard(fieldFormart));

			}
			// 非主键
			for (MySqlColBean colObj : nokeyCols) {
				allSqlBuf.append(String.format("`%s` %s REPLACE NULL comment \"%s\",",
						colObj.getColumnNameStandard(fieldFormart),
						createDbCallBack == null ? colObj.getColumnType()
								: createDbCallBack.getColumnType(colObj, bigColNames),
						StringUtil.hasNull(colObj.getColumnComment())));
			}
			// 处理附加的列
			if (CollectionUtils.isNotEmpty(addCols)) {
				for (MySqlColBean addCol : addCols) {
					if (!colnamelist.contains(addCol.getColumnNameStandard(fieldFormart))) {
						colnamelist.add(addCol.getColumnNameStandard(fieldFormart));
						allSqlBuf.append(
								String.format("`%s` %s comment \"%s\",", addCol.getColumnNameStandard(fieldFormart),
										addCol.getColumnType(), StringUtil.hasNull(addCol.getColumnComment())));
					}
				}
			}
			// hack :tenantid要做分桶（分区）时需要出现在主键列中
			List<String> prikeylistTrue = new ArrayList<String>();
			prikeylistTrue.addAll(prikeylist);
			if (tenantidCol != null && !prikeylist.contains(tenantidCol.getColumnNameStandard(fieldFormart))) {
				prikeylistTrue.add(tenantidCol.getColumnNameStandard(fieldFormart));
			}

			String createKeySql = toDbType.getCreateKeySql(prikeylistTrue);
			if (StringUtil.isNull(createKeySql)) {
				allSqlBuf.deleteCharAt(allSqlBuf.length() - 1);
			} else {
				allSqlBuf.append(createKeySql);
			}
			allSqlBuf.append(")");
			if (createDbCallBack != null) {
				allSqlBuf.append(createDbCallBack.getOtherInfo(colnamelist, prikeylist, tenantidCol));
			}
			allSqlBuf.append(";");
			retlist.add(allSqlBuf.toString());
		}
		return retlist;
	}

	public static Map<Pair<String, String>, List<MySqlColBean>> splitGroupByDbTb(List<MySqlColBean> cols,
			FieldFormart fieldFormart) {
		Map<Pair<String, String>, List<MySqlColBean>> colsBytable = new HashMap<Pair<String, String>, List<MySqlColBean>>();

		for (MySqlColBean colEle : cols) {
			Pair<String, String> key = Pair.of(colEle.getDb(), colEle.getTb());
			List<MySqlColBean> temp = colsBytable.containsKey(key) ? colsBytable.get(key)
					: new ArrayList<MySqlColBean>();
			colEle.setColumnName(colEle.getColumnNameStandard(fieldFormart));
			temp.add(colEle);
			colsBytable.put(key, temp);
		}
		return colsBytable;
	}

	/**
	 * 新的得到列信息包含comment
	 * 
	 * @param conn
	 * @param db
	 * @param tb
	 * @param isRds
	 * @return L:列名, M：类型(varchar), R：Comment
	 */
	public static List<Triple<String, String, String>> getColsComment(Connection conn, String db, String tb,
			YesOrNo isRds) {
		try {
			PreparedStatement prepCols = conn.prepareStatement(
					"select   column_name,data_type,column_comment from  information_schema.columns  where  table_schema=? and table_name=? order by ORDINAL_POSITION");
			List<Triple<String, String, String>> retlist = new ArrayList<Triple<String, String, String>>();
			JdbcAssit.setPreParam(prepCols, null, true, db, tb);
			ResultSet rs = prepCols.executeQuery();
			while (rs.next()) {
				Triple<String, String, String> data = Triple.of(rs.getString(1), rs.getString(2), rs.getString(3));
				retlist.add(data);
			}
			rs.close();
			if (CollectionUtils.isEmpty(retlist)) {// 库名或表名错误
				throw new RuntimeException("请确认数据为库存在库名：[" + db + "],表名：[" + tb + "]");
			}
			if (YesOrNo.yes == isRds && ArrayUtils.isEmpty(getPrimary(conn, db, tb))) {// 是rds
				Triple<String, String, String> data = Triple.of("_rowkey_", "varchar", "varchar(255)");
				retlist.add(data);
			}
			return retlist;
		} catch (Exception e) {
			log.error("获取cols错误", e);
			throw new RuntimeException("获取cols错误");
		}
	}

	// 设置local_infile变量
	public static Result setLocalInfile(Connection conn) {
		Statement statement = null;
		try {
			statement = conn.createStatement();
			statement.execute("set global local_infile=on");
			return Result.getSuc();
		} catch (Exception e) {
			return Result.getError(e.getMessage());
		} finally {
			try {
				statement.close();
			} catch (SQLException e) {
				log.error("close statement error", e);
			}
		}
	}

}
