/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common-apiext
 * 项目描述：工具
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.apiext.jdbc;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.sql.Array;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.ArrayUtils;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.constant.dbType.BinlogType;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

@Slf4j
public abstract class JdbcAssit {

	public static ResultSet querySql(Connection conn, String sql) {
		Statement stmt = null;
		try {
			if (conn == null || conn.isClosed()) {
				return null;
			}
			stmt = conn.createStatement();
			ResultSet rs = stmt.executeQuery(sql);
			return rs;
		} catch (SQLException e) {
			String errormsg = String.format("查询sql出错，sql:[%s]", sql);
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_sql_package,errormsg,e);
		}
	}

	public static Result execSql(Connection conn, String sql) {
		Statement stmt = null;
		try {
			if (conn == null || conn.isClosed()) {
				return Result.getError("拿不到连接");
			}
			stmt = conn.createStatement();
			stmt.execute(sql);
			return Result.getSuc();
		} catch (SQLException e) {
			String format = String.format("查询sql出错，sql:[%s]", sql);
			log.error(format, e);
			return Result.getError(format);
		}
	}

	/***
	 * 设置PreparedStatement的参数
	 * 
	 * @param stmt        要设置参数的PreparedStatement
	 * @param queryParams 要设置的参数，与stmt出现的问题要对应的上
	 * @throws SQLException sql异常
	 */
	public static void setPreParam(PreparedStatement stmt, BinlogType[] typeAry, boolean isQuery, Object... queryParams)
			throws SQLException {
		if (stmt == null) {
			return;
		}
		int needparamscount = stmt.getParameterMetaData().getParameterCount();
		if (needparamscount == 0 && ArrayUtils.isEmpty(queryParams)) {
			return;
		}
		if (needparamscount != queryParams.length) {
			throw new SQLException(
					String.format("参数的个数不匹配，需要[%s]个参数，但传进来[%s]个参数", needparamscount, queryParams.length));
		}
		for (int i = 0; i < queryParams.length; i++) {
			Object queryParam = queryParams[i];
			if (queryParam == null) {
				if (ArrayUtils.isEmpty(typeAry) || typeAry.length < i + 1) {
					throw new SQLException(String.format("第%s个参数不合法，没有传入typeAry却存在null值", i));
				}
				stmt.setNull(i + 1, typeAry[i].getTypes());
			} else if (queryParam instanceof String) {
				String valueStr = String.valueOf(queryParam);
				if (isQuery) {
					// 查询如何使用Bytes会查不出结果 。
					stmt.setString(i + 1, valueStr);
				} else {
					// "ߐ±暖暖猫"等表情符需要这么处理
					stmt.setBytes(i + 1, valueStr.getBytes());
				}
			} else if (queryParam instanceof Integer) {
				stmt.setInt(i + 1, (Integer) queryParam);
			} else if (queryParam instanceof Timestamp) {// Timestamp是Date的子类，需要出现在前面
				stmt.setTimestamp(i + 1, (Timestamp) queryParam);
			} else if (queryParam instanceof Date) {
				// 20220121 使用Date会导致丢失“时间”,后来在BinlogType.getValue里修改
				// stmt.setTimestamp(i + 1, new java.sql.Timestamp(((Date)
				// queryParam).getTime()));
				stmt.setDate(i + 1, new java.sql.Date(((Date) queryParam).getTime()));
			} else if (queryParam instanceof Long) {
				stmt.setLong(i + 1, (Long) queryParam);
			} else if (queryParam instanceof Double) {
				stmt.setDouble(i + 1, (Double) queryParam);
			} else if (queryParam instanceof Boolean) {
				stmt.setBoolean(i + 1, (Boolean) queryParam);
			} else if (queryParam instanceof byte[]) {
				stmt.setBlob(i + 1, StringUtil.convertInputStream((byte[]) queryParam));
			} else if (queryParam instanceof BigDecimal) {
				stmt.setBigDecimal(i + 1, (BigDecimal) queryParam);
				// stmt.setBlob(i + 1, StringUtil.convertInputStream((byte[]) queryParam));
			} else if (queryParam instanceof Array) {// mysql不支持，oracle和postgreSQL支持
				stmt.setArray(i + 1, (Array) queryParam);
			}
//			else if (queryParam.getClass().isArray()) {// 由于参数的原因，只能传二组数组类型
//				Class<?> componentType = queryParam.getClass().getComponentType();
//				Array arrayParams = null;
//				if (componentType.getName().equals("java.lang.String")) {
//					arrayParams = stmt.getConnection().createArrayOf("varchar", (String[]) queryParam);
//				} else if (componentType.getName().equals("java.lang.Long")) {// "long"
//					arrayParams = stmt.getConnection().createArrayOf("varchar", (Long[]) queryParam);
//				} else {
//					String valueStr = String.valueOf(queryParam);
//					throw new SQLException(String.format("不支持的数组类型,参数值[%s]", valueStr));
//				}
//				stmt.setArray(i + 1, arrayParams);
//			}
//			else if(Collection.class.isAssignableFrom(queryParam.getClass())) {
//				ParameterizedType parameterizedType = (ParameterizedType)  queryParam.getClass().getGenericSuperclass();
//		        // 获取具体的泛型类型 getActualTypeArguments获取具体的泛型的类型
//		        // 这个方法会返回一个Type的数组
//		        Type[] types = parameterizedType.getActualTypeArguments();
//		        // 获取具体的泛型的类型·
//				System.out.println(types[0]);
//			} 
			else {
				String valueStr = String.valueOf(queryParam);
				throw new SQLException(String.format("不支持的类型,参数值[%s]", valueStr));
			}
		}

	}

	public static void setPreParam(PreparedStatement stmt, BinlogType[] typeAry, boolean isQuery, Object queryParams)
			throws SQLException {
		setPreParam(stmt, null, isQuery, queryParams);
	}

	public static void setPreParam(PreparedStatement stmt, boolean isQuery, Object... queryParams) throws SQLException {
		setPreParam(stmt, null, isQuery, queryParams);
	}

	public static List<Map<String, String>> querySqlMap(Connection conn, String sql, boolean isConvertKey) {
		ResultSet rs = querySql(conn, sql);
		try {
			return rsToMap(rs, isConvertKey);
		} finally {
			try {
				if (rs != null) {
					rs.close();
				}
			} catch (Exception e) {
			}
		}
	}

	/***
	 * 
	 * @param conn
	 * @param sql
	 * @param isConvertKey 是否转为驮峰
	 * @param queryParams  查询参数
	 * @return
	 */
	public static List<Map<String, String>> querySqlMapPre(Connection conn, String sql, boolean isConvertKey,
			Object... queryParams) {
		ResultSet rs = null;
		try {
			PreparedStatement stmt = conn.prepareStatement(sql);
			setPreParam(stmt, true, queryParams);
			rs = stmt.executeQuery();
			return rsToMap(rs, isConvertKey);
		} catch (Exception e) {
			throw new ProjectExceptionRuntime(ExceptAll.Project_default, "查询sql出错,sql:" + sql);
		} finally {
			try {
				if (rs != null) {
					rs.close();
				}
			} catch (Exception e) {
			}
		}
	}

	public static void execSqlPre(Connection conn, String sql, Object... queryParams) throws SQLException {
		PreparedStatement stmt = conn.prepareStatement(sql);
		String selectStr = StringUtil.trimSpace(sql).substring(0, 6);
		setPreParam(stmt, selectStr.equalsIgnoreCase("select"), queryParams);
		stmt.execute();
	}

	public static List<Map<String, String>> rsToMap(ResultSet rs, boolean isConvertKey) {
		List<Map<String, String>> retlist = new ArrayList<>();
		try {
			int nums = rs.getMetaData().getColumnCount();
			while (rs.next()) {
				Map<String, String> retmap = new HashMap<>();
				for (int i = 0; i < nums; i++) {
					String colName = rs.getMetaData().getColumnLabel(i + 1);
					String value = rs.getString(colName);
					if (isConvertKey) {
						retmap.put(StringUtil.convertStr(colName), value);
					} else {
						retmap.put(colName, value);
					}
				}
				retlist.add(retmap);
			}
			return retlist;
		} catch (Exception e) {
			log.error("转换ResultSet出错", e);
			return null;
		}
	}

	public static void exeSQLScript(Connection connection, String filePath, Class<?> classz, boolean autoCommit,
			boolean stopOnError) {
		InputStream inputStream = IOUtil.fileToInputStream(filePath, classz);
		exeSQLScript(connection, inputStream, autoCommit, stopOnError);
	}

	/***
	 * 批量执行SQL脚本
	 * 
	 * @param connection
	 * @param input
	 * @param autoCommit
	 */
	public static void exeSQLScript(Connection connection, InputStream input, boolean autoCommit, boolean stopOnError) {
		log.info("=============batch exec sql begin=============================================");
		// 遇到错误是否停止执行
		// boolean stopOnError = autoCommit ? false : true;// 非自动提交时错误退出。
		ScriptRunner runner = new ScriptRunner(connection);
		runner.setAutoCommit(autoCommit);
		runner.setStopOnError(stopOnError);
		try {
//			runner.setLogWriter(log.getLogWiter());
//			runner.setErrorLogWriter(log.getLogWiter());
			// 执行脚本
			runner.runScript(new InputStreamReader(input));
		} catch (Exception e) {
			log.error("批量执行SQL失败", e);
			if (!autoCommit) {
				try {
					connection.rollback();
				} catch (SQLException e1) {
					log.error("批量执行SQL回滚失败", e1);
				}
			}
		}
		if (!autoCommit) {
			try {
				connection.commit();
			} catch (SQLException e1) {
				log.error("批量执行SQL提交事务失败", e1);
			}
		}
		log.info("============batch exec sql end==============================================");
	}
}
